# mutexm_test.rb - mutexm unit test
#
# $Date: 2001/06/05 17:33:55 $
# Copyright (c) 2000 Masatoshi SEKI
#
# mutexm_test.rb is copyrighted free software by Masatoshi SEKI.
# You can redistribute it and/or modify it under the same term as Ruby.

require 'runit/testcase'
require 'runit/cui/testrunner'
require 'mutexm'

class Balance
  include MutexM
  def initialize(f,b)
    @foo = f
    @bar = b
#    super
  end
  synchronize_reader(:foo, :bar)
  
  def move(d)
    @foo = @foo + d
    Thread.pass		# dummy
    @bar = @bar - d
    [@foo, @bar]
  end
  synchronized :move
end


class Foo
  include MutexM
  
  def initialize
    super
    @foo = nil
    @bar = nil
    @count = 0
  end
  synchronize_accessor(:foo)
  lock_accessor("bar")
  
  def commit
    in_synchronize
    [@foo, @bar]
  end
  
  def sleep_and_set
    sleep 0.5
    @foo = @foo + @foo
    sleep 0.5
    @bar = @bar + @foo
    [@foo, @bar]
  end
  synchronized :sleep_and_set
end

class Unsafe
  def initialize
    super
    @key = []
    @value = []
  end

  def get(k)
    idx = @key.index(k)
    Thread.pass
    idx ? @value[idx] : nil
  end

  def set(k, v)
    idx = @key.index(k)
    Thread.pass
    if v.nil? 
      if idx
	@key.delete_at(idx)
	@value.delete_at(idx)
      end
    else
      if idx
	@value[idx] = v
      else
	@key.unshift(k)
	@value.unshift(v)
      end
    end
  end

  def each
    sz = @key.size
    if sz > 0
      for i in 0...sz
	yield(@key[i], @value[i])
      end
    end
  end

  def [](k)
    get(k)
  end

  def []=(k, v)
    set(k, v)
  end
end

class Safe < Unsafe
  include MutexM

  synchronized(:get, :set, :each)
end

class MutexMTest < RUNIT::TestCase
  def balance_thread(b, n)
    Thread.new do
      for i in 1..10
	pair = b.move(i)
	assert_equal(pair[0]+pair[1], n)
	pair = b.move(-i)
	assert_equal(pair[0]+pair[1], n)
      end
    end
  end

  def test_00_balance
    b = Balance.new(10, 0)
    pair = b.move(5)
    assert_equal(pair[0]+pair[1], 10)

    th = []
    for i in 1..10
      th.push balance_thread(b, 10)
    end
    
    th.each do |t| t.join end

    b.synchronize do
      assert_equal(b.foo + b.bar, 10)
    end
  end

  def test_01_foo
    f = Foo.new
    f.foo = 1
    assert_equal(f.foo, 1)
    assert_exception(ThreadError) do 
      f.bar = 1
    end
    assert_exception(ThreadError) do 
      a = f.bar
    end
    f.lock
    f.bar = 2
    assert_equal(f.bar, 2)
    f.unlock
    f.synchronize do
      f.bar = 3
      assert_equal(f.bar, 3)
    end
    
    th = Thread.new do 
      a = f.sleep_and_set
      assert_equal(a, [2, 5])
    end
    sleep 0.1
    f.foo = 'hello'
    assert_equal(f.foo, 'hello')

    th.join
  end

  def unsafe_thread(it, n)
    Thread.new do
      while n > 0
	it[n] = n.to_s
	assert_equal(it[n], n.to_s)
	n = n - 1
      end
      sum = 0
      it.each do |k, v|
	sum += k
      end
      sum
    end
  end

  def test_02_unsafe
    th = []
#   it = Unsafe.new
    it = Safe.new
    for i in 1..10
      th.push unsafe_thread(it, 10)
    end
    
    th.each do |t| t.join end
  end

  def extend_thread(it, v)
    Thread.new do 
      it.synchronize do
	for i in 0 ... 10
	  Thread.pass
	  it[i] = v
	end
      end
    end
  end

  def test_03_extend
    it = Hash.new
    it.extend MutexM
    
    th = []
    th.push extend_thread(it, 0)
    th.push extend_thread(it, 1)
    th.each do |t| t.join end
    
    assert_equal(it[0], it[9])
  end
  
  def test_04_kill
    it = Hash.new
    it.extend MutexM

    th = nil

    it.synchronize do
      th = Thread.new do 
	it.synchronize do
	  assert(false)
	end
      end
      th.run
      sleep 1
      Thread.kill th
      th = Thread.new do 
	it.synchronize do
	  it[0] = 2
	  assert_equal(it[0], 2)
	end
      end
    end
    
    th.join
  end

  def test_05_unlock
    it = Hash.new
    it.extend MutexM

    th = nil

    it.lock
    it.in_synchronize

    th = Thread.new do 
      it.unlock		# no effect ...
    end
    th.join

    it.in_synchronize
    it.unlock
    assert_exception(ThreadError) do
      it.in_synchronize
    end
  end
end

if __FILE__ == $0
  RUNIT::CUI::TestRunner.run(MutexMTest.suite)
end

