/***************************************************************************
 *   Copyright (C) 2006 by Peter Penz (peter.penz@gmx.at) and              *
 *   and Patrice Tremblay                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "statusbarspaceinfo.h"

#include <qpainter.h>
#include <qtimer.h>
#include <kglobalsettings.h>
#include <kdiskfreesp.h>
#include <klocale.h>
#include <kio/job.h>

StatusBarSpaceInfo::StatusBarSpaceInfo(QWidget* parent) :
    QWidget(parent),
    m_gettingSize(false),
    m_kBSize(0),
    m_kBAvailable(0)
{
    setMinimumWidth(200);

    // Update the space information each 10 seconds. Polling is useful
    // here, as files can be deleted/added outside the scope of Dolphin.
    QTimer* timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(refresh()));
    timer->start(10000);
}

StatusBarSpaceInfo::~StatusBarSpaceInfo()
{
}

void StatusBarSpaceInfo::setURL(const KURL& url)
{
    m_url = url;
    refresh();
    update();
}

void StatusBarSpaceInfo::paintEvent(QPaintEvent* /* event */)
{
    QPainter painter(this);
    const int barWidth = width();
    const int barTop = 2;
    const int barHeight = height() - 4;

    QString text;

    const QColor c1 = colorGroup().background();
    const QColor c2 = KGlobalSettings::buttonTextColor();
    const QColor frameColor((c1.red()   + c2.red())   / 2,
                            (c1.green() + c2.green()) / 2,
                            (c1.blue()  + c2.blue())  / 2);
    painter.setPen(frameColor);
    painter.setBrush(KGlobalSettings::baseColor());
    painter.drawRect(QRect(0, barTop + 1 , barWidth - 3, barHeight));

    if ((m_kBSize > 0) && (m_kBAvailable > 0)) {
        // draw 'used size' bar
        painter.setPen(Qt::NoPen);
        painter.setBrush(KGlobalSettings::buttonBackground());
        int usedWidth = barWidth - static_cast<int>((m_kBAvailable *
                                                    static_cast<float>(barWidth)) / m_kBSize);
        painter.drawRect(QRect(1, barTop + 2, usedWidth - 4, barHeight - 2));

        text = i18n("%1% of %2 used")
               .arg( 100 - (int)(100.0 * m_kBAvailable / m_kBSize))
               .arg(KIO::convertSizeFromKB(m_kBSize));
    }
    else {
        text = m_gettingSize ? i18n("Getting size...") : i18n("Unknown size");
    }

    // draw text (usually 'X% of Y GB used')
    painter.setPen(KGlobalSettings::textColor());
    painter.drawText(QRect(1, 1, barWidth - 2, barHeight + 4),
                     Qt::AlignHCenter | Qt::AlignVCenter | Qt::WordBreak,
                     text);
}


void StatusBarSpaceInfo::slotFoundMountPoint(const unsigned long& kBSize,
                                             const unsigned long& /* kBUsed */,
                                             const unsigned long& kBAvailable,
                                             const QString& /* mountPoint */)
{
    m_gettingSize = false;
    m_kBSize = kBSize;
    m_kBAvailable = kBAvailable;

    // Bypass a the issue (?) of KDiskFreeSp that for protocols like
    // FTP, SMB the size of root partition is returned.
    // TODO: check whether KDiskFreeSp is buggy or Dolphin uses it in a wrong way
    const QString protocol(m_url.protocol());
    if (!protocol.isEmpty() && (protocol != "file")) {
        m_kBSize = 0;
        m_kBAvailable = 0;
    }

    update();
}

void StatusBarSpaceInfo::slotDone()
{
    m_gettingSize = false;
    update();
}

void StatusBarSpaceInfo::refresh()
{
    m_gettingSize = true;
    m_kBSize = 0;
    m_kBAvailable = 0;

    const QString mountPoint(KIO::findPathMountPoint(m_url.path()));

    KDiskFreeSp* job = new KDiskFreeSp(this);
    connect(job, SIGNAL(foundMountPoint(const unsigned long&,
                                        const unsigned long&,
                                        const unsigned long&,
                                        const QString& )),
            this, SLOT(slotFoundMountPoint(const unsigned long&,
                                           const unsigned long&,
                                           const unsigned long&,
                                           const QString& )));
    connect(job, SIGNAL(done()),
            this, SLOT(slotDone()));

    job->readDF(mountPoint);
}

#include "statusbarspaceinfo.moc"
