"""
Various strategies for mesh marking

Options:

- fixed_fraction    : Refine a fraction N of cells
- equidistribution  : Refine cells that contribute more than averaged tolerance
- maximal           : Refine cells greater than fraction of max error
- dorfler           : Refine cells corresponding to fraction of total error

"""

__author__ = "Marie E. Rognes (meg@simula.no)"
__copyright__ = "Copyright (C) 2009 - Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# Modified by Anders Logg, 2010.

# Last changed: 2010-05-09

from dolfin import info, warning
from dolfin.cpp import MeshFunction, File, cells

import numpy

def mark_cells(mesh, error_indicators, parameters):
    """
    Construct markers based on strategy given by parameters['strategy']
    """

    # Get markers depending on strategy
    markers = None
    strategy = parameters["strategy"]
    if strategy == "fixed_fraction":
        fraction = parameters["fraction"]
        markers = fixed_fraction(mesh, error_indicators, fraction)
    elif strategy == "equidistribution":
        tolerance = parameters["tolerance"]
        fraction = parameters["fraction"]
        markers = equidistributed_marking(mesh, error_indicators, tolerance,
                                          fraction)
    elif strategy == "maximal":
        fraction = parameters["fraction"]
        markers = maximal_marking(mesh, error_indicators, fraction)
    elif strategy == "dorfler":
        fraction = parameters["fraction"]
        markers = dorfler_marking(mesh, error_indicators, fraction)

    # Count the number of marked cells
    num_marked = 0
    for i in range(mesh.num_cells()):
        if markers[i]:
            num_marked += 1

    # Report the number of marked cells
    info("Marking %d cells out of %d (%.1f%%) for refinement" % \
         (num_marked, mesh.num_cells(), 100.0*num_marked/mesh.num_cells()))

    return markers

def dorfler_marking(mesh, error_indicators, fraction):
    """
    Mark a set A of cells such that the error of that set
    is greater than a fraction of the total error

    \eta_A = \sum_{T \in A} \eta_T \geq (1 - fraction) \eta_{T_h}

    cf. D\"orfler, SIAM Numer. Anal. 1996
    """
    # FIXME: Cf. D\"orfler1996 for O(#cells) strategy

    info("Using Dorfler marking, fraction %g" % fraction)

    values = numpy.array([abs(v) for v in error_indicators.vector().array()])
    indices = list(numpy.argsort(values))
    indices.reverse()

    eta_A = 0.0
    eta_T_h = sum(values)
    stop = fraction*eta_T_h

    # Initialize cell markers
    cell_markers = MeshFunction("bool", mesh, mesh.topology().dim())
    for i in range(mesh.num_cells()):
        cell_markers[i] = False

    for i in indices:
        if eta_A >= stop:
            return cell_markers

        eta_A += values[i]
        # FIXME: Fix problem when i is of type numpy.int64
        cell_markers[int(i)] = True

    warning("Dorfler marking did not converge! Very strange")
    return cell_markers

def maximal_marking(mesh, error_indicators, fraction):
    """
    Mark cells for which the error is greater than the given fraction
    of the largest error

    Let \eta_max = max_T \eta_T

    For T in T_h:

        mark T if \eta_T > fraction*\eta_max
    """

    info("Using maximal marking, fraction %g" % fraction)

    values = [abs(v) for v in error_indicators.vector().array()]
    largest = max(values)

    cell_markers = MeshFunction("bool", mesh, mesh.topology().dim())
    for c in cells(mesh):
        cell_markers[c] = values[c.index()] > (fraction*largest)
    return cell_markers

def fixed_fraction(mesh, error_indicators, fraction):
    """
    Refine the N% cells with the largest error

    Let n = #cells. Define m = fraction*n, the fraction of cells to be
    refined. Refine the m cells with the largest error.
    """

    info("Using fixed fraction marking, fraction %g" % fraction)

    values = [abs(v) for v in error_indicators.vector().array()]
    limit = sorted(values, reverse=True)[int(len(values)*fraction)]

    cell_markers = MeshFunction("bool", mesh, mesh.topology().dim())
    for c in cells(mesh):
        cell_markers[c] = abs(values[c.index()]) > limit
    return cell_markers


def equidistributed_marking(mesh, error_indicators, fraction, tolerance):
    """
    Refine cells for which the error indicator is greater than
    the ratio fraction*tolerance/(#cells)

    """

    info("Using equidistributed marking, fraction %g, tolerance %g" % \
         (fraction, tolerance))

    values = [abs(v) for v in error_indicators.vector().array()]
    limit = fraction*tolerance/len(values)

    cell_markers = MeshFunction("bool", mesh, mesh.topology().dim())
    for c in cells(mesh):
        cell_markers[c] = (values[c.index()] > limit)
    return cell_markers


