"""
Special error estimators based on DualEstimatorBase.
"""

__author__ = "Marie E. Rognes (meg@simula.no)"
__copyright__ = "Copyright (C) 2009 - Marie E. Rognes"
__license__  = "GNU GPL version 3 or any later version"

# First added:  2009-10-20
# Last changed: 2010-05-09
#
# Modified by Anders Logg, 2009-2010

__all__ = ["ErrorRepresentationEstimator", "DualWeightedResidualEstimator",
           "BothResidualsEstimator",
           "CauchySchwarzEstimator", "BeckerRannacherEstimator"]

import numpy

from dolfin import FunctionSpace, TestFunction, Function
from dolfin import assemble, info, CellSize, interpolate
from dolfin import inner, lhs, dx, dS, ds, avg

from dolfin.adaptivity.residual import *
from dolfin.adaptivity.errorestimator import GoalEstimatorBase
from dolfin.adaptivity.formmanipulation import increase_order, increase_bc_order

class ErrorRepresentationEstimator(GoalEstimatorBase):
    """
    M(u) - M(u_h) = r(z) \approx r(\tilde z)
    """

    def estimate_error(self, u_h):
        info("Computing error estimators")

        # Estimate true dual and store
        V_h = u_h.function_space()
        z = self.construct_dual_approximation(V_h)
        self.dual_approximation = z

        # Create residual functional (as lambda function)
        r = residual(self.F, u_h)

        # Assemble error, that is r(z)
        error = assemble(r(z), mesh = z.function_space().mesh())

        return abs(error)

    def __str__(self):
        return "error_representation"

class DualWeightedResidualEstimator(GoalEstimatorBase):

    """
    M(u) - M(u_h) \approx  \sum_T |<R_T, z - z_h>_T + <R_dT, z - z_h>_dT|

    with variations in the form of z and z_h

    """

    def estimate_error(self, u_h):

        # Assemble error indicators
        error_indicators = self.assemble_error_indicators(u_h)

        # Sum to get total error
        error = sum(e for e in error_indicators.vector())

        return error

    def assemble_error_indicators(self, u_h):
        info("Assembling error indicators")

        # If error indicators already computed, use those
        if self.error_indicators:
            return self.error_indicators

        # Compute residual representation
        (R_T, R_dT) = compute_residual_representation(self.F, u_h)

        # Compute w \approx z - z_h
        w = self.construct_dual_difference(u_h.function_space())

        # Variable to localize error contributions
        Constants = FunctionSpace(R_T.function_space().mesh(), "DG", 0)
        v = TestFunction(Constants)

        contribution = v*inner(R_T, w)*dx \
                       + avg(v)*(inner(R_dT('+'), w('+'))
                                 + inner(R_dT('-'), w('-')))*dS \
                       + v*inner(R_dT, w)*ds

        # Assemble contributions
        indicators = Function(Constants)
        assemble(contribution, tensor=indicators.vector())
        indicators.vector()[:] = numpy.abs(indicators.vector().array())

        # Store error indicators for later use
        self.error_indicators = indicators

        return indicators

    def __str__(self):
        return "dual_weighted_residual"


class BothResidualsEstimator(DualWeightedResidualEstimator):
    """
    This is just for testing purposes for now. Could probably be
    simplified by better design.
    """

    def assemble_error_indicators(self, u_h):
        info("Assembling error indicators")

        # If error indicators already computed, use those
        if self.error_indicators:
            return self.error_indicators

        # Compute dual approximation
        z_h = self.dual.solve(u_h)

        # (1a) Compute primal residual representation
        info("-- Computing primal residual representation")
        (R_T, R_dT) = compute_residual_representation(self.F, u_h)

        # (1b) Compute improved dual approximation
        V_h = u_h.function_space()
        W_h = increase_order(V_h)
        new_bcs = increase_bc_order(self.dual.bcs, V_h)
        z = Function(W_h)
        z.extrapolate(z_h, new_bcs)
        w_z = z - interpolate(z, V_h)

        # (2a) Compute dual residual representation
        info("-- Computing dual residual representation")
        from dolfin import adjoint
        # FIXME: This is a bit disturbing:
        (S_T, S_dT) = compute_residual_representation(adjoint(adjoint(self.dual.F)), z_h)

        # (2b) Compute improved primal approximation
        u = Function(W_h)
        u.extrapolate(u_h, new_bcs)
        w_u = u - interpolate(u, V_h)

        # Variable to localize error contributions
        Constants = FunctionSpace(R_T.function_space().mesh(), "DG", 0)
        v = TestFunction(Constants)

        # Forms for error estimator
        primal_r = v*inner(R_T, w_z)*dx \
                   + avg(v)*(inner(R_dT('+'), w_z('+'))
                             + inner(R_dT('-'), w_z('-')))*dS \
                             + v*inner(R_dT, w_z)*ds
        dual_r = v*inner(S_T, w_u)*dx \
                 + avg(v)*(inner(S_dT('+'), w_u('+'))
                           + inner(S_dT('-'), w_u('-')))*dS \
                           + v*inner(S_dT, w_u)*ds
        contribution = primal_r + dual_r

        # Assemble contributions
        indicators = Function(Constants)
        assemble(contribution, tensor=indicators.vector())
        indicators.vector()[:] = 0.5*numpy.abs(indicators.vector().array())

        # Store error indicators for later use
        self.error_indicators = indicators
        return indicators

    def __str__(self):
        return "both_residuals"

class CauchySchwarzEstimator(DualWeightedResidualEstimator):
    """
    Indicators

    e_T = ||z - z_h||_T ||R_T||_T + ||z - z_h|T||_dT ||avg(R_dT)|e||_dT
    """
    def assemble_error_indicators(self, u_h):
        """ Assemble error indicators """

        error("This estimator has not been updated!")

        # If error indicators already computed, use those
        if self.error_indicators is not None:
            return self.error_indicators

        # Compute residual representation
        (R_T, R_dT) = compute_residual_representation(self.F, u_h)

        # Compute w \approx z - z_h
        w = self.construct_dual_difference(u_h.function_space())

        # Define forms for contributions
        Constants = FunctionSpace(R_T.function_space().mesh(), "DG", 0)
        v = TestFunction(Constants)

        # Assemble contributions from residual
        cell_res = assemble(v*inner(R_T, R_T)*dx)
        facet_res = assemble(2*avg(v)*inner(avg(R_dT), avg(R_dT))*dS)
        boundary_res = assemble(v*inner(R_dT, R_dT)*ds)

        # Assemble contributions from dual
        cell_du = assemble(v*inner(w, w)*dx)
        facet_du = assemble(2*avg(v)*inner(avg(w), avg(w))*dS)
        boundary_du = assemble(v*inner(w, w)*ds)

        # Multiply, take square root and sum up contributions
        cell_errors = numpy.sqrt(cell_res*cell_du)
        facet_errors = numpy.sqrt(facet_res*facet_du)
        boundary_errors = numpy.sqrt(boundary_res*boundary_du)
        indicators = cell_errors + facet_errors + boundary_errors

        error_indicators = Function(Constants)
        error_indicators.vector()[:] = indicators # Is this ok?
        return error_indicators


    def __str__(self):
        return "cauchy_schwarz"


class BeckerRannacherEstimator(DualWeightedResidualEstimator):
    """
    Indicators

    e_T = (||R_T||_T + h_T^{-1/2} ||R_dT||_dT) (||z - z_h||_T + h_T^{1/2} ||z - z_h||_dT)
    """

    def assemble_error_indicators(self, u_h):
        """ Assemble error indicators """

        error("This estimator has not been updated!")

        # If error indicators already computed, use those
        if self.error_indicators is not None:
            return self.error_indicators

        # Compute residual representation
        (R_T, R_dT) = compute_residual_representation(self.F, u_h)

        # Compute w \approx z - z_h
        w = self.construct_dual_difference(u_h.function_space())

        # Define forms for contributions
        mesh = R_T.function_space().mesh()
        h = CellSize(mesh)
        Constants = FunctionSpace(mesh, "DG", 0)
        v = TestFunction(Constants)

        # Assemble contributions from residual
        cell_res = assemble(v*inner(R_T, R_T)*dx)
        facet_res = assemble(2*avg(v)*(1.0/avg(h))*inner(avg(R_dT), avg(R_dT))*dS)
        boundary_res = assemble(v*1.0/h*inner(R_dT, R_dT)*ds)

        # Assemble contributions from dual
        cell_du = assemble(v*inner(w, w)*dx)
        facet_du = assemble(2*avg(v)*avg(h)*inner(avg(w), avg(w))*dS)
        boundary_du = assemble(v*h*inner(w, w)*ds)

        # Sum square-roots of contributions
        residual_factor = numpy.sqrt(cell_res) + numpy.sqrt(facet_res) + numpy.sqrt(boundary_res)
        dual_factor = numpy.sqrt(cell_du) + numpy.sqrt(facet_du) + numpy.sqrt(boundary_du)

        # Take product of residual and dual factors
        indicators = residual_factor*dual_factor

        error_indicators = Function(Constants)
        error_indicators.vector()[:] = indicators # Is this ok?
        return error_indicators

    def __str__(self):
        return "becker_rannacher"



