/*
 * Dmxdummy.c
 * a driver that does nothing but registering a dummy driver.
 *
 * Copyright (C) Michael Stickel <michael@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


#include <linux/init.h>
#include <linux/module.h>
#include <linux/proc_fs.h>

#include <dmx/dmxdev.h>

#define DEBUG
#undef  DEBUG

#ifdef DEBUG
#define DO_DEBUG(x...) (x)
#else
#define DO_DEBUG(x...)
#endif


#define MAXUNIVERSES  10

#ifdef proc_mkdir
#undef proc_mkdir
#endif
#define proc_mkdir(name,root) create_proc_entry(name,S_IFDIR|S_IRUGO|S_IXUGO,root)


int in_universes = 1;    /* create 1 input universe by default */
MODULE_PARM(in_universes,"i");
MODULE_PARM_DESC(in_universes, "Number of input universes to be cretated on module-startup");


int out_universes = 1;    /* create 1 output universe by default */
MODULE_PARM(out_universes,"i");
MODULE_PARM_DESC(out_universes, "Number of output universes to be cretated on module-startup");

MODULE_AUTHOR ("(c) 1999-2001 Michael Stickel");
MODULE_DESCRIPTION ("Dummy interface used for test purpose if no hardware is available or needed. version " DMXVERSION);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,17)
MODULE_LICENSE("GPL");
#endif




typedef struct
{
  struct proc_dir_entry *proc_entry;
  struct proc_dir_entry *cocked_proc_entry;
  unsigned char          buffer[512];
  DMXUniverse           *universe;
  char                   data_avail;
} DummyUniverse;


#define GetDummyUniverse(u) (((u)->conn_id>=0 && (u)->conn_id<MAXUNIVERSES)?((u)->kind?(input_universe[(u)->conn_id]):(output_universe[(u)->conn_id])):NULL)



static struct proc_dir_entry   *outdir_entry = NULL;
static DMXFamily               *dummy_family = NULL;
static char                     loopback[MAXUNIVERSES];
static DummyUniverse           *input_universe[MAXUNIVERSES],
                               *output_universe[MAXUNIVERSES];



/*
 *-----------------[ dmxdev access functions ]-------------------
 */

/*
 * This method is called by the dmxdev module
 * if slots have been changed by the userspace
 * for e.g. by a write to /dev/dmx.
 */
static int  write_universe (DMXUniverse *u, off_t offs, DMXSlotType *buff, size_t size)
{
    if (!u || !buff || size<=0 || offs+size >= 512 /*dev->channels*/) return -EINVAL;
    memcpy (&output_universe[u->conn_id]->buffer[offs], buff, size);

    /*
     * signal the intput universe that data has arived if loopback mode is on.
     */
    if (loopback[u->conn_id] && input_universe[u->conn_id] && input_universe[u->conn_id]->universe)
      {
	int id = u?u->conn_id:-1;
	memcpy (&input_universe[id]->buffer[offs], &output_universe[id]->buffer[offs], size);
	input_universe[id]->data_avail=1;
	input_universe[id]->universe->signal_changed (input_universe[id]->universe, 0, size);
      }

    /*
     * signal the output universe too.
     */
    if (output_universe[u->conn_id] && output_universe[u->conn_id]->universe)
      {
	int id = u?u->conn_id:-1;
	output_universe[id]->data_avail=1;
	output_universe[id]->universe->signal_changed (output_universe[id]->universe, 0, size);
      }
    return size;
}


/*
 * read slots from the output universe.
 * This function can be used for both, the input and the output universe.
 */
static int read_universe (DMXUniverse *u, off_t start, DMXSlotType *buff, size_t size)
{
  if (u && start >= 0 && start < 512)
    {
      int id = u->conn_id;
      if (id>=0 && id<10)
	{
	  if (start+size>512)
	    size = 512-start;
	  if (u->kind)
	    {
	      memcpy (buff, &input_universe[id]->buffer[start], size);
	      input_universe[u->conn_id]->data_avail = 0;
	    }
	  else
	    {
	      memcpy (buff, &output_universe[id]->buffer[start], size);
	      output_universe[u->conn_id]->data_avail = 0;
	    }
	  return size;
	}
    }
  return -EINVAL;
}

/*
 * returns < 0 for error, > 0 if new data is available and 0
 * if no new data is available for that universe.
 */
int  dummy_data_available (DMXUniverse *u, uint start, uint size)
{
  if (!u || u->conn_id<0 || u->conn_id>=10) return 0;
  if (u->kind)
    return input_universe[u->conn_id]->data_avail;
  else
    return 1;
  /* return output_universe[u->conn_id]->data_avail; */ /* output is by definition at any time ready to read */
}


/*
 *------------[ proc-fs access functions ]---------------------
 */

/*
 * This method is called if someone writes to
 * the file /proc/dmxdummy/in<connector>
 */
static int in_write_universe_data (struct file *file, const char *buf, unsigned long count, void *data)
{
  DMXUniverse *u = (DMXUniverse *)data;

  if (u && u->kind)
    {
      DummyUniverse *du = GetDummyUniverse(u);

      if (!du)
	return -EBUSY;

      if (count > 512)
	count = 512;

      memcpy (&(du->buffer[0]), buf, count);
      du->data_avail=1;
      u->signal_changed (u, 0, count); /* a signal for the channel that has changed */
      return count;
    }
  return -EINVAL;
}

/*
 * This method is called if someone reads from
 * the file /proc/dmxdummy/out<connector>
 */
static int read_universe_data (char *buf, char **start, off_t offset, int length, int *eof, void *data)
{
  DMXUniverse *u = (DMXUniverse *)data;
  if (u && u->conn_id>=0 && u->conn_id<10)
    {
      DummyUniverse *du = GetDummyUniverse(u);

      if (!du)
	return -EINVAL;

      if (length+offset > 512)
        length = 512 - offset;

      if (length > 0)
        memcpy (buf, &(du->buffer[offset]), length);

      return length;
    }
  return -EINVAL;
}

/*
 * This method is called if someone reads from
 * the file /proc/dmxdummy/outc<connector>
 */
static int read_universe_data_cocked (char *buf, char **start, off_t offset, int length, int *eof, void *data)
{
  DMXUniverse *u = (DMXUniverse *)data;
  if (u && u->conn_id>=0 && u->conn_id<10)
    {
      DummyUniverse *du = GetDummyUniverse(u);
      char *p = buf;
      int i;

      if (!du)
	return -EINVAL;

      for (i=0; i<512; i++)
        p += sprintf (p, "%02x ", du->buffer[i]);

      return p- buf;
    }
  return -EINVAL;
}


/*
 *---------------[ setter / getter functions ]------------------
 */


static int loopback_get_long (DMXProperty *p, long *val)
{
  if (p && val && p->type == (DMXPTYPE_LONG|DMXPTYPE_USER))
    {
      DMXUniverse *u = (DMXUniverse *)p->data;
      if (u && u->conn_id>=0 && u->conn_id<MAXUNIVERSES)
        {
          *val = loopback[u->conn_id];
          return 0;
        }
    }
  return -1;
}

static int loopback_set_long (DMXProperty *p, long val)
{
  if (p && val && p->type == (DMXPTYPE_LONG|DMXPTYPE_USER))
    {
      DMXUniverse *u = (DMXUniverse *)p->data;
      if (u && u->conn_id>=0 && u->conn_id<MAXUNIVERSES)
        {
          loopback[u->conn_id] = val?1:0;
          return 0;
        }
    }
  return -1;
}



/*
 *---------------[ universe creation / deletion ]------------------
 */

/*
 * This method is called before the universe will be deleted
 */
static int delete_universe (DMXUniverse *u)
{
  if (u && outdir_entry && u->conn_id>=0 && u->conn_id<MAXUNIVERSES)
    {
      char e_name[20];

      if (u->kind)
	sprintf (e_name, "in%d", u->conn_id);
      else
	sprintf (e_name, "out%d", u->conn_id);
      remove_proc_entry (e_name, outdir_entry);

      if (!u->kind)
        {
          sprintf (e_name, "outc%d", u->conn_id);
          remove_proc_entry (e_name, outdir_entry);
        }

      printk (KERN_INFO "dummy-universe deleted\n");

      if (u->kind)
	{
	  DMX_FREE(input_universe[u->conn_id]);
	  input_universe[u->conn_id] = NULL;
	}
      else
	{
	  DMX_FREE(output_universe[u->conn_id]);
	  output_universe[u->conn_id] = NULL;
	}
   }
  return 0;
}


/*
 * return the id of the interface.
 *
 */
int misc_getUniverseID (DMXUniverse *u, char *id, size_t size)
{
  if (u && id && size >= 10)
    {
      char *p = id;
      p += sprintf(id, "%d", u->conn_id);
      return p-id;
    }
  return -1;
}




static int create_universe (DMXUniverse *u, DMXPropList *pl)
{
  long iobase = 0L;
  if (pl)
    {
	DMXProperty *p = pl->find(pl, "iobase");
        if (p) p->get_long(p, &iobase);
    }

  printk ("dmxdummy:create_universe_%s(iobase=%ld) called\n", 
	  (u->kind==0)?"out":"in",
	  iobase);
  if (u)
    {
      int maxid = 0;
      DMXUniverse *tu = NULL;
      DummyUniverse *du = NULL;

      if ((u->kind==0 && output_universe[u->conn_id]) || (u->kind==1 && input_universe[u->conn_id]))
	return -EBUSY;

      u->user_delete = delete_universe;

      /*
       *
       */
      for (tu = u->interface ? u->interface->universes : NULL; tu; tu = tu->next)
        {
          if (tu->conn_id >= maxid && tu->kind==u->kind)
            maxid = tu->conn_id + 1;
        }

      strcpy (u->connector, "none"); /* it's an interface that has definitly no connector */
      u->conn_id = maxid;

      if (u->conn_id >=0 && u->conn_id<10)
	{
	  int  i;
	  char e_name[10];
	  
	  if (u->kind)
	    du = input_universe[u->conn_id] = DMX_ALLOC(DummyUniverse);
	  else
	    du = output_universe[u->conn_id] = DMX_ALLOC(DummyUniverse);
	  if (!du)
	    {
	      printk (KERN_INFO "unable to create internal-dmxdummy structures\n");
	      return -EBUSY;
	    }

	  u->data_available = dummy_data_available;

	  sprintf (e_name, u->kind?"in%d":"out%d", u->conn_id);
	  du->proc_entry = create_proc_entry (e_name, S_IFREG | S_IRUGO | (u->kind?S_IWUGO:0), outdir_entry);
	  if (u->kind)
	    du->proc_entry->write_proc = in_write_universe_data;
	  du->proc_entry->read_proc  = read_universe_data;
	  du->proc_entry->data       = (void *)u;

          if (!u->kind) /* out */
            {
	      sprintf (e_name, "outc%d", u->conn_id);
              du->cocked_proc_entry = create_proc_entry (e_name, S_IFREG | S_IRUGO, outdir_entry);
              du->cocked_proc_entry->read_proc  = read_universe_data_cocked;
	      du->cocked_proc_entry->data       = (void *)u;
            }
          else
            du->cocked_proc_entry = NULL;
  
	  du->universe = u;

	  for (i=0; i<512; i++)
	    du->buffer[i]=0;
	  du->data_avail = 0;
	  
	  u->read_slots   = read_universe;
	  if (!u->kind)
	    u->write_slots =  write_universe;


          if (pl)
            {
              DMXProperty *p = pl->find(pl, "loopback");
              if (p)
                {
                  dmxprop_user_long (p, loopback_get_long, loopback_set_long, (void *)u);
                  p->data = (void *)u;
                }
            }

	}
    }
  return 0;
}



/*
 *----------------[ module initialition / cleanup ]------------------
 */

/*
 * Called by the kernel-module-loader after the module is loaded.
 */
static int __init dummy_init(void)
{
  int  i;

  printk ("dmxdummy:init_module called\n");

  if (in_universes>MAXUNIVERSES || out_universes>MAXUNIVERSES)
    {
      printk ("in_universes or out_universes is greater than %d\n", MAXUNIVERSES);
      return -EINVAL;
    }

  for (i=0; i<MAXUNIVERSES; i++)
    input_universe[i] = output_universe[i] = NULL;

  outdir_entry = proc_mkdir ("dmxdummy", 0);

  dummy_family = dmx_create_family ("dummy");
  if (dummy_family)
    {
      DMXDriver *drv = dummy_family->create_driver (dummy_family, "dummy", create_universe, NULL);

      printk (KERN_INFO "dmx_find_driver(dummy_family,\"dummy\") => %p\n", dmx_find_driver(dummy_family, "dummy") );


      if (drv)
        drv->getUniverseID = misc_getUniverseID;

      if (drv)
        {
          DMXInterface *dif = drv->create_interface (drv, NULL);
          if (dif)
            {
              int  j;
              for (j=0; j<out_universes; j++)
		  dif->create_universe (dif, 0, dmxproplist_vacreate("loopback=0"));

              for (j=0; j<in_universes; j++)
		  dif->create_universe (dif, 1, dmxproplist_vacreate("test=1,aaaa=0"));
              return 0;
            }
        }
      dummy_family->delete(dummy_family, 0);
    }
  remove_proc_entry ("dmxdummy", 0);
  return -1;
}



/*
 * Called by the kernel-module-loader before the module is unloaded.
 */
static void __exit dummy_exit(void)
{
  printk ("dmxdummy:cleanup_module called\n");
  if (dummy_family)
    dummy_family->delete(dummy_family, 0);
  remove_proc_entry ("dmxdummy", 0);
}

#ifdef MODULE
module_init(dummy_init);
module_exit(dummy_exit);
EXPORT_NO_SYMBOLS;
#endif
