/*
 * Fader.c
 * This is a fader-widget for studio purpose
 *
 * Copyright (C) Michael Stickel <michael@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "Fader.h"

#include "FaderP.h"

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/xpm.h>

/*============[ local definitions ]=============*/

#define DEFAULT_THUMB_HEIGHT  31

#define DISPLAYVALUE          1 /* A 1 causes the value to be displayed */
#define DISPLAYVALUE_ONBACK   0 /* A 1 causes the value to be displayed on the back */
#define DISPLAYVALUE_ONTHUMB  1 /* A 1 causes the value to be displayed on the thumb */


/*============[ local Prototypes ]==============*/
static void     select_fader ();
static void     Initialize ();
static void     Redisplay ();
static void     Resize ();
static void     Destroy ();
static Boolean  SetValues ();

static void     XsFaderSetThumb (XsFaderWidget);

static void     StartFade (XsFaderWidget w, int pos);
static void     StopFade  (XsFaderWidget w);

static void     fade_up (XtPointer w_p, XtIntervalId *iid);
static void     FaderSendChanged (XsFaderWidget w, XEvent *event);



/*==============[ local data-definitions ]============*/

static char defaultTranslations[] =
"<BtnDown> : select()\n"
"<BtnUp>   : select()\n"
"<Motion>  : select()";

static XtActionsRec actionsList[] =
{
  {"select",  (XtActionProc) select_fader},
};



/*=============[ Recources ]====================*/

static XtResource	resources[] =
{
  /*------[ Colors ]-------*/
  {XtNforeground,   XtCForeground, XtRPixel,sizeof (Pixel), 
   XtOffsetOf(XsFaderRec, fader.foreground),
   XtRString, "Black" },
#if 0
  {XtNbackground,   XtCBackground, XtRPixel,sizeof (Pixel), 
   XtOffsetOf(XsFaderRec, fader.background),
   XtRString, "Grey" },
#endif
  {XtNsliderColor,  XtCColor, XtRPixel, sizeof(Pixel), 
   XtOffsetOf(XsFaderRec, fader.slider_color),
   XtRString, "SteelBlue"},

  {XtNidentColor,   XtCColor, XtRPixel, sizeof(Pixel),
   XtOffsetOf(XsFaderRec, fader.ident_color),
   XtRString, "Grey"},

  {XtNshadow1Color, XtCForeground, XtRPixel,sizeof (Pixel), 
   XtOffsetOf(XsFaderRec, fader.shadow1_color),
   XtRString, "White" },

  {XtNshadow2Color, XtCForeground, XtRPixel,sizeof (Pixel), 
   XtOffsetOf(XsFaderRec, fader.shadow2_color),
   XtRString, "Black" },


  /*------[ Values ]------*/
  {XtNposition,     XtCPosition, XtRPosition, sizeof(Position), 
   XtOffsetOf(XsFaderRec, fader.position),
   XtRString, "1000"},

  {XtNfaderType,    XtCPosition,   XtRPosition, sizeof(int),
   XtOffsetOf(XsFaderRec, fader.th.type),
   XtRString, "0"},

  {XtNfadeTo,       XtCPosition,   XtRPosition, sizeof(Position),
   XtOffsetOf(XsFaderRec, fader.fadeto_pos),
   XtRString, "0"},

  /*------[ Callbacks ]------*/
  {XtNselectCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffsetOf(XsFaderRec, fader.select),
   XtRCallback, NULL },

  {XtNchangedCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffsetOf(XsFaderRec, fader.changed),
   XtRCallback, NULL },


  /*====[ The Shadow-value bar ]==========*/

/*------[ Colors ]-------*/
  {XtNshadowColor,  XtCForeground, XtRPixel,sizeof (Pixel), 
   XtOffsetOf(XsFaderRec, fader.shadowColor),
   XtRString, "Green" },

  /*------[ Values ]------*/
  {XtNshadowPos,    XtCPosition, XtRPosition, sizeof(Position), 
   XtOffsetOf(XsFaderRec, fader.shadowPos),
   XtRString, "0"},

  /*------[ Callbacks ]------*/
#if 0
  {XtNselectCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffset Of(XsFaderRec, fader.select),
   XtRCallback, NULL },
#endif
};


/*============[ class record ]============*/

XsFaderClassRec XsfaderClassRec =
{		/* CoreClassPart */
  {
    (WidgetClass) &widgetClassRec,
    "Fader",
    sizeof(XsFaderRec),
    NULL,
    NULL,
    FALSE,
    Initialize,
    NULL,
    XtInheritRealize,
    actionsList,
    XtNumber(actionsList),
    resources,
    XtNumber(resources),
    NULLQUARK,
    TRUE,	
    XtExposeCompressMaximal,
    TRUE,
    TRUE,
    Destroy,
    Resize,
    Redisplay,
    SetValues,
    NULL,
    XtInheritSetValuesAlmost,
    NULL,
    NULL,
    XtVersion,
    NULL,
    defaultTranslations,
    NULL,
    NULL,
    NULL,
  },
  /* Dial class fields */
  {
    0,
  }
};


WidgetClass	XsfaderWidgetClass = (WidgetClass) &XsfaderClassRec;

XpmImage faderknob_image;



/*==============[ methods ]==============*/

static void Initialize (XsFaderWidget request, XsFaderWidget new)
{
  XGCValues	values;
  XtGCMask	valueMask;

  if (request->core.width==0)  new->core.width  = 100;
  if (request->core.height==0) new->core.height = 100;

  new->fader.autofadeId = 0;
  new->fader.hit = 0;

  new->fader.position = new->core.height - DEFAULT_THUMB_HEIGHT;
  new->fader.fadeto_pos = new->fader.position;
  new->fader.th.val_win = 0;

  valueMask = GCForeground;
  values.foreground = new->core.background_pixel;
  new->fader.GC_bg = XtGetGC ((Widget)new, valueMask, &values);

  values.foreground = new->fader.foreground;
  new->fader.GC_fg = XtGetGC ((Widget)new, valueMask, &values);

  values.foreground = new->fader.slider_color;
  new->fader.GC_slider = XtGetGC ((Widget)new, valueMask, &values);

  values.foreground = new->fader.ident_color;
  new->fader.GC_ident = XtGetGC ((Widget)new, valueMask, &values);

  values.foreground = new->fader.shadow1_color;
  new->fader.GC_shadow1 = XtGetGC ((Widget)new, valueMask, &values);

  values.foreground = new->fader.shadow2_color;
  new->fader.GC_shadow2 = XtGetGC ((Widget)new, valueMask, &values);

  new->fader.th.w  = 0;
  new->fader.th.pm = 0;


  new->fader.shadowPos = new->fader.lastShadowPos = 0;
  values.foreground    = new->fader.shadowColor;
  new->fader.GC_shadow = XtGetGC ((Widget)new, valueMask, &values);
  new->fader.GC_shadow_thumb = XtGetGC ((Widget)new, valueMask, &values);
  /*	Resize ();  */


  new->fader.fast_motion=0;
}

#include "XmsBuildThumb.h"




static void XsFaderCreateThumb (XsFaderWidget new)
{
  if (new)
    {
      Display        *disp = XtDisplay (new);
      int             depth;
      XmsFaderThumb  *th = &(new->fader.th);

      new->fader.cont = XtWidgetToApplicationContext ((Widget)new);
      if (XtWindow (new))
        {
          th->width = new->core.width;
          th->height = DEFAULT_THUMB_HEIGHT;
          th->hx = th->width/2;
          th->hy = th->height/2;

          th->xoffs = new->core.width/2 - th->width/2;
          XtVaGetValues ((Widget)new, XtNdepth, &depth, NULL);
          th->w = XCreateSimpleWindow (disp, XtWindow(new),
                                       th->xoffs, new->fader.position, 
                                       th->width, th->height,
                                       0, new->fader.foreground,
                                       new->core.background_pixel);

          if (th->hx==-1 || th->hy==-1) /* set Hotspot to [0,0] if no one is defined */
            th->hx = th->hy = 0;

          if (th->w)
            {
              th->pm = XCreatePixmap (disp, th->w, th->width, th->height, depth);
              XmsBuildThumb (new, th->width, th->height, th->type);
              XSetWindowBackgroundPixmap (disp, th->w, th->pm);
            }
          XMapWindow (disp, th->w);
        }
    }
}



static void Destroy (XsFaderWidget w)
{
  printf ("Destroy (0x%08X) .window = %08X\n", (int)w, (int)XtWindow(w));
  if (w->fader.th.w)
	{
      XDestroyWindow (XtDisplay(w), w->fader.th.w);
      if (w->fader.th.pm)
        XFreePixmap (XtDisplay(w), w->fader.th.pm);
	}
  XtReleaseGC ((Widget)w, w->fader.GC_bg);
  XtReleaseGC ((Widget)w, w->fader.GC_fg);
  XtReleaseGC ((Widget)w, w->fader.GC_slider);
  XtReleaseGC ((Widget)w, w->fader.GC_ident);
  XtReleaseGC ((Widget)w, w->fader.GC_shadow1);
  XtReleaseGC ((Widget)w, w->fader.GC_shadow2);
  XtRemoveCallbacks ((Widget)w, XtNselectCallback, w->fader.select);
  XtRemoveCallbacks ((Widget)w, XtNchangedCallback, w->fader.changed);
}




static void calculate_thumb_pos (XsFaderWidget w)
{
  int pos = w->fader.position;

  if (pos < 0)
    pos = 0;

  if (pos > w->core.height - w->fader.th.height) 
    pos = w->core.height - w->fader.th.height;

  w->fader.position = pos;
}


static void Resize (XsFaderWidget w)
{
  w->fader.th.xoffs = w->core.width/2 - w->fader.th.width / 2;
  calculate_thumb_pos (w);
}


static void Redisplay (XsFaderWidget w, XEvent *event, Region region)
{
  char *mask_bits = NULL;
#if WID_DEBUG
  printf ("Redisplay (0x%08X)\n", w);
#endif

  if (!w->fader.th.w)
    {
      Pixmap bitmask;
      int  len = 0, width = 0;
      int  i;

      XsFaderCreateThumb (w);

      width = (w->fader.th.width/8) + ((w->fader.th.width%8)?1:0);
      len = width * w->fader.th.height;

      mask_bits = (char *)malloc(len);

      for (i=0; i<len; i++)  mask_bits[i] = ((i / width) % 2)?0x55:0xAA;
      for (i=0; i<len; i++)  mask_bits[i] = 0xff;

      bitmask  = XCreateBitmapFromData (XtDisplay(w), w->fader.th.w,
                                        mask_bits, width, w->fader.th.height);
      if (bitmask)
        XSetClipMask (XtDisplay(w), w->fader.GC_shadow_thumb, bitmask);
      else
        printf ("unable to create bit-mask\n");
    }

  if (w->core.visible && w->fader.th.w)
    XsFaderSetThumb (w);
}



static GC FaderOtherGC (XsFaderWidget w, GC oldgc, Pixel newpix) 
{
  XGCValues		values;
  XtGCMask		valueMask;

  valueMask = GCForeground;
  values.foreground = newpix;
  XtReleaseGC ((Widget)w, oldgc);
  return XtGetGC ((Widget)w, valueMask, &values);
}


static Boolean SetValues (XsFaderWidget current, XsFaderWidget request, XsFaderWidget new)
{
  Boolean			rebuild = FALSE;
  Boolean			redraw_indicator = FALSE;

  new->fader.hit = current->fader.hit;

  new->fader.hit = current->fader.hit;


  if (new->core.background_pixel != current->core.background_pixel) {
    new->fader.GC_bg = FaderOtherGC(new, new->fader.GC_bg, new->core.background_pixel);
    rebuild = TRUE;
  }

  if (new->fader.foreground != current->fader.foreground) {
    new->fader.GC_fg = FaderOtherGC (new, new->fader.GC_fg,
                                     new->fader.foreground);
    rebuild = TRUE;
  }

  if (new->fader.slider_color != current->fader.slider_color) {
    new->fader.GC_slider = FaderOtherGC(new, new->fader.GC_slider,
                                        new->fader.slider_color);
    rebuild = TRUE;
  }

  if (new->fader.ident_color != current->fader.ident_color) {
    new->fader.GC_ident = FaderOtherGC(new, new->fader.GC_ident,
                                       new->fader.ident_color);
    rebuild = TRUE;
  }

  if (new->fader.shadow1_color != current->fader.shadow1_color) {
    new->fader.GC_shadow1 = FaderOtherGC(new, new->fader.GC_shadow1,
                                         new->fader.shadow1_color);
    rebuild = TRUE;
  }

  if (new->fader.shadow2_color != current->fader.shadow2_color) {
    new->fader.GC_shadow2 = FaderOtherGC(new, new->fader.GC_shadow2,
                                         new->fader.shadow2_color);
    rebuild = TRUE;
  }

  if (new->fader.th.type != current->fader.th.type)
    rebuild = TRUE;

  if (rebuild)
    { /* Fader-Thumb has been modified => rebuild Pixmap */
      XmsBuildThumb (new, new->fader.th.width, new->fader.th.height,
                     new->fader.th.type);
      XClearWindow (XtDisplay (new), new->fader.th.w);
    }

  if (new->fader.position != current->fader.position)
    {
      if (new->fader.hit==4)
        {
          new->fader.hit=0;
          new->fader.fadeto_pos = -1;
        }
      calculate_thumb_pos (new);
      redraw_indicator = TRUE;
    }

  /* Fadevorgang initiieren */
  if (new->fader.fadeto_pos != current->fader.fadeto_pos
      && new->fader.fadeto_pos != new->fader.position)
    {
      StartFade (new, new->fader.fadeto_pos);
    }


  if (new->fader.shadowPos != current->fader.shadowPos)
    {
      redraw_indicator = TRUE;
    }


  if (redraw_indicator && XtIsRealized((Widget)new) && new->core.visible)
    XsFaderSetThumb (new);

  return 0;
}



/*
 +----+
 |    |
 |    |
 |    |
 | ** | w->fader.position  = ttop
 | || |
 | || |
 | ** | w->fader.th.height = tbot
 |    |
 |#   | w->core.height-w->fader.shadowPos
 |#   |
 |#   |
 |#   |
 |#   |
 +----+ w->core.height
*/
static void XsFaderSetShadow (XsFaderWidget new)
{
  int rpos  = (new->core.height-new->fader.shadowPos)
    - new->fader.position;


  /* redraw the shadow Fader */
  if (new->fader.shadowPos < new->fader.lastShadowPos)  /* clear a small part */
    {
      XClearArea     (XtDisplay(new), XtWindow(new),
                      3,
                      new->core.height - new->fader.lastShadowPos,
                      (new->core.width-6),
                      new->core.height - new->fader.shadowPos,
                      0);
    }

  XFillRectangle (XtDisplay(new), XtWindow(new), new->fader.GC_shadow,
                  3,
                  new->core.height - new->fader.shadowPos,
                  (new->core.width-6),
                  new->core.height);
#if 0
  if (new->core.height - new->fader.shadowPos <= new->fader.position+new->fader.th.height)
#endif
    {
#if 0
      printf ("rpos=%d, lrpos=%d\n", rpos, lrpos);
#endif
      if (rpos > 0)
        XClearArea (XtDisplay(new), new->fader.th.w,
                    3,
                    0,
                    (new->core.width-6),
                    rpos,
                    0);


      if (rpos <= new->fader.th.height)
        XFillRectangle (XtDisplay(new), new->fader.th.w, new->fader.GC_shadow_thumb,
                        3,
                        rpos,
                        (new->core.width-6),
                        new->fader.th.height);
    }

  new->fader.lastShadowPos = new->fader.shadowPos;
}


static void XsFaderSetThumb (XsFaderWidget new)
{
  calculate_thumb_pos (new);

  if (new->fader.th.w)
    XMoveWindow (XtDisplay (new), new->fader.th.w,
                 new->fader.th.xoffs, new->fader.position);

  XFillRectangle (XtDisplay(new), XtWindow(new), new->fader.GC_fg,
                  new->core.width/2, 4, 2, new->core.height-8);


  XsFaderSetShadow (new);



#undef DISPLAYVALUE
#if DISPLAYVALUE
  {
    char s[4];

    /* hier evtl callback-Funktion aufrufen die die Umrechnung macht */
    /*   p = (callback_function)?callback_function(w):interne_umrechnung(w); */

    int p = 100 - ((new->fader.position*100) / (new->core.height-new->fader.th.height));
    if (p==100)
      strcpy (s, "FF");
    else
      sprintf (s, "%02d", p);
#if DISPLAYVALUE_ONBACK
    XClearArea  (XtDisplay(new), XtWindow(new), 1,1,new->core.width-2, 10, 0);
    XDrawString (XtDisplay(new), XtWindow(new), new->fader.GC_fg, 1,10, s, 2);
#endif

#if DISPLAYVALUE_ONTHUMB
    XClearArea  (XtDisplay(new), new->fader.th.w, 1,1,new->core.width-2, new->fader.th.height/2+6, 0);
    XDrawString (XtDisplay(new), new->fader.th.w, new->fader.GC_fg, 3,new->fader.th.height/2+5, s, 2);
#endif
  }
#endif
}

/* not used
static void button_state (XEvent *event)
{
  if (!event) return;
  printf (" xkey: state=%04d, btn=%04d, x=%03d, y=%03d\n",
          event->xbutton.state, 
          event->xbutton.button, event->xbutton.x, event->xbutton.y);
}
*/


static void continue_fade (XsFaderWidget w);

static void fade_up (XtPointer w_p, XtIntervalId *iid)
{
  XsFaderWidget w = (XsFaderWidget)w_p;
  if (!w) return;

  w->fader.autofadeId = 0; /* Timeout is invoked => timer deleted */

  switch (w->fader.hit) {
  case 4:
    if (w->fader.fadeto_pos==w->fader.position) {
      StopFade (w);
    }
    if (w->fader.fadeto_pos>=w->fader.position
        && w->fader.position >= w->core.height - w->fader.th.height) {
      StopFade (w);
    }
    if (w->fader.hit!=0) {
      if (w->fader.position < w->fader.fadeto_pos)
        w->fader.position++;
      else
        w->fader.position--;
      continue_fade (w);
    }
    break;


  case 3:
    if (w->fader.position >= w->core.height - w->fader.th.height) {
      w->fader.hit=0;
      XtCallCallbacks ((Widget)w, XtNselectCallback, (XtPointer)0);
    }
    else {
      w->fader.position+=2;
      continue_fade (w);
    }
    break;

  case 2:
    if (w->fader.position <= 0) {
      w->fader.hit=0;
      XtCallCallbacks ((Widget)w, XtNselectCallback, (XtPointer)0);
    }
    else {
      w->fader.position-=2;
      continue_fade (w);
    }
    break;
  }
}

static void continue_fade (XsFaderWidget w) {
  w->fader.autofadeId = XtAppAddTimeOut (w->fader.cont, 10, fade_up, (XtPointer)w);
  XsFaderSetThumb (w);
  FaderSendChanged (w, NULL);
}


static void StartFade (XsFaderWidget w, int pos)
{
  if (w->fader.cont)
    {
      /* stop an autofade if one is running */
      if (w->fader.hit==4) {
        StopFade(w);
      }

      /* clip the value */
      if (pos < 0) pos = 0;
      if (pos > w->core.height - w->fader.th.height)
        pos = w->core.height - w->fader.th.height;

      /* set target position and start the timer */
      w->fader.fadeto_pos = pos;
      w->fader.hit = 4;
      w->fader.autofadeId = XtAppAddTimeOut (w->fader.cont, 50, fade_up, (XtPointer) w);

      /* signal an "in-fade" to the Application */
      XtCallCallbacks ((Widget)w, XtNselectCallback, (XtPointer)1);
    }
}

static void StopFade (XsFaderWidget w)
{
  /* Stop running timeout */
  if (w->fader.autofadeId)
    {
      XtRemoveTimeOut(w->fader.autofadeId);
      w->fader.autofadeId = 0;
    }

  w->fader.hit=0;
  w->fader.fadeto_pos = w->fader.position;
  XtCallCallbacks ((Widget)w, XtNselectCallback, (XtPointer)0);
}


static void select_fader (XsFaderWidget w, XEvent *event, char *args[], int n_args)
{
  switch (event->type)
    {
    case ButtonRelease:
      switch (event->xbutton.button)
        {
        case 3: break;
        case 2: w->fader.fast_motion=0; break;
        case 1: /* release motion */
          if (w->fader.hit==1)
            {
              w->fader.hit = 0;
              w->fader.fadeto_pos = w->fader.position;
              XtCallCallbacks ((Widget)w, XtNselectCallback, (XtPointer)0);
            }
          break;
        }
      break;


    case ButtonPress:
      switch (event->xbutton.button)
        {
        case 5:  /* wheel down */
                if (w->fader.hit==4)
                  {
                    int pos = w->fader.fadeto_pos + (w->fader.fast_motion?15:7);
                    if (pos > w->core.height - w->fader.th.height)
                      pos = w->core.height - w->fader.th.height;
                    w->fader.fadeto_pos = pos;
                  }
                else
                  StartFade (w, w->fader.position + (w->fader.fast_motion?15:7));
                break;
        case 4: /* wheel up */
                if (w->fader.hit==4)
                  {
                    int pos = w->fader.fadeto_pos - (w->fader.fast_motion?15:7);
                    if (pos < 0)
                      pos = 0;
                    w->fader.fadeto_pos = pos;
                  }
                else
                  StartFade (w, w->fader.position - (w->fader.fast_motion?15:7));
                break;

        case 3: /* start autofade */
                if (w->fader.hit==0)
                  {
                    StartFade (w, event->xbutton.y - w->fader.th.hy);
                  }
                else
                  if (w->fader.hit==4)
                    {
                      StopFade(w);
                    }
                break;

        case 2: 
                w->fader.fast_motion=1;
                break;

        case 1:
                if (w->fader.hit==4)
                  StopFade(w);

                XtCallCallbacks ((Widget)w, XtNselectCallback, (XtPointer)1);

                if (!w->fader.hit)
                  {
                    w->fader.hit = 1;
                    w->fader.position = event->xbutton.y - w->fader.th.hy;
                    XsFaderSetThumb (w);
                    FaderSendChanged (w, event);
                  }
                break;
        }
      break;

    case MotionNotify:
      if (w->fader.hit==1)
        {
          w->fader.position = event->xbutton.y - w->fader.th.hy;
          w->fader.fadeto_pos = w->fader.position;
          XsFaderSetThumb (w);
          FaderSendChanged (w, event);
        }
    }
}


static void FaderSendChanged (XsFaderWidget w, XEvent *event) 
{
  XsfaderCallbackStruct  cb;
  int max = w->core.height - w->fader.th.height;

  cb.event    = event;
  cb.position = w->fader.position;
  cb.val      = (float)(max-cb.position) / (float)max;
  if (cb.val < 0.0) cb.val = 0.0;
  if (cb.val > 1.0) cb.val = 1.0;
  XtCallCallbacks ((Widget)w, XtNchangedCallback, &cb);
}
