/*
 * dmxpanel.c
 * Copyright (C) Michael Stickel <michael@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <Fader.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <getopt.h>
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <dmx/dmx.h>

/******************************/

#define MAX_CHANNELS 512

static int   dmxPort = -1;
static dmx_t dmx[MAX_CHANNELS];

int GetDMX (int dmxbase, unsigned char *dmx, int num)
{
  if (!dmx || num<=0)
    return -1;
  lseek (dmxPort, dmxbase, SEEK_SET);
  read (dmxPort, dmx, num);
  return 0;
}

static void CloseDMX ()
{
  if (dmxPort>=0)
    close (dmxPort);
  dmxPort=-1;
}

int OpenDMX (const char *devName)
{
  dmxPort = open (devName, O_WRONLY);
  if (dmxPort>=0)
    {
      GetDMX (0, dmx, MAX_CHANNELS);
      atexit (CloseDMX);
      return 1;
    }
  return 0;
}

void SetDMX (int nr, unsigned char val)
{
  if (nr>=0 && nr<MAX_CHANNELS && val!=dmx[nr])
    {
      dmx[nr] = val;
      if (dmxPort>=0)
	{
	  lseek (dmxPort, nr, SEEK_SET);
	  write (dmxPort, &val, 1);
	}
    }
}

int GetDMXChannel (int channel)
{
  unsigned char v=0;
  if (channel>=MAX_CHANNELS || channel<0)
    return -1;
  lseek (dmxPort, channel, SEEK_SET);
  read (dmxPort, &v, 1);
  return v;
}

/******************************/

#define COLOR(r,g,b) ((r<<16)|(g<<8)|b)

#define BORDER(a)     XtNhSpace, (a), XtNvSpace, (a)
#define BORDERSIZE(a) XtNborderWidth,(a)
#define BORDER0       BORDER(0), BORDERSIZE(0)

#define FADER_SIZE XtNwidth,17,XtNheight,180
#define LABEL_SIZE XtNwidth,22,XtNheight,14

#define VERTICAL   XtNorientation, XtorientVertical
#define HORIZONTAL XtNorientation, XtorientHorizontal

int COLS=25;
int ROWS=2;
int KANALZAHL=0;

struct _Fader
{
  int     nr;
  Widget  box;
  Widget  label;
  Widget  fader;
  Widget  value;
};
typedef struct _Fader Fader;

static int *chanvals;
static float master_val = 1.0;
Widget master_label;
int offset=0;

char num[5];

void exit_proc (Widget w, XtPointer client_data, XtPointer call_data)
{
  exit(0);
}

void CHECK(void *p)
{
  if(p==NULL)
    {
      fprintf(stderr, "could not create widget\n");
      exit(1);
    }
}

void master_callback (Widget w, XtPointer client_data, XtPointer call_data)
{
  XsfaderCallbackStruct *cb = (XsfaderCallbackStruct *)call_data;
  int i;
  master_val = cb->val;

  i = (int)(master_val*100.0f);
  if (i>100) i=100;
  if (i<0) i=0;
  sprintf (num, (i==100)?"FL":"%02d", i);
  XtVaSetValues (master_label, XtNlabel, num, NULL);

  for (i=0; i<KANALZAHL; i++)
    SetDMX (i, master_val * chanvals[i]);
}

static void select_callback (Widget w, XtPointer client_data, XtPointer call_data)
{
  XsfaderCallbackStruct *cb = (XsfaderCallbackStruct *)call_data;
  Fader *f  = (Fader *)client_data;
  const int nr = f->nr;
  if (nr>=0 && nr < KANALZAHL)
    {
      int val = (int)(cb->val*100.0f);
      if (val>100) val=100;
      if (val<0) val=0;
      sprintf (num, (val==100)?"FL":"%02d", val);
      XtVaSetValues (f->value, XtNlabel, num, NULL);

      chanvals[nr]=cb->val * 255;
      SetDMX (nr, chanvals[nr] * master_val);
    }
}

int main (int argc, char **argv)
{
  int i,x,y;
  const char *arg;

  Widget channel_frame;
  Widget frame;
  Widget master;
  Widget master_frame;
  Widget toplevel;
  Widget wexit;

  Pixel backgroundcolor=COLOR(0xdd, 0xdd, 0xe0);

  if (!OpenDMX (DMXdev(&argc, (const char **)argv)))
    printf ("cannot open dmx-port\n");

  if(DMXgetarg(&argc, (const char **)argv, "--cols", 1, &arg)==1)
    COLS=atoi(arg);
  if(COLS<1)
    COLS=1;

  if(DMXgetarg(&argc, (const char **)argv, "--rows", 1, &arg)==1)
    ROWS=atoi(arg);
  if(ROWS<1)
    ROWS=1;

  if(DMXgetarg(&argc, (const char **)argv, "--offset", 1, &arg)==1)
    offset=atoi(arg);
  if(offset<0)
    offset=0;

  KANALZAHL=ROWS*COLS + offset;
  chanvals=calloc(KANALZAHL, sizeof(int));
  CHECK(chanvals);

  for (i=0; i<KANALZAHL; i++)
    chanvals[i] = GetDMXChannel(i);

  toplevel = XtInitialize ("DMXPanel", "DMXPanel", NULL, 0, &argc, argv);
  CHECK(toplevel);

  frame = XtVaCreateManagedWidget ("frame", boxWidgetClass, toplevel,
				   HORIZONTAL,
				   BORDER0,
				   XtNbackground, backgroundcolor,
				   NULL);
  CHECK(frame);

  /* Left Frame */
  channel_frame = XtVaCreateManagedWidget ("ch_frame", boxWidgetClass, frame,
					   VERTICAL,
					   BORDER0,
					   XtNbackground, backgroundcolor,
					   NULL);
  CHECK(channel_frame);

  i=offset;
  for(y=0; y<ROWS; y++)
    {
      Widget hbox=XtVaCreateManagedWidget ("Row", boxWidgetClass, channel_frame,
					   XtNorientation, XtorientHorizontal,
					   BORDER0,
					   XtNbackground, backgroundcolor,
					   NULL);
      CHECK(hbox);
      for(x=0; x<COLS; x++, i++)
	{
	  Fader *f=(Fader *)malloc(sizeof(Fader));
	  CHECK(f);
	  f->nr=i;

	  f->box=XtVaCreateManagedWidget ("Col", boxWidgetClass, hbox,
					  BORDERSIZE(1),
					  BORDER(1),
					  XtNbackground, backgroundcolor,
					  NULL);
	  CHECK(f->box);

	  sprintf(num, "%02d", i+1);
          f->label = XtVaCreateManagedWidget ("Label", labelWidgetClass, f->box,
					      XtNlabel, num,
					      LABEL_SIZE,
					      BORDER0,
					      XtNbackground, backgroundcolor,
					      NULL);
	  CHECK(f->label);

	  strcpy(num, "00");
	  f->value = XtVaCreateManagedWidget ("Value", labelWidgetClass, f->box,
					      XtNlabel, num,
					      LABEL_SIZE,
					      BORDER0,
					      XtNbackground, backgroundcolor,
					      NULL);
	  CHECK(f->value);

	  f->fader = XtVaCreateManagedWidget ("fader", XsfaderWidgetClass, f->box,
					      FADER_SIZE,
					      BORDER0,
					      XtNbackground, backgroundcolor,
					      NULL);
	  CHECK(f->fader);
	  XtAddCallback (f->fader, XtNchangedCallback, select_callback, f);
	}
    }

  /* Right Frame */
  master_frame = XtVaCreateManagedWidget ("mframe", boxWidgetClass, frame,
					  BORDERSIZE(1),
					  BORDER(1),
					  XtNbackground, backgroundcolor,
					  NULL );
  CHECK(master_frame);

  wexit = XtVaCreateManagedWidget ("exit", commandWidgetClass, master_frame,
				   XtNlabel, "Quit",
				   BORDER(0),
				   XtNbackground, backgroundcolor,
				   NULL);
  CHECK(wexit);
  XtAddCallback (wexit, XtNcallback, exit_proc, NULL);

  strcpy(num, "FL");
  master_label = XtVaCreateManagedWidget ("Label", labelWidgetClass, master_frame,
					  XtNlabel, num,
					  BORDER0,
					  XtNbackground, backgroundcolor,
					  NULL);

  master = XtVaCreateManagedWidget ("master", XsfaderWidgetClass, master_frame,
				    FADER_SIZE+((ROWS>1)?224:10),
				    XtNidentColor, "blue",
				    BORDER0,
				    XtNbackground, backgroundcolor,
				    NULL);
  CHECK(master);
  XtAddCallback (master, XtNchangedCallback, master_callback, NULL);
  XtVaSetValues (master, XtNposition, 0, NULL);

  /* mainloop */
  XtRealizeWidget (toplevel);
  XtMainLoop ();
  return 0;
}
