/*
 * Detect a Library for hardware detection
 *
 * Copyright (C) 1998-2000 MandrakeSoft
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/stat.h>
#include <linux/unistd.h>
#include <dirent.h>

#include "discover.h"
#include "utils.h"

#define CDROM_IDE_ID "cdrom"
#define FLOPPY_IDE_ID "floppy"
#define DISK_IDE_ID "disk"
#define TAPE_IDE_ID "tape"


extern struct identf_info *ideinterface_detect(struct bus_lst *bus){
  struct pci_info *pci = (struct pci_info *)NULL;
  struct isa_info *isa = (struct isa_info *)NULL;
  struct pcmcia_info *pcmcia = (struct pcmcia_info *)NULL;
  static struct identf_info *first = (struct identf_info *)NULL;
  struct identf_info *result = (struct identf_info *)NULL;

  if(first){
    return first;
  }/*endif*/
  
  if(debug){
    fprintf(stdout, "\nProbing IDE interface...\n");
  }/*endif*/
  /********************************************************************/
  /********************* PCI IDE-INTERFACE DETECTION ******************/
  /********************************************************************/
  if(debug){
    fprintf(stdout, "\tProbing PCI interface...\n");
  }/*endif*/
  if(!kernel_verif(2, 1, 120)){
    for(pci = bus->pci; pci; pci = pci->next){
      if(pci->type == IDEINTERFACE){
        if(!first){
          first = result = (struct identf_info *) 
                                  my_malloc(sizeof(struct identf_info));
        }else {
          result->next = (struct identf_info *) 
                                  my_malloc(sizeof(struct identf_info));
          result = result->next;
        }/*endif*/
        result->next = (struct identf_info *)NULL;
        
        result->vendor = pci->vendor;
        result->model = pci->model;
        result->module = pci->modulename;
        result->bus = PCI;
        result->long_id = pci->id;
        if(debug){
          fprintf(stdout, "\t\tFound %s %s\n", 
                                         result->vendor, result->model);
        }/*endif*/
      }/*endif*/
    }/*next pci*/
  }/*endif*/

  /********************************************************************/
  /********************* ISA IDE-INTERFACE DETECTION ******************/
  /********************************************************************/
  if(debug){
    fprintf(stdout, "\tProbing ISA interface...\n");
  }/*endif*/
  for(isa = bus->isa; isa; isa = isa->next){
    if(isa->type == IDEINTERFACE){
      if(!first){
        first = result = (struct identf_info *)
                                  my_malloc(sizeof(struct identf_info));
      }else{
        result->next = (struct identf_info *)
                                  my_malloc(sizeof(struct identf_info));
        result = result->next;
      }/*endif*/
      result->next = (struct identf_info *)NULL;
      
      result->board_num = isa->board_num;
      result->board_id = isa->board_id;
      result->dev_num = isa->dev_num;
      result->dev_id = isa->dev_id;
      result->vendor = isa->vendor;
      result->model = isa->model;
      result->module = isa->modulename;
      result->bus = ISA;
      result->io = isa->io;
      result->irq = isa->irq;
      if(debug){
        fprintf(stdout, "\t\tFound %s %s\n", 
                                         result->vendor, result->model);
     }/*endif*/
   }/*endif*/
  }/*next isa*/

  /********************************************************************/
  /******************* PCMCIA IDE-INTERFACE DETECTION *****************/
  /********************************************************************/
  if(debug){
    fprintf(stdout, "\tProbing PCMCIA interface...\n");
  }/*endif*/
  for(pcmcia = bus->pcmcia; pcmcia; pcmcia = pcmcia->next){
    if(pcmcia->type == IDEINTERFACE){
      if(!first){
        first = result = (struct identf_info *) 
                                  my_malloc(sizeof(struct identf_info));
      }else{
        result->next = (struct identf_info *)
                                  my_malloc(sizeof(struct identf_info));
        result = result->next;
      }/*endif*/
      result->next = (struct identf_info *)NULL;
      
      result->vendor = pcmcia->vendor;
      result->model = pcmcia->model;
      result->module = pcmcia->modulename;
      result->long_id = pcmcia->id;
      result->bus = PCMCIA;
      if(debug){
        fprintf(stdout, "\t\tFound %s %s\n", 
                                         result->vendor, result->model);
      }/*endif*/
    }/*endif*/
  }/*next pcmcia*/
  return first;
}/*endfunc ideinterfce_detect*/


extern struct ide_info *ide_detect(void){
  char path[256];
  char *line;
  int cylinders;
  short heads, sectors;
  char *ide_link;
  FILE *f;
  struct stat state;
  int length;                    /* To compute length of strings      */
  int nb_disk;                   /* number of disk                    */
  size_t len = 0;
  DIR *ideDir;                   /* a pointer to a directory          */
  struct ide_info *first;        /* "first" disk in the list          */
  struct ide_info *result;       /* current disk                      */
  struct dirent *ide_link_entry; /* ide_link is a possible link to    */
                                 /* the hdx disk                      */

  ideDir = opendir(PATH_PROC_IDE);
  if(ideDir == NULL){
    return NULL;
  }/*endif*/

  first = result = (struct ide_info *)NULL;
  nb_disk = 0;

  /* For each link in the PATH_PROC_IDE */
  for(ide_link_entry = readdir(ideDir); ide_link_entry;
      ide_link_entry = readdir(ideDir)){
    ide_link = ide_link_entry->d_name;
    if(strncmp(ide_link, "hd", 2) == 0){
      /* Make test for this hdx possible disk */
      sprintf(path,"%s/%s",PATH_PROC_IDE,ide_link);
      if(!lstat(path, &state)){
        /*
         * Not so fast!  Make sure that the device isn't being
         * handled by the ide-scsi driver; otherwise, the device
         * might show up twice (e.g. CD burners with CD-ROM
         * detection).
         *       - Jeff Licquia, Progeny (jlicquia@progeny.com) -
         */
        sprintf(path, "%s/%s/driver", PATH_PROC_IDE, ide_link);
        if (!(f = fopen(path, "r"))) {
          return NULL;
        }/*endif*/
        getline(&line, &len, f);
        fclose(f);
        line[8] = 0x00;
        if (!strcmp(line, "ide-scsi")) {
          continue;
        }/*endif*/

        /* one more */
        nb_disk++;
        if(result){
          result->next = (struct ide_info *)
                                     my_malloc(sizeof(struct ide_info));
          result = result->next;
        }else{
          first = result = (struct ide_info *)
                                     my_malloc(sizeof(struct ide_info));
        }/*endif*/
        result->next = (struct ide_info *)NULL;
        
        /* Drive name */
        length = strlen(ide_link) + strlen("/dev/") + 1;
        result->device = (char *)my_malloc(length);
        sprintf(result->device,"/dev/%s",ide_link);
        
        /* Drive model */
        sprintf(path, "%s/%s/model", PATH_PROC_IDE,ide_link);
        if(!(f = fopen(path, "r"))){
          return NULL;
        }/*endif*/
        getline(&line, &len, f);
        fclose(f);
        result->model = (char *)my_malloc(strlen(line));
        result->model[0] = 0x00;
        strncat(result->model, line, strlen(line) - 1);
        
        /* Determine the drive type (floppy, hd...) */
        sprintf(path, "%s/%s/media", PATH_PROC_IDE,ide_link);
        if (!(f = fopen(path, "r"))){
          return NULL;
        }/*endif*/
        getline(&line, &len, f);
        fclose(f);
        if (strstr(line, CDROM_IDE_ID)){
          result->type = CDROM;
        }else if(strstr(line, FLOPPY_IDE_ID)){
          result->type = FLOPPY;
        }else if(strstr(line, TAPE_IDE_ID)){
          result->type = TAPE;
        }else if(strstr(line, DISK_IDE_ID)){ 
          result->type = DISK; 
        }else{
            result->type = UNKNOWN_DEVICE;
        }/*endif*/
        
        /* Geometry for DISK */
        if(result->type == DISK){
          sprintf(path, "%s/%s/geometry", PATH_PROC_IDE,ide_link);
          if (!(f = fopen(path, "r"))){
            return NULL;
          }/*endif*/
          getline(&line, &len, f);  /* Physical geometry              */
          getline(&line, &len, f);  /* Logical geometry               */
          sscanf(line, "%*s%d/%hd/%hd", (int *)&cylinders, 
                 (short *)&heads, (short *)&sectors);
          result->cylinders = cylinders;
          result->heads = heads;
          result->sectors = sectors;
          fclose(f);
        }/*endif*/
      }/*endif*/
    }/*endif*/
  }/*next ide_link_entry*/
  return first;
}/*endfunc ide_detect*/
