/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_suite_traversal.hpp
 *
 * @brief [LEVEL: beta] @ref
 * diagnostics::unittest::Test_Suite_Traversal middle-class
 *
 * $Id: test_suite_traversal.hpp,v 1.18 2005/06/23 09:54:27 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/test_system/test_suite_traversal.t.cpp
 */
#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_SUITE_TRAVERSAL_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_SUITE_TRAVERSAL_HPP__INCLUDE_GUARD

#include <diagnostics/unittest/namespace.hpp>

// used in the interface by value
#include <diagnostics/frame/level.hpp>

// used in the interface by reference
#include <string>

// used in the interface by reference
#include <vector>


DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

// used in the interface by reference
// <diagnostics/unittest/test_suite.hpp>
class Test_Suite;

// used in the interface by reference
// <diagnostics/unittest/test_case.hpp>
class Test_Case;

/**
 * @class Test_Suite_Traversal diagnostics/unittest/test_system/test_suite_traversal.hpp
 *
 * @brief Provides a Traversal through a @ref Test_Suite, whereby it
 * is possible to select Test_Suites and Test_Cases which are going to
 * be visited by a string mask and the build-level they are supposed
 * to work at (LEVEL_TEST matches all). Concrete Traversals must
 * implement the hook methods.
 *
 * The traversal order is as follows:
 * @arg visit_hook /S1
 * @arg visit_hook /S1/C1
 * @arg visit_hook /S1/S2
 * @arg visit_hook /S1/S2/C1
 * @arg visit_hook /S1/S2/C2
 * @arg leave_hook /S1/S2
 * @arg visit_hook /S1/S3
 * @arg visit_hook /S1/S3/C1
 * @arg visit_hook /S1/S3/C2
 * @arg leave_hook /S1/S3
 * @arg leave_hook /S1
 * where S* are suites and C* are cases -- and the suites and cases have
 * been ADDED in this ORDER, i.e., S2 before S3, the C1s before C2s.
 * 
 * In general, the mask is of the form
 * /X_1/X_2/.../X_n. Each X_i is a string, if X_i=="*" then it matches
 * any string. If X_i!="*" then it has to match exactly the name of
 * the corresponding suite or case. If X_n=="*" then whole
 * corresponding subtree is matched. For example, the mask /A (we use
 * A for * to avoid problems with the C++ preprocessor) matches
 * everything. The mask /A/S3/A would give the following trace:
 * @arg visit_hook /S1
 * @arg visit_hook /S1/S3
 * @arg visit_hook /S1/S3/C1
 * @arg visit_hook /S1/S3/C2
 * @arg leave_hook /S1/S3
 * @arg leave_hook /S1
 *
 * @note There exist nameless Test_Suites. However, they can only be
 * the root of a Test_Suite-Tree. In such a case, the first element in
 * the path is an empty string.
 *
 * @nosubgrouping
 */
class Test_Suite_Traversal
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Test_Suite_Traversal Self;
public:
    typedef ::std::vector< ::std::string> Path_t;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief do nothing
     *
     * @throw never
     */
    virtual ~Test_Suite_Traversal();
protected:
    /**
     * @brief Only used as a base class.
     *
     * @throw never
     */
    Test_Suite_Traversal();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Modifiers
     * @{
     */
public:
    /**
     * @brief Traverse the @a test_suite and visit all Test_Suites and Test_Cases which
     * have a path that matches the @a mask and which are testable at @a level.
     *
     * @pre @ref ::diagnostics::mask_adheres_convention (mask)==true
     *
     * @throw anything (because of the called hooks)
     * @throw Test_System_Exception if the precondition is not met.
     */
    void traverse(Test_Suite const & test_suite,
		  ::std::string const & mask,
		  Level_t const level);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Hooks
     * @{
     */
protected:
    /**
     * @brief Called direclty after entering @ref traverse [default:
     * noop]
     *
     * The parameters are excatly the same as in for traverse -- they
     * are handed through.
     *
     * @throw anything
     */
    virtual void enter_hook(Test_Suite const & test_suite,
			    ::std::string const & mask,
			    Level_t const level);

    /**
     * @brief Called as last hook of a @ref traverse invocation
     * [default: noop]
     *
     * The parameters are excatly the same as in for traverse -- they
     * are handed through. @a exception_going_through==true if
     * exit_hook is called from an exception handler.
     *
     * @throw anything
     */
    virtual void exit_hook(Test_Suite const & test_suite,
			   ::std::string const & mask,
			   Level_t const level,
			   bool const exception_going_through);
	
    /**
     * @brief Called when a @ref Test_Suite is entered for the first
     * time [default: noop]
     * 
     * No hook for any @ref Test_Suite or @ref Test_Case which is
     * contained in the currently visited Test_Suite has been called
     * before.
     * 
     * A @ref Test_Suite is visited iff its path is matched by the
     * mask.
     * 
     * @param test_suite the visited @ref Test_Suite 
     * @param path the names of the Test_Suites along the path to the
     * current one (does not include the name of the current
     * Test_Suite).
     * @param level the build-level we are searching for
     *
     * @throw anything
     */
    virtual void visit_hook(Test_Suite const & test_suite,
			    Path_t const & path,
			    Level_t const level);

    /**
     * @brief Called when a @ref Test_Suite is completely processed
     * [default: noop]
     * 
     * The hooks for every @ref Test_Suite or @ref Test_Case which are
     * contained in the currently visited Test_Suite have been called
     * before.
     * 
     * The meaning and values of the parameters match exactly the
     * parameters of the corresponding @ref visit_hook invocation.
     *
     * @throw anything
     */
    virtual void leave_hook(Test_Suite const & test_suite,
			    Path_t const & path,
			    Level_t const level);
	
    /**
     * @brief Called when a @ref Test_Case is visited [default: noop]
     *
     * A @ref Test_Case is only visited iff
     * @arg @a level == @ref LEVEL_TEST or if
     * test_case.is_testable_at(level) == true.
     * @arg its path is matched by the mask
     *
     * @param test_case the visited @ref Test_Case 
     * @param path the names of the Test_Suites along the path to the
     * current one (does not include the name of the current
     * Test_Case).
     * @param level the build-level we are searching for
     *
     * @throw anything
     */
    virtual void visit_hook(Test_Case const & test_case,
			    Path_t const & path,
			    Level_t const level);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated Modifiers 
     * @{
     */
private:
    /**
     * @brief traverese a @ref Test_Suite -- decide whether to go on
     * recursively, and whether to visit
     *
     * @throw anything
     */
    void p_traverse(Test_Suite const & test_suite,
		    ::std::string mask,
		    Path_t & path,
		    Level_t const level);
    /**
     * @brief traverse a @ref Test_Case -- decide whether to visit
     *
     * @throw anything
     */
    void p_traverse(Test_Case const & test_case,
		    ::std::string const & mask,
		    Path_t const & path,
		    Level_t const level);
    // @}
};


UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
