/* datastructures.h
 *
 * the important data structures and defines for denemo, a gtk+ frontend to
 * Lilypond, the GNU music typesetter
 *
 * (c) 1999-2005 Matthew Hiller, Adam Tee
 *
 */

#ifndef DENEMO_DATASTRUCTURES
#define DENEMO_DATASTRUCTURES
#ifdef __cplusplus
extern "C" {
#endif

/* Include the internationalization stuff.  */
/* Only if not on Cygwin, because libintl does not work with mingw */
/* G_OS_WIN32 is defined in gtk.h for mingw */


#ifndef G_OS_WIN32
# include <libintl.h>
# include <locale.h>
# define _(String) gettext (String)
# ifdef gettext_noop
#  define N_(String) gettext_noop (String)
# else
#  define N_(String) (String)
# endif
#else
# define _(String) (String)
# define N_(String) (String)
# define setlocale(x,y)
# define bindtextdomain(x,y)
# define textdomain(x)
#endif

#include "denemo_objects.h"
#include <glib.h>
#include <gmodule.h>

#if GTK_MAJOR_VERSION <= 1
#define GtkTextBuffer GtkWidget
#define g_string_append_printf g_string_sprintfa
#endif

typedef void (*GActionCallback) (GtkAction *action, gpointer data);
#define G_ACTIONCALLBACK(f) ((GActionCallback)(f))

#define MAX_UNDOS 50
/* This unit of memory size is used all over the place.  */

#define SEVENGINTS (sizeof (gint) * 7)

/* and the following typedefs are basically here for so that it's
 * possible to understand what my code is doing -- just as much for
 * my sake as yours!
 *
 * What can I say; I've done a lot of programming in Java and
 * SML/NJ; 
 * I like my type names to convey information. */

/* The ->data part of each objnode presently points to a mudelaobject */

typedef GList objnode;


/* The idea here is to make everything recursive.  The dominant
   paradigm is a linked list.  Handy that there's such a nice
   precooked implementation of them in glib, eh?  Note, however, that
   the whole score isn't treated as a linked list of notes and other
   objects as it is in, say Rosegarden; instead, the program has a
   linked list of musical objects for each measure, and only then are
   measures linked into staffs.  That's my main beef with Rosegarden
   -- I don't tend to compose my stuff in the order in which it will
   eventually be played. As such, I like being able to start entering
   music in the middle of my score, then the beginning, then the end,
   or whatever, as appropriate.  */

enum objtype
{
  CHORD,
  TUPOPEN,
  TUPCLOSE,
  CLEF,
  TIMESIG,
  KEYSIG,
  BARLINE,
 // LILYDIRECTIVE,
 // COMMENT,
  STEMDIRECTIVE,
  MEASUREBREAK,
  DYNAMIC,
  GRACE_START,
  GRACE_END,
  LYRIC,
  FIGURE
};


typedef enum tuplet_type {
	DUPLET,
	TRIPLET,
	QUADTUPLET,
	QUINTUPLET,
	SEXTUPLET,
	SEPTUPLET	
}tuplet_type;

enum input_mode {
  INPUTNORMAL,
  INPUTREST,
  INPUTBLANK,
  REPLACE,
  TRAVERSE
};

enum action_type {
  ACTION_INSERT,
  ACTION_DELETE,
  ACTION_CHANGE
};

/**
 * Contains all the top-level information of an musical object
 * the object pointer contains the actual object
 */
typedef struct mudelaobject
{
  enum objtype type;
  gchar *user_string;/* holds user's original text parsed to generated this 
			object */
  gint basic_durinticks;
  gint durinticks;
  gint starttick;
  gint starttickofnextnote;
  /* Allots extra space for accidentals or reverse-aligned notes if
   * the stem is down */
  gint space_before;
  gint minpixelsalloted;
  /* Holds x co-ordinate relative to the beginning of the measure. */
  gint x;
  gboolean isstart_beamgroup;
  gboolean isend_beamgroup;
  /* mudelaobjunion u; has become a pointer*/
  gpointer object;
  gboolean isinvisible;
}
mudelaobject;

typedef enum denemocontext 
{
  DENEMO_NONE=1,
  DENEMO_PIANO,
  DENEMO_GROUP,
  DENEMO_CHOIR
}denemocontext;
/* The ->data part of each measurenode points to an objlist, which is
   a list of the musical objects in that measure. */

typedef GList measurenode;

typedef struct staff
{
  measurenode *measures;
  gint sclef;
  gint skey;
  gint skey_isminor;  //25112005 Changed use to incorporate modes values 0 - major, 1 - minor, 2 - mode
  gint skeyaccs[7];
  gint stime1;
  gint stime2;
  gint leftmost_clefcontext;
  gint leftmost_keysigcontext;
  gint leftmost_keywidth;
  gint leftmost_time1context;
  gint leftmost_time2context;
  gint leftmost_keyaccs[7];
  gint leftmost_stem_directive;
  /*New variable to define the context in which this staff appears*/
  denemocontext context; 
  /*
   * Staff Parameters
   * Added Adam Tee 27/1/2000, 2001 
   */
  gint no_of_lines;
  gint transposition;
  gint pos_in_half_lines;
  gint volume;
  /* Back to Hiller stuff */
  GString *staff_name;
  /* RTS: I've introduced the staff name here, the other two are versions
     of the voice name; however I'm still sticking to the unwritten convention
     that each staff's voices are contiguous in si->thescore. Without this you
     can't have same named voices in different staffs. */
  GString *denemo_name;
  GString *lily_name;
  GString *midi_instrument;

  gint space_above;
  gint space_below;
  gint voicenumber;
  /* presently set to 2 for any non-primary voices; we might want to
   * change that, though */
  measurenode ** is_parasite; /* points to address of host staff's measures 
				 field if measures are owned by another 
				 staff */
}
staff;

/* The ->data part of each staffnode points to a staff structure */

typedef GList staffnode;
typedef staffnode *score;

/* a pair of staffs, used to relate two staffs together */
typedef struct staff_info
{
  staff *main_staff; /* eg the bass line or the vocal part */
  staff *related_staff; /* eg the figures for the bass or the lyrics*/
}
staff_info;


/* This structure is what's stored in the keymap data structure, both
   in the array and in the hash tables.  Note that some redundant
   information is stored in either case, but either one structure or
   the other will need or find it useful to have each piece of information. */

union func_union
{
  GtkFunction nocallback;
  GActionCallback callback;
};

typedef struct keybinding_info
{
  gint keyval;
  gint state;
  gint command_number;
  gint callback_action;
  union func_union func;
}keybinding_info;

typedef struct keymap
{
  GList **commands;
  GHashTable *quick_lookup_hashes;
}keymap;


#define MAX_HISTORY 10
/**
 * prefinfo holds information on user preferences. 
 */
typedef struct prefinfo
{
  GString *lilypath;
  GString *midiplayer;
  GString *audioplayer;
  gboolean playbackoutput;
  gboolean immediateplayback;
  gboolean lilyentrystyle;
  gboolean createclones;
  gboolean articulation_palette;
  gboolean notation_palette;
  /*Auto save data*/
  gboolean autosave;
  gint autosave_timeout;
  GString *browser;
  GString *csoundcommand;
  GString *csoundorcfile;
  gboolean rtcs;
  GString *pdfviewer;
  GString *texteditor;
  GQueue *history; //Queue to contain recently opened files
  keymap *the_keymap;
}prefinfo;

/**
 * Contains all the lilypond header information
 *
 */
typedef struct mudelaheaderinfo {
/* Mudela file header information */
  GString *title;
  GString *subtitle;
  GString *poet;
  GString *composer;
  GString *meter;
  GString *opus;
  GString *arranger;
  GString *instrument;
  GString *dedication;
  GString *piece;
  GString *head;
  GString *copyright;
  GString *footer;
  GString *tagline;
/*  GString *; */

}mudelaheaderinfo;

typedef enum InstrumentType
{
  WOODWIND,
  BRASS,
  STRINGS,
  VOCALS,
  PITCHEDPERCUSSION,
  PLUCKEDSTRINGS,
  KEYBOARDS,
  NONE
}InstrumentType;

/**
 * Structure to contain the list of Instruments for the score
 * configuration wizard
 *
 */
typedef struct InstrumentConfig
{
	GString *name;
	GString *midiinstrument;
	gint sclef;
	gint transposition;
	gint numstaffs;

}InstrumentConfig;

/**
 * Stores global instrument type and a list of InstrumentConfig structures
 */
typedef struct InstrumentList
{
	InstrumentType type;
	GList *instruments;  // List to contain a list of Instruments of given type
}InstrumentList;

/**
 * Contains data required for undo operation 
 * Borrowed idea from GScore
 */
typedef struct undo_data
{
  gpointer object;    /* pointer to object deleted */
  gint staffnum;      /* staff number */     
  gint measurenum;    /* measure number */
  gint position;      /* position in bar */
  enum action_type action; /*action type */

}undo_data;
 
/**
 * Contains data required for redo operation 
 * contents as per undo_data
 */
typedef struct redo_data
{
  gpointer object;
  gint staffnum;
  gint measurenum;
  gint position;
  enum action_type action;
}redo_data;


/**
 * Structure to hold bookmark information
 * Id - gint
 * Bar - gint
 * Staff - gint
 */
typedef struct bookmark
{
  gint id;
  gint bar;
  gint staff;
}bookmark;

/** 
 * Contains paper score configuration
 *
 */
typedef struct scoreconfig 
{
	GString *papersize;
	gint fontsize;
	GString	*lilyversion;
	gboolean orientation;
	
}scoreconfig;

/**
 * struct scoreinfo is actually a somewhat misleading name.  It contains
 * within it a struct prefinfo *, and when Denemo supports editing
 * multiple scores simultaneously, each score's scoreinfo structure will
 * contain a pointer to the same single prefinfo structure.
 *
 * I originally hit on this idea as a hack -- everything in Denemo was
 * coded to pass around struct scoreinfo *s, and I didn't want to
 * change this around -- but then I realized that it was a reasonably
 * good idea.  Basically all operations need access to the scoreinfo
 * structure, and a handful need access to the prefinfo structure as
 * well. Only the preferences dialog needs access to _just_ the
 * prefinfo structure.  So, why not keep things simple and always pass
 * around scoreinfos?
  */

typedef struct scoreinfo
{
	/*Indicator for open file readonly*/
  gint readonly;
  /* Fields used fairly directly for drawing */
  GtkWidget *window;
  GtkWidget *scorearea;
  GdkPixmap *pixmap;
  GtkObject *vadjustment;
  GtkWidget *vscrollbar;
  GtkObject *hadjustment;
  GtkWidget *hscrollbar;
  GtkWidget *menubar;
  GtkWidget *statusbar;
  /* text of lily input string corresponding to curobj */
  GtkTextBuffer *textbuffer; 
  GtkWidget *textwindow; /* top level window containing lily textwindow */
  GtkWidget *textview; /* text entry containing lily text to be edited */
  GtkWidget *musicdatabutton; /* button for editing music data textually */
  gulong sigid; /* signal handler id for when text is changed */

  GtkWidget *musictext; /* status bar with text of lily input string 
			   corresponding to music item at cursor */
  GList *curlilynode; /* the node of the lily parse tree on display 
			 in textwindow */
  GList *lily_file; /* root of lily file parse, see lilyparser.y etc  */
	scoreconfig *config; //Contains the paper configuration of the score
  gint status_context_id;
  gint leftmeasurenum; /* start at 1 */
  gint rightmeasurenum;/* start at 1 */
  gint top_staff;
  gint bottom_staff;
  gint measurewidth;
  /* List of all minimum measure widths */
  GList *measurewidths;
  gint widthtoworkwith;
  gint staffspace;

  /* Fields that have more to do with the data model and its manipulation,
   * though they may be modified by side-effects of the drawing routines */
  score thescore;
  
  staffnode *currentprimarystaff;
  staffnode *currentstaff;
  gint currentstaffnum;
  measurenode *currentmeasure;
  gint currentmeasurenum;/* start at 1 */
  objnode *currentobject;
  /* currentobject points to the note preceding the cursor when the
   * cursor is appending. == NULL only when currentmeasure is empty. */
  gint cursor_x;
  gint cursor_y;
  gint staffletter_y;
  gint maxkeywidth;
  gboolean cursor_appending;
  enum input_mode mode;
  gboolean cursoroffend;
  gint cursorclef;
  gint cursoraccs[7];
  gint cursortime1;
  gint cursortime2;
  gint curmeasureclef;
  gint curmeasurekey;
  gint curmeasureaccs[7];
  gint nextmeasureaccs[7];
  /* These are used for determining what accidentals should be there
   * if the cursor advances to the next measure from the next "insert chord"
   * operation */
  gint curmeasure_stem_directive;

  gboolean is_grace_mode;
  /* Is there a figured bass present, is so this relates the bass
   * with its figures staff, if one is present */
  staff_info * has_figures;
  /* Now stuff that's used for marking areas */
  gint markstaffnum;
  gint markmeasurenum;
  gint markcursor_x;
  gint firststaffmarked;
  gint laststaffmarked;
  gint firstmeasuremarked;
  gint lastmeasuremarked;
  gint firstobjmarked;
  gint lastobjmarked;

  gboolean haschanged;

  /*Temp field for testing for a slur*/
  gboolean is_slured;

  /* Fields used for MIDI playback */
  gint tempo;
  gint start;
  gint end;
  gint stafftoplay;

  struct mudelaheaderinfo *headerinfo;
  
  /* Filename */
  GString *filename;
  GString *autosavename;

  /* And the prefinfo pointer */
  struct prefinfo *prefs;
  
  GList *savebuffer;
  /* Holds List of Plugin Names */
  GList *plugins;
  gint plugincounter;

  /* a list of scoreinfos for each scoreblock in the (lilypond) file
     the last one is always the one currently at the address hardwired
     into the display functions.
     NULL if just one scoreblock in file */
  GList *scoreblocks;

  /*list of undo data */
  GQueue *undodata;
  gint undo_level; /*denotes undo level */
  GQueue *redodata;
  gint redo_level;
  gint undo_redo_mode;

  /* UI manager */
  GtkUIManager *ui_manager;
  scoreconfig *sconfig;
  /* Articulation palette window */
  GtkWidget* articulation_palette;

  GtkWidget *progressbar;
  GList *bookmarks;
  
  GList *Instruments;
  
}scoreinfo;



/*Forward Declaration*/
typedef struct _PluginData PluginData;
/**
 * Structure borrowed from Gnumeric which 
 * allows us to unload a plugin.
 * This should render the denemo_plugin struct redundant.
 */
struct _PluginData {
  GModule *handle;
  void (*initialise) (struct scoreinfo *si, PluginData *pd);
  void (*clean_up)   (struct scoreinfo *si, PluginData *pd);
  gchar *title;
};


#define g_malloc(a) g_malloc0(a) /* for extensible debuggable code */

#ifdef __cplusplus
}
#endif

#endif  /* #ifndef DENEMO_DATASTRUCTURES  */
