/**
 * Tree data type.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include "parsetree.h"
#include "util.h"
#include <stdlib.h>
#include <string.h>


/** Maximum number of symbols. */
#define SYMBOLS 256


/** Destroys this parse tree. */
void
destroy_parse_tree( Tree* tree)
{
	if( tree->count == 0)
		free( (char*)tree->data);
	while( tree->count)
		destroy_tree( tree->children[--tree->count]);
	free( tree);
}


/** Prints this parse tree. */
void
print_parse_tree( const Tree* tree, const Dictionary* symbols)
{
	int i;
	if( tree == NULL)
		return;
	if( tree->data <= symbols->count) {
		printf( "(%s", get_key( symbols, tree->data));
		for( i = 0; i < tree->count; i++) {
			putchar( ' ');
			print_parse_tree( tree->children[i], symbols);
		}
		putchar( ')');
	} else
		print_escaped_string( (const char*)tree->data, " ()");
}


/** Reads a parse tree. */
Tree*
read_parse_tree( FILE* file, const Dictionary* symbols)
{
	char c = getchar();
	assumex( c != EOF, "unexpected end of input");

	if( c != '(') {
		ungetc( c, file);
		return create_node( (int)read_string( file, " \n()"));
	} else {
		Tree* tree;
		char* symbol = read_string( file, " \n()");
		int a = find( symbols, symbol);
		assumex( a != 0, "unknown symbol %s", symbol);
		free( symbol);
		tree = create_node( a);
		while( (c = getchar()) != ')') {
			assumex( c != EOF, "unexpected end of input");
			scanf( " ");
			tree->children[tree->count++] =
				read_parse_tree( file, symbols);
		}
		return tree;
	}
}


/** Prints the lexemes of this parse tree. */
void
print_lexemes( const Tree* tree)
{
	int i;
	if( tree == NULL)
		return;
	if( tree->data <= SYMBOLS) {
		for( i = 0; i < tree->count; i++)
			print_lexemes( tree->children[i]);
	} else {
		print_string( (const char*)tree->data);
		putchar( ' ');
	}
}
