/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#include <iostream>

#include "DirectoryService.hh"

// Class to be instanciatated by factory()
#include "LDAP.hh"
#include "LDAPCache.hh"
#include "ConfigDatabase.hh"

namespace Cryptonit 
{

DirectoryService::DirectoryService() 
{
  clear();
}


DirectoryService::~DirectoryService() 
{
    DirectoryService::iterator itr = DirectoryService::begin();

    while( itr != DirectoryService::end() ) {
	if( itr.second() ) {
	    delete itr.second();
	}
 	itr++;
    }
    clear();
}


DirectoryService* DirectoryService::factory( const std::string uri )
{
    // Check if we were called with an URI
    unsigned int i = uri.find("://");

    // "Simple ID" call
    if( i == std::string::npos ) {
      if( uri == "ldap" ) return new ldap();
      if( uri == "ldapcache" ) return new LDAPCache();
      if( uri == "config" ) return new ConfigDatabase();
      else if( uri == "" ) return new DirectoryService();
      else {
#ifdef DEBUG
 	std::cerr << "Invalid DirectoryService id '" << uri << "'." << std::endl;
#endif 
	return new DirectoryService();
      }
    } else { // URI call
      const std::string backend_name( uri, 0, i );
      //PRINT(backend_name);

	std::string params( uri, i+3, uri.size() - 1 );
	//PRINT(params);

	if( backend_name == "ldap" ) { 	// LDAP backend
	    return new ldap( params );
	} else if( backend_name == "ldapcache" ) { // LDAP Cache backend
	  return new LDAPCache( params );
	} else if ( backend_name == "config" ) { // Config Backend
	  return new ConfigDatabase( params );
	} else { // Unknow backend
	  return NULL;
	}
    }
}



Entry* DirectoryService::append( const std::string entry_name )
{
    if( entry_name != "" )
	return entryList[entry_name] = new Entry( entry_name );
    else return NULL;
}



bool DirectoryService::append( const std::string entry_name, Entry* entry )
{
    if( entry_name != "" ) {
	entryList[entry_name] = entry;
	return true;
    }
    else return false;
}


bool DirectoryService::append( const std::string entry_name, 
			       const std::string attribute_name, 
			       const std::string& val)
{
    if( entry_name != "" && attribute_name != "" && val != "" ) {
	if( entryList.find(entry_name) == entryList.end() ) {
	    entryList[entry_name] = new Entry( attribute_name, val );
	    return true;
	}
	else {
	    if( entryList[entry_name]->getAttribute( attribute_name ) != NULL )
		return entryList[entry_name]->getAttribute( attribute_name )->append( val );
	    else {
		entryList[entry_name]->append( attribute_name, 
					       new Attribute(val) );
		return true;
	    }
	}
    }
    else return false;
}



bool DirectoryService::append( const std::string entry_name, 
			       const std::string attribute_name, 
			       const std::string& val,
			       AttributeType type)
{
    if( entry_name != "" && attribute_name != "" && val != "" ) {
	if( entryList.find(entry_name) == entryList.end() ) {
	    entryList[entry_name] = new Entry( attribute_name, val, type );
	    return true;
	}
	else {
	    if( entryList[entry_name]->getAttribute( attribute_name ) != NULL )
		return entryList[entry_name]->getAttribute( attribute_name )->append( val, type );
	    else {
		entryList[entry_name]->append( attribute_name, 
					       new Attribute(val, type) );
		return true;
	    }
	}
    }
    else return false;
}



bool DirectoryService::append( const std::string entry_name, 
			       const std::string attribute_name, 
			       char* val,
			       unsigned long len,
			       AttributeType type)
{
    if( entry_name != "" && attribute_name != "" && val != "" ) {
	if( entryList.find(entry_name) == entryList.end() ) {
	    entryList[entry_name] = new Entry( attribute_name, val, len, type );
	    return true;
	}
	else {
	    if( entryList[entry_name]->getAttribute( attribute_name ) != NULL )
		return entryList[entry_name]->getAttribute( attribute_name )->append( val, len, type );
	    else {
		entryList[entry_name]->append( attribute_name, 
					       new Attribute(val, len, type) );
		return true;
	    }
	}
    }
    else return false;
}



bool DirectoryService::erase( const std::string name )
{
    if( name != "" && entryList.find(name) != entryList.end() ) 
	if( entryList[name]->clear() ) {
	    entryList.erase(name);
	    return true;
	}
	else return false;
    else
	return false;
}


bool DirectoryService::clear()
{
//     <std::hash_map<const std::string, Entry*,hash<const char*>, DirectoryService_eqstr>>::iterator i;

//      for( i = entryList.begin(); i != entryList.end(); i++ )
//  	i->second->clear();

    entryList.clear();

    return true;
}



Entry* DirectoryService::getEntry( const std::string name )
{
    if( name != "" && entryList.find(name) != entryList.end() )
	return entryList[name];
    else
	return NULL;
}


unsigned int DirectoryService::getNbEntry() const
{
    return entryList.size();
}


bool DirectoryService::find( const std::string entry_name,
					       Entry& entry )
{
    if( entryList.find(entry_name) == entryList.end() ) return false;

    Entry* e = entryList[entry_name];

    if( e->fillEntry( entry ) )
	return true;
    else return false;
}


} // Namespace
