import py
from py.__.rest.rst import rest, Out, itertext, RestTag
from py.__.misc import rest as pyrest

temp = py.test.ensuretemp('check_rest')
#temp = py.path.local.mkdtemp()

def check_rest(content, include_dir = None):
    " try to convert content to html "
    if isinstance(content, RestTag):
        content = content.text()
    content = unicode(content)
    print content
    tempdir = py.path.local.make_numbered_dir(rootdir=temp)
    if include_dir is None:
        include_dir = tempdir
    #logging
    tempdir.ensure('input.txt').write(content)
    try:
        output = pyrest.convert_rest_html(content, include_dir)
    except:
        fail_msg = '''
        failed to convert %s to html, probably not valid reStructuredText
        see recorded stderr for error message'''
        
        py.test.fail(fail_msg % tempdir.join('input.txt') + '\n\n' + str(py.code.ExceptionInfo()))
    tempdir.ensure('output.html').write(output)
    return True

def render_xml(arg):
    'try to generate a xml representation of arg'
    return arg.unicode()

def strip_lines(line_list):
    'filter line_list and remove trailing whitespaces'
    return [line.rstrip() for line in line_list if line.strip()]

def print_lines(line_list):
    'pretty print line_list'
    py.std.pprint.pprint(strip_lines(line_list))

class TestSplit:

    def test_empyt_string(self):
        assert list(itertext('')) == ['']

    def test_whitespace(self):
        assert list(itertext('  ')) == ['  ']

    def test_single_word(self):
        assert list(itertext('word')) == ['word']
        
    def test_word_with_whitespace(self):
        assert list(itertext('word ')) == ['word', ' ']
        
    def test_two_words(self):
        assert list(itertext('first second')) == ['first', ' ', 'second']

    def test_trailing_newline(self):
        assert list(itertext('\nfirst word')) == ['\n', 'first', ' ', 'word']
    def test_newline_and_space_are_seperated(self):
        assert list(itertext('\n third_item')) == ['\n', ' ', 'third_item']

class TestOut:
    
    def test_write_nothing(self):
        assert Out().write('').render() == ''

    def test_write_returns_self(self):
        out = Out()
        assert out.write('') is out
        assert out.write('') is not Out().write('')

    def test_write_newline(self):
        out = Out()
        out.write('\n')
        assert len(out.lines) == 2
        assert out.render() == '\n'

    def test_write_one_line(self):
        text = "'[B]ut assume that I have some other use case' isn't a valid use case. - Fredrik Lundh"
        out = Out(width=None)
        out.write(text)
        assert out.lines == [text]
        assert out.render() == text

    def test_write_and_width(self):
        text = "This sentence is 36 characters wide."
        out = Out(width = 36)
        out.write(text)
        assert len(out.lines) == 1
        out = Out(width = 35)
        out.write(text)
        assert len(out.lines) == 2

    def test_write_and_newline(self):
        text = "1234567890\n1234567890"
        out = Out(width=30)
        out.write(text)
        assert len(out.lines) == 2
        assert len(out.lines[0]) == 10
        assert out.render() == text
        out.write(text)
        assert len(out.lines) == 3

    def test_write_with_trailing_newline(self):
        text = "0123456789\n"
        out = Out()
        out.write(text)
        assert len(out.lines) == 2
        assert out.render() == text

    def test_write_long_word(self):
        text = '12345678901234567890'
        out = Out(width=19)
        out.write(text)
        assert len(out.lines) == 1
        assert text == out.render()
        out.write('1')
        assert len(out.lines) == 1
        out.write(' 2')
        assert len(out.lines) == 2

    def test_long_literal_and_newline(self):
        text = '12345678901234567890'
        out = Out(width=10)
        out.write_literal(text)
        assert len(out.lines) == 1
        text += '\n1234567890'
        out.write_literal(text)
        assert len(out.lines) == 2

    def test_append(self):
        out = Out()
        out.write('First line\n')
        out.write('Second line')

        root_out = Out()
        root_out.write('Root')
        root_out.append(out)

        assert len(root_out.lines) == 2

    def test_extend_empty_list(self):
        out = Out()
        text = out.render()
        out.extend([])
        assert text == out.render()

    def test_max_length(self):
        out = Out()
        out.write('1234567890')
        out.writeln()
        out.write('123456789')
        assert out.max_length() == 10
        

class TestRest:
    disabled = False
    def setup_method(self, method):
        self.text = {}

        self.text['paragraph'] = "Paragraphs consist of blocks of "
        "left-aligned text with no markup indicating any other body "
        "element. Blank lines separate paragraphs from each other and "
        "from other body elements. Paragraphs may contain inline markup."
       
    def test_paragraph(self):
        para = rest.paragraph(self.text['paragraph'])
        print render_xml(para)
        text = para.text()
        check_rest(text)
        assert text[0] == '\n'
        assert text[-1] == '\n'
        
    def test_paragraph_with_whitespaces(self):
        phrase =  "Perhaps if we wrote                 programs from "
        "childhood on, as                 adults                     "
        "               we'd be able to read them."
        para = rest.paragraph(phrase)
        check_rest(para)
        assert True not in [line.startswith(' ') for line in para.text().splitlines()]
        
    def test_emph(self):
        emph = rest.emph('strong')
        assert emph.text() == '*strong*'
        assert check_rest(emph.text())

    def test_paragraph_adds_whitespace(self):
        para = rest.paragraph('Starttext', rest.emph('EMPHASIS'), 'endtext')
        assert para.text() == para.sep +'Starttext *EMPHASIS* endtext'+ para.sep

    def test_nested_emph(self):
        "Nested Inline Markup not allowed in ReST, don't try at home ;-)"
        emph = rest.emph('start', rest.emph('middle'), 'end')
        check_rest(emph.text())
        assert emph.text() == '*start *middle* end*'
    
    def test_strongemph(self):
        phrase = 'failure is not an option'
        emph = rest.strongemph(phrase)
        assert emph.text() == '**' + phrase + '**'

    def test_title(self):
        phrase = 'Everything should be built top-down, except the first time.'
        title = rest.title(phrase)
        expected = title.sep + title.start_string * len(phrase) \
                   +'\n' + phrase + '\n' + title.end_string *len(phrase)\
                   + '\n' + title.sep
        assert title.text() == expected
        check_rest(title.text())


    def test_list_item(self):
        item_text = 'A short item.'
        item = rest.list_item(item_text)
        assert item.text() == item.sep + item.start_string + item_text \
               + item.end_string + item.sep
        check_rest(item)

    def test_list_item_multiline(self):
        item_text = '01234567890 1234567890'
        item = rest.list_item(item_text)
        text = item.text(width = 15)
        assert len([l for l in text.splitlines() if l.strip()]) == 2
        check_rest(text)
        out = Out(width = 15)
        item.__rest__(out)
        assert out.max_length() <= 15

    def test_list_item_custom_bullet(self):
        item_text = '12345678901234567890'
        item = rest.list_item(item_text, bullet='+')
        assert item.text().strip()[0] == '+'
        check_rest(item)

    def test_auto_enumerated_list(self):
        item_text = '12345678901234567890'
        item = rest.list_item(item_text, auto_enumerate = True)
        assert item.text().strip()[0:2] == '#.'
        check_rest(item)

    def test_literal_block(self):
        text = '''\
        This line is only 45 characters wide.
          This one is even longer (two spaces).\
        '''
        block = rest.literal_block(text)
        assert block.text()[:6] == block.sep + '::' + block.sep
        out = Out()
        block.__rest__(out)
        assert out.max_length() == len(block.quote) + max([len(l) for l in text.splitlines()])
        check_rest(block)
        
        block = rest.literal_block(text, quote= rest.literal_block.quotes[3])
        assert block.text().strip()[4] == rest.literal_block.quotes[3]
        check_rest(block)
        
        

    def test_interpreted_text(self):

        itext = rest.interpreted_text('just text')
        assert itext.text() == '`just text`'
        itext_role = rest.interpreted_text('just text with role', role = 'red')
        assert itext_role.text().startswith(':red:`just')

    def test_directive(self):
        dir = rest.directive.image('img.png')
        assert dir.text() == dir.sep + '.. image:: img.png\n' + dir.sep
        check_rest(dir.text())

    def test_directive_with_options(self):
        expected = """\
.. figure:: picture.png
   :alt: map to buried treasure
   :scale: 50

   This is the caption of the figure (a simple paragraph).

   The legend consists of all elements after the caption.  In this
   case, the legend consists of this paragraph."""

        legend = """ The legend consists of all elements after the caption.  In this\n case, the legend consists of this paragraph."""
        
        figure = rest.directive.figure('picture.png', rest.paragraph('This is the caption of the figure (a simple paragraph).'), rest.paragraph(legend), scale=50, alt= 'map to buried treasure')
             
        check_rest(expected)
        check_rest(figure.text())
        print_lines(expected.splitlines())
        print_lines(figure.text().splitlines())

        assert strip_lines(expected.splitlines()) == strip_lines(figure.text().splitlines())
        
        
    def test_directive_replace_underscore_in_directive_name(self):
        # should we replace underscore in keyword arguments?
        expected = '''\
.. csv-table:: Frozen Delights!
   :header: "Treat", "Quantity", "Description"
   :widths: 15, 10, 30

   Albatross,2.99,On a stick!
   Crunchy Frog,1.49,"If we took the bones out, it wouldn't be crunchy,
   now would it?"
   Gannet Ripple,1.99,On a stick!
''' #"

        data =[["Albatross", 2.99, "On a stick!"],
               ["Crunchy Frog", 1.49, "If we took the bones out, it wouldn't be crunchy, now would it?"],
               ["Gannet Ripple", 1.99, "On a stick!"]
               ]

        out = Out(width = None)
        py.std.csv.writer(out).writerows(data)
        text = out.render()
        table = rest.directive.csv_table('Frozen Delights!', text,
                                         header = '"Treat", "Quantity", "Description"',
                                         widths = "15, 10, 30")

        print_lines(expected.splitlines())
        print_lines(table.text().splitlines())
        check_rest(expected)
        check_rest(table.text())
        assert strip_lines(expected.splitlines()) == strip_lines(table.text().splitlines())


##     def test_block_quote(self):
##         block ="""\
## This is an ordinary paragraph, introducing a block quote.

##     "It is my business to know things.  That is my trade."

##     -- Sherlock Holmes
##     """
##         assert check_rest(block)


    
##     def test_quoted_line_block(self):
##         text = """\
## Take it away, Eric the Orchestra Leader!

##     | A one, two, a one two three four
##     |
##     | Half a bee, philosophically,
##     |     must, *ipso facto*, half not be.
##     | But half the bee has got to be,
##     |     *vis a vis* its entity.  D'you see?
##     |
##     | But can a bee be said to be
##     |     or not to be an entire bee,
##     |         when half the bee is not a bee,
##     |             due to some ancient injury?
##     |
##     | Singing...
##     """
##         assert check_rest(text)

#def test_temdir_output():
#    py.test.skip('tempdir is %s' % temp)
