/*
*
* ls.c - ls program for COBEX.
*
* Copyright (c) 2004,2005,2006 Fredrik Srensson
*
* History:
* v0.1 - fsn - 04-01-20 - First version
* v0.5 - fsn - 04-02-22 - Spring cleaning
* v0.6 - fsn - 04-06-16 - Expat parsing
* v0.7 - fsn - 05-07-19 - Using general signal handling and general path handling
* v0.8 - fsn - 06-07-29 - Code cleanup
*
* Source:
*
*/

#include <stdio.h>
#include <ezV24/ezV24.h>
#include <signal.h>
#include <string.h>
#include <expat.h>

#include <unistd.h>
#include <langinfo.h>
#include <locale.h>
#include <iconv.h>

#include "cobex_core.h"
#include "cobex_tools.h"
#include "cobex_serial.h"

#define MAXCHARS			256

void help () {
	printf ("cobex_ls v0.7\n\n");
	printf ("Usage: cobex_ls <device> <filename>\n\n");
	printf ("All arguments are compulsory and order is important. Returncodes are used to\n");
	printf ("indicate errors.\n\n");
}

// Deal with XML parsing
static void startElement(void *data, const char *el, const char **attr) {

	int nameidx=0;
	int sizeidx=0;
	int i;
	
	
	
	u_int inBytes;
	char inOrigMsg[MAXCHARS];
	u_int outBytes = MAXCHARS;
	char outOrigMsg[MAXCHARS];
	
	char *inMsgP = inOrigMsg;
	char *outMsgP = outOrigMsg;

	iconv_t iconvHandle;
	size_t sizeHandle;
	
	setlocale(LC_CTYPE, "");
	
	iconvHandle = iconv_open(nl_langinfo(CODESET), "UTF-8" );
		
	if (strncmp(el, "parent-folder", imin(strlen(el), 13)) == 0 ) {
		printf ("..	directory\n");
	} else if ( (strlen(el) == 6) && (strncmp(el, "folder", 6 ) == 0) ) {
		for (i = 0; attr[i]; i += 2) {
			if (strncmp(attr[i], "name", imin(strlen(attr[i]), 4)) ==0) {nameidx=i+1;}
		}
		inBytes=strlen(attr[nameidx])+1;
		strncpy(inOrigMsg, attr[nameidx], MAXCHARS);
		sizeHandle = iconv(iconvHandle, &inMsgP, &inBytes , &outMsgP, &outBytes);
		printf ("%s\tdirectory\n",outOrigMsg);
	} else if (strncmp(el, "file", imin(strlen(el), 4)) == 0 ) {
		for (i = 0; attr[i]; i += 2) {
			if (strncmp(attr[i], "name", imin(strlen(attr[i]), 4)) ==0) {nameidx=i+1;}
			if (strncmp(attr[i], "size", imin(strlen(attr[i]), 4)) ==0) {sizeidx=i+1;}
		}
		inBytes=strlen(attr[nameidx])+1;
		strncpy(inOrigMsg, attr[nameidx], MAXCHARS);
		sizeHandle = iconv(iconvHandle, &inMsgP, &inBytes, &outMsgP, &outBytes);
		printf ("%s\t%s\n",outOrigMsg,attr[sizeidx]);
	}
	iconv_close(iconvHandle);

}	

static void endElement(void *data, const char *name) {

	return;
}

int setupExpat ( XML_Parser *inParser ) {
	*inParser = XML_ParserCreate(NULL);
	if (!*inParser) {return 1;}
	XML_SetElementHandler(*inParser, startElement, endElement);
	return 0;
}

int teardownExpat ( XML_Parser *inParser ) {
	XML_ParserFree(*inParser);
	return 0;
}

// Parse the body for file data.

int parseBody(obex_packet *inPacket, XML_Parser *inParser) {
	u_int p;
	parsedData pD;
	
	p=cobex_parseStart(inPacket);

	// Hooray! This proved MUCH cleaner with the new, simple parser.

	while ( p < inPacket->l ) {
		pD = cobex_parseNext( inPacket,p ); 
		switch (pD.HI) {
			case OBEX_HI_END_OF_BODY:
				if (XML_Parse (*inParser, pD.HIdata, pD.HIlen, 1) == XML_STATUS_ERROR ) {return COBEX_ERR;}
				break;
			case OBEX_HI_BODY:
				if (XML_Parse (*inParser, pD.HIdata, pD.HIlen, 0) == XML_STATUS_ERROR ) {return COBEX_ERR;}
				break;
		}
		p = pD.nextHIp;
	}
	return COBEX_OK;
}

// Request file by type (ie. file listing type)

int ls_getFileByType( obex_packet *packet, char *type, v24_port_t *port, XML_Parser *inParser) {

	int rc;

	rc = obex_opcode_get ( packet );
	if (rc != COBEX_OK) return rc;
	rc = obex_hi_type ( packet, type, strlen(type)+1 );
	if (rc != COBEX_OK) return rc;
	rc = cobex_set_final_bit ( packet );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packlgt ( packet );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packet_send( packet, port );
	if (rc != COBEX_OK) return rc;
	rc = cobex_packet_recieve_t( packet, port, 200); // 200... something timeout
	if (rc != COBEX_OK) return rc;
	rc = parseBody( packet, inParser);

	return COBEX_OK;
}

// Fetch body. Parse it using expat

int ls_getRecBody( obex_packet *packet, v24_port_t *port, XML_Parser *inParser ) {
	
	int rc;
		
	do {
		rc = obex_opcode_get ( packet );
		if (rc != COBEX_OK) return rc;
		rc = cobex_set_final_bit ( packet );
		if (rc != COBEX_OK) return rc;
		rc = cobex_packlgt ( packet );
		if (rc != COBEX_OK) return rc;
		rc = cobex_packet_send( packet, port);
		if (rc != COBEX_OK) return rc;
		rc = cobex_packet_recieve_t( packet, port,200);
		if (rc != COBEX_OK) return rc;
		rc = parseBody( packet, inParser);
	} while (cobex_response_code(packet) == (OBEX_RESPONSE_CONTINUE|OBEX_FINAL_BIT) );
	
	return COBEX_OK;
}

// End code copying

void fakeTelecom() {
	printf("..	directory\n");
	printf("cal	directory\n");
	printf("cal.vcs	0\n");
	printf("devinfo.txt	0\n");
	printf("pb	directory\n");
	printf("pb.vcf	0\n");
	printf("rtc.txt	0\n");
}

// Main routine

int main (int argc, char *argv[]) {
	
	static char fsBrowser[]={ 0xf9, 0xec, 0x7b, 0xc4, 0x95, 0x3c, 0x11, 0xd2, 0x98, 0x4e, 0x52, 0x54, 0x00, 0xdc, 0x9e, 0x09, 0x00};
	v24_port_t *UsedPort=NULL;
	int rc;

	//char fileName[80];
	char aBuffer[513];
	obex_packet aPacket;

	aPacket.max=512;
	aPacket.buffer=aBuffer;
	
	XML_Parser aParser;
	
	if ( setupExpat( &aParser ) ) {fputs ("ERR: Could not allocate memory for XML parser.\n",stderr);return COBEX_ERR;}
	if (argc != 3) {help();return COBEX_ERR;}

	// Set up the port

	rc = ctools_commonInit( &UsedPort, &aPacket, argv[1], fsBrowser );
	if (rc != COBEX_OK) { exit (COBEX_ERR); }	

	// Do the specific things
	
	if (  (strlen(argv[2])==7||strlen(argv[2])==8) && strncmp(argv[2],"telecom",7)==0 ) {
		fakeTelecom();
	} else {

		rc = ctools_recursePath( &aPacket, argv[2], UsedPort, OBEX_SETPATH_DONTCREATE );

		if (cobex_response_code(&aPacket) != (OBEX_RESPONSE_OK|OBEX_FINAL_BIT) ) { 
			fprintf (stderr, "ERR: %s \n",
			cobex_respstring(cobex_response_code(&aPacket)) ) ;
			rc=COBEX_ERR;
		} else {

		// Do ls if and only if recursePath was ok.
			ls_getFileByType( &aPacket, "x-obex/folder-listing", UsedPort, &aParser );
	
			if (cobex_response_code(&aPacket) != (OBEX_RESPONSE_OK|OBEX_FINAL_BIT) &&
				cobex_response_code(&aPacket) != (OBEX_RESPONSE_CONTINUE|OBEX_FINAL_BIT) ) { 
				fprintf (stderr, "ERR: %s \n",
					cobex_respstring(cobex_response_code(&aPacket)) ) ;
				rc=COBEX_ERR;
			} else {
				ls_getRecBody( &aPacket, UsedPort, &aParser );
			}
		}
	}
			
	// Bye, y'all!

	ctools_disconnect( &aPacket, UsedPort );
	teardownExpat( &aParser );
	cobex_closePort(UsedPort);

	return rc;
}
