
#include "Core/precomp.h"

#ifdef USE_NETWORK

#include <API/Core/System/error.h>
#include <API/Core/System/system.h>
#include <API/Core/Network/netgame.h>
#include <API/Core/Network/network.h>
#include <Core/Network/Generic/network_delivery_impl.h>
#include <Core/Network/Generic/netgame_client.h>
#include <Core/Network/Generic/netgame_server.h>
#include <Core/Network/Generic/network_generic.h>

#ifndef WIN32
	#include <sys/socket.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#include <netdb.h>
#endif

CL_Network_Generic *CL_Network_Generic::self = NULL;

CL_Network_Generic::CL_Network_Generic(CL_ConnectionProvider *provider)
{
	this->provider = provider;
	self = this;
	udp_connection = provider->create_udp_connection(0);
}

CL_Network_Generic::~CL_Network_Generic()
{
	delete udp_connection;
	delete provider;
	self = NULL;
}

void CL_Network_Generic::find_games_broadcast(
	std::string game_id,
	int port)
{
	CL_OutputSource_MemoryGeneric msg;
	msg.write_int32(0); // ping message.
	msg.write_int32(game_id.length());
	msg.write(game_id.c_str(), game_id.length());

	CL_UDPConnectionPacket netmsg;
	netmsg.ip_addr = 0;
	netmsg.port = port;
	netmsg.size = msg.size();
	netmsg.data = msg.get_data();

	udp_connection->broadcast(netmsg);
}

void CL_Network_Generic::find_game_at(
	std::string game_id,
	const char *host_address,
	int port)
{
	// lookup dns name, if needed.
	unsigned int addr = 0;
	if (host_address[0] >= '0' && host_address[0] <= '9') // a.b.c.d
	{
		addr = inet_addr(host_address);
	}
	else
	{
		hostent *host = gethostbyname(host_address);
		if (host == NULL) throw CL_Error("Could not lookup DNS name");

		addr = *((unsigned int*) host->h_addr_list[0]);
	}

	queue.push(
		new CL_NetGame_Client(
			addr,
			port,
			game_id,
			this));
}

bool CL_Network_Generic::peek_game_found()
{
	if (udp_connection->peek())
	{
		CL_UDPConnectionPacket netmsg = udp_connection->receive();
		CL_InputSource_MemoryGeneric input(netmsg.data, netmsg.size);

		if (input.read_int32() == 1)
		{
			std::string game_id = input.read_string();
			queue.push(
				new CL_NetGame_Client(
					netmsg.ip_addr,
					netmsg.port,
					game_id,
					this));
		}
		
		delete[] netmsg.data;
	}

	return !queue.empty();
}

CL_NetGame *CL_Network_Generic::receive_game_found(int timeout_millis)
{
	if (timeout_millis > 0) CL_System::sleep(timeout_millis);
	peek_game_found();

	if (queue.empty()) throw CL_Error("Game not found");
	CL_NetGame *game = queue.front();
	queue.pop();
	return game;
}

void CL_Network_Generic::clear_games_found()
{
	while (queue.empty() == false)
	{
		delete queue.front();
		queue.pop();
	}
}

CL_NetGame *CL_Network_Generic::create_game(std::string game_id, int port)
{
	return new CL_NetGame_Server(
		this,
		game_id.c_str(),
		port);
}

void CL_Network_Generic::keep_alive()
{
/*	CL_Iterator<CL_NetGame_Generic> it(netgames);
	while (it.next() != NULL)
	{
		it()->keep_alive();
	}
	*/
}

/*
	CL_Network API static function calls:
	-------------------------------------
*/

void CL_Network::find_games_broadcast(
	std::string game_id,
	int port)
{
	if (CL_Network_Generic::self == NULL)
		throw CL_Error("Network not initialized!");

	CL_Network_Generic::self->find_games_broadcast(
		game_id,
		port);
}
	
void CL_Network::find_game_at(
	std::string game_id,
	const char *host_address,
	int port)
{
	if (CL_Network_Generic::self == NULL)
		throw CL_Error("Network not initialized!");

	CL_Network_Generic::self->find_game_at(
		game_id,
		host_address,
		port);
}
	
bool CL_Network::peek_game_found()
{
	if (CL_Network_Generic::self == NULL)
		throw CL_Error("Network not initialized!");

	return CL_Network_Generic::self->peek_game_found();
}

CL_NetGame *CL_Network::receive_game_found(int timeout)
{
	if (CL_Network_Generic::self == NULL)
		throw CL_Error("Network not initialized!");

	return CL_Network_Generic::self->receive_game_found(timeout);
}

void CL_Network::clear_games_found()
{
	if (CL_Network_Generic::self == NULL)
		throw CL_Error("Network not initialized!");

	CL_Network_Generic::self->clear_games_found();
}

CL_NetGame *CL_Network::create_game(
	std::string game_id,
	int port)
{
	if (CL_Network_Generic::self == NULL)
		throw CL_Error("Network not initialized!");

	return CL_Network_Generic::self->create_game(
		game_id,
		port);
}

#endif
