/*
	
	 ------------------------------------------------------------------------
	 ClanLib, the platform independent game SDK.

	 This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	 version 2. See COPYING for details.

	 For a total list of contributers see CREDITS.

	 ------------------------------------------------------------------------
*/

//! component="Lua scripting"

#ifndef header_lua_lib
#define header_lua_lib

extern "C" {
#define LUA_COMPAT2_5	// to include old Lua macros
#include <lua/lua.h>
#include <lua/lualib.h>
}

/* These c++ classes were created by Waldemer Celes. Additions and changes were made
*  to adapt it to ClanLib.
*/

//* Lua++: Lua API for C++
// These classes implement an API for accessing Lua from C++ code.
// It is based on the Lua 3.0a API for C programs.
// Developed by Waldemar Celes.
// May 1997 - version 1.1 (beta)

class CL_Lua;
class CL_LuaValue;
class CL_LuaObject;

class CL_LuaValue
//: Lua value
// Represents a value that can be passed to Lua.
// When calling Lua functions or indexing Lua tables,
// objects of this class are expected.
// However, they are implicitly constructed from C++ primitive types.
// Thus, in general,
// there is no need to directly create objects of this class.
{
                        //* Copying from LuaObject
                        // To convert a LuaValue from a LuaObject
                        // is _not_ allowed.
	CL_LuaValue (const CL_LuaObject&);  // not implemented

                        //* Assigning a LuaObject
                        // Assigning a LuaObject is _not_ allowed.
	CL_LuaValue& operator= (const CL_LuaObject&); // not implemented


protected:
	enum {
		LUA_PP_USERDATA=0,
		LUA_PP_NIL,
		LUA_PP_NUMBER,
		LUA_PP_STRING,
		LUA_PP_CFUNCTION,
		LUA_PP_REFERENCE
	};

	int d_type;
	union {
		float         d_num;
		char*         d_str;
		const char*   d_cstr;
		lua_CFunction d_fnc;
		void*         d_ptr;
		const void*   d_cptr;
		lua_Object    d_lo;   // used by derived class
		int           d_ref;  // used by derived class
	};

                        // Protected constructor to create references
	CL_LuaValue (int ref, int mark) : d_type(mark), d_ref(ref) {}

public:
 //* CREATORS
 // The following constructors are available,
 // allowing automatic convertion from
 // C++ primitive types.


	CL_LuaValue (void) : d_type(LUA_PP_NIL) {}
                        //: Create a 'nil' object

	CL_LuaValue (int v) : d_type(LUA_PP_NUMBER),   d_num(v)  {}
                        //: Create a 'number' from an 'int'

	CL_LuaValue (float v) : d_type(LUA_PP_NUMBER),   d_num(v)  {}
                        //: Create a 'number' from a 'float'

	CL_LuaValue (double v) : d_type(LUA_PP_NUMBER),   d_num(v)  {}
                        //: Create a 'number' from a 'double'

	CL_LuaValue (char* v) : d_type(LUA_PP_STRING),   d_str(v)  {}
                        //: Create a 'string' from a 'char*'

	CL_LuaValue (const char* v) : d_type(LUA_PP_STRING),   d_cstr(v) {}
                        //: Create a 'string' from a 'const char*'

	CL_LuaValue (lua_CFunction v) : d_type(LUA_PP_CFUNCTION),d_fnc(v)  {}
                        //: Create a 'cfunction' from a 'lua_CFunction}

	CL_LuaValue (void* v, int tag=LUA_PP_USERDATA) : d_type(tag), d_ptr(v)  {}
                        //: Create a 'userdata' from a 'void*'

	CL_LuaValue (const void* v, int tag=LUA_PP_USERDATA) : d_type(tag), d_cptr(v) {}
                        //: Create a 'userdata' from a 'const void*'

	CL_LuaValue (const CL_LuaValue& v)  { *this = v; }
                        //: Copy and assignment	

	CL_LuaValue& operator= (const CL_LuaValue& v)
	{
		d_type = v.d_type;
		d_num = v.d_num;
		return *this;
	}

	virtual ~CL_LuaValue (void) {}

 //* MANIPULATORS
	virtual void push (void) const
	{
		switch (d_type)
		{
			case LUA_PP_NIL:       lua_pushnil();                       break;
			case LUA_PP_NUMBER:    lua_pushnumber(d_num);               break;
			case LUA_PP_STRING:    lua_pushstring((char*)d_str);        break;
			case LUA_PP_CFUNCTION: lua_pushcfunction(d_fnc);            break;
			default:               lua_pushusertag((void*)d_ptr,d_type); break;
		}
	}
                        //: Push
                        // Pushs the value onto Lua stack.

	void store (char* varname)
	{
		push();
		lua_storeglobal(varname);
	}
                        //: Store
                        // Stores the value in a given Lua global variable.

 //* ACCESSORS


	virtual int isNil(void) const { return type()==LUA_PP_NIL; }
                        //: Query if 'nil'	

	virtual int isNumber(void) const { return type()==LUA_PP_NUMBER; }
                        //: Query if 'number'

	virtual int isString(void) const { return type()==LUA_PP_STRING; }
                        //: Query if 'string'	

	virtual int isTable(void) const { return 0; } // never happens
                        //: Query if 'table'

	virtual int isFunction(void) const { return type()==LUA_PP_CFUNCTION; }
                        //: Query if 'function'

	virtual int isUserData(void) const { return type()>=LUA_PP_USERDATA; }
                        //: Query if 'userdata'

	virtual int type(void) const { return d_type; }
                        //: Return the corresponding Lua type
 //* CONVERSORS
 // Allow conversion to C++ primitive types.


	virtual operator float() const
		{ return d_type==LUA_PP_NUMBER ? d_num : 0; }
                        //: Convert to 'float'
	virtual operator char*() const
		{ return d_type==LUA_PP_STRING ? (char*)d_str : 0; }
                        //: Convert to {char*}

	virtual operator const char*() const
		{ return d_type==LUA_PP_STRING ? d_str : 0; }
                        //: Convert to {const char*}

	virtual operator void*() const
		{ return d_type>=LUA_PP_USERDATA ? (void*)d_ptr : 0; }
                        //: Convert to {void*}		

	virtual operator const void*() const
		{ return d_type>=LUA_PP_USERDATA ? d_ptr : 0; }
                        //: Convert to {const void*}
};


class CL_LuaObject : public CL_LuaValue
//: Lua object
// Represents a general Lua object, that is, an object stored at Lua stack.
{

	lua_Object d_table;    // used for subsructor to create subscript

                        // Private constructor to create subscripts
	CL_LuaObject (lua_Object lo, const CL_LuaValue& v)
		: CL_LuaValue(v), d_table(lo) {}

protected:

friend class CL_Lua;

	CL_LuaObject (lua_Object lo)
		: d_table(LUA_NOOBJECT) { d_lo = lo; }
//: Protected constructor to create objects from the stack values
	
	CL_LuaObject (int ref, int mark)
		: CL_LuaValue(ref,mark), d_table(LUA_NOOBJECT) {}
//: Protected constructor to create references

public:
 //* CREATORS

	CL_LuaObject (void)
		: CL_LuaValue(-1,LUA_PP_REFERENCE), d_table(LUA_NOOBJECT) {}
//: Create a 'nil' object (default)
// After creating a 'nil' object,
// we use the Lua class methods to
// access values from the Lua stack	
	
	CL_LuaObject (const CL_LuaObject& org) { *this = org; }

	virtual ~CL_LuaObject (void) {}

	CL_LuaObject& operator= (const CL_LuaObject& value)
	{
		d_table = value.d_table;
		CL_LuaValue& base1 = *this;
		const CL_LuaValue& base2 = value;
		base1 = base2;
		return *this;
	}
//: operator=

	CL_LuaObject& operator= (const CL_LuaValue& value)
	{
		if (d_table != LUA_NOOBJECT)
		{
			lua_beginblock();
			lua_pushobject(d_table);     // push table
			CL_LuaValue::push();            // push index
			value.push();                // push value
			lua_storesubscript();
			lua_endblock();
		}
		else
		{
			lua_error("lua++: cannot assign a CL_LuaValue to a CL_LuaObject.");
		}
		return *this;
	}
//: operator=

	virtual lua_Object getobject (void) const
	{
		if (d_table == LUA_NOOBJECT)
		{
			if (d_type == LUA_PP_REFERENCE) return lua_getref(d_ref);
			else return d_lo;
		}
		else
		{
			lua_pushobject(d_table);
			CL_LuaValue::push();
			return  lua_getsubscript();
		}
	}

//: Get the corresponding Lua object

	void push (void) const
	{
		if (d_table == LUA_NOOBJECT)
		{
			if (d_type == LUA_PP_REFERENCE) lua_pushref(d_ref);
			else lua_pushobject(d_lo);
		}
		else
		{
			lua_beginblock();
			lua_pushobject(d_table);
			CL_LuaValue::push();
			lua_Object lo = lua_getsubscript();
			lua_endblock();
			lua_pushobject(lo);
		}
	}
	
//: Base virtual functions
	int isNil        (void) const
	{
		lua_beginblock();
		int result = lua_isnil(getobject());
		lua_endblock();
		return result;
		}

	int isNumber     (void) const
	{
		lua_beginblock();
		int result = lua_isnumber(getobject());
		lua_endblock();
		return result;
	}

	int isString     (void) const
	{
		lua_beginblock();
		int result = lua_isstring(getobject());
		lua_endblock();
		return result;
	}
	
	int isFunction   (void) const
	{
		lua_beginblock();
		int result = lua_isfunction(getobject());
		lua_endblock();
		return result;
	}
	
	int isUserData   (void) const
	{
		lua_beginblock();
		int result = lua_isuserdata(getobject());
		lua_endblock();
		return result;
	}

	int isTable      (void) const
	{
		lua_beginblock();
		int result = lua_istable(getobject());
		lua_endblock();
		 return result;
	}

	int type         (void) const
	{
		lua_beginblock();
		int result = lua_type(getobject());
		lua_endblock();
		return result;
	}

	operator float() const
	{
		lua_beginblock();
		lua_Object lo = getobject();
		float res = lua_getnumber(lo);
		lua_endblock();
		return res;
	}
//: Convert lua value to ordinary C types

	operator char*() const
	{
		lua_beginblock();
		lua_Object lo = getobject();
		char* res = lua_getstring(lo);
		lua_endblock();
		return res;
	}
//: Convert lua value to ordinary C types
	
	operator const char*() const
	{
		lua_beginblock();
		lua_Object lo = getobject();
		const char* res = lua_getstring(lo);
		lua_endblock();
		return res;
	}
//: Convert lua value to ordinary C types

	operator void*() const
	{
		lua_beginblock();
		lua_Object lo = getobject();
		void* res = lua_getuserdata(lo);
		lua_endblock();
		return res;
	}
//: Convert lua value to ordinary C types

	operator const void*() const
	{
		lua_beginblock();
		lua_Object lo = getobject();
		const void* res = lua_getuserdata(lo);
		lua_endblock();
		return res;
	}
//: Convert lua value to ordinary C types

 // TABLES

	CL_LuaObject operator[] (const CL_LuaValue& index)
	{
		return CL_LuaObject(getobject(),index);
	}
                        //: Subscription
                        // If the object is a Lua table,
                        // the {operator[]} can be used to indexed the Lua
                        // corresponding value.

	CL_LuaObject operator[] (int index)
	{
		return CL_LuaObject(getobject(),CL_LuaValue(index));
	}
                        //: Subscription
                        // If the object is a Lua table,
                        // the {operator[]} can be used to indexed the Lua
                        // corresponding value.	

 // FUNCTIONS
 // If the object is a Lua function,
 // the operator () can be used to call it.


	int operator() (void) const
	{
		lua_Object lo = getobject();
		return lua_callfunction(lo);
	}
                        //: Call function with no argument

	int operator() (const CL_LuaValue& arg1)
	{
		lua_Object lo = getobject();
		arg1.push();
		return lua_callfunction(lo);
	}
                        //: Call function with 1 argument
	
	int operator() (const CL_LuaValue& arg1, const CL_LuaValue& arg2)
	{
		lua_Object lo = getobject();
		arg1.push(); arg2.push();
		return lua_callfunction(lo);
	}
                        //: Call function with 2 arguments

	int operator() (const CL_LuaValue& arg1,
                 const CL_LuaValue& arg2,
                 const CL_LuaValue& arg3)
	{
		lua_Object lo = getobject();
		arg1.push(); arg2.push(); arg3.push();
		return lua_callfunction(lo);
	}
                        //: Call function with 3 arguments

	int operator() (const CL_LuaValue& arg1,
				const CL_LuaValue& arg2,
                const CL_LuaValue& arg3,
                 const CL_LuaValue& arg4)
	{
		lua_Object lo = getobject();
		arg1.push(); arg2.push(); arg3.push(); arg4.push();
		return lua_callfunction(lo);
	}
                        //: Call function with 4 arguments

	int operator() (const CL_LuaValue& arg1,
                 const CL_LuaValue& arg2,
                 const CL_LuaValue& arg3,
                 const CL_LuaValue& arg4,
                 const CL_LuaValue& arg5)
	{
		lua_Object lo = getobject();
		arg1.push(); arg2.push(); arg3.push(); arg4.push(); arg5.push();
		return lua_callfunction(lo);
	}
                        //: Call function with 5 arguments
};


class CL_LuaTable : public CL_LuaObject
//: Lua table
// Allows a Lua table to be explicitly created from C++ code.
{
public:
	CL_LuaTable (void)
		: CL_LuaObject(lua_createtable()) {}
//: Creator
// An initialy empty Lua table is created.

	~CL_LuaTable (void) {}
};



class CL_LuaReference : public CL_LuaObject
//: Lua reference
// If the user needs to store references to LuaObject,
// this reference must be explicitly created, creating objects of
// this class.
{
public:
	CL_LuaReference (void)
        : CL_LuaObject() {}
//: Create a reference to a 'nil' object (default)

	CL_LuaReference (const CL_LuaObject& obj, int lock=0)
        :  CL_LuaObject()
	{
		ref(obj,lock);
	}
//: Create a reference to the given Lua object
// An optinal parameter specifies if the
// reference is not to be garbage collected by Lua.
// The default, {lock=0}, means the reference becomes
// invalid if Lua collects the referenced object.
//!param: lock - to lock or not to lock, that is the question

	void ref (const CL_LuaObject& obj, int lock=0)
	{
		lua_beginblock();
		obj.push();
		d_ref = lua_ref(lock);
		lua_endblock();
	}
//: Replace the reference
// This method allows the user to replace the
// referenced object; it is useful when the object
// is created with the default constructor.
// _Note_: if there is already a reference to a
// non-'nil' object, this reference, if not used
// by other LuaReference objets, should be
// eliminated (using the 'unref' method) before
// calling this method.

	void unref (void)
	{
		if (d_type == LUA_PP_REFERENCE)
			lua_unref(d_ref);
		d_ref = -1;
	}
//: Unreference an object
};


class CL_Lua
//: CL_Lua class
// Binds global functions of Lua API.
// Refer to Lua manual for details.
{
public:
	static int dofile (char *filename)
		{ return lua_dofile(filename); }
//: Execute a Lua chunk stored in a file
//!param: filename - the script file name
//!retval: the return code

	static int dostring (char *string)
		{ return lua_dostring(string); }
//: Execute a Lua chunk stored in a string
//!param: string - the lua string
//!retval: the return code

	static void error (char* message)
		{ lua_error(message); }
//: Generate a Lua error, with the given message
//!param: message - the error message

	static CL_LuaObject getparam (int number)
		{ return CL_LuaObject(lua_getparam(number)); }
//: Get a parameter from Lua stack
//!param: number: the number
//!retval: the return code

	static CL_LuaObject getresult (int number)
		{ return CL_LuaObject(lua_getresult(number)); }
//: Get a result from Lua stack
//!param: number: the number
//!retval: the return code

	static CL_LuaObject getglobal (char* var)
		{ return CL_LuaObject(lua_getglobal(var)); }
//: Get a Lua global variable
//!param: var: the name of the global variable
//!retval: the return code

	static CL_LuaObject setfallback (char* name, lua_CFunction fallback)
		{ return CL_LuaObject(lua_setfallback(name,fallback)); }
//: Set Lua fallback		
//!param: name - the name of the fallback
//!param: fallback - pointer to the function
//!retval: the return code

	static void record (char* name, lua_CFunction func)
	{
		lua_beginblock();
  		lua_register(name,func);
		lua_endblock();
	}
//: Register a C++ function to be called from Lua
//!param: name - the name of the function
//!param: func - pointer to the function

	static CL_LuaObject createObject (lua_Object lo)
		{ return CL_LuaObject(lo); }
//: Create a LuaObject from a lua_Object
//!param: obj - the object
//!retval: the return code
	
	static void iolibopen(void) { lua_iolibopen(); }
//: Open the IO library

	static void strlibopen(void) { lua_strlibopen(); }
//: Open the string library

	static void mathlibopen(void) { lua_mathlibopen(); }
//: Open the math library

	static void dblibopen(void) { lua_dblibopen(); }
//: Open the db library
};

#endif








