#include "cp_types.h"
#include "cp_proto.h"

/* (Derived from old version comp_pack_centers.)
Find centers based on current radii by going thru faces in order, 
but allow various options and error cutoffs in computation and 
analysis of plotting errors. Start with alpha vert; at end, normalize 
so gamma vert is on y>0 axis. Only compute each circle's center once. 
If errflag, only circles placed w/o too much error have their plot_flag 
set. Record rad/cent in redlist, if it exists, BUT use the cent/rad 
data from pack (rather than redlist data) in those computations. 
Then use the redlist data to set side-pairing Mobius transforms. 

Options: 
 errflag true: plot_flag not set if centers doesn't meet placement
             criteria; use only verts with plot_flag's set 
	     when computing later centers.

 dflag true: write diagnostic file /tmp/layout.log of problem placements.

 opt: 1 = use drawing-order only (ie. go to next face, compute
          center of third vert from other two -- original 'fix'
	  method)
      2 = use average of already plotted neighbors
      3 = ???

 crit: this is a double giving accuracy criteria (if errflag set). 
       Eg., don't use computed center of v if nghbs u,w actual 
       distance apart is off by factor > crit. Should have default 
       value to use for crit in general; can experiment with it.

Return number of failed placements.

*/

int comp_pack_centers(struct p_data *p,int errflag,int dflag,int opt,
		     double crit)
{
  int nf,i,j,n0,n1,n,vert,count=1,v,indx,lastface,flick=1;
  int v_ind,wflag=0,debug=0;
  struct K_data *pK_ptr;
  f_data *faces;
  struct RedList *trace;
  FILE *diagfp=NULL;

  pK_ptr=p->packK_ptr;faces=p->faces;
  for (j=1;j<=p->nodecount;j++) pK_ptr[j].plot_flag=0;
  for (j=1;j<=p->facecount;j++) faces[j].plot_flag=1;
  nf=p->first_face;
  place_face(p,nf,faces[nf].index_flag);  
  for (j=0;j<3;j++) pK_ptr[faces[nf].vert[j]].plot_flag=1;

  if (dflag && errflag)
    {
      if (!(diagfp=fopen("/tmp/layout.log","a")))
	{
	  sprintf(msgbuf,"couldn't open /tmp/layout.log");
	  emsg();
	  return count;
	}
      fprintf(diagfp,"Diagnostic file created in 'new_pack_centers'.\n"
	      "Packing %s, nodecount %d, 'critical' tolerance set at %f\n\n"
	      "Following are failed placements:\n"
	      "\tat count \tvert \tface\n\n",
	      p->file_name,p->nodecount,crit);
    }

  /* ------------ simply connected case ----------- */
  if ((p->euler==1 || p->euler==2) && p->genus==0)
    {
      while ( (nf=faces[nf].next_face)!=p->first_face 
	      && nf>0 && nf<=p->facecount && count<2*p->facecount)
	{
	  vert=faces[nf].vert[(indx=(faces[nf].index_flag +2) % 3)];
	  n0=nghb(p,vert,faces[nf].vert[(indx+1)%3]);
	  if (!pK_ptr[vert].plot_flag && (pK_ptr[vert].plot_flag=
	       fancy_comp_center(p,vert,n0,0,pK_ptr[vert].num,
				 errflag,opt,0,crit)))
	    count++;
	  else if (dflag && errflag)
	    fprintf(diagfp,"\t%d \t%d \t%d   \n",count,vert,nf);
	}
      if ((trace=p->redfaces)) /* record rad/cent data in redchain */
	while (trace!=p->redfaces || !(wflag++))
	  {
	    v=p->faces[trace->face].vert[trace->v_flag];
	    trace->center=p->packR_ptr[v].center;
	    trace->rad=p->packR_ptr[v].rad;
	    trace=trace->next;
	  }
    } /* end of simply connected case  */

  /* -------- multiply connected case ----------------- */

  else 
    {
      for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;

      if (opt==2) /* want to use average, so have to mark outer verts 
		     of red chain using util_flag; need to find fans. */
	{
	  trace=p->redfaces;
	  wflag=0;
	  while (trace!=p->redfaces || !(wflag++))
	    {
	      indx=nghb_tri(p,trace->next->face,trace->face);
	      v=p->faces[trace->face].vert[indx];
	      if (!pK_ptr[v].util_flag) pK_ptr[v].util_flag=1;
	      if (trace->next->face==trace->prev->face) /* blue? */
		{
		  indx=nghb_tri(p,trace->next->face,trace->face);
		  v=p->faces[trace->face].vert[(indx+2) % 3];
		  pK_ptr[v].util_flag=1;
		}
	      trace=trace->next;
	    }
	}

      lastface=p->first_face;
      nf=p->first_face;
      while (nf>0 && nf<=p->facecount && count<3*p->facecount)
	{
	  if (flick && nf==p->first_red_face) 
	    /* When we first reach first_red_face, we switch to layout 
	       whole red chain in order. This is:
	       1) necessary because some faces will occur twice, placing 
	       possibly a different one of its circles each time, and 
	       2) to load rad/cent data in redfaces data structure
	       for other uses. 
	       Note: circles of pack ends up where last placed. */
	  		    
	    {
	      flick=0;
	      vert=faces[nf].vert[(indx=(faces[nf].index_flag +2) % 3)];
	      n0=nghb(p,vert,faces[nf].vert[(indx+1)%3]);
	      if (pK_ptr[vert].util_flag) /* this is redchain outer vert
					     and we're using averaging. */
		red_fan(p,p->redfaces,vert,&n1,&n);
	      else 
		{
		  n1=0;
		  n=pK_ptr[vert].num;
		}
	      if (!pK_ptr[vert].plot_flag && (pK_ptr[vert].plot_flag=
		  fancy_comp_center(p,vert,n0,n1,n,errflag,opt,0,crit)))
		count++;
	      else if (dflag && errflag)
		fprintf(diagfp,"\t%d \t%d \t%d   \n",count,vert,nf);

	      p->redfaces->center=
		p->packR_ptr[p->faces[p->redfaces->face].
			    vert[p->redfaces->v_flag]].center;
	      p->redfaces->rad=
		p->packR_ptr[p->faces[p->redfaces->face].
			    vert[p->redfaces->v_flag]].rad;

	      /* Now proceed through rest of redfaces.
		 Which center to compute is based on indices stored 
		 in f_data. This should fit 'next_face' order (which 
		 reflects redfaces order). Note, this can differ 
		 from v_flag in redlist. Record rad/cent whether 
		 or not error is within tolerance.  */

	      /* This is very confusing. Only problem with going
		 through redfaces in order, each depending only on
		 last, is that errors will build up and bdry may
		 detach from white faces. Indices in f_data try to
		 be smart in choosing which of three verts to plot,
		 but I'm not sure if it might rely on outdated 
		 locations for the other two. May have to pad the
		 'next_face' order with sections of redfaces list
		 which it skips (because those faces already had
		 a 'next_face' -- see 'final_order'). */
	  

	      lastface=p->first_red_face;
	      trace=p->redfaces->next;
	      while (trace!=p->redfaces)
		{
		  nf=trace->face;
		  if ((indx=nghb_tri(p,lastface,nf))<0)
		    {
		      sprintf(msgbuf,"Problem with redfaces order.");
		      emsg();
		      lastface=nf;
		      goto CARRYON;
		    }
		  indx=(p->faces[nf].index_flag+2) % 3;
		  vert=p->faces[nf].vert[indx];
		  n0=nghb(p,vert,faces[nf].vert[(indx+1)%3]);
		  if (pK_ptr[vert].util_flag) /* this is redchain outer vert
						 and we're using averaging. */
		    red_fan(p,p->redfaces,vert,&n1,&n);
		  else 
		    {
		      n1=0;
		      n=pK_ptr[vert].num;
		    }
		  if ((pK_ptr[vert].plot_flag=
		       fancy_comp_center(p,vert,n0,n1,n,0,1,0,crit)))
		    /* Note: since the red faces are needed downstream
		       and it's difficult to keep track of which 
		       nghbs should be used to plot a vert at each visit,
		       we use values errflag=0 and opt=1 in this call. */

		    /* fixup: Can we rearrange drawing order so as many
		       verts as possible not on outside of red chain 
		       are laid out first, then the red chain? */
		       
		    count++;
		  else if (dflag && errflag)
		    fprintf(diagfp,"\t%d \t%d \t%d   \n",count,vert,nf);

		  /* Now record rad/cent in redface data. This may not 
		     be the data we just computed, so we may have to
		     do yet another computation here (depends on v_flag). */

		  v_ind=p->faces[trace->face].vert[trace->v_flag];
		  if (v_ind!=vert) /* need to update center for circle
				      associated with v_flag of redface. */
		    {
		      n0=nghb(p,v_ind,faces[trace->face].
			      vert[((trace->v_flag)+1)%3]);
		      fancy_comp_center(p,v_ind,n0,0,0,0,1,1,crit);
		    }
		  trace->center=p->packR_ptr[v_ind].center;
		  trace->rad=p->packR_ptr[v_ind].rad;

		  lastface=nf;
		  trace=trace->next;
		}
	      /* debug */
	      if (debug) red_bug(p->redfaces);

	    }

	  /* pick up order from f_data again */

	CARRYON: 
	  if ((nf=p->faces[lastface].next_face)==p->first_face) 
	    break;
	  vert=faces[nf].vert[(indx=(faces[nf].index_flag +2) % 3)];
	  n0=nghb(p,vert,faces[nf].vert[(indx+1)%3]);
	  if (!pK_ptr[vert].plot_flag && (pK_ptr[vert].plot_flag=
	       fancy_comp_center(p,vert,n0,0,pK_ptr[vert].num,
				 errflag,opt,0,crit)))
	    count++;
	  else if (dflag && errflag)
	    fprintf(diagfp,"\t%d \t%d \t%d   \n",count,vert,nf);

	  lastface=nf;
	  nf=p->faces[lastface].next_face;
	}
    } /* end of multiply connected case  */

  if (dflag && errflag)
    fprintf(diagfp,"\nFinished original pass with %d successful"
	    " placements.\n\n",count);
  
  /* If errflag is set and simply connected, make second pass (perhaps
     even more?) to catch any additional verts possible.
     Note: count may get large if errflag is set, since a circle may have
     center computed several times */

  if (errflag && count<(5*(p->nodecount))
      && (p->euler==1 || p->euler==2) && p->genus==0) 
    {
      nf=p->first_face;
      while ( (nf=faces[nf].next_face)!=p->first_face 
	      && nf>0 && nf<=p->facecount && count<(2*p->facecount))
	{
	  for (j=0;j<3;j++) /* potential to plot a vert? */
	    {
	      vert=faces[nf].vert[(indx=(faces[nf].index_flag +2+j) % 3)];
	      if (!pK_ptr[vert].plot_flag)
		{
		  n0=nghb(p,vert,faces[nf].vert[(indx+1)%3]);
		  if (!pK_ptr[vert].plot_flag && (pK_ptr[vert].plot_flag=
		     fancy_comp_center(p,vert,n0,0,pK_ptr[vert].num,
				 errflag,opt,0,crit)))
		    {
		      count++;
		      break;      /* get out of for loop */
		    }
		}
	    }
	} /* end of while */
      if (dflag && errflag)
	fprintf(diagfp,"\nFinished second pass with %d successful"
	    " placements.\n\n",count);
  
    }

  /* set face plot_flags; precision here is rather experimental. */
  for (j=1;j<=p->facecount;j++) faces[j].plot_flag=1;
  if (errflag) for (j=1;j<=p->facecount;j++)
    if (!pK_ptr[faces[j].vert[0]].plot_flag
	|| !pK_ptr[faces[j].vert[1]].plot_flag
	|| !pK_ptr[faces[j].vert[2]].plot_flag
	|| fabs(invdist_err(p,faces[j].vert[0],faces[j].vert[1]))> 100000*toler
	|| fabs(invdist_err(p,faces[j].vert[1],faces[j].vert[2]))> 100000*toler
	|| fabs(invdist_err(p,faces[j].vert[2],faces[j].vert[0]))> 100000*toler)
      faces[j].plot_flag=0;
  /* I no longer understand this old message:
       fixup: to handle multiply connected, have to go around redlist
       for special treatment -- circle(s) of face may have been moved */
  /* normalize */
  if (nf<=0 || nf>p->facecount || count>(5*(p->nodecount))) 
    /* probably some error in data */
    {
      if (diagfp) fclose(diagfp);
      return 0;    
    }

  norm_any_pack(p,p->packR_ptr[p->alpha].center,
		p->packR_ptr[p->gamma].center); 
  /* fixup? could be problem if errflag and gamma poorly plotted */

  /* now record any edge-pairing Mobius transforms */
  update_pair_mob(p);

  if (diagfp) fclose(diagfp);
  return (count);

} /* comp_pack_centers */

