/* cp_string.c  various string manipulation and parsing routines.*/

#include "cp_head.h"
#include <sys/times.h>
#include <X11/Xutil.h>

extern char *get_selection();
static Region region;

/* ====================== utility routines ====================== */

int
grab_next(ptr,next) /* get next string of <256 non-space, non-semicolon
characters from ptr, put in next, reset ptr. 
Return 1 if next not empty. Note: changes ptr in calling routine, but not
contents it originally pointed to.*/
char **ptr,*next;
{
	int i=0;

	if (*ptr==NULL) return 0;
	while (**ptr == ' ' || **ptr == '\n' || **ptr == '\t' 
		|| **ptr == '\r')
		(*ptr)++;
	if (**ptr=='\0') return 0;
	while (*((*ptr)+i)!='\0' && *((*ptr)+i)!=' ' 
		&& *((*ptr)+i)!='\n' && *((*ptr)+i)!='\r' 
		&& *((*ptr)+i)!=';' && *((*ptr)+i)!='\t' && i<255)
	 {
		*(next+i)=*((*ptr)+i);
		i++;
	 }
	*(next+i)='\0';
	*ptr += i;
	return i;
} /* grab_next */

stripsp(datastr) /* strip initial 'space' chars, shift rest down. */
char *datastr;
{
	int i=0;
	char *ptr;

	ptr=datastr;
	while (*ptr==' ' || *ptr=='\n' || *ptr=='\r' || *ptr=='\t') ptr++;
	if (ptr==datastr) return;
	if (*ptr=='\0') 
	 {
		*datastr='\0';
		return;
	 }
	do {*(datastr+i)=*(ptr+i); i++;}
	while (*(ptr+i-1)!='\0');
	return;
} /* stripsp */

int
notspace(ptr) /* returns number of steps to next nonspace char or to end */
char *ptr;
{
	int n=0;
	while (*(ptr+n)==' ' || *(ptr+n)=='\n' || *(ptr+n)=='\r' 
		|| *(ptr+n)=='\t') n++;
	return n;
} /* notspace */

/* ====================== making linked lists ===================== */

struct Vertlist *	
node_link_parse(p,dpoint,endptr,hits) /* Vertices: read string specified by 
dpoint of vertex indices from p, create link list, return pointer, and
set endptr to NULL or first inapprop char encountered.
 hits=0 ==> no expressions parsed. */
int *hits;
struct p_data *p;
char *dpoint,**endptr;
{
	int count,nextvert,stopvert,v1,v2,n,j,cflg,
		q,qcount,q_flag,cf_flag=1;
	char next[256],*nextptr,*lastptr,phold[1];
	struct K_data *pK_ptr,*qK_ptr;
	struct Vertlist *vertlist=NULL,*trace,*clobber,*tmp_ptr; 
	extern struct Vertlist *brace_parse();

	pK_ptr=p->packK_ptr;
	nextptr=lastptr=dpoint;
	*endptr=NULL;
	count=0;
	*hits=0;
	if (*nextptr == '\0' || !grab_next(&nextptr,next)) return NULL;
	*hits=1;
	vertlist=(struct Vertlist *)calloc(1,sizeof(struct Vertlist));
	trace=vertlist;
	clobber=trace;
		/* note: trace points to allocated space, but without
			node data. */
	do
	 {
		lastptr += notspace(lastptr);
		if (*next=='a') /* 'all' specification */
		 {
			lastptr++;
			n=parse_parens(lastptr,&v1,&v2);
			if ( n == (-1) || (n>0 && v2<v1))
				/* malformed expression */
			 {
				free(trace);
				clobber->next=NULL;
				if (!count) return NULL;
				return vertlist;
			 } 
			else if (n==0)
				/* run thru whole complex */
			 {
				v1=1;
				v2=p->nodecount;
			 }
			if (v1<1) v1=1;
			if (v2>p->nodecount) v2=p->nodecount;
				/* reaching here, should have v1 and v2 */
			if (lastptr>nextptr) nextptr=lastptr;
			for (n=v1;n<=v2;n++)
			 {
				trace->v=n;
				trace->next=(struct Vertlist *)
					calloc(1,sizeof(struct Vertlist));
				clobber=trace;
				trace=trace->next;
				count++;
			 }
		 } /* finished with 'all' */
		else if (next[0]=='b') /* boundary verts */
		 {
			lastptr++;
			n=parse_parens(lastptr,&v1,&v2);
			if ( n == (-1) || (n>0 && (v1<1 || v2>p->nodecount 
				|| !pK_ptr[v1].bdry_flag 
				|| !pK_ptr[v2].bdry_flag)) )
				/* malformed expression */
			 {
				free(trace);
				clobber->next=NULL;
				if (!count) return NULL;
				return vertlist;
			 } 
			else if (n==0)
				/* run thru whole bdry */
			 {
			   for (j=1;j<=p->num_bdry_comp;j++)
			    {
				v1=p->bdry_starts[j];
				v2=pK_ptr[v1].flower[pK_ptr[v1].num];
				trace->v=v1;
				trace->next=(struct Vertlist *)
					calloc(1,sizeof(struct Vertlist));
				clobber=trace;
				trace=trace->next;
				count++;
				if (v2!=v1)
				 {
				   stopvert=pK_ptr[v2].flower[0];
				   nextvert=v1;
				   while 
				     ( (nextvert=pK_ptr[nextvert].flower[0]) 
					!= stopvert )
					 {
					  trace->v=nextvert;
					  trace->next=(struct Vertlist *)
					    calloc(1,sizeof(struct Vertlist));
					  clobber=trace;
					  trace=trace->next;
					  count++;
					 }
				 }
			    }
			 } /* end whole bdry case */
			else if (n==1)
			 {
			      /* reaching here, should have v1 and v2 */
				trace->v=v1;
				trace->next=(struct Vertlist *)
					calloc(1,sizeof(struct Vertlist));
				clobber=trace;
				trace=trace->next;
				count++;
				if (v2==v1) 
				   v2=pK_ptr[v1].flower[pK_ptr[v1].num];
				stopvert=pK_ptr[v2].flower[0];
				nextvert=v1;
				while ( (nextvert=pK_ptr[nextvert].flower[0]) 
				   != stopvert && count < p->nodecount)
				 {
					trace->v=nextvert;
					trace->next=(struct Vertlist *)
					    calloc(1,sizeof(struct Vertlist));
					clobber=trace;
					trace=trace->next;
					count++;
				 }
			 } /* end (v1,v2) case */
			if (lastptr>nextptr) nextptr=lastptr;
		 } /* finished with boundary case */
		else if (*next=='i') /* interior */
		 {
			lastptr++;
			n=parse_parens(lastptr,&v1,&v2);
			if ( n == (-1) || (n>0 && v2<v1))
				/* malformed expression */
			 {
				free(trace);
				clobber->next=NULL;
				if (!count) return NULL;
				return vertlist;
			 } 
			else if (n==0)
				/* run thru whole complex */
			 {
				v1=1;
				v2=p->nodecount;
			 }
			if (v1<1) v1=1;
			if (v2>p->nodecount) v2=p->nodecount;
			if (lastptr>nextptr) nextptr=lastptr;
				/* reaching here, should have v1 and v2 */
			for (n=v1;n<=v2;n++)
				if (!pK_ptr[n].bdry_flag)
				 {
					trace->v=n;
					trace->next=(struct Vertlist *)
					  calloc(1,sizeof(struct Vertlist));
					clobber=trace;
					trace=trace->next;
					count++;
				 }
		 } 
		else if (*next=='m') /* marked */
		 {
			if (cflg=(next[1]=='c')) lastptr++; 
				/* 1==>complement of marked */
			if (next[1]=='p' || (cflg && next[2]=='p') )
				/* take marking from pack q */
			 {
				lastptr++;
				if (next[1]=='p') phold[0]=next[2];
				else phold[0]=next[3];
				q=atoi(phold);
				if (q>=0 && q<NUM_PACKS && packdata[q].status)
				 {
					lastptr++;
					q_flag=1;
					qK_ptr=packdata[q].packK_ptr;
					qcount=packdata[q].nodecount;
				 }
				else q_flag=0;
			 }
			else q_flag=0;				
			lastptr++;
			n=parse_parens(lastptr,&v1,&v2);
			if ( n == (-1) || (n>0 && v2<v1))
				/* malformed expression */
			 {
				free(trace);
				clobber->next=NULL;
				if (!count) return NULL;
				return vertlist;
			 } 
			else if (n==0)
				/* run thru whole complex */
			 {
				v1=1;
				v2=p->nodecount;
			 }
			if (v1<1) v1=1;
			if (v2>p->nodecount) v2=p->nodecount;
			if (lastptr>nextptr) nextptr=lastptr;
				/* reaching here, should have v1 and v2 */
			for (n=v1;n<=v2;n++)
				if ( (!cflg && !q_flag && pK_ptr[n].mark) 
				   || (cflg && !q_flag && !pK_ptr[n].mark) 
				   || (!cflg && q_flag && n<=qcount
					&& qK_ptr[n].mark)
				   || (cflg && q_flag && n<=qcount
					&& !qK_ptr[n].mark) )
				 {
					trace->v=n;
					trace->next=(struct Vertlist *)
					  calloc(1,sizeof(struct Vertlist));
					clobber=trace;
					trace=trace->next;
					count++;
				 }
		 } 
		else if (*next=='{')
		 {
			tmp_ptr=brace_parse(p,lastptr,&cf_flag,&lastptr);
			if (tmp_ptr!=NULL) /* fit into growing list */
			 {
				if (vertlist==trace) vertlist=tmp_ptr;
				free(trace);
				clobber=tmp_ptr;
				count++;
				while (clobber->next!=NULL)
				 {
					count++;
					clobber=clobber->next;
				 }
				clobber->next=(struct Vertlist *)
					calloc(1,sizeof(struct Vertlist));
				trace=clobber->next;
			 }
		 }
		else if (sscanf(next,"%d",&v1) && v1>0 && v1<=p->nodecount) 
		 {
			trace->v=v1;
			trace->next=(struct Vertlist *)
				calloc(1,sizeof(struct Vertlist));
			clobber=trace;
			trace=trace->next;
			count++;
		 }
		else /* inappropriate entry */
		 {
			*endptr=lastptr;
			if (trace!=clobber) /* some valid hits */
				clobber->next=NULL;
			free(trace);trace=NULL;
			if (!count) return NULL;
			return vertlist;
		 }
		lastptr=nextptr;
	 }
	while (grab_next(&nextptr,next));
	if (trace!=clobber) clobber->next=NULL;
	free(trace);trace=NULL;
	if (!count) return NULL;
	return vertlist;
} /* node_link_parse */

struct Vertlist *
face_link_parse(p,dpoint,endptr,hits) /* read face numbers for p,
put in linked list. endptr=NULL or pts to inappropriate char, if such is
encountered. hits=0 ==> no expressions parsed.*/ 
int *hits;
char *dpoint,**endptr;
struct p_data *p;
{
	int count,v1,v2,n,m,flag,i,cflg,localcount,fhold,
		q,qcount,q_flag,cf_flag=0;
	char next[256],*nextptr,*lastptr,phold[1];
	struct Vertlist *facelist=NULL,*trace,*clobber,*tmp_ptr;
	extern struct Vertlist *brace_parse();
	f_data *qfaces;

	nextptr=lastptr=dpoint;
	*endptr=NULL;
	count=0;
	*hits=0;
	if (*nextptr == '\0' || !grab_next(&nextptr,next)) return NULL;
	*hits=1;
	facelist=(struct Vertlist *)calloc(1,sizeof(struct Vertlist));
	trace=facelist;
	clobber=trace;
	do
	 {
		lastptr += notspace(lastptr);
		if (*next=='a') /* all */
		 {
			lastptr++;
			n=parse_parens(lastptr,&v1,&v2);
			if ( n == (-1) || (n>0 && v2<v1))
				/* malformed expression */
			 {
				free(trace);
				clobber->next=NULL;
				if (!count) return NULL;
				*hits=count;
				return facelist;
			 } 
			else if (n==0)
				/* run thru whole complex */
			 {
				v1=1;
				v2=p->facecount;
			 }
			if (v1<1) v1=1;
			if (v2>p->facecount) v2=p->facecount;
			if (lastptr>nextptr) nextptr=lastptr;
				/* reaching here, should have v1 and v2 */
			for (n=v1;n<=v2;n++)
			 {
				trace->v=n;
				trace->next=(struct Vertlist *)
					calloc(1,sizeof(struct Vertlist));
				clobber=trace;
				trace=trace->next;
				count++;
			 }
		 } /* finished with 'all' */
		else if (*next=='b') /* boundary */
		 {
			lastptr++;
			n=parse_parens(lastptr,&v1,&v2);
			if ( n == (-1) )
				/* malformed expression */
			 {
				free(trace);
				clobber->next=NULL;
				if (!count) return NULL;
				*hits=count;
				return facelist;
			 } 
			else if (n==0)
				/* run thru all bdry */
			 {
				v1=1;
				v2=p->facecount;
			 }
			if (v1<1) v1=1;
			if (v2>p->facecount) v2=p->facecount;
			for (n=v1;n<=v2;n++)	
			 {
			   flag=0;
			   for (i=0;i<3;i++) 
			     if (p->packK_ptr[p->faces[n].vert[i]].bdry_flag)
				flag++;
			   if (flag)
				 {
					trace->v=n;
					trace->next=(struct Vertlist *)
					   calloc(1,sizeof(struct Vertlist));
					clobber=trace;
					trace=trace->next;
					count++;
				 }
			 }
		 } /* finished with bdry */
		else if (*next=='r' 
			&& (n=p->first_red_face)!=0)    /* red faces */
		 {
			localcount=0;
			lastptr++;
			if (sscanf(next,"%d",&v1) && v1>0 && v1<=p->facecount
				&& p->faces[v1].rwb_flag) n=v1; 
					/* start with n */
			v2=n;
			trace->v=n;
			trace->next=(struct Vertlist *)
				calloc(1,sizeof(struct Vertlist));
			clobber=trace;
			trace=trace->next;
			localcount++;
			fhold=n;
			n=p->faces[n].next_red;
			while ( n>0 && n<=p->facecount 
				&& n!=v2 && localcount<(2*p->facecount) )
			 {
				trace->v=n;
				trace->next=(struct Vertlist *)
				  calloc(1,sizeof(struct Vertlist));
				clobber=trace;
				trace=trace->next;
				localcount++;
				if (p->faces[n].rwb_flag==2) 
				   /* back-track one face when blue is hit */
				 {
					trace->v=fhold;
					trace->next=(struct Vertlist *)
					  calloc(1,sizeof(struct Vertlist));
					clobber=trace;
					trace=trace->next;
					localcount++;
				 }
				fhold=n;
				n=p->faces[n].next_red;
			 }
			count += localcount;
		 }
		else if (*next=='w') /* white faces in draw order, as possib.*/
		 {
			lastptr++;
			n=parse_parens(lastptr,&v1,&v2);
			if ( n == (-1) )
				/* malformed expression */
			 {
				free(trace);
				clobber->next=NULL;
				if (!count) return NULL;
				*hits=count;
				return facelist;
			 } 
			else if (n==0)
				/* run thru all faces */
			 {
				v1=1;
				v2=p->facecount;
			 }
			if (v1<1) v1=1;
			if (v2>p->facecount) v2=p->facecount;
			n=m=p->first_face;
			do
			 {
				if (n>=v1 && n<=v2 && !p->faces[n].rwb_flag)
				 {
				   trace->v=n;
				   trace->next=(struct Vertlist *)
					calloc(1,sizeof(struct Vertlist));
				   clobber=trace;
				   trace=trace->next;
				   count++;
				 }
				n=p->faces[n].next_face;
			 }
			while (n!=m);
		 }  
		else if (*next=='m') /* marked */
		 {
			if (cflg=(next[1]=='c')) lastptr++; 
				/* 1==>complement of marked */
			if (next[1]=='p' || (cflg && next[2]=='p') )
				/* take marking from pack q */
			 {
				lastptr++;
				if (next[1]=='p') phold[0]=next[2];
				else phold[0]=next[3];
				q=atoi(phold);
				if (q>=0 && q<NUM_PACKS && packdata[q].status)
				 {
					lastptr++;
					q_flag=1;
					qfaces=packdata[q].faces;
					qcount=packdata[q].facecount;
				 }
				else q_flag=0;
			 }				
			else q_flag=0;				
			lastptr++;
			n=parse_parens(lastptr,&v1,&v2);
			if ( n == (-1)  || (n>0 && v2<v1))
				/* malformed expression */
			 {
				free(trace);
				clobber->next=NULL;
				if (!count) return NULL;
				*hits=count;
				return facelist;
			 } 
			else if (n==0)
				/* run thru whole complex */
			 {
				v1=1;
				v2=p->facecount;
			 }
			if (v1<1) v1=1;
			if (v2>p->facecount) v2=p->facecount;
			if (lastptr>nextptr) nextptr=lastptr;
				/* reaching here, should have v1 and v2 */
			for (n=v1;n<=v2;n++)
				if ( (!cflg && !q_flag && p->faces[n].mark) 
				   || (cflg && !q_flag && !p->faces[n].mark) 
				   || (!cflg && q_flag && n<=qcount
					&& qfaces[n].mark)
				   || (cflg && q_flag && n<=qcount
					&& !qfaces[n].mark) )
				 {
					trace->v=n;
					trace->next=(struct Vertlist *)
					  calloc(1,sizeof(struct Vertlist));
					clobber=trace;
					trace=trace->next;
					count++;
				 }
		 } 
		else if (*next=='{')
		 {
			tmp_ptr=brace_parse(p,lastptr,&cf_flag,&lastptr);
			if (tmp_ptr!=NULL) /* fit into growing list */
			 {
				if (facelist==trace) facelist=tmp_ptr;
				free(trace);
				clobber=tmp_ptr;
				count++;
				while (clobber->next!=NULL)
				 {
					count++;
					clobber=clobber->next;
				 }
				clobber->next=(struct Vertlist *)
					calloc(1,sizeof(struct Vertlist));
				trace=clobber->next;
			 }
		 }
		else if (sscanf(next,"%d",&v1) && v1>0 && v1<=p->facecount) 
		 {
			trace->v=v1;
			trace->next=(struct Vertlist *)
				calloc(1,sizeof(struct Vertlist));
			clobber=trace;
			trace=trace->next;
			count++;
		 }
		else /* inappropriate entry */
		 {
			*endptr=lastptr;
			if (trace!=clobber) /* some valid hits */
				clobber->next=NULL;
			free(trace);trace=NULL;
			if (!count) return NULL;
			*hits=count;
			return facelist;
		 }
		lastptr=nextptr;
	 }
	while (grab_next(&nextptr,next));
	if (trace!=clobber) clobber->next=NULL;
	free(trace);trace=NULL;
	if (!count) return NULL;
	*hits=count;
	return facelist;
} /* face_link_parse */

struct Edgelist *
node_pair_link(p,dpoint,endptr,hits) /* read string of node pairs into
linked list, endptr=NULL or pts to inappropriate char, it such is
encountered or odd number of nodes is listed. hits=0 ==> no expressions 
parsed.*/
int *hits;
char *dpoint,**endptr;
struct p_data *p;
{
	int count,nextvert,stopvert,v1,v2,i,j,k,v;
	char next1[256],next2[256],*nextptr,*lastptr;
	struct K_data *pK_ptr;
	struct Edgelist *edgelist=NULL,*trace,*clobber;

	pK_ptr=p->packK_ptr;
	stripsp(dpoint);
	nextptr=lastptr=dpoint;
	*endptr=NULL;
	count=0;
	*hits=0;
	if (*nextptr == '\0' || !grab_next(&nextptr,next1) ) return NULL;
	*hits=1;
	edgelist=(struct Edgelist *)malloc(sizeof(struct Edgelist));
	edgelist->next=NULL;
	trace=edgelist;
	clobber=trace;
	do
	 {
		lastptr += notspace(lastptr);
		if (next1[0]=='a') /* do all edges */
		 {
			for (i=1;i<p->nodecount;i++) 
			for (j=0;j<=pK_ptr[i].num;j++)
			   if ((k=pK_ptr[i].flower[j])>i)
			    {
				trace->v=i;trace->w=k;
				trace->next=(struct Edgelist *)
					calloc(1,sizeof(struct Edgelist));
				clobber=trace;
				trace=trace->next;
				count++;
			     }
		 }
		else if (next1[0]=='b') /* all boundary edges */
		 {
		   lastptr++;
		   for (j=1;j<=p->num_bdry_comp;j++)
		    {
			v1=stopvert=p->bdry_starts[j];
			nextvert=pK_ptr[v1].flower[0];
			trace->v=v1;trace->w=nextvert;
			trace->next=(struct Edgelist *)
				calloc(1,sizeof(struct Edgelist));
			clobber=trace;
			trace=trace->next;
			count++;
			do
			 {
				v1=nextvert;
				nextvert=pK_ptr[v1].flower[0];
				trace->v=v1;trace->w=nextvert;
				trace->next=(struct Edgelist *)
					calloc(1,sizeof(struct Edgelist));
				clobber=trace;
				trace=trace->next;
				count++;
			 }
			while (nextvert!=stopvert);
		    }
		 }
		else if (next1[0]=='m') /* all with both ends marked */
		 {
		   lastptr++;
		   for (j=1;j<p->nodecount;j++)
		    {
			if (pK_ptr[j].mark) 
			 {
			   for (k=0;k<=pK_ptr[j].num;k++)
			    {
				if (pK_ptr[(v=pK_ptr[j].flower[k])].mark
					&& v>j)
				 {
				   trace->v=j;trace->w=v;
				   trace->next=(struct Edgelist *)
				      calloc(1,sizeof(struct Edgelist));
				   clobber=trace;
				   trace=trace->next;
				   count++;
				 }
			    }
			 }
		    }
		 }
		else
		 {
			if (sscanf(next1,"%d",&v1)==1  
				&& v1>0 && v1<=p->nodecount
				&& grab_next(&nextptr,next2)
				&& sscanf(next2,"%d",&v2)==1 
				&& v2>0 && v2<=p->nodecount)
			 {
				if (nghb(p,v1,v2)!=-1)
				 {
				   trace->v=v1;trace->w=v2;
				   trace->next=(struct Edgelist *)
					calloc(1,sizeof(struct Edgelist));
				   clobber=trace;
				   trace=trace->next;
				   count++;
				 }
			 }
			else /* inappropriate entry */
			 {
				*endptr=lastptr;
				if (trace!=clobber) /* some valid hits */
				   clobber->next=NULL;
				free(trace);trace=NULL;
				if (!count) return NULL;
				*hits=count;
				return edgelist;
			 }
		 }
		lastptr=nextptr;
	 }
	while (grab_next(&nextptr,next1) );
	if (trace!=clobber) clobber->next=NULL;
	free(trace);trace=NULL;
	if (!count) return NULL;
	*hits=count;
	return edgelist;
} /* node_pair_link */

struct Edgelist *
get_extended_edge(p,v1,v2,lgth) /* build edgelist from v1 to v2
through up to 'lgth' hex interior vertices (if possible). */
struct p_data *p; 
int v1,v2,lgth;
{
	int next=0,i,dir,w;
	struct Edgelist *elist=NULL,*trace;
	struct K_data *pK_ptr;
	extern int hex_proj();
	extern void edge_free();

	if (!p->status || v1==v2) return NULL ;
	pK_ptr=p->packK_ptr;
	if ((dir=nghb(p,v1,v2))!=-1) /* immediate neighbor */
	 {
		elist=(struct Edgelist *)calloc(1,sizeof(struct Edgelist));
		elist->v=v1;
		elist->w=v2;
		return elist;
	 }
	for (dir=0;dir<pK_ptr[v1].num+pK_ptr[v1].bdry_flag;dir++)
		/* search in this direction */
	 {
	   elist=(struct Edgelist *)calloc(1,sizeof(struct Edgelist));
	   elist->v=v1;
	   elist->w=pK_ptr[v1].flower[dir];
	   trace=elist;
	   i=1;
	   while (i<lgth && (next=hex_proj(p,trace->w,trace->v)) 
		&& next!=v2)
	    {
		w=trace->w;
		trace=trace->next=
		   (struct Edgelist *)calloc(1,sizeof(struct Edgelist));
		trace->v=w;
		trace->w=next;
		i++;
	    }
	   if (next==v2)   /* found it! */
	    {
 		w=trace->w;
		trace=trace->next=
		   (struct Edgelist *)calloc(1,sizeof(struct Edgelist));
		trace->v=w;
		trace->w=next;
		return elist;
	    }
	   edge_free(&elist);
	 }
	return NULL;
} /* get_extended_edge */

void vert_free(vertlist)
struct Vertlist **vertlist;
{
	struct Vertlist *trace,*clobber;

	if (*vertlist==NULL) return;
	trace=*vertlist;
	while (trace!=NULL)
	 {
		clobber=trace;
		trace=trace->next;
		free(clobber);
	 }
	*vertlist=NULL;
} /* vert_free */

void path_free(pathlist)
struct Pathlist **pathlist;
{
	struct Pathlist *trace,*clobber;

	if (*pathlist==NULL) return;
	trace=*pathlist;
	while (trace!=NULL)
	 {
		clobber=trace;
		trace=trace->next;
		free(clobber);
	 }
	*pathlist=NULL;
} /* path_free */

void edge_free(edgelist)
struct Edgelist **edgelist;
{
	struct Edgelist *trace,*clobber;

	if (*edgelist==NULL) return;
	trace=*edgelist;
	while (trace!=NULL)
	 {
		clobber=trace;
		trace=trace->next;
		free(clobber);
	 }
	*edgelist=NULL;
} /* edge_free */

int
cookie_cutter(p) /* uses current path to cut out part of pack p */
struct p_data *p;
{
	int i,j,k,pnum,Nodecount,Alpha,Beta,*newnum=NULL,length;
	int tick,tock,level, toplevel,finalcount,count;
	struct K_data *pK_ptr=NULL,*nK_ptr=NULL;
	struct R_data *pR_ptr,*packRbkp=NULL;
	XPoint *Xptr;
	extern XPoint *path_XPoints();
	extern int alloc_pack_space(),find_next();


	if (pathlist==NULL || p->locks) return 0; /* no path or pack locked */
	newnum=(int *)calloc((size_t)(p->nodecount+1),sizeof(int));
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	Nodecount=p->nodecount;Alpha=p->alpha;Beta=p->beta;
	pnum=pack_num(p);
	if ((nK_ptr=(struct K_data *)calloc
		((size_t)(Nodecount+1),sizeof(struct K_data)))==NULL
	   || (packRbkp=
		(struct R_data *)malloc
			((Nodecount+1)*sizeof(struct R_data)))==NULL)
		   goto ABORT;
	for (i=1;i<=Nodecount;i++)
	 {
		nK_ptr[i]=pK_ptr[i];
		packRbkp[i]=pR_ptr[i];
		nK_ptr[i].flower=(int *)malloc((nK_ptr[i].num+1)*sizeof(int));
		for (j=0;j<=nK_ptr[i].num;j++) 
			nK_ptr[i].flower[j]=pK_ptr[i].flower[j];
	 }
	tick=0;
	length=pathlength;
	Xptr=path_XPoints(p->screen,pathlist,&length);
	region=XPolygonRegion(Xptr,length,WindingRule);
	free(Xptr);

/* set flags. plot_flag=1 means inside. */
	for (i=1;i<=Nodecount;i++) 
	 {
		if (path_wrap(pnum,pR_ptr[i].center))
			nK_ptr[i].plot_flag=tick=1;
		else 
		 {
			nK_ptr[i].plot_flag=0;
			tock=1;
		 }
		nK_ptr[i].bdry_flag=-1;
	 }
	XDestroyRegion(region);
	if (!tick || !tock) goto ABORT; 
			/* all points are in/outside the path. */
	if (!nK_ptr[Alpha].plot_flag) /* ensure that Alpha is "inside" */ 
		for (i=1;i<=Nodecount;i++) 
		 {
			if (nK_ptr[i].plot_flag) nK_ptr[i].plot_flag=0;
			else nK_ptr[i].plot_flag=1;
		 }
	for (i=1;i<=Nodecount;i++) /* find interior ones */
	 {
		k=nK_ptr[i].plot_flag;
		for (j=0;j<=nK_ptr[i].num;j++)
			k *= nK_ptr[nK_ptr[i].flower[j]].plot_flag;
		if (k && nK_ptr[i].flower[0]==nK_ptr[i].flower[nK_ptr[i].num]) 
			nK_ptr[i].bdry_flag=0;
	 }
	for (i=1;i<=Nodecount;i++) /* find bdry ones = inside, not interior */
		if (nK_ptr[i].plot_flag && nK_ptr[i].bdry_flag) 
			nK_ptr[i].bdry_flag=1;
	if (nK_ptr[Alpha].bdry_flag)
	 {
		strcpy(msgbuf,"Alpha vertex is not interior to this path.");
		emsg();
		goto ABORT;
	 }

/* label interiors by generation, starting at alpha */
	for (i=1;i<=Nodecount;i++) nK_ptr[i].plot_flag=-1;
	nK_ptr[Alpha].plot_flag=1;
	level=1;
	do
	 {
		tick=0;
		for (i=1;i<=Nodecount;i++)
		if (nK_ptr[i].plot_flag==level)
		 {
			tick++;
			for (j=0;j<=nK_ptr[i].num;j++)
			   if (nK_ptr[nK_ptr[i].flower[j]].plot_flag<0
				&& nK_ptr[nK_ptr[i].flower[j]].bdry_flag==0)
			      nK_ptr[nK_ptr[i].flower[j]].plot_flag=level+1;
		 }
		level++;
	 }
	while (tick);
	toplevel=level-2;

/* identify bdry verts without interior neighbors. */
	for (i=1;i<=Nodecount;i++) 
	 {
		k=0;
		if (nK_ptr[i].bdry_flag==1)
		 {
			for (j=0;j<=nK_ptr[i].num;j++) 
			   if (nK_ptr[nK_ptr[i].flower[j]].plot_flag>0) k++;
			if (!k) nK_ptr[i].bdry_flag=-1;
		 }
	 }

/* Now have:
--	plot_flag>0 gives generation of interiors connected to alpha;
--	plot_flag is -1 for other vertices;
--	interiors have bdry_flag 0;
--	boundary nodes next to interiors have bdry_flag 1;
--	all non-interior, non-bdry vertices have bdry_flag -1;
*/

	for (i=1;i<=Nodecount;i++) 
		if (nK_ptr[i].bdry_flag==1) { Beta=i;break;}
	if (i==Nodecount+1) goto ABORT; 
				/* no boundary? */ 
		
/* Give new node numbers for interior verts. */

	newnum[Alpha]=1;
	count=1;
	level=2;
	while (level<=toplevel)
	 {
		for (i=1;i<=Nodecount;i++) 
			if (!newnum[i] && (nK_ptr[i].plot_flag==level))
				newnum[i]=++count;
		level++;
	 }
	
/* also for boundary verts (while fixing their flowers). */

	tick=Beta;newnum[Beta]=++count;
	while ( (tick=find_next(nK_ptr,tick))>0 && tick!=Beta) 
		newnum[tick]=++count;
	if (!tick) goto ABORT;	
			/* error while going around boundary */
	do /* catch rest of boundary components */
	 {
		tock=0;
		for (i=1;i<=Nodecount;i++)
			if (!newnum[i] && (nK_ptr[i].bdry_flag==1) )
			 {
			   tock=i;
			   newnum[i]=++count;
			   while ( (tock=find_next(nK_ptr,tock))>0 && tock!=i)
				newnum[tock]=++count;
			   if (!tock) goto ABORT;
						/* error */
			 }
	 }
	while (tock);
	finalcount=count;
	if (finalcount==Nodecount) goto ABORT;	
			/* there has been no change */

/* old data corrupted */

	p->alpha=p->active_node=1;
	p->beta=newnum[Beta];
	if (newnum[p->gamma]) p->gamma=newnum[p->gamma];
	else p->gamma=p->beta;
/* fixup ??: could try to save overlaps */
	free_overlaps(p);
	sprintf(p->file_name,"Cookie"); 

/* move data into p */
          
	for (i=1;i<=Nodecount;i++) if (newnum[i]) /* fix flower */
		for (j=0;j<=nK_ptr[i].num;j++)
			if (newnum[nK_ptr[i].flower[j]])
			   nK_ptr[i].flower[j]=newnum[nK_ptr[i].flower[j]];

	for (i=1;i<=Nodecount;i++) 		
	   free(pK_ptr[i].flower);pK_ptr[i].flower=NULL;

	for (i=1;i<=Nodecount;i++) 		
	 {
	   if (newnum[i]) 
	    {
		pK_ptr[newnum[i]]=nK_ptr[i];
		pR_ptr[newnum[i]]=packRbkp[i]; /* get old data from backup */
	    }
	   else free(nK_ptr[i].flower); 
	 }
	if (nK_ptr) free(nK_ptr);
	if (packRbkp) free(packRbkp);
	free(newnum);

	p->nodecount=finalcount;
	alloc_pack_space(p,p->nodecount,1); /* may have gotten smaller */
	complex_count(p,FALSE);
	facedraworder(p,NULL);
	fillcurves(p);
	set_aim_default(p);
	return 1;

ABORT: /* original p should be okay */
	if (nK_ptr)
	 {
		for (i=1;i<=Nodecount;i++)
			if (nK_ptr[i].flower) free(nK_ptr[i].flower);
		free(nK_ptr);
	 }
	if (packRbkp) free(packRbkp);
	free(newnum);
	return 0;
	
} /* cookie_cutter */
		
int
find_next(pK,v) /* Called by cookie_cutter: given a bdry vert v, 
find next new bdry vertex ctr-clkwise from v and fix flower of v. */
struct K_data *pK;
int v;
{
	int fnum,startv=0,endv,k,period,newstart,j;
	int *newflower;
	int *pf; /* pointer to flower */
	
	pf=pK[v].flower;
	fnum=pK[v].num;

/* check for problems */

	k=0;
	for (j=0;j<fnum;j++)
		if (pK[pf[j]].plot_flag*pK[pf[j+1]].plot_flag<0) k++;
	if (k>2) return 0; /* v hits interior in two places, error. */
	if (pf[0]!=pf[fnum] 
		&& (pK[pf[0]].plot_flag>0 || pK[pf[fnum]].plot_flag>0) ) 
			return 0; /* shouldn't be possible */
	k=0;
	for (j=0;j<pK[v].num;j++)
		if (pK[pf[j]].bdry_flag==1) k++;
	if (pf[0]!=pf[pK[v].num] && pK[pf[pK[v].num]].bdry_flag==1) k++;
	if (k<=1) return 0; /* v has only one bdry neighbor */

/* rotate petal so first and last are not interior. */

	while (startv<(fnum-1) && pK[pf[startv]].bdry_flag!=1) startv++;
	if (startv==(fnum-1)) return 0;
	if (pK[pf[startv+1]].plot_flag<1) newstart=startv+1;
	else if (startv>0 && pK[pf[startv-1]].plot_flag<1) newstart=startv;
	else if (startv==0 && pf[0]==pf[fnum] && pK[pf[fnum-1]].plot_flag<1)
		newstart=startv;
	else if (startv==0 && pf[0]!=pf[fnum]) newstart=startv;
	else return 0;
	if (pf[0]==pf[fnum]) fnum--; /* break petal, if needed */
	period=fnum+1;
	newflower=(int *)malloc(period*sizeof(int));
	for (j=0;j<=fnum;j++) newflower[j]=pf[(newstart+j) % period];
	free(pf);
	pf=newflower;

/* now find bdry petal with interior next petal */
	
	startv=0;
	while (startv<(fnum-1) && pK[pf[startv+1]].plot_flag<1) startv++;
	if (startv==(fnum-1)) return 0; /* are no interior neighbors */
	if (pK[pf[startv]].bdry_flag!=1) return 0; /* error */
	endv=startv+2;
	while (endv<=fnum && pK[pf[endv]].plot_flag>0) endv++;
	if (endv==fnum && pK[pf[endv]].plot_flag>0) return 0;
	if (pK[pf[endv]].bdry_flag!=1) return 0;
	pK[v].num=endv-startv;
	newflower=(int *)malloc((pK[v].num+1)*sizeof(int));
	for (j=0;j<=pK[v].num;j++) newflower[j]=pf[j+startv];
	free(pf);
	pK[v].flower=newflower;
	return pK[v].flower[0];
} /* find_next */

int
path_wrap(q,pt) /* return 1 if path is wraped around pt */	
int q;
complex pt;
{
	complex normpt;	
	int x,y;

	r_to_pix(pt,&normpt,screendata[q].pix_box,
		screendata[q].box,Aspect);
	x=(int)normpt.re;y=(int)normpt.im;
	return (XPointInRegion(region,x,y));
} /* path_wrap */

int
find_path_int(p) /* identify vertices interior to path */
struct p_data *p;
{
	int i,length;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	XPoint *Xptr;
	extern XPoint *path_XPoints();

	if (pathlist==NULL || !p->status) return 0;	
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	length=pathlength;
	Xptr=path_XPoints(p->screen,pathlist,&length);
	region=XPolygonRegion(Xptr,length,WindingRule);
	free(Xptr);
	for (i=1;i<=p->nodecount;i++) /* set plot_flag for interior */
	 {
		if (path_wrap(pack_num(p),pR_ptr[i].center))
			pK_ptr[i].plot_flag=1;
		else pK_ptr[i].plot_flag=0;
	 }
	XDestroyRegion(region);
	if (!pK_ptr[p->alpha].plot_flag) 
		for (i=1;i<=p->nodecount;i++)
		 {
			if (pK_ptr[i].plot_flag) 
				pK_ptr[i].plot_flag=0;
			else pK_ptr[i].plot_flag=1;
		 }
	return 1;
} /* find_path_int */

int
parse_parens(datastr,v1,v2) /* find/handle '(n,m)' range indicators. 
return 0 if none found, -1 if malformed, 1 if okay. reset datastr.*/
char *datastr;
int *v1,*v2;
{
	int m,n;
	char *temp,*lastptr;

	lastptr=datastr;
	lastptr=lastptr+notspace(lastptr);
	if ( *lastptr=='\0' 
	   || *lastptr!='(' ) 
	 {
		datastr=lastptr;
		return 0;
	 }
	lastptr += notspace(lastptr)+1; 
			/* pt after '(' */
	temp=lastptr;
	n=0;
	while (*(temp+n)!=',' && *(temp+n)!=')' 
	   && *(temp+n)!='\0' ) n++;
	if ( *(temp+n)==')' || *(temp+n)=='\0'
	   || !sscanf(temp,"%d",v1)
	   || *(temp=(temp+n+1))=='\0' ) 
			/* malformed pt after ',' */
	 {
		datastr=lastptr+n;
		return -1;
	 }
	m=0;
	while (*(temp+m)!=')' && *(temp+m)!='\0') m++; 
	if (*(temp+m)=='\0' || !sscanf(temp,"%d",v2))
			/* malformed */	
	 {
		datastr=temp+m;
		return -1;
	 }
	datastr=temp+m;
	return 1;
} /* parse_parens */

struct Vertlist *
brace_parse(p,datastr,n_flag,lastptr) /* find vert/face numbers
from set-builder form. Format is very precise. 
n_flag indicates circles/faces. Incoming:0==>faces, 
1==>circles, -1==>either. Return value same, only
-1 indicates malformed. */
struct p_data *p;
char *datastr,**lastptr;
int *n_flag;
{
	int cond_count=0,tar_p,click,i,node,first_one=1,cc,
		bool_flag,cum_result,count,
		stop_flag=0,cond_connect[5];
	char full_expr[1024],*fe,*exp_ptr,*endptr,*holdptr,
		*tar_str[5],*cond_str[5],next[256],*nextptr,
		*next_expr=NULL;
	float val[5];
	struct Vertlist *nodelist=NULL,*trace;
	extern char *pair_picking();

/* verify existence and proper form of six
	main portions of description: 
	1. Outer curly brackets.	2. Target object: v, f, 
	3. Target packing	(boolean recursive)
	4. Target quantity: rad=r, degree=d, bdry=b, int=i, 
		angle sum=s, aim=a, marked=m, ratio(p,q)=epq,
		ratio(p,q)=cpq, modulus of (eucl) center=z (or ze).
	5. Comparison: =, <=, <, >=, >
	6. value  	7. Connective: &&, || (inclusive), 
		or ! (meaning 'and not')  */

	for (i=0;i<5;i++) tar_str[i]=cond_str[i]=NULL;
/* pick out full expression */

	if ( (fe=pair_picking(datastr,'{','}',lastptr))== NULL) 
		/* not well-formed set */
	 {*n_flag=-1;return NULL;}
	strcpy(full_expr,fe);
	free(fe);
	stripsp(full_expr);
	if ( (full_expr[0]=='c' || full_expr[0]=='v') 
		&& (*n_flag==(-1) || *n_flag==1) ) *n_flag=1; 
			/* look for circles */
	else if (*full_expr=='f'  
		&& (*n_flag==(-1) || *n_flag==0) ) *n_flag=0; 
			/* look for faces */
	else {*n_flag=-1;return NULL;}

/* prefix: pick off target packing. e.g., v -p1: */

	exp_ptr=full_expr+1;
	stripsp(exp_ptr);
	if (*exp_ptr==':') tar_p=pack_num(p);
		/* use calling pack */
	else 
	 {
		if (*exp_ptr!='-' || *(exp_ptr+1)!='p' 
		   || (tar_p=atoi(exp_ptr+2))< 0 
		   || tar_p>=NUM_PACKS || !packdata[tar_p].status)
		 {*n_flag=-1;return NULL;}
		exp_ptr += 3;
		stripsp(exp_ptr);
		if (*exp_ptr!=':') 
		 {*n_flag=-1;return NULL;}
	 }
	exp_ptr++;
	if (*n_flag) count=packdata[tar_p].nodecount;
	else count=packdata[tar_p].facecount;

/* pick off various conditions */

	endptr=exp_ptr;
	stripsp(endptr);
	while ( cond_count<5 && !stop_flag )
	 {
		stop_flag=0;
		if (*exp_ptr!='[') stop_flag=1;
			/* no brackets? treat as single cond'n */
		if (!stop_flag 
			&& (next_expr=pair_picking(endptr,
			   '[',']',&endptr))==NULL)
		 {*n_flag=-1;return NULL;}
		if (stop_flag) nextptr=endptr;
		else nextptr=next_expr;
/* parse within [] */
	/* get target */
		holdptr=nextptr;
		stripsp(holdptr);
		if (!grab_next(&nextptr,next)) /* no target */
		 {*n_flag=-1;free(next_expr);return NULL;}
		i=0;
		while (next[i]!='<' && next[i]!='=' && next[i]!='>'
			&& next[i]!='\0') i++;
		next[i]='\0';
		nextptr=holdptr+i;
		tar_str[cond_count]=
			(char *)malloc(strlen(next)*sizeof(char));
		strcpy(tar_str[cond_count],next);
	/* get condition */
		holdptr=nextptr;
		stripsp(holdptr);
		if (grab_next(&nextptr,next)) /* condition */
		 {
			i=0;
			while (next[i]=='<' 
				|| next[i]=='=' || next[i]=='>') i++;
			next[i]='\0';
			nextptr=holdptr+i;
			cond_str[cond_count]=
			   (char *)malloc(strlen(next)*sizeof(char));
			strcpy(cond_str[cond_count],next);
	/* get value */
			if (grab_next(&nextptr,next))
			 {
				if (!sscanf(next,"%lf",&val[cond_count]))
				 {
					free(cond_str[cond_count]);
					cond_str[cond_count]=NULL;
				 } /* no value, so invalid */
			 }
		 }
		stripsp(endptr);
/* parse logical connective */
		if (*endptr=='\0') stop_flag=1;
		if (!stop_flag)
		 {
			if (*endptr=='&' && *(endptr+1)=='&')
			 {
				endptr += 2;
				cond_connect[cond_count]=1;
			 }
			else if (*endptr=='|' && *(endptr+1)=='|')
			 {
				endptr += 2;
				cond_connect[cond_count]=2;
			 }
			else if (*endptr=='!')
			 {
				endptr += 1;
				cond_connect[cond_count]=3;
			 }
			else stop_flag++;
				/* some error */
		 }
		stripsp(endptr);
		cond_count++;
		free(next_expr);next_expr=NULL;
	 } /* end of while */
	if (!cond_count) {*n_flag=-1;return NULL;} 
		/* no conditions */

/* now, cycle through indices */

	for (node=1;node<=count;node++)
	 {
		cum_result=1;
		click=0;
		while ( (click < cond_count) )
		 {
			bool_flag=cond_test(&packdata[tar_p],
				tar_str[click],cond_str[click],
				val[click],node,*n_flag);
			if (bool_flag==(-1)) /* test failed */
			 {
				cum_result=0;
				click=cond_count;
			 }
			else if (click==0) /* first pass thru */
			 {
				cum_result=bool_flag;
				click++;
			 }
			else /* need logical connector */
			 {
				cc=cond_connect[click-1];
				if ( (cc==1 && bool_flag && cum_result)
				   || (cc==2 && 
				   (bool_flag || cum_result))
				   || (cc==3 && 
				   cum_result && !bool_flag) )
					cum_result=1;
				else cum_result=0;
				click++;
			 }
		 } /* end of while */	
		if (cum_result) /* add to list */
		 {
			if (first_one)
			 {
			   nodelist=(struct Vertlist *)
				calloc(1,sizeof(struct Vertlist));
			   nodelist->v=node;
			   trace=nodelist;
			   first_one=0;
			 }
			else
			 {
			   trace->next=(struct Vertlist *)
				    calloc(1,sizeof(struct Vertlist));
			   trace=trace->next;
			   trace->v=node;
			 }
		 }
	 } /* end of for loop */
	for (i=0;i<5;i++)
	 { free(tar_str[i]);free(cond_str[i]);}
	return nodelist;
} /* brace_parse */

int
cond_test(p,tar_str,cond_str,val,node,cf) 
/* parse keystr to see if node fits. cf=1==>circle,
cf=0==>face */
struct p_data *p;
char *tar_str,*cond_str;
float val;
int node,cf;
{
	int pp,pq;
	float target_quant,rp,rq;
	complex ctrp,ctrq;

	switch (*tar_str)
	 {
case 'a': /* aim? */
	if (!cf) return -1;
	return comparison(cond_str,(p->packR_ptr[node].aim/M_PI),val);
case 'b': /* boundary object? */
 {
	if (cf) return p->packK_ptr[node].bdry_flag;
	else return (
	   p->packK_ptr[p->faces[node].vert[0]].bdry_flag
	   || p->packK_ptr[p->faces[node].vert[1]].bdry_flag 
	   || p->packK_ptr[p->faces[node].vert[2]].bdry_flag);
 }
case 'd': /* degree? */
	if (!cf) return -1;
	target_quant=(float)(p->packK_ptr[node].num+
		(p->packK_ptr[node].bdry_flag!=0)); /* add 1 if bdry vert */
	return comparison(cond_str,target_quant,val);
case 'c': /* ratio of radii? */
case 'e': /* ratio using euclidean radii? */
 {
	if (!cf) return -1;
	if (*(tar_str+1)>='0' && *(tar_str+1)<='9') pp=*(tar_str+1)-'0';
	if (*(tar_str+2)>='0' && *(tar_str+1)<='9') pq=*(tar_str+2)-'0';
	if (pp<0 || pq<0 || pp>=NUM_PACKS || pq>=NUM_PACKS)
		return -1;
	if (node>packdata[pp].nodecount 
	   || node>packdata[pq].nodecount) 
		return 0;
	rq=packdata[pq].packR_ptr[node].rad;
	rp=packdata[pp].packR_ptr[node].rad;
	if (*tar_str=='e') /* compare in eucl geom */ 
	 {
		if (packdata[pp].hes<0) 
			h_to_e_data(packdata[pp].packR_ptr[node].center,
			rp,&ctrp,&rp);
		if (packdata[pq].hes<0) 
			h_to_e_data(packdata[pq].packR_ptr[node].center,
			rq,&ctrq,&rq);
	 }
	if (rq<okerr) /* too small */
		return 0;
	target_quant=rp/rq;
	return comparison(cond_str,target_quant,val);
 }
case 'i': /* interior object? */
 {
	if (cf) return !p->packK_ptr[node].bdry_flag;
	else return (
	   !p->packK_ptr[p->faces[node].vert[0]].bdry_flag
	   && !p->packK_ptr[p->faces[node].vert[1]].bdry_flag 
	   && !p->packK_ptr[p->faces[node].vert[2]].bdry_flag);
 }
case 'm': /* marked object? */
 {
	if (cf) return p->packK_ptr[node].mark;
	else return p->faces[node].mark;
 }
case 'n': /* vert or face index */
 {
	return comparison(cond_str,node,val);
 }
case 'r': /* rad? */
	if (!cf) return -1;
	return comparison(cond_str,p->packR_ptr[node].rad,val);
case 's': /* angle sum? */
	if (!cf) return -1;
	return comparison(cond_str,(p->packR_ptr[node].curv/M_PI),val);
case 'z': /* modulus of center */
 {
	if (!cf || p->hes>0) return -1; /* not yet for sph case */
	ctrp=p->packR_ptr[node].center;
	if (p->hes<0 && *(tar_str+1)=='e') /* want eucl center */
		h_to_e_data(ctrp,p->packR_ptr[node].rad,&ctrp,&rp);
	target_quant=cAbs(ctrp); 
	return comparison(cond_str,target_quant,val);
 }
	 } /* end of switch */
	return 0;
} /* cond_test */

int
comparison(datastr,x,y) /* compare x to y: return 0
for failure or error.*/
char *datastr;
float x,y;
{
	float diff;

	diff=x-y;
	if (datastr==NULL) return 0; /* no compare */
	if (*datastr=='=') return (fabs(diff)<okerr);
	if (*datastr=='>')
	 {
		if (*(datastr+1)=='=') return (diff>=0.0);
		else return (diff>0.0);
	 }
	if (*datastr=='<')
	 {
		if (*(datastr+1)=='=') return (diff<=0.0);
		else return (diff<0.0);
	 }
	return 0;
 } /* comparison */

char *
pair_picking(datastr,start_char,end_char,endptr) 
/* return string between start_char and end_char. 
NULL means one was not found. */
char *datastr,start_char,end_char,**endptr;
{
	int n=0,m=0;
	char *expression;

	while (*(datastr+n)!='\0' 
		&& *(datastr+n)!=start_char) n++;
	if (*(datastr+n)=='\0') /* no start_char */
	 { *endptr=datastr; return NULL;}
	n++;
	while (*(datastr+n+m)!='\0'
		&& *(datastr+n+m)!=end_char) m++;
	if (*(datastr+n+m)=='\0') /* no end_char */
	 { *endptr=datastr+n; return NULL;}
	if (m==0) /* empty string */
	 { *endptr=datastr+n+1;return NULL;}
	if (m>1023) /* too big */
	 { *endptr=datastr+n-1;return NULL;}
	expression=(char *)malloc((m+1)*sizeof(char));
	strncpy(expression,datastr+n,m);
	*(expression+m)='\0';
	*endptr=datastr+n+m+1;
	return expression;
} /* pair_picking */


int
cmd_search(datastr,shift) /* Process datastr to pick out full cmd; 
takes care of line continuations, etc. Shift gives end of cmd. */
char *datastr;
int *shift;
{
	int flag=0,done=0,high_water=0;

	*shift=0;
	while (!done)
	 {
		while (datastr[flag]!='\n' 
			&& datastr[flag]!='\0'
			&& datastr[flag]!='\\') flag++;
		if (datastr[flag]=='\0') 
		 {
			*shift=flag;
			if (flag==0) return 0; /* vacuous command */
			done=1;
		 }
		else if (datastr[flag]=='\n')
		 {
			datastr[flag]='\0';
			done=*shift=flag;
			if (flag==0) return 0; /* vacuous command */
		 }
		else /* continuation indicator */
		 {
			high_water=flag+1;
			while (datastr[high_water]==' '
				|| datastr[high_water]=='\t') high_water++;
			if (datastr[high_water]=='\n') flag=high_water+1;
			else /* improper termination of cmd, but okay */
			 {
				datastr[high_water]='\0';
				*shift=high_water;
				done=1;
			 }
		 }
	 }
	flag=0;
	while (flag<*shift) /* replace line breaks/continues by spaces */
	 {
		if (datastr[flag]=='\\' || datastr[flag]=='\n') 
			datastr[flag]=' ';
		flag++;
	 }
	return flag;
} /* cmd_search */

