/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_LIBNETCDF
#include "netcdf.h"
#endif

#include "cdo_int.h"
#include "grid.h"
#include "remap_vars.h"
#include "remap.h" // SubmapType

#ifdef HAVE_LIBNETCDF
static void
nce(int istat)
{
  // This routine provides a simple interface to NetCDF error message routine.
  if (istat != NC_NOERR) cdoAbort(nc_strerror(istat));
}

size_t
cdfReadDimlen(int ncfileid, const char *dimname)
{
  size_t dimlen = 0;
  int ncdimid;
  const int status = nc_inq_dimid(ncfileid, dimname, &ncdimid);
  if (status == NC_NOERR) nce(nc_inq_dimlen(ncfileid, ncdimid, &dimlen));
  return dimlen;
}

void
cdfReadAttText(int ncfileid, int ncvarid, const char *name, char *text)
{
  size_t attlen;
  nce(nc_get_att_text(ncfileid, ncvarid, name, text));
  nce(nc_inq_attlen(ncfileid, ncvarid, name, &attlen));
  text[attlen] = 0;
}

void
cdfReadVarInt(int ncfileid, const char *name, int *array)
{
  int ncvarid;
  nce(nc_inq_varid(ncfileid, name, &ncvarid));
  nce(nc_get_var_int(ncfileid, ncvarid, array));
}

void
cdfReadVarSize(int ncfileid, const char *name, size_t len, size_t *array)
{
  if (len < 0x7FFFFC00)  // 2GB
    {
      std::vector<int> iarray(len);
      cdfReadVarInt(ncfileid, name, iarray.data());
      for (size_t i = 0; i < len; ++i) array[i] = (size_t) iarray[i];
    }
#ifdef HAVE_NETCDF4
  else
    {
      int ncvarid;
      nce(nc_inq_varid(ncfileid, name, &ncvarid));
      nce(nc_get_var_ulonglong(ncfileid, ncvarid, (unsigned long long *) array));
    }
#endif
}

void
cdfReadVarDouble(int ncfileid, const char *name, double *array)
{
  int ncvarid;
  nce(nc_inq_varid(ncfileid, name, &ncvarid));
  nce(nc_get_var_double(ncfileid, ncvarid, array));
}

void
cdfReadCoordinateRadian(int ncfileid, const char *name, size_t size, double *array)
{
  int ncvarid;
  nce(nc_inq_varid(ncfileid, name, &ncvarid));
  nce(nc_get_var_double(ncfileid, ncvarid, array));

  char grid_units[64];
  cdfReadAttText(ncfileid, ncvarid, "units", grid_units);
  grid_to_radian(grid_units, size, array, name);
}

struct RemapGridW
{
  int gridID;
  int rank;                 // rank of the grid
  size_t size;              // total points on the grid
  size_t num_cell_corners;  // number of corners for each grid cell

  bool lneed_cell_corners;
  bool luse_cell_corners;  // use corners for bounding boxes

  bool lextrapolate;
  bool non_global;
  bool is_cyclic;

  size_t dims[2];  // size of grid dimension

  int nvgp;               // size of vgpm
  std::vector<int> vgpm;  // flag which cells are valid

  std::vector<int> mask;  // flag which cells participate

  std::vector<double> cell_center_lon;  // lon/lat coordinates for
  std::vector<double> cell_center_lat;  // each grid center in radians
  std::vector<double> cell_corner_lon;  // lon/lat coordinates for
  std::vector<double> cell_corner_lat;  // each grid corner in radians

  std::vector<double> cell_area;  // total area of each grid cell
  std::vector<double> cell_frac;  // fractional area of grid cells participating in remapping
};

void
remapGridWInit(RemapGridW &grid)
{
  grid.num_cell_corners = 0;
  grid.luse_cell_corners = false;
  grid.lneed_cell_corners = false;

  grid.nvgp = 0;
}

void
remapGridWAlloc(RemapMethod mapType, RemapGridW &grid)
{
  if (grid.nvgp) grid.vgpm.resize(grid.nvgp);

  grid.mask.resize(grid.size);

  grid.cell_center_lon.resize(grid.size);
  grid.cell_center_lat.resize(grid.size);

  if (mapType == RemapMethod::CONSERV_SCRIP || mapType == RemapMethod::CONSERV)
    {
      grid.cell_area.resize(grid.size, 0.0);
    }

  grid.cell_frac.resize(grid.size, 0.0);

  if (grid.lneed_cell_corners)
    {
      if (grid.num_cell_corners > 0)
        {
          size_t nalloc = grid.num_cell_corners * grid.size;
          grid.cell_corner_lon.resize(nalloc, 0.0);
          grid.cell_corner_lat.resize(nalloc, 0.0);
        }
    }
}

struct RemapVarsW
{
  bool sort_add;
  bool pinit;               // true: if the pointers are initialized
  RemapMethod mapType;      // identifier for remapping method
  NormOpt normOpt;          // option for normalization (conserv only)
  long links_per_value;
  size_t max_links;         // current size of link arrays
  size_t num_links;         // actual number of links for remapping
  size_t num_wts;           // num of weights used in remapping
  size_t resize_increment;  // default amount to increase array size

  std::vector<size_t> src_cell_add;  // source grid address for each link
  std::vector<size_t> tgt_cell_add;  // target grid address for each link
  std::vector<double> wts;           // map weights for each link [max_links*num_wts]
};

NormOpt
getNormOpt(const char *normOptStr)
{
  NormOpt normOpt(NormOpt::NONE);
  // clang-format off
  if      (cstrIsEqual(normOptStr, "none"))     normOpt = NormOpt::NONE;
  else if (cstrIsEqual(normOptStr, "fracarea")) normOpt = NormOpt::FRACAREA;
  else if (cstrIsEqual(normOptStr, "destarea")) normOpt = NormOpt::DESTAREA;
  else
    {
      cdoPrint("normalize_opt = %s", normOptStr);
      cdoAbort("Invalid normalization option");
    }
  // clang-format on

  if (Options::cdoVerbose) cdoPrint("normalize_opt = %s", normOptStr);

  return normOpt;
}

void
remapVarsWInit(RemapMethod mapType, int remapOrder, RemapVarsW &rv)
{
  // Initialize all pointer
  if (!rv.pinit) rv.pinit = true;

  rv.sort_add = (mapType == RemapMethod::CONSERV_SCRIP);

  // Determine the number of weights
  rv.num_wts = (mapType == RemapMethod::CONSERV_SCRIP) ? 3 : ((mapType == RemapMethod::BICUBIC) ? 4 : 1);
  if (mapType == RemapMethod::CONSERV && remapOrder==2) rv.num_wts = 3;

  rv.links_per_value = -1;
  rv.num_links = 0;
  rv.max_links = 0;
  rv.resize_increment = 1024;
}

RemapMethod getMapType(int ncfileid, SubmapType *submapType, int *numNeighbors, int &remapOrder);

static void
remapReadDataScrip(const char *interp_file, RemapMethod *mapType, SubmapType *submapType,
                   int *numNeighbors, int &remapOrder, RemapGridW &src_grid, RemapGridW &tgt_grid, RemapVarsW &rv)
{
  // The routine reads a NetCDF file to extract remapping info in SCRIP format

  bool lgridarea = false;
  size_t dimlen;

  // Open file and read some global information
  // nce(nc_open(interp_file, NC_NOWRITE, &ncfileid));
  int ncfileid = cdf_openread(interp_file);

  // Map name
  char map_name[1024];
  cdfReadAttText(ncfileid, NC_GLOBAL, "title", map_name);

  if (Options::cdoVerbose)
    {
      cdoPrint("Reading remapping: %s", map_name);
      cdoPrint("From file: %s", interp_file);
    }

  // Map Type
  *mapType = getMapType(ncfileid, submapType, numNeighbors, remapOrder);
  if (*mapType == RemapMethod::CONSERV_SCRIP) lgridarea = true;

  remapVarsWInit(*mapType, remapOrder, rv);

  rv.mapType = *mapType;
  rv.links_per_value = -1;
  rv.sort_add = false;

  // Normalization option
  char normalize_opt[64];  // character string for normalization option
  cdfReadAttText(ncfileid, NC_GLOBAL, "normalization", normalize_opt);
  rv.normOpt = getNormOpt(normalize_opt);

  // File convention
  char convention[64]; // character string for output convention
  cdfReadAttText(ncfileid, NC_GLOBAL, "conventions", convention);

  if (!cstrIsEqual(convention, "SCRIP"))
    {
      cdoPrint("convention = %s", convention);
      if (cstrIsEqual(convention, "NCAR-CSM"))
        cdoAbort("Unsupported file convention: %s!", convention);
      else
        cdoAbort("Unknown file convention!");
    }

  // Read some additional global attributes

  // Source and destination grid names
  char src_grid_name[64];  // grid name for source grid
  cdfReadAttText(ncfileid, NC_GLOBAL, "source_grid", src_grid_name);

  char tgt_grid_name[64];  // grid name for dest grid
  cdfReadAttText(ncfileid, NC_GLOBAL, "dest_grid", tgt_grid_name);

  if (Options::cdoVerbose) cdoPrint("Remapping between: %s and %s", src_grid_name, tgt_grid_name);

  // Initialize remapgrid structure
  remapGridWInit(src_grid);
  remapGridWInit(tgt_grid);

  // Read dimension information
  src_grid.size = cdfReadDimlen(ncfileid, "src_grid_size");
  tgt_grid.size = cdfReadDimlen(ncfileid, "dst_grid_size");

  dimlen = cdfReadDimlen(ncfileid, "src_grid_corners");
  if (dimlen > 0)
    {
      src_grid.num_cell_corners = dimlen;
      src_grid.luse_cell_corners = true;
      src_grid.lneed_cell_corners = true;
    }

  dimlen = cdfReadDimlen(ncfileid, "dst_grid_corners");
  if (dimlen > 0)
    {
      tgt_grid.num_cell_corners = dimlen;
      tgt_grid.luse_cell_corners = true;
      tgt_grid.lneed_cell_corners = true;
    }

  src_grid.rank = cdfReadDimlen(ncfileid, "src_grid_rank");;
  tgt_grid.rank = cdfReadDimlen(ncfileid, "dst_grid_rank");;
  rv.num_links = cdfReadDimlen(ncfileid, "num_links");
  //  if ( rv.num_links == 0 ) cdoAbort("Number of remap links is 0, no remap weights found!");
  rv.num_wts = cdfReadDimlen(ncfileid, "num_wgts");

  remapGridWAlloc(rv.mapType, src_grid);
  remapGridWAlloc(rv.mapType, tgt_grid);

  // Allocate address and weight arrays for mapping 1
  if (rv.num_links > 0)
    {
      rv.max_links = rv.num_links;
      rv.src_cell_add.resize(rv.num_links);
      rv.tgt_cell_add.resize(rv.num_links);
      rv.wts.resize(rv.num_wts * rv.num_links);
    }

  // Read all variables of source grid
  cdfReadVarSize(ncfileid, "src_grid_dims", 2, src_grid.dims);

  cdfReadVarInt(ncfileid, "src_grid_imask", src_grid.mask.data());

  cdfReadCoordinateRadian(ncfileid, "src_grid_center_lat", src_grid.size, src_grid.cell_center_lat.data());
  cdfReadCoordinateRadian(ncfileid, "src_grid_center_lon", src_grid.size, src_grid.cell_center_lon.data());

  if (src_grid.num_cell_corners)
    {
      const size_t size = src_grid.num_cell_corners * src_grid.size;
      cdfReadCoordinateRadian(ncfileid, "src_grid_corner_lat", size, src_grid.cell_corner_lat.data());
      cdfReadCoordinateRadian(ncfileid, "src_grid_corner_lon", size, src_grid.cell_corner_lon.data());
    }

  if (lgridarea) cdfReadVarDouble(ncfileid, "src_grid_area", src_grid.cell_area.data());
  cdfReadVarDouble(ncfileid, "src_grid_frac", src_grid.cell_frac.data());

  // Read all variables of target grid
  cdfReadVarSize(ncfileid, "dst_grid_dims", 2, tgt_grid.dims);

  cdfReadVarInt(ncfileid, "dst_grid_imask", tgt_grid.mask.data());

  cdfReadCoordinateRadian(ncfileid, "dst_grid_center_lat", tgt_grid.size, tgt_grid.cell_center_lat.data());
  cdfReadCoordinateRadian(ncfileid, "dst_grid_center_lon", tgt_grid.size, tgt_grid.cell_center_lon.data());

  if (tgt_grid.num_cell_corners)
    {
      const size_t size = tgt_grid.num_cell_corners * tgt_grid.size;
      cdfReadCoordinateRadian(ncfileid, "dst_grid_corner_lat", size, tgt_grid.cell_corner_lat.data());
      cdfReadCoordinateRadian(ncfileid, "dst_grid_corner_lon", size, tgt_grid.cell_corner_lon.data());
    }

  if (lgridarea) cdfReadVarDouble(ncfileid, "dst_grid_area", tgt_grid.cell_area.data());
  cdfReadVarDouble(ncfileid, "dst_grid_frac", tgt_grid.cell_frac.data());

  if (rv.num_links > 0)
    {
      cdfReadVarSize(ncfileid, "src_address", rv.num_links, rv.src_cell_add.data());
      cdfReadVarSize(ncfileid, "dst_address", rv.num_links, rv.tgt_cell_add.data());

      for (size_t i = 0; i < rv.num_links; ++i) rv.src_cell_add[i]--;
      for (size_t i = 0; i < rv.num_links; ++i) rv.tgt_cell_add[i]--;

      cdfReadVarDouble(ncfileid, "remap_matrix", rv.wts.data());
    }

  // Close input file
  nce(nc_close(ncfileid));
}  // remapReadDataScrip

void
remapWeightsCheckAreas(size_t n_a, const std::vector<double> &area_a, const std::vector<double> &area_b,
                       size_t n_s, const size_t *col, const size_t *row, const double *S)
{
  std::vector<double> sum(n_a, 0.0);

  // sum weighted ratio of true areas 
  //      ’a’ is source; ’b’ is destination
  for (size_t i = 0; i < n_s; ++i) // loop over all elements of S (the weights)
    sum[col[i]] = sum[col[i]] + S[i] * area_a[col[i]] / area_b[row[i]];

  // check that sums are equal to 1 (within tolerance of 1.e-6)
  for (size_t i = 0; i < n_a; ++i) // loop over all source cells
    if (std::fabs(sum[i] - 1.0) > 1.e-6)
      printf("ERROR\n");
  printf("OK\n");
}

static
void verifyWeights(const char *weights_file)
{
  // int ncfileid = cdf_openread(weights_file);
  // nce(nc_close(ncfileid));

  RemapMethod mapType(RemapMethod::UNDEF);
  SubmapType submapType(SubmapType::NONE);
  int numNeighbors = 0;
  int remapOrder = 0;
  RemapGridW src_grid, tgt_grid;
  RemapVarsW remapvars;

  remapReadDataScrip(weights_file, &mapType, &submapType, &numNeighbors, remapOrder, src_grid, tgt_grid, remapvars);

  remapWeightsCheckAreas(src_grid.size, src_grid.cell_area, tgt_grid.cell_area,
                         remapvars.num_links, remapvars.src_cell_add.data(), remapvars.tgt_cell_add.data(), remapvars.wts.data());
}
#endif

void *
Remapweights(void *argument)
{
  cdoInitialize(argument);

#ifdef HAVE_LIBNETCDF
  int VERIFYWEIGHTS = cdoOperatorAdd("verifyweights", 0, 0, nullptr);

  int operatorID = cdoOperatorID();

  const char *weights_file = cdoGetStreamName(0);

  if (operatorID == VERIFYWEIGHTS) verifyWeights(weights_file);
#else
  cdoAbort("NetCDF support not compiled in!");
#endif

  cdoFinish();

  return nullptr;
}
