/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "cdo_int.h"
#include "pstream_int.h"

void *
Importfv3grid(void *process)
{
  cdoInitialize(process);

  int streamID1 = cdoStreamOpenRead(cdoStreamName(0));

  int vlistID1 = cdoStreamInqVlist(streamID1);
  int nvars = vlistNvars(vlistID1);
  if (nvars != 5) cdoAbort("Found %d variables, expected 5 variables!", nvars);

  std::vector<std::string> vars(nvars);
  vars[0] = "grid_lon";
  vars[1] = "grid_lat";
  vars[2] = "grid_lont";
  vars[3] = "grid_latt";
  vars[4] = "area";

  char varname[CDI_MAX_NAME];
  for (int varID = 0; varID < nvars; ++varID)
    {
      vlistInqVarName(vlistID1, varID, varname);
      if (strcmp(varname, vars[varID].c_str()) != 0)
        cdoAbort("Found variable %s, expected variable %s!", varname, vars[varID].c_str());
    }

  int ngrids = vlistNgrids(vlistID1);
  if (ngrids != 2) cdoAbort("Found %d grids, expected 2 grids!", nvars);

  int gridIDi1 = vlistGrid(vlistID1, 0);
  int gridIDi2 = vlistGrid(vlistID1, 1);

  size_t nx = gridInqXsize(gridIDi1);
  size_t ny = gridInqYsize(gridIDi1);
  size_t gridsize1 = gridInqSize(gridIDi1);
  size_t gridsize2 = gridInqSize(gridIDi2);

  cdoStreamInqTimestep(streamID1, 0);

  int gridIDo = gridCreate(GRID_UNSTRUCTURED, gridsize2);
  gridDefNvertex(gridIDo, 4);

  std::vector<double> buffer(gridsize1);
  int varID, levelID;
  size_t nmiss;

  {
    std::vector<double> grid_corner(4 * gridsize2);

    pstreamInqRecord(streamID1, &varID, &levelID);  // grid_lon
    pstreamReadRecord(streamID1, buffer.data(), &nmiss);

    for (size_t j = 1; j < ny; ++j)
      for (size_t i = 1; i < nx; ++i)
        {
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 0] = buffer[j * nx + i];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 1] = buffer[j * nx + (i - 1)];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 2] = buffer[(j - 1) * nx + (i - 1)];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 3] = buffer[(j - 1) * nx + i];
        }

    gridDefXbounds(gridIDo, grid_corner.data());

    pstreamInqRecord(streamID1, &varID, &levelID);  // grid_lat
    pstreamReadRecord(streamID1, buffer.data(), &nmiss);

    for (size_t j = 1; j < ny; ++j)
      for (size_t i = 1; i < nx; ++i)
        {
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 0] = buffer[j * nx + i];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 1] = buffer[j * nx + (i - 1)];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 2] = buffer[(j - 1) * nx + (i - 1)];
          grid_corner[((j - 1) * (nx - 1) + (i - 1)) * 4 + 3] = buffer[(j - 1) * nx + i];
        }

    gridDefYbounds(gridIDo, grid_corner.data());
  }

  pstreamInqRecord(streamID1, &varID, &levelID);  // grid_lont
  pstreamReadRecord(streamID1, buffer.data(), &nmiss);
  gridDefXvals(gridIDo, buffer.data());

  pstreamInqRecord(streamID1, &varID, &levelID);  // grid_latt
  pstreamReadRecord(streamID1, buffer.data(), &nmiss);
  gridDefYvals(gridIDo, buffer.data());

  pstreamInqRecord(streamID1, &varID, &levelID);  // area
  pstreamReadRecord(streamID1, buffer.data(), &nmiss);

  double sfclevel = 0;
  int surfaceID = zaxisCreate(ZAXIS_SURFACE, 1);
  zaxisDefLevels(surfaceID, &sfclevel);

  int vlistID2 = vlistCreate();
  varID = vlistDefVar(vlistID2, gridIDo, surfaceID, TIME_CONSTANT);
  vlistDefVarName(vlistID2, varID, "area");
  vlistDefVarStdname(vlistID2, varID, "area");
  vlistDefVarLongname(vlistID2, varID, "cell area");
  vlistDefVarUnits(vlistID2, varID, "m2");
  vlistDefVarDatatype(vlistID2, varID, CDI_DATATYPE_FLT32);

  int taxisID = cdoTaxisCreate(TAXIS_ABSOLUTE);
  vlistDefTaxis(vlistID2, taxisID);

  int streamID2 = cdoStreamOpenWrite(cdoStreamName(1));
  pstreamDefVlist(streamID2, vlistID2);
  pstreamDefTimestep(streamID2, 0);
  pstreamDefRecord(streamID2, 0, 0);
  pstreamWriteRecord(streamID2, buffer.data(), 0);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);
  gridDestroy(gridIDo);
  zaxisDestroy(surfaceID);

  cdoFinish();

  return nullptr;
}
