/*
 * packages.c
 *
 * Copyright (C) 2003 Bastian Blank <waldi@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $LastChangedBy: bastian $
 * $LastChangedDate: 2008-02-23 22:46:32 +0100 (Sa, 23 Feb 2008) $
 * $LastChangedRevision: 1487 $
 */

#include <config.h>

#include <ar.h>
#include <ctype.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "download.h"
#include "execute.h"
#include "log.h"
#include "packages.h"
#include "target.h"

const char *package_get_local_filename (di_package *package)
{
  const char *tmp = strrchr (package->filename, '/');
  if (tmp)
    return tmp + 1;
  return package->filename;
}

static long package_extract_self_parse_header_length (const char *inh, size_t len)
{
  char lintbuf[15];
  long r;
  char *endp;

  if (memchr (inh, 0, len))
    return -1;
  memcpy (lintbuf, inh, len);
  lintbuf[len] = ' ';
  *strchr(lintbuf,' ') = 0;
  r = strtoul(lintbuf, &endp, 10);
  if (*endp)
    return -1;
  return r;
}

static int package_extract_self (const char *file, const char *command)
{
  char buf[8192], *infobuf = NULL, versionbuf[40];
  ssize_t memberlen, copied = 0;
  FILE *ar = NULL, *pi = NULL;
  char *cur;
  struct ar_hdr arh;
  int header_done = 0;
  int ret = -1;

  ar = fopen (file, "r");
  if (!ar)
    goto error;
  if (!fgets (versionbuf, sizeof (versionbuf), ar))
    goto error;

  if (!strcmp (versionbuf, ARMAG))
  {
    while (1)
    {
      if (fread (&arh, 1, sizeof(arh), ar) != sizeof (arh))
        goto error;
      if (memcmp (arh.ar_fmag, ARFMAG, sizeof (arh.ar_fmag)))
        goto error;
      memberlen = package_extract_self_parse_header_length (arh.ar_size, sizeof (arh.ar_size));
      if (memberlen < 0)
        goto error;

      if (!header_done)
      {
        if (memcmp (arh.ar_name,"debian-binary   ",sizeof (arh.ar_name)) &&
            memcmp (arh.ar_name,"debian-binary/  ",sizeof (arh.ar_name)))
          goto error;
        infobuf = di_new (char, memberlen + 1);
        if (fread (infobuf, 1, memberlen + (memberlen & 1), ar) == (size_t) (memberlen + (memberlen & 1)))
        infobuf[memberlen]= 0;
        cur = strchr (infobuf, '\n');
        if (!cur)
          goto error;
        *cur = 0;
        cur = strchr (infobuf, '.');
        if (!cur)
          goto error;
        *cur = 0;
        if (strcmp (infobuf, "2"))
          goto error;
        *cur = '.';
        strncpy (versionbuf, infobuf, sizeof(versionbuf));
        versionbuf[sizeof (versionbuf) - 1]= 0;
        header_done = 1;
      }
      else if (!memcmp (arh.ar_name,"data.tar.gz     ",sizeof (arh.ar_name)) ||
               !memcmp (arh.ar_name,"data.tar.gz/    ",sizeof (arh.ar_name)))
        break;
      else
        fseek (ar, memberlen + (memberlen & 1), SEEK_CUR);
    }
  }
  else
    goto error;

  log_text (DI_LOG_LEVEL_DEBUG, "Execute \"%s\"", command);

  pi = popen (command, "w");
  if (!pi)
    goto error;

  while (copied < memberlen)
  {
    size_t r, s;
    r = memberlen - copied;
    if (r > sizeof (buf))
      r = sizeof (buf);
    r = fread (buf, 1, r, ar);
    if (!r)
      goto error;
    s = fwrite (buf, 1, r, pi);
    if (r != s)
      goto error;
    copied += s;
  }

  ret = fclose (pi);
  pi = NULL;

  log_text (DI_LOG_LEVEL_DEBUG, "Return code: %d", ret);

error:
  di_free (infobuf);
  if (ar)
    fclose (ar);
  if (pi)
    fclose (pi);

  return ret;
}

int package_extract (di_package *package)
{
  char buf_file[PATH_MAX];
  char buf[PATH_MAX];

  build_target_deb (buf_file, PATH_MAX, package_get_local_filename (package));
  snprintf (buf, PATH_MAX, "tar -xz -C %s -f -", target_root);
  return package_extract_self (buf_file, buf);
}

