/**
  \class COptionsXML
  \brief Store configuration options in an XML tree.
  \author Nemosoft Unv.

  With this class configuration options for applications can be easily
  stored and retrieved in XML format. Rather than wading through a complex
  XML tree, this class lets you store a simple list of options and
  references their values by their name.

  This class won't do recursive storage or grouping. It's a simple 1-to-1
  list. For 90% of the cases, that is enough.

  The class should be initialized with a list of variable names and their
  defaults using \ref DeclareVariable . Trying to set a variable name
  that isn't in the list will be ignored; getting a variable that isn't
  there will return a null variable.

*/


#include "OptionsXML.h"

COptionsXML::COptionsXML()
{
   m_Variables.setAutoDelete(true);
}

COptionsXML::~COptionsXML()
{
   /* Empty dummy destructor */
}



// private

// protected


/**
  \brief Declare a variable in this class
  \param name The name by which the variable is known
  \param defvalue A default (start) value

  With this function you can declare a variable that can be retrieved
  and stored in an XML subtree.
*/
void COptionsXML::DeclareVariable(const QString &name, const QString &defvalue)
{
   m_Variables.insert(name, new QString(defvalue));
}


// public

QString COptionsXML::Get(const QString &name) const
{
   return *m_Variables[name];
}

int COptionsXML::GetInt(const QString &name) const
{
   return Get(name).toInt();
}

bool COptionsXML::GetBool(const QString &name) const
{
   return (Get(name) == "true");
}

void COptionsXML::Set(const QString &name, const QString &value)
{
   if (m_Variables.find(name) == 0)
     return;
   m_Variables.replace(name, new QString(value)); // autodelete is very handy :)
}

void COptionsXML::Set(const QString &name, int value)
{
   Set(name, QString::number(value));
}

void COptionsXML::Set(const QString &name, bool value)
{
   Set(name, value ? QString("true") : QString("false"));
}

/**
  \brief Parse XML tree

  For every variable declared, retrieve the value from this node
  and store it.
*/
void COptionsXML::SetXML(const QDomNode &node)
{
   QDictIterator<QString> it(m_Variables);
   QString name;
   QDomNode v;

   while (it.current()) {
     name = it.currentKey();
     ++it; // to prevent since replace() below from skipping
     v = node.namedItem(name);
     if (!v.isNull() && v.isElement()) {
       m_Variables.replace(name, new QString(v.toElement().text()));
     }
   }
}

/**
  \brief Return XML tree

  For every variable declared, set its value as a subnode in the XML node.
  In the XML file, it looks like this:
  <name>value</name>

  \note No subnodes are cleared, so existing subnodes that are not
  in our declared variables remain intact.
*/
void COptionsXML::GetXML(QDomNode &node) const
{
   QDictIterator<QString> it(m_Variables);
   QString name;
   QDomNode v, w;

   while (it.current()) {
     name = it.currentKey();
     w = node.ownerDocument().createElement(name);
     w.appendChild(node.ownerDocument().createTextNode(*it.current()));
     v = node.namedItem(name);
     if (v.isNull()) {
       node.appendChild(w);
     }
     else {
       node.replaceChild(w, v);
     }
     ++it;
   }
}
