/*  audiodevs: Abstraction layer for audio hardware & samples
    Copyright (C) 2003-2004 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

/**
  \class CAudioCollector
  \brief Class that detects the present audio devices in the system


  Use a singleton pattern, i.e. there can only be one instance of this class present;
  get a pointer to this instance through the \ref Instance() member function.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include "AudioCollector.h"

#if defined(_OS_LINUX_) || defined(Q_OS_LINUX)
  #ifdef HAVE_ALSA
    #define ALSA_PCM_NEW_HW_PARAMS_API
    #define ALSA_PCM_NEW_SW_PARAMS_API
    #include <alsa/asoundlib.h>
  #endif
  #include "../linux/AudioDeviceLinux.h"
#endif

CAudioCollector *CAudioCollector::s_pAudioCollector = 0;

CAudioCollector::CAudioCollector()
{
   qDebug("CAudioCollector::CAudioCollector()");
}

// private

void CAudioCollector::Scan()
{
#if defined(_OS_LINUX_) || defined(Q_OS_LINUX)
#ifdef HAVE_ALSA
   int snd_card_number = -1;
   CAudioDeviceLinux *a;

   while (true) {
      snd_card_next(&snd_card_number);
      if (snd_card_number < 0)
        break;
      a = new CAudioDeviceLinux(snd_card_number);
      if (a->IsValid())
        m_Devices.append(a);
   };
#else
#error Only ALSA is supported at the moment!
#endif
#endif
}

// public

CAudioCollector *CAudioCollector::Instance()
{
   if (s_pAudioCollector == 0) {
     s_pAudioCollector = new CAudioCollector();
     s_pAudioCollector->Scan();
   }
   return s_pAudioCollector;
}

uint CAudioCollector::NumberOfAudioDevices() const
{
   return m_Devices.count();
}

CAudioDevice *CAudioCollector::GetAudioDevice(uint n)
{
   if (n >= m_Devices.count())
     return 0;
   return m_Devices.at(n);
}

/* \brief Find an AudioDevice, based on name and node
   \param device_name The name, as returned by \ref CAudioDevice::GetName()
   \param node_name The node, as return by \ref CAudioDevice::GetNode()
   \return A pointer to a \ref CAudioDevice, or 0 if no match was found

This function searchs the currently available audio devices and tries to find a
match based on device name and node name. Since it is a-priory not known which
number an audio devices gets, we cannot use hardcoded device numbers. The function
first matches on device_name, and in case of multiple candidates, uses the node_name
as well to find a best match. If no device with a matching name is found, 0 is returned.

*/
CAudioDevice *CAudioCollector::Find(const QString &device_name, const QString &node_name)
{
   uint u;
   CAudioDevice *best = 0, *dev;

   for (u = 0; u < m_Devices.count(); u++)
   {
      dev = m_Devices.at(u);
      if (dev == 0)
        continue;
      if (dev->GetName() == device_name)
      {
        if (dev->GetNodeName() == node_name || best == 0)
          best = dev;
      }
   }
   return best;
}

