/**
  \class CVideoDeviceTuner
  \brief Tuner on V4L device
  
  This class sets the frequency on the TV or radio tuner on a TV/radio card.
  The frequency is represented as a float number in Hz.
  
*/  

#include "VideoDeviceInput.h"
#include "VideoDeviceTuner.h"
#include "VideoDevice.h"

/**
  \brief Constructor
  \param _video Master VideoDevice object
  \param _tuner Tuner number (should rarely be > 0)
*/
CVideoDeviceTuner::CVideoDeviceTuner(CVideoDevice *_video, int _tuner)
{
   struct video_tuner vtun;
   
   pVideo = _video;
   Tuner = _tuner;

   FreqLow = 0;
   FreqHigh = 0;
   FreqStep = 1;

   memset(&vtun, 0, sizeof(vtun));
   vtun.tuner = Tuner;
   if (ioctl(pVideo->CamFD, VIDIOCGTUNER, &vtun) < 0)
     return;
   Name = vtun.name;
   if (vtun.flags & VIDEO_TUNER_LOW)
     FreqStep = 1.0e3 / 16;
   else
     FreqStep = 1.0e6 / 16;
   FreqLow  = vtun.rangelow * FreqStep;
   FreqHigh = vtun.rangehigh * FreqStep;
}


const QString CVideoDeviceTuner::GetName()
{
   return Name;
}

float CVideoDeviceTuner::GetLowestFrequency() const
{
   return FreqLow;
}

float CVideoDeviceTuner::GetHighestFrequency() const
{
   return FreqHigh;
}

/**
  \brief Return frequency of tuner, in Hz
  \return -1.0 in case of failure
*/
float CVideoDeviceTuner::GetFrequency() const
{
   if (ioctl(pVideo->CamFD, VIDIOCGFREQ, &ufreq) < 0)
     return -1.0;
   return ufreq * FreqStep;
}

/**
  \brief Set frequency
  \param freq Desired frequency, in Hz
  \return FALSE if device refused the frequency or was not able to, TRUE with success
*/  
bool CVideoDeviceTuner::SetFrequency(float freq)
{
   ufreq = (ulong)(freq / FreqStep);
   if (ioctl(pVideo->CamFD, VIDIOCSFREQ, &ufreq) < 0)
     return FALSE;
   return TRUE;
}

