/** 
  \struct SVideoOptions 
  \brief Options per video device
*/

#define ENABLE_TRACING
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "VideoOptions.h"

enum {
  BaseName,
  TextFont,
  TextColor,
  TimeInImage,
  FileFormat,
  SaveToDisk,
  FTPToServer,
  SaveOption,
  MaxSequence,
  Sequence,
  FTPServer,
  FTPPath,
  FTPUser,
  FTPPass,

  TagMax
};

static const char *option[][2] = { 
   { "basename",     "snapshot" },
   { "textfont",     "system" },
   { "textcolor",    "#ffff00" },
   { "timeinimage",  "false" },
   { "fileformat",   "PNG" },
   { "savetodisk",   "true" },
   { "ftptoserver",  "false" },
   { "saveoption",   "1" },
   { "maxsequence",  "1000" },
   { "sequence",     "0" },
   { "ftpserver",    "" },
   { "ftppath",      "" },
   { "ftpuser",      "" },
   { "ftppass",      "" },
};

SVideoOptions::SVideoOptions(QDomDocument &doc)
	
{
   QDomElement el;

//qDebug("SVideoOptions::SVideoOptions(QDomDocument &doc)");
  // populate tree
   pValues = NULL;
   Me = doc.createElement("videoconfiguration");
   Populate(doc);
}



/**
 \brief Copy constructor; makes deep copy
*/
SVideoOptions::SVideoOptions(const SVideoOptions &org)
{
   QDomDocument doc;

//qDebug("SVideoOptions::SVideoOptions(const SVideoOptions &org)");
   pValues = NULL;
   doc = org.Me.ownerDocument();
   Me = doc.createElement("videoconfiguration");
   Populate(doc);
   SetXML(org.Me);
}


SVideoOptions::~SVideoOptions()
{
//   if (pValues)
//     delete pValues;
}


// private

void SVideoOptions::Populate(QDomDocument &doc)
{
   unsigned int i;
   QDomElement elem;

//qDebug("SVideoOptions::Populate(QDomDocument &doc)");
   if (pValues)
     delete pValues;
   pValues = new QDomText[TagMax];
   Opts.clear();
   for (i = 0; i < TagMax; i++) {
      elem = doc.createElement(option[i][0]);
      pValues[i] = doc.createTextNode(option[i][1]);
      elem.appendChild(pValues[i]);
      Me.appendChild(elem);
      Opts.insert(option[i][0], &pValues[i]);
   }
}

// public


/**
  \brief Parse contents of node, apply to our own tree

 */
void SVideoOptions::SetXML(const QDomElement &el)
{
   QDomNode n;
   QDomElement e;
   int i;

//qDebug("SVideoOptions::SetXML(QDomElement &el)");
   // Go through tags, bind them to the 'variables' we have
   n = el.firstChild();
   while (!n.isNull()) {
      if (n.isElement()) {
        e = n.toElement();
//        qDebug("  Parsing videoconfiguration " + e.tagName() + "=" + e.text());
        for (i = 0; i < TagMax; i++) {
           if (e.tagName() == option[i][0]) {
//             qDebug("  Binding to variable %d", i);
             pValues[i].setData(e.text());
           }
        }
      }
      n = n.nextSibling();
   }
   Me.setAttribute("name", el.attribute("name"));
   Me.setAttribute("node", el.attribute("node"));
}


/**
  \brief Returns an XML structure with the current settings
*/
QDomNode SVideoOptions::GetXML()
{
//qDebug("SVideoOptions::GetXML(): %s", (const char *)Me.tagName());
   return Me;
}



QString SVideoOptions::GetDeviceName() const
{
   return Me.attribute("name");
}

void SVideoOptions::SetDeviceName(const QString &name)
{
//qDebug("SVideoOptions::SetDeviceName(%s)", (const char *)name);
   Me.setAttribute("name", name);
}

QString SVideoOptions::GetNodeName() const
{
   return Me.attribute("node");
}

void SVideoOptions::SetNodeName(const QString &node)
{
//qDebug("SVideoOptions::SetNodeName(%s)", (const char *)node);
   Me.setAttribute("node", node);
}


QString SVideoOptions::GetBaseName() const
{
   return pValues[BaseName].data();
}

void SVideoOptions::SetBaseName(const QString &basename)
{
//qDebug("SVideoOptions::SetBaseName(%s)", (const char *)basename);
   pValues[BaseName].setData(basename);
}

QColor SVideoOptions::GetTextColor() const
{
   return QColor(pValues[TextColor].data());
}

void SVideoOptions::SetTextColor(const QColor &color)
{
//qDebug("SVideoOptions::SetTextColor(%s)", (const char *)color.name());
   pValues[TextColor].setData(color.name());
}

void SVideoOptions::SetTextColor(const QString &color_name)
{
//qDebug("SVideoOptions::SetTextColor(%s)", (const char *)color_name);
   pValues[TextColor].setData(color_name);
}


QFont SVideoOptions::GetTextFont() const
{
   QFont f;
   
   f.setRawName(pValues[TextFont].data());
   return f;
}

void SVideoOptions::SetTextFont(const QFont &font)
{
//qDebug("SVideoOptions::SetTextFont(%s)", (const char *)font.rawName());
   pValues[TextFont].setData(font.rawName());
}

void SVideoOptions::SetTextFont(const QString &font_name)
{
//qDebug("SVideoOptions::SetTextFont(%s)", (const char *)font_name);
   pValues[TextFont].setData(font_name);
}

bool SVideoOptions::GetTimeInImage() const
{
   return (pValues[TimeInImage].data() == "true");
}

void SVideoOptions::SetTimeInImage(bool value)
{
   pValues[TimeInImage].setData(value ? "true" : "false");
}


QString SVideoOptions::GetFileFormat() const
{
   return pValues[FileFormat].data();
}

void SVideoOptions::SetFileFormat(const QString &format)
{
//qDebug("SVideoOptions::SetFileFormat(%s)", (const char *)format);
   pValues[FileFormat].setData(format);
}


int SVideoOptions::GetSaveOption() const
{
   return pValues[SaveOption].data().toInt();
}

void SVideoOptions::SetSaveOption(int value)
{
   pValues[SaveOption].setData(QString::number(value));
}
   
int SVideoOptions::GetSequence() const
{
   return pValues[Sequence].data().toInt();
}

void SVideoOptions::SetSequence(int value)
{
   pValues[Sequence].setData(QString::number(value));
}
   
int SVideoOptions::GetMaxSequence() const
{
   return pValues[MaxSequence].data().toInt();
}

void SVideoOptions::SetMaxSequence(int max)
{
   pValues[MaxSequence].setData(QString::number(max));
}
   
bool SVideoOptions::GetSaveToDisk() const
{
   return (pValues[SaveToDisk].data() == "true");
}

void SVideoOptions::SetSaveToDisk(bool value)
{
   pValues[SaveToDisk].setData(value ? "true" : "false");
}

bool SVideoOptions::GetFTPToServer() const
{
   return (pValues[FTPToServer].data() == "true");
}

void SVideoOptions::SetFTPToServer(bool value)
{
   pValues[FTPToServer].setData(value ? "true" : "false");
}

QString SVideoOptions::GetFTPServer() const
{
   return pValues[FTPServer].data();
}

void SVideoOptions::SetFTPServer(const QString &server)
{
   pValues[FTPServer].setData(server);
}

QString SVideoOptions::GetFTPPath() const
{
   return pValues[FTPPath].data();
}

void SVideoOptions::SetFTPPath(const QString &path)
{
   pValues[FTPPath].setData(path);
}

QString SVideoOptions::GetFTPUser() const
{
   return pValues[FTPUser].data();
}

void SVideoOptions::SetFTPUser(const QString &user)
{
   pValues[FTPUser].setData(user);
}

QString SVideoOptions::GetFTPPass() const
{
   return pValues[FTPPass].data();
}

void SVideoOptions::SetFTPPass(const QString &pass)
{
   pValues[FTPPass].setData(pass);
}

