/*  camserv - An internet streaming picture application
 *
 *  Copyright (C) 1999  Jon Travis (jon@dsndata.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <unistd.h>

#include "fmt_text.h"
#include "config.h"

static const char *Weekdays[] = { "Sunday", "Monday", "Tuesday", "Wednesday",
				  "Thursday", "Friday", "Saturday" };

static const char *Months[] = { "January", "February", "March", "April", "May",
				"June", "July", "August", "September", 
				"October", "November", "December" };

/*
 * cam_fmt_buf:  Format a text string using information about time, dates,
 *               and other various 'info'.
 *
 * Arguments:    fmt = Format to fulfill.  Every char is copied directly
 *                     to 'dest', unless it is a '$', in which case a following
 *                     meta-char will be replaced by filled in information
 *                     (read the source)
 *               dest = Destination to place all of the text into
 *               destlen = # of bytes allocated for dest
 */

void cam_fmt_buf( const char *fmt, char *dest, int destlen ){
  const char *cp;
  char  *destcp, *overdest;
  struct tm *loctime;
  time_t curtime;

  time( &curtime );
  loctime = localtime( &curtime );

  overdest = dest + destlen;
  for( cp = fmt, destcp = dest; *cp && destcp < overdest; cp++ ){
    if( *cp == '$' && *(cp+1))	{
      cp++;
      switch( *cp ) {
      case '$' : *destcp++ = '$';
	break;
				
      case 'H' : destcp += snprintf( destcp, overdest - destcp,
				     "%02d", loctime->tm_hour );
	break;
      case 'k' : destcp += snprintf( destcp, overdest - destcp,
				    "%2d", loctime->tm_hour );
        break;
      case 'I' : destcp += snprintf( destcp, overdest - destcp, "%02d", 
				    loctime->tm_hour > 12 ? 	
				    loctime->tm_hour - 12 :
				    loctime->tm_hour );
        break;
      case 'l' : destcp += snprintf( destcp, overdest - destcp,"%2d", 
				    loctime->tm_hour > 12 ? 	
				    loctime->tm_hour - 12 :
				    loctime->tm_hour );
        break;
      case 'M' : destcp += snprintf( destcp, overdest - destcp,"%02d",  
				     loctime->tm_min );
	break;

      case 'p' : destcp += snprintf( destcp, overdest - destcp, "%s", 
				    (loctime->tm_hour < 12 ) ? "AM" :
				    "PM" );
        break;
      case 'S' : destcp += snprintf( destcp, overdest - destcp,"%02d",  
				     loctime->tm_sec );
	break;
      case 's' : 
	destcp += snprintf( destcp, overdest - destcp, "%d",  
				     (int)curtime );
        break;
						
      case 'T' : destcp += snprintf( destcp, overdest - destcp,
				     "%02d:%02d:%02d",
				     loctime->tm_hour,
				     loctime->tm_min,
				     loctime->tm_sec );
        break;
      case 'X' : destcp += snprintf( destcp, overdest - destcp,
				     "%02d:%02d:%02d %s",
				    (loctime->tm_hour > 12 ) ? 
				    loctime->tm_hour - 12 :
				    loctime->tm_hour,
				    loctime->tm_min,
				    loctime->tm_sec,
				    (loctime->tm_hour < 12 ) ? "AM" :
				    "PM" );
        break;
      case 'a' : destcp += snprintf( destcp, overdest - destcp,"%.3s", 
				    Weekdays[ loctime->tm_wday ] );
        break;
      case 'A' : destcp += snprintf( destcp, overdest - destcp,"%s", 
				    Weekdays[ loctime->tm_wday ] );
        break;
      case 'b' : destcp += snprintf( destcp, overdest - destcp,"%.3s",
				    Months[ loctime->tm_mon ] );
        break;
      case 'B' : destcp += snprintf( destcp, overdest - destcp,"%s",
				    Months[ loctime->tm_mon ] );
        break;
      case 'c' : destcp += snprintf( destcp, overdest - destcp,
				     "%s", ctime( &curtime));
	break;
      case 'v' : destcp += snprintf( destcp, overdest - destcp,
				     "%s", VERSION );
        break;
      default :
	break;
      }
    } else {
      *destcp++ = *cp;
    }
  }
  *destcp = '\0';
}
