#ifndef _SYSTEMITERATOR_CPP_
#define _SYSTEMITERATOR_CPP_

#include "systemIterator.h"
#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <string.h>
#include "track.h"
#include "part.h"
#include "event.h"
#include "note.h"
#include "prPartEditor.h"
#include "prScoreEditor.h"
#include "scoreBar.h"
#include "reference.h"
#include "symbol.h"
#include "tuplet.h"

struct InitState {
  Position   cur_pos;
  Position   free_pos;
  Position   next_bar;
  ScoreBar * bar;
  ScoreBar * used_bar;
  int        tupletDisplay;
  int        tupletDuration;
  bool       no_newgroup;
};


SystemIterator::SystemIterator(const SystemIterator & i) : Iterator(i), _system(i._system), _res(i._res), _no_overlap(i._no_overlap) { cout << "+++" << endl; }


SystemIterator::SystemIterator(ScoreType sctype, Track * tr, PrScoreEditor * editor, Part * ap, long lpos, long rpos)
  : Iterator(tr, lpos==-1?editor->left():lpos, rpos==-1?editor->right():rpos), _editor(editor), _score_type(sctype), _system(),
    _res(editor->resolution()), _no_overlap(editor->noOverlap()), _width(editor->editorWidth()), _active_part(ap) {
  init();
}

SystemIterator::SystemIterator(Part * pt, int editorwidth, int resolution, bool nooverlap, Position left, Position right)
  : Iterator(pt,left,right), _system(), _res(resolution), _no_overlap(nooverlap), _width(editorwidth) {
  init();
}

SystemIterator::SystemIterator(PrPartEditor * editor) : Iterator(editor), _system(), _res(editor->resolution()),
						      _no_overlap(editor->noOverlap()), _width(editor->editorWidth()) {
  init();
}

SystemIterator::~SystemIterator() {
  delete state;
}

void SystemIterator::add(Event * event, long dur, int dis, Position pos) {
  // if no position specified, take free_pos:
  if (pos==0) pos = state->free_pos;

  // if no length specified, take nextbar-curpos:
  if (dur==0) dur = state->next_bar - state->free_pos.ticks();
  if (dis==0) dis = dur;

  if (adjust) { dis += int(_res*0.49); dis -= dis%_res; }
  // cout << "add in! curpos: " << state->cur_pos << ", freepos: " << state->free_pos << ", dur: " << dur << ", dis: " << dis << ", pos: " << pos << endl;

  if (dur>0 && pos < _right) { // only for lenghts!=0 and inside the range!
    // cout << "add "; if (event==0) cout << "BREAK"; else cout << event->ctype(); cout << " at " << pos << " - " << len << endl;
    if (state->bar==0) {
      state->bar = new ScoreBar(_score_type, _part, Iterator::change()!=0, _active_part==_part);
      state->bar->add(event, pos, dur, dis, _no_overlap, state->no_newgroup);
      _system.add(state->bar);
    } else {
      state->bar->add(event, pos, dur, dis, _no_overlap, state->no_newgroup);
    }
    if (state->used_bar==0 && event!=0) {
      state->used_bar = state->bar; // remember bar in case notes starting at the same position, but ending in another bar are entered
      // cout << "used_bar: " << state->used_bar << ", event: " << event << endl;
    }
    if (event==0 /* break */ || event->isA()==NOTE) {
      //
      // the grouping/timing mechanism is affected only by notes and breaks
      // (not by symbols!)
      //
      Position foo = pos+dur;
      // if (event!=0 && ((Note*)event)->tuplet()!=0) foo = pos + event->duration();
      if (foo > state->free_pos) state->free_pos = foo;
      state->cur_pos = pos;
    }
  }
  // cout << "add out! curpos: " << state->cur_pos << ", freepos: " << state->free_pos << ", dur: " << dur << ", dis: " << dis << ", pos: " << pos << endl;
}


int smallest(int len) {
  int min = 0;
  len /= 3;
  if (len > 0) for (min = 1; (len&1) == 0; len = len >> 1) min = min << 1;
  // cout << "smallest of " << 3*len << " is " << 3*min << endl;
  return 3*min;
}

int greatest(int len) {
  int max = 0;
  len /= 3;
  if (len > 0) for (max = 1; len != 1; len = len >> 1) max = max << 1;
  // cout << "greatest of " << 3*len << " is " << 3*max << endl;
  return 3*max;
}

int min(int a, int b) {
  if ( a < b ) return a;
  else return b;
}

void SystemIterator::splitAndAdd(int brk, Tuplet * prevTuplet, Tuplet * nextTuplet) {
  //
  // this adds a break of length "len" to the system
  // it requires the break to be fully inside the bar
  // it splits the break according to the beats
  //
  // the situation is ("*" means: here is a note, "-" its length):
  //
  //    *----------                    *------------            |
  // curpos    freepos               notepos      noteend     barend
  //               <------- brk ------>
  //               <------------------- rest ------------------->
  //
  // the algorithm for splitting breaks is rather easy:
  // split the "rest" (the space between the free position and the end
  // of the bar!) into "integer" breaks (meaning no dotted values!)
  // and start displaying them in increasing order up to the point until
  // the total amount of breakspace to be displayed succeeds the
  // original break value to be displayed ("brk").
  // Reduce the "brk" value by the space that is already displayed so far.
  // The remaining space has to be split into "integer" breaks again, which
  // are now to be displayed in decreasing order.
  //
  // the only subtle point is to know, when a break has to be adjusted due
  // to a preceeding or following tuplet note. The preceeding tuplet only
  // affects the first part of the algorithm (the increasing breaks),
  // the following tuplet only affects the second part (the decreasing breaks)

  long rest = state->next_bar - state->free_pos.ticks();
  int b;


  if (brk==0) brk = rest;
  if (brk > rest) cout << "PANIC: break exceeds bar" << endl;

  if (prevTuplet != 0 && state->tupletDuration > 0) {
    int bb = min(state->tupletDuration, brk);
    // cout << "brk: " << brk << ", dur: " << bb << ", dis: " << prevTuplet->display(bb) << " - state: " << state->tupletDuration << endl;
    add(0, bb, prevTuplet->display(bb)); // TODO: loop (see below)
    rest -= bb;
    brk  -= bb;
    state->tupletDuration -= bb; if (state->tupletDuration<0) state->tupletDuration = 0;
  }

  if (brk > 0 && nextTuplet != 0) {
    // we assume the tuplet starts with the break at the beat. This break is to be drawn!
    int delta = notepos.tick();
    if (delta>0 && delta <= brk) {
      state->no_newgroup = false;
      // cout << "brk: " << brk << ", dur: " << delta << ", dis: " << nextTuplet->display(delta) << " - state: " << state->tupletDuration << endl;
      if (delta>0) add(0,delta, nextTuplet->display(delta)); // TODO: loop (see below)
      rest -= delta;
      brk  -= delta;
      state->tupletDuration = nextTuplet->duration() - delta; if (state->tupletDuration<0) state->tupletDuration = 0;
    }
  }

  //
  // fill up gap in increasing order
  //
  for ( ; (b=smallest(rest)) < brk && b!=0; ) {
    brk -= b;
    rest -= b;
    if (_res==0 || _res <= b) add(0, b);
  }

  //
  // draw remaining breaks in decreasing order
  //
  for ( ; brk > 0 ; ) {
    b = greatest(brk);
    if (b>0) {
      brk -= b;
      if (_res==0 || _res <= b) add(0, b);
    } else {
      brk = 0;
    }
  }

}

void SystemIterator::nextbar() {
  state->cur_pos.nextBar();
  state->next_bar.nextBar();
  state->free_pos = state->cur_pos;
  state->bar = 0;
  state->tupletDisplay = 0;
}

void SystemIterator::init() {
  //
  // init
  //

  Note *         note = 0;
              notepos = -1;
  Position old_notepos =0;
  Position    noteend = 0;
  long            brk = 0;
               adjust = (_res>0);
  Tuplet * prevTuplet = 0; // whether the previous note is a tuplet
  Tuplet *     tuplet = 0; // whether the next note is a tuplet
  long            dur = 0;
  int             dis = 0;
  state = new InitState;
  state->cur_pos  = _left;
  state->free_pos = _left;
  state->next_bar = _left;
  state->next_bar.nextBar();
  state->bar = 0;
  state->used_bar = 0;
  state->tupletDisplay  = 0; // remaining tuplet space
  state->tupletDuration = 0; // remaining tuplet duration
  state->no_newgroup    = false; // tuplets may not be splitted over groups!

  for (; !reallyDone(tuplet); Iterator::operator++()) {
    if ( _event->isA() == NOTE ) {
      //
      // remember tuplet of previous note to adjust breaks!
      //
      prevTuplet = tuplet;

      //
      // here we have the new note
      //
      note        = (Note*) _event;
      tuplet      = note->tuplet();
      old_notepos = notepos;
      notepos     = _part->start(note);
      noteend     = truncEnd(note);
      if (adjust) notepos.snap(_res, tuplet);
      if (adjust) { noteend.snap(_res, tuplet); if (noteend==notepos) noteend += _res; }
      state->no_newgroup = false;
      if (notepos==old_notepos) {
	//
	// note at same position as last note
	//
	if (state->used_bar!=0)
	  state->used_bar->add(note, notepos.ticks(), noteend-notepos.ticks(),  noteend-notepos.ticks(), _no_overlap, state->no_newgroup);
	else
	  cout << "PANIC: used bar" << endl;
      } else {
	state->used_bar = 0; // invokes used_bar being set with adding this note!
	//
	// note at new position
	//
	while ( !state->cur_pos.sameBarOrGreater(notepos) ) { // while bar(curpos) < bar(note)
	  splitAndAdd(0, prevTuplet, tuplet);
	  nextbar();
	}
	if (notepos < state->free_pos) {
	  //
	  // in case the new note starts before the previous note ends:
	  //
	  state->free_pos = notepos;
	} else {
	  //
	  // freepos is, where the previous note ends:
	  //
	  if (adjust) state->free_pos.snap(_res, prevTuplet);
	  brk = notepos - state->free_pos.ticks();
	  if (brk<0)          cout << "SystemIterator::PANIC " << brk << " = " << notepos << " - " << state->free_pos << endl;
	  else if (brk > 0)   splitAndAdd(brk, prevTuplet, tuplet);
	}
	
	if (tuplet!=0) {
	  if (state->tupletDuration == 0) {
	    state->tupletDuration = tuplet->duration();
	    state->tupletDisplay  = state->tupletDuration;
	  } else { state->no_newgroup = true; }
	  state->tupletDuration -= note->duration();
	  state->tupletDisplay  -= note->display(_res);
	  if (state->tupletDuration <= 1) state->tupletDuration = 0;
	}
	
	while ( noteend > state->next_bar ) {
	  add(note);
	  nextbar();
	}
	dur = noteend - state->free_pos.ticks();
	dis = dur;
	
	
	if (tuplet!=0) dis = tuplet->display(dur);
	if (dis>0) add(note, dur, dis);
      }

    } else if ( _event->isA() == SYMBOL ) {
      // cout << "add symbol" << endl;
      add(_event);
    }

  }

  _ptr = 0;
  _max = _system.size();

  //
  // To calculate the scaling coefficient: SUM over bars: [ c*len(bar) + indent(bar)] = width
  //

  int width = _width;
  int sum = 0;
  for (ScoreBar * b = (ScoreBar*) _system.first(); b!=0; b = (ScoreBar*) _system.next(b)) {
    width -= b->systemIndent();
    sum += b->rawWidth();
  }
  //
  // now: c * sum = width => c = width/sum
  //
  _scale = width*1.0/sum;
  for (ScoreBar * b = (ScoreBar*) _system.first(); b!=0; b = (ScoreBar*) _system.next(b)) {
    b->setScale(_scale);
    // cout << "scale: " << b->scale() << ", indent: " << b->indent() << endl << endl;
  }


  /* cout << "size: " << _system.size() << endl;
  for (int i=0; i< _system.size(); i++) {
    cout << *_system.get(i) << endl;
    }*/


}

Position SystemIterator::truncEnd(Note * note) {
  Note * nx = note;
  Position pNote = _part->start(note);
  long eNote = _part->end(note);

  //
  // we need the nearest following note, which starts at a different position than "note".
  //
  for (; (nx != 0) && (_part->start(nx) == pNote); nx = (Note*) _part->next(nx)) {}
  //
  // now, we have to make sure, nx is a note (and not a symbol, or else)
  //
  for (; (nx != 0) && (nx->isA()!=NOTE); nx = (Note*) _part->next(nx)) {}

  if (nx==0) return Position(eNote);
  else {
    long pNext = _part->start(nx).ticks();
    if ( eNote < pNext) return Position(eNote);
    else return Position(pNext);
  }
}


bool SystemIterator::done() const { return _ptr >= _max; }

bool SystemIterator::reallyDone(Tuplet * tuplet) {
 if (!Iterator::done()) return false;
  else {
    while ( !state->free_pos.sameBarOrGreater(_right) ) { // bar(free_pos) < bar(_right)
      splitAndAdd(0, tuplet, 0); // since no note follows, it cannot be a tuplet
      nextbar();
    }
    return true;
  }
}

Element * SystemIterator::operator *() {
  if (_ptr < _max) return _system.get(_ptr);
  else return 0;
}

Iterator& SystemIterator::operator++() {
  _ptr++;
  return *this;
}

Iterator SystemIterator::operator++(int) {
  _ptr++;
  return *this;
}


#endif
