/***************************************************************************
                          bknode.cpp  -  description
                             -------------------
    begin                : Sun Sep 15 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   BkNode, BkFolder, and BkBookmark class implementations.  See bknode.h *
 *   for more information on each class.                                   *
 *                                                                         *
 ***************************************************************************/

#include "bknode.h"

#include <algorithm>

using namespace std;

/***************************************************************************
 *                                                                         *
 *   BkNode::BkNode()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      BkNode class constructor.                                          *
 *                                                                         *
 ***************************************************************************/

BkNode::BkNode()
{
	m_ignore = false;
	m_ValidFields = 0;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::BkNode()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkNode &                                                           *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      BkNode class copy constructor.                                     *
 *                                                                         *
 ***************************************************************************/

BkNode::BkNode(const BkNode &node) :
	m_path(node.m_path),
	m_DiffDesc(node.m_DiffDesc),
	m_desc(node.m_desc),
	m_title(node.m_title),
	m_parentPath(node.m_parentPath),
	m_attrs(node.m_attrs),
	m_order(node.m_order)
{
	m_ignore = node.m_ignore;
	m_browserFound = node.m_browserFound;
	m_browserSaved = node.m_browserSaved;
	m_ValidFields = node.m_ValidFields;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::~BkNode()                                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      BkNode class destructor.                                           *
 *                                                                         *
 ***************************************************************************/

BkNode::~BkNode()
{
}

/***************************************************************************
 *                                                                         *
 *   BkNode::operator=()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BkNode &                                                     *
 *   Return:                                                               *
 *      BkNode &                                                           *
 *   Description:                                                          *
 *      BkNode overloaded assignment operator.                             *
 *                                                                         *
 ***************************************************************************/

BkNode & BkNode::operator=(const BkNode &node)
{
	if(this != &node)
	{
		m_title = node.m_title;
		m_desc = node.m_desc;
		m_ignore = node.m_ignore;
		m_browserFound = node.m_browserFound;
		m_browserSaved = node.m_browserSaved;
		m_ValidFields = node.m_ValidFields;
		m_path = node.m_path;
		m_parentPath = node.m_parentPath;
		m_DiffDesc = node.m_DiffDesc;

		m_attrs = node.m_attrs;
		m_order = node.m_order;
	}

	return *this;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::attr()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      const char *name                                                   *
 *      string &value                                                      *
 *      BRWSNUM browser                                                    *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function retrieves a browser's private attribute.  The        *
 *      attribute's name and browser are specified.  If the attribute is   *
 *      located then the value parameter will contain the attribute's      *
 *      on return.  See bknode.h for more information about these          *
 *      attributes.                                                        *
 *                                                                         *
 ***************************************************************************/

void BkNode::attr(const QString &name, QString &value, BRWSNUM browser) const
{
	vector<BkAttr>::const_iterator it, start;

	// Search through the list of attributes for this node and attempt to
	// locate the named attribute for the specified browser.
	
	for(start=m_attrs.begin();;)
	{
		// start the search from where we left off from the last iteration
		// or the beginning if this is the first iteration.
		
		it = find(start, m_attrs.end(), name);

		// was the attribute not found?
		
		if(it == m_attrs.end())
		{
			value = "";
			return;
		}

		// is this is right browser?
		
		if(it->m_browser == browser)
			break;

		// start at the next node for the next iteration
		
		start = ++it;
	}

	// A match was found.  Return the value.
	
	value = it->m_value;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::copyOrder()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BkNode &node                                                 *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function copies the browser order vector from the node        *
 *      supplied in the argument to the order vector for this node.        *
 *                                                                         *
 ***************************************************************************/

void BkNode::copyOrder(const BkNode &node)
{
	vector<BkOrder>::const_iterator it;
	
	for(it=node.m_order.begin(); it!=node.m_order.end(); ++it)
		setOrder(it->browser(), it->order());
}

/***************************************************************************
 *                                                                         *
 *   BkNode::order()                                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      BRWSNUM browser                                                    *
 *   Return:                                                               *
 *      The order for the browser, or zero if unknown                      *
 *   Description:                                                          *
 *      This function returns the order for the node for the browser       *
 *      supplied in the first argument.                                    *
 *                                                                         *
 ***************************************************************************/

int BkNode::order(BRWSNUM browser) const
{
	vector<BkOrder>::const_iterator it;
	
	it = find(m_order.begin(), m_order.end(), browser);
	return it!=m_order.end() ? it->order() : 0;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::setOrder()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      BRWSNUM browser                                                    *
 *      int order                                                          *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function sets the order for the node for the browser supplied.*
 *                                                                         *
 ***************************************************************************/

void BkNode::setOrder(BRWSNUM browser, int order)
{
	vector<BkOrder>::iterator it;
	
	it = find(m_order.begin(), m_order.end(), browser);
	if(it != m_order.end())
		it->setOrder(order);
	else
		m_order.push_back(BkOrder(browser,order));
}

/***************************************************************************
 *                                                                         *
 *   BkNode::attr()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      int iAttr                                                          *
 *      string &name                                                       *
 *      string &value                                                      *
 *      BRWSNUM &browser                                                   *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs.                         *
 *   Description:                                                          *
 *      This function retrieves a browser's private attribute.  The index  *
 *      of the desired attribute is given in the first parameter.  The     *
 *      attributes name, value, and browser are returned.  See bknode.h    *
 *      for more information about these attributes.                       *                                *
 *                                                                         *
 ***************************************************************************/

void BkNode::attr(int iAttr, QString &name, QString &value, BRWSNUM &browser) const
			throw(BkException)
{
	// make sure the requested index is valid
	
	if(iAttr<0 || iAttr>=(int)m_attrs.size())
		BKEXCEPT("Attribute Index Out of Range");

	name = m_attrs[iAttr].m_name;
	value = m_attrs[iAttr].m_value;
	browser = m_attrs[iAttr].m_browser;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::setAttr()                                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      const char *name                                                   *
 *      const char *value                                                  *
 *      BRWSNUM browser                                                    *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs.                         *
 *   Description:                                                          *
 *      This function set a browser's private attribute for this node.     *
 *      The name, value, and browser are specified as parameters.  The     *
 *      function checks to see if this attribute is already specified for  *
 *      the browser and simply replaces the value if found.  Otherwise, a  *
 *      new attribute is saved.  See bknode.h for more information about   *
 *      these attributes.                                                  *
 *                                                                         *
 ***************************************************************************/

void BkNode::setAttr(const QString &name, const QString &value, BRWSNUM browser)
{
	BkAttr	temp;

	// look to see if this attribute already exists

	vector<BkAttr>::iterator it, start;

	for(start=m_attrs.begin();;)
	{
		// start the search from where we left off from the last iteration
		// or the beginning if this is the first iteration.

		it = find(start, m_attrs.end(), name);

		// the attribute wasn't found
		
		if(it == m_attrs.end())
			break;

		// the attribute already exists.  replace the value if
		// the browser is the same.

		if(browser == it->m_browser)
		{
			it->m_value = value;
			return;
		}

		// start at the next node for the next iteration

		start = ++it;
	}

	// allocate a new slot for the attribute

	temp.m_name  = name;
	temp.m_value = value;
	temp.m_browser = browser;
	m_attrs.push_back(temp);
}

/***************************************************************************
 *                                                                         *
 *   BkNode::isDeleted()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      true if this node was stored in a browser but has been deleted.    *
 *   Description:                                                          *
 *      This function determines if a bookmark (or folder) was previously  *
 *      stored in a browser but was deleted by the user.  It figures this  *
 *      out by removing any browsers that were found during this merge     *
 *      from a copy of the list of saved browsers.  If any browsers remain *
 *      in the saved browser list that means that the bookmark or folder   *
 *      used to be stored in a browser but wasn't found this time.         *
 *                                                                         *
 ***************************************************************************/

bool BkNode::isDeleted(void) const
{
	BrowserOrdLst bsaved(m_browserSaved);
	
	BrowserOrdLst::const_iterator fit;
	BrowserOrdLst::iterator sit;
	for(fit=m_browserFound.begin(); fit!=m_browserFound.end(); ++fit)
	{
		sit = remove(bsaved.begin(), bsaved.end(), *fit);
		if(sit != bsaved.end())
			bsaved.erase(sit, bsaved.end());
	}

	return bsaved.size() >= 1;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::setParentPath()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &parentPath                                          *
 *   Return:                                                               *
 *      true if error, false otherwise                                     *
 *   Description:                                                          *
 *      This function stores the path from this node to the root node as a *
 *      string.                                                            *
 *                                                                         *
 ***************************************************************************/

void BkNode::setParentPath(const QString &parentPath)
{
	m_parentPath = parentPath;
	m_path = m_parentPath + "/" + m_title;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::setTitle()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &title                                               *
 *   Return:                                                               *
 *      true if error, false otherwise                                     *
 *   Description:                                                          *
 *      This function stores the title of the data represented in this     *
 *      node.  For a bookmark, this would be the title of the bookmarked   *
 *      page.  For a folder, this would be the name of the folder.         *
 *                                                                         *
 ***************************************************************************/

void BkNode::setTitle(const QString &title)
{
	m_title = title;
	m_path = m_parentPath + "/" + m_title;
}

/***************************************************************************
 *                                                                         *
 *   BkNode::setDesc()                                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &desc                                                *
 *   Return:                                                               *
 *      true if error, false otherwise                                     *
 *   Description:                                                          *
 *      This function stores the description of the data represented in    *
 *      this node.                                                         *
 *                                                                         *
 ***************************************************************************/

void BkNode::setDesc(const QString &desc)
{
	m_desc = desc;
}

/***************************************************************************
 *                                                                         *
 *   BkFolder::BkFolder()                                                  *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      BkFolder class constructor.                                        *
 *                                                                         *
 ***************************************************************************/

BkFolder::BkFolder()
{
}

/***************************************************************************
 *                                                                         *
 *   BkFolder::operator=()                                                 *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BkFolder &                                                   *
 *   Return:                                                               *
 *      BkFolder &                                                         *
 *   Description:                                                          *
 *      BkFolder assignment operator.                                      *
 *                                                                         *
 ***************************************************************************/

BkFolder & BkFolder::operator=(const BkFolder & folder)
{
	if(this != &folder)
	{
		m_bookmarks = folder.m_bookmarks;
		m_folders = folder.m_folders;
		m_separators = folder.m_separators;
		BkNode::operator=(folder);
	}

	return *this;
}

/***************************************************************************
 *                                                                         *
 *   BkFolder::addBookmark()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkBookmark &bookmark                                               *
 *   Return:                                                               *
 *      BkBookmark &                                                       *
 *   Description:                                                          *
 *      This function adds a node, passed as a parameter, to this folder's *
 *      child list and returns a reference to the node.                    *
 *                                                                         *
 ***************************************************************************/

BkBookmark & BkFolder::addBookmark(const BkBookmark &bookmark)
{
	m_bookmarks.push_back(bookmark);
	BkBookmark &temp = m_bookmarks.back();
	temp.setParentPath(m_path);

	return temp;
}

/***************************************************************************
 *                                                                         *
 *   BkFolder::addFolder()                                                 *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder &folder                                                   *
 *   Return:                                                               *
 *      BkFolder &                                                         *
 *   Description:                                                          *
 *      This function adds a node, passed as a parameter, to this folder's *
 *      child list and returns a reference to the node.                    *
 *                                                                         *
 ***************************************************************************/

BkFolder & BkFolder::addFolder(const BkFolder &folder)
{
	m_folders.push_back(folder);
	BkFolder & temp = m_folders.back();
	temp.setParentPath(m_path);

	return temp;
}

/***************************************************************************
 *                                                                         *
 *   BkFolder::addSeparator()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkSeparator &separator                                             *
 *   Return:                                                               *
 *      BkSeparator &                                                      *
 *   Description:                                                          *
 *      This function adds a node, passed as a parameter, to this folder's *
 *      child list and returns a reference to the node.                    *
 *                                                                         *
 ***************************************************************************/

BkSeparator & BkFolder::addSeparator(const BkSeparator &separator)
{
	m_separators.push_back(separator);
	BkSeparator & temp = m_separators.back();
	temp.setParentPath(m_path);

	return temp;
}

/***************************************************************************
 *                                                                         *
 *   BkFolder::rollUpDifferences()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs.                         *
 *   Description:                                                          *
 *      This function walks through this folder's children and calls       *
 *      rollUpDifferences() for each them.  See the description of         *
 *      BkBookmark::rollUpDifferences() for more information.              *
 *                                                                         *
 ***************************************************************************/

void BkFolder::rollUpDifferences(void) throw(BkException)
{
	// first, roll up any differences in this folder node

	rollUpDifferencesPvt();

	BookmarkLst::iterator bit;
	FolderLst::iterator fit;

	// roll up differences for each bookmark in this folder

	for(bit = BookmarksBegin();
		bit != BookmarksEnd();
		++bit)
	{
		bit->rollUpDifferences();
	}

	// roll up differences for any child folder
	
	for(fit = FoldersBegin();
		fit != FoldersEnd();
		++fit)
	{
		fit->rollUpDifferences();
	}
}

/***************************************************************************
 *                                                                         *
 *   BkFolder::rollUpDifferencesPvt()                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs.                         *
 *   Description:                                                          *
 *      This function scans this node's differences chain and determines   *
 *      which node in the differences chain wins.  This is decided by      *
 *      using the differences node most recently modified.  Once the       *
 *      winner is chosen, the changed are rolled up into the node itself.  *
 *      Once the differences are rolled in, the chain of differences are   *
 *      purged.                                                            *
 *                                                                         *
 ***************************************************************************/

void BkFolder::rollUpDifferencesPvt(void) throw(BkException)
{
	if(!diffDesc().isNull())
	{
		setDesc(diffDesc());
		setValidField(BKVALID_DESC);
	}
}

/***************************************************************************
 *                                                                         *
 *   BkFolder::sortContents()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.                                                              *
 *   Description:                                                          *
 *      This function sorts the folders and bookmarks by title then        *
 *      iterates through the child folders asking them to do the same.     *
 *                                                                         *
 ***************************************************************************/

void BkFolder::sortContents(void)
{
	// Sort the folders first

	sort(m_folders.begin(), m_folders.end());

	// Sort the bookmarks next

	sort(m_bookmarks.begin(), m_bookmarks.end());

	// Sort the contents of each child folder

	FolderLst::iterator fit;

	for(fit=m_folders.begin(); fit!=m_folders.end(); ++fit)
		fit->sortContents();
}

/***************************************************************************
 *                                                                         *
 *   BkBookmark::BkBookmark()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      BkBookmark class constructor.                                      *
 *                                                                         *
 ***************************************************************************/

BkBookmark::BkBookmark()
{
}

/***************************************************************************
 *                                                                         *
 *   BkBookmark::BkBookmark()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkBookmark &                                                       *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      BkBookmark class copy constructor.                                 *
 *                                                                         *
 ***************************************************************************/

BkBookmark::BkBookmark(const BkBookmark &bmark)
	: BkNode(bmark), m_url(bmark.m_url), m_DiffTitle(bmark.m_DiffTitle)
{
}

/***************************************************************************
 *                                                                         *
 *   BkBookmark::operator=()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BkBookmark &                                                 *
 *   Return:                                                               *
 *      BkBookmark &                                                       *
 *   Description:                                                          *
 *      BkBookmark assignment operator.                                    *
 *                                                                         *
 ***************************************************************************/

BkBookmark & BkBookmark::operator=(const BkBookmark & bookmark)
{
	if(this != &bookmark)
	{
		m_url = bookmark.m_url;
		m_DiffTitle = bookmark.m_DiffTitle;
		BkNode::operator=(bookmark);
	}

	return *this;
}

/***************************************************************************
 *                                                                         *
 *   BkBookmark::operator==()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BkBookmark &                                                 *
 *   Return:                                                               *
 *      true if equal, false otherwise                                     *
 *   Description:                                                          *
 *      BkBookmark equality operator.                                      *
 *                                                                         *
 ***************************************************************************/

bool BkBookmark::operator==(const BkBookmark & bookmark) const
{
	if(bookmark.isFieldValid(BKVALID_TITLE))
	{
		if( !isFieldValid(BKVALID_TITLE)
		 || title() != bookmark.title())
			return false;
	}

	if(bookmark.isFieldValid(BKVALID_DESC))
	{
		if( !isFieldValid(BKVALID_DESC)
		 || desc() != bookmark.desc())
			return false;
	}

	return true;
}

/***************************************************************************
 *                                                                         *
 *   BkBookmark::rollUpDifferences()                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs.                         *
 *   Description:                                                          *
 *      This function scans this node's differences chain and determines   *
 *      which node in the differences chain wins.  This is decided by      *
 *      using the differences node most recently modified.  Once the       *
 *      winner is chosen, the changed are rolled up into the node itself.  *
 *      Once the differences are rolled in, the chain of differences are   *
 *      purged.                                                            *
 *                                                                         *
 ***************************************************************************/

void BkBookmark::rollUpDifferences(void) throw(BkException)
{
	if(!diffDesc().isNull())
	{
		setDesc(diffDesc());
		setValidField(BKVALID_DESC);
		clearBrowserSaved();
	}

	if(!diffTitle().isNull())
	{
		setTitle(diffTitle());
		setValidField(BKVALID_TITLE);
		clearBrowserSaved();
	}
}
