/*
 * Copyright (C) 1999  Ross Combs (rocombs@cs.nmsu.edu)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <ctype.h>
#include <errno.h>
#include "compat/strerror.h"
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <netdb.h>
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#include <termios.h>
#include <sys/types.h>
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include "compat/netinet_in.h"
#include <arpa/inet.h>
#include "compat/inet_ntoa.h"
#include "packet.h"
#include "init_protocol.h"
#include "bot_protocol.h"
#include "bn_type.h"
#include "field_sizes.h"
#include "network.h"
#include "version.h"
#include "util.h"
#ifdef CLIENTDEBUG
#include "eventlog.h"
#endif
#include "client.h"


#define SCREEN_WIDTH 80 /* FIXME need to determine real terminal width here */


static void usage(char const * progname);


static void usage(char const * progname)
{
    fprintf(stderr,"usage: %s [<servername> [<TCP portnumber>]]\n"
            "    -h, --help, --usage         show this information and exit\n"
            "    -v, --version               print version number and exit\n",
	    progname);
    exit(1);
}


int main(int argc, char * argv[])
{
    int                a;
    int                sd;
    struct sockaddr_in saddr;
    t_packet *         packet;
    t_packet *         rpacket;
    char const *       servname=NULL;
    unsigned short     servport=0;
    char               text[MAX_MESSAGE_LEN];
    struct hostent *   host;
    unsigned int       commpos;
    struct termios     in_attr_old;
    struct termios     in_attr_new;
    int                changed_in;
    unsigned int       currsize;
    int                fd_stdin;
    
    if (argc<1 || !argv || !argv[0])
    {
	fprintf(stderr,"bad arguments\n");
	return 1;
    }
    
    for (a=1; a<argc; a++)
	if (servname && isdigit((int)argv[a][0]) && a+1>=argc)
	{
	    if (str_to_ushort(argv[a],&servport)<0)
	    {
		fprintf(stderr,"%s: \"%s\" should be a positive integer\n",argv[0],argv[a]);
		usage(argv[0]);
	    }
	}
	else if (!servname && argv[a][0]!='-' && a+2>=argc)
	    servname = argv[a];
	else if (strcmp(argv[a],"-v")==0 || strcmp(argv[a],"--version")==0)
	{
            printf("version "BNETD_VERSION"\n");
            return 0;
	}
	else if (strcmp(argv[a],"-h")==0 || strcmp(argv[a],"--help")==0 || strcmp(argv[a],"--usage")==0)
            usage(argv[0]);
	else
	{
	    fprintf(stderr,"%s: unknown option \"%s\"\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
    
    if (servport==0)
        servport = BNETD_SERV_PORT;
    if (!servname)
       servname = BNETD_DEFAULT_HOST;
    
    if (!(host = gethostbyname(servname)))
    {
	fprintf(stderr,"%s: unknown host \"%s\"\n",argv[0],argv[1]);
	return 1;
    }
    
    fd_stdin = fileno(stdin);
    if (tcgetattr(fd_stdin,&in_attr_old)>=0)
    {
        in_attr_new = in_attr_old;
        in_attr_new.c_lflag &= ~(ECHO | ICANON); /* turn off ECHO and ICANON */
	in_attr_new.c_cc[VMIN]  = 0; /* don't require reads to return data */
        in_attr_new.c_cc[VTIME] = 1; /* timeout = .1 second */
        tcsetattr(fd_stdin,TCSANOW,&in_attr_new);
        changed_in = 1;
    }
    else
    {
	fprintf(stderr,"%s: could not set terminal attributes for stdin\n",argv[0]);
	changed_in = 0;
    }
    
    if ((sd = socket(PF_INET,SOCK_STREAM,0))<0)
    {
	fprintf(stderr,"%s: could not create socket (socket: %s)\n",argv[0],strerror(errno));
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    memset(&saddr,0,sizeof(saddr));
    saddr.sin_family = AF_INET;
    saddr.sin_port   = htons(servport);
    memcpy(&saddr.sin_addr.s_addr,host->h_addr_list[0],host->h_length);
    if (connect(sd,(struct sockaddr *)&saddr,sizeof(saddr))<0)
    {
	fprintf(stderr,"%s: could not connect to server \"%s\" port %hu (connect: %s)\n",argv[0],servname,servport,strerror(errno));
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    if (fcntl(sd,F_SETFL,O_NONBLOCK)<0)
    {
	fprintf(stderr,"%s: could not set TCP socket to non-blocking mode (fcntl: %s)\n",argv[0],strerror(errno));
	close(sd);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    printf("Connected to %s:%hu.\n",inet_ntoa(saddr.sin_addr),servport);
#ifdef CLIENTDEBUG
    eventlog_set(stderr);
#endif
    
    {
	t_connect_header temp;
	
	bn_byte_set(&temp.class,CONNECT_CLASS_BOT);
	if (write(sd,&temp,sizeof(temp))!=sizeof(temp))
	{
	    fprintf(stderr,"%s: could not write to server (write: %s)\n",argv[0],strerror(errno));
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	    return 1;
	}
    }
    
    if (!(rpacket = packet_create(packet_class_raw)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    if (!(packet = packet_create(packet_class_raw)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_append_data(packet,"\004",1);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    {
	int    highest_fd;
	fd_set rfds;
	
	FD_ZERO(&rfds);
	highest_fd = fd_stdin;
	if (sd>highest_fd)
	    highest_fd = sd;
	
	commpos = 0;
	text[0] = '\0';
	
	for (;;)
	{
	    FD_SET(fd_stdin,&rfds);
	    FD_SET(sd,&rfds);
	    
	    if (select(highest_fd+1,&rfds,NULL,NULL,NULL)<0)
	    {
		if (errno!=EINTR)
		    fprintf(stderr,"%s: select failed (select: %s)\n",argv[0],strerror(errno));
		continue;
	    }
	    
	    if (FD_ISSET(sd,&rfds)) /* got network data */
	    {
		packet_set_size(rpacket,MAX_PACKET_SIZE-1);
		currsize = 0;
		if (net_recv_packet(sd,rpacket,&currsize)<0)
		{
		    close(sd);
		    sd = -1;
		}
		
		if (currsize>0)
		{
		    char * str=packet_get_raw_data(rpacket,0);
		    
		    str[currsize] = '\0';
		    printf("%s",str);
		    fflush(stdout);
		}
		
		if (sd==-1) /* if connection was closed */
		{
		    printf("Connection closed by server.\n");
		    if (changed_in)
			tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
 		    packet_del_ref(rpacket);
		    return 0;
		}
	    }
	    
	    if (FD_ISSET(fd_stdin,&rfds)) /* got keyboard data */
	    {
		switch (client_get_comm("",text,MAX_MESSAGE_LEN,&commpos,-1,SCREEN_WIDTH))
		{
		case -1: /* cancel */
		    if (changed_in)
			tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		    return 1;
		    
		case 0: /* timeout */
		    break;
		    
		case 1:
		    if (!(packet = packet_create(packet_class_raw)))
		    {
			fprintf(stderr,"%s: could not create packet\n",argv[0]);
			if (changed_in)
			    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
 			packet_del_ref(rpacket);
			return 1;
		    }
		    packet_append_data(packet,text,strlen(text));
		    packet_append_data(packet,"\r\n",2);
		    client_blocksend_packet(sd,packet);
		    packet_del_ref(packet);
		    commpos = 0;
		    text[0] = '\0';
		}
	    }
	}
    }
    
    /* not reached */
    return 0;
}
