/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/cports.c                */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Thu Jul 23 15:34:53 1992                          */
/*    Last change :  Thu Nov  8 09:29:57 2001 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Input ports handling                                             */
/*=====================================================================*/
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#if( !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   ifdef HAVE_TERMIO
#      include <termio.h>
#   endif
#endif
#if( !defined( sony_news ) && \
     !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   include <unistd.h>
#endif
#include <sys/file.h>
#include <bigloo.h>
#if( defined( sony_news ) || (defined( NeXT ) && defined( mc68000 )) )
#   include <ctype.h>
#endif
 
/*---------------------------------------------------------------------*/
/*    isascii                                                          */
/*---------------------------------------------------------------------*/
#if( !defined( isascii ) )
#   define isascii( c ) (!((c) & ~0177))
#endif

/*---------------------------------------------------------------------*/
/*    Global variables                                                 */
/*---------------------------------------------------------------------*/
obj_t current_output_port, current_input_port, current_error_port;
long default_io_bufsiz;

/*---------------------------------------------------------------------*/
/*    External definitions.                                            */
/*---------------------------------------------------------------------*/
extern obj_t string_to_bstring();
extern obj_t string_to_keyword();
extern obj_t make_real();
extern void  c_error();
extern obj_t make_input_port( char *, FILE *, obj_t, long );
extern obj_t make_string();

extern int bgl_fgetc( FILE * );
extern int bgl_feof( FILE * );

#if( !defined( fgetc ) )
#  define  bgl_fgetc fgetc
#endif

#if( !defined( feof ) )
#  define  bgl_feof feof
#endif

/*---------------------------------------------------------------------*/
/*    Prototypes                                                       */
/*---------------------------------------------------------------------*/
static bool_t pipe_name_p( char * );
static char *pipe_name( char * );

/*---------------------------------------------------------------------*/
/*     make_output_port ...                                            */
/*---------------------------------------------------------------------*/
obj_t
make_output_port( char *name, FILE *file, obj_t kindof ) {
   obj_t new_output_port;

   new_output_port = GC_MALLOC( OUTPUT_PORT_SIZE );
   
   new_output_port->output_port_t.header = MAKE_HEADER( OUTPUT_PORT_TYPE, 0 );
   new_output_port->output_port_t.file   = file;
   new_output_port->output_port_t.name   = name;
   new_output_port->output_port_t.kindof = kindof;
   
   return BGL_HEAP_DEBUG_MARK_OBJ( BREF( new_output_port ) );
}

/*---------------------------------------------------------------------*/
/*    open_output_file ...                                             */
/*---------------------------------------------------------------------*/
obj_t
open_output_file( obj_t name ) {
   FILE *file;

#if( HAVE_PIPE )
   if( pipe_name_p( (char *)BSTRING_TO_STRING( name ) ) ) {
      if( !(file = popen( pipe_name( (char *)BSTRING_TO_STRING( name ) ),
			  "w" )) )
	 return BFALSE;
      
      return make_output_port( (char *)BSTRING_TO_STRING( name ),
			       file,
			       KINDOF_PIPE );
   } else
#endif
   {
      if( !(file = fopen( (char *)BSTRING_TO_STRING( name ), "wb" )) )
	 return BFALSE;
      
      return make_output_port( (char *)BSTRING_TO_STRING( name ),
			       file,
			       KINDOF_FILE );
   }
}

/*---------------------------------------------------------------------*/
/*    append_output_file ...                                           */
/*---------------------------------------------------------------------*/
obj_t
append_output_file( obj_t name ) {
   FILE *file;
   
   if( !(file = fopen( (char *)BSTRING_TO_STRING( name ), "a+b" )) )
      return BFALSE;

   return make_output_port( (char *)BSTRING_TO_STRING( name ),
			    file,
			    KINDOF_FILE );
}
 
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_output_string ...                                           */
/*---------------------------------------------------------------------*/
obj_t
open_output_string() {
   char *buffer;
   obj_t port;

   port = GC_MALLOC( OUTPUT_STRING_PORT_SIZE );

   /* We allocate a buffer filled of zero */
   buffer = BGL_HEAP_DEBUG_MARK_STR(
      (char *)(GC_MALLOC_ATOMIC( OUTPUT_STRING_PORT_BUFFER_SIZE + 1 )) );
   bzero( buffer, OUTPUT_STRING_PORT_BUFFER_SIZE + 1 );

   port->output_string_port_t.header = MAKE_HEADER( OUTPUT_STRING_PORT_TYPE,0);
   port->output_string_port_t.buffer = buffer;
   port->output_string_port_t.size   = OUTPUT_STRING_PORT_BUFFER_SIZE;
   port->output_string_port_t.offset = 0;
   
   return BGL_HEAP_DEBUG_MARK_OBJ( BREF( port ) );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    get_output_string ...                                            */
/*---------------------------------------------------------------------*/
obj_t
get_output_string( obj_t port ) {
   if( OUTPUT_STRING_PORTP( port ) ) {
      if( OUTPUT_STRING_PORT( port ).size == 0 )
	 return make_string( 0, ' ' );
      else
	 return string_to_bstring_len( OUTPUT_STRING_PORT( port ).buffer,
				       OUTPUT_STRING_PORT( port ).offset );
   }
   else
      C_FAILURE( "get-output-string", "Not a string port", port );
}

/*---------------------------------------------------------------------*/
/*    close_output_port ...                                            */
/*---------------------------------------------------------------------*/
obj_t
close_output_port( obj_t port ) {
   if( OUTPUT_STRING_PORTP( port ) ) {
      if( OUTPUT_STRING_PORT( port ).size == 0 ) {
	 C_FAILURE( "close-output-port", "port already closed", port );
      
	 return port;
      } else {
	 obj_t res;

	 res = string_to_bstring_len( OUTPUT_STRING_PORT( port ).buffer,
				      OUTPUT_STRING_PORT( port ).offset );
	 OUTPUT_STRING_PORT( port ).buffer = 0L;
	 OUTPUT_STRING_PORT( port ).size = 0;

	 return res;
      }
   } else {
      /* We do not close console ports (e.g. stdout, stderr) */
      switch( (long)(OUTPUT_PORT( port ).kindof) ) {
	 case (long)KINDOF_FILE:
	    OUTPUT_PORT( port ).kindof = KINDOF_CLOSED;
	    fclose( OUTPUT_PORT( port ).file );
	    break;
	    
#if( HAVE_PIPE )
	 case (long)KINDOF_PROCPIPE:
	    OUTPUT_PORT( port ).kindof = KINDOF_CLOSED;
	    fclose( OUTPUT_PORT( port ).file );
	    break;
	    
	 case (long)KINDOF_PIPE:
	    OUTPUT_PORT( port ).kindof = KINDOF_CLOSED;
	    pclose( OUTPUT_PORT( port ).file );
	    break;
#endif
	    
	 case (long)KINDOF_CLOSED:
	    break;
      }
      
      return port;
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    make_input_port ...                                              */
/*---------------------------------------------------------------------*/
obj_t
make_input_port( char *name, FILE *file, obj_t kindof, long bufsiz ) {
   obj_t new_input_port;

   /* An input port cannot be allocated as an atomic object    */
   /* because it holds a buffer and a name that are GC objects */
   new_input_port = GC_MALLOC( INPUT_PORT_SIZE );
   
   new_input_port->input_port_t.header      = MAKE_HEADER( INPUT_PORT_TYPE,0 );
   new_input_port->input_port_t.kindof      = kindof;
   new_input_port->input_port_t.name        = name;
   new_input_port->input_port_t.file        = file;
   new_input_port->input_port_t.filepos     = 0;
   new_input_port->input_port_t.bufsiz      = bufsiz;
   new_input_port->input_port_t.eof         = 0;
   new_input_port->input_port_t.matchstart  = 0;
   new_input_port->input_port_t.matchstop   = 0;
   new_input_port->input_port_t.forward     = 0;
   new_input_port->input_port_t.abufsiz     = 1;
   new_input_port->input_port_t.lastchar    = '\n';
   new_input_port->input_port_t.syseof      = bgl_feof;

   switch( (long)kindof ) {
      case (long)KINDOF_CONSOLE:
      case (long)KINDOF_SOCKET:
	 new_input_port->input_port_t.sysread = bgl_fgetc;
	 break;

      case (long)KINDOF_FILE:
      case (long)KINDOF_PIPE:
      case (long)KINDOF_PROCPIPE:
	 new_input_port->input_port_t.sysread = (int (*)())fread;
	 break;

      default:
	 new_input_port->input_port_t.sysread = (int (*)())fread;
   }
      
   
   if( bufsiz > 0 ) {
      new_input_port->input_port_t.buffer = BGL_HEAP_DEBUG_MARK_STR(
	 (unsigned char *)GC_MALLOC_ATOMIC(bufsiz+1) );
      new_input_port->input_port_t.buffer[ 0 ] = '\0';
   }
   else
      new_input_port->input_port_t.buffer = '\0';

   return BGL_HEAP_DEBUG_MARK_OBJ( BREF( new_input_port ) );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_input_pipe ...                                              */
/*---------------------------------------------------------------------*/
obj_t
open_input_pipe( obj_t name, obj_t bbufsiz ) {
#if( HAVE_PIPE )
   FILE *file;
   char *cname = (char *)BSTRING_TO_STRING( name );

   if( !(file = popen( cname, "r" )) )
      return BFALSE;

   return make_input_port( (char *)cname, file, KINDOF_PIPE, CINT( bbufsiz ) );
#else
   return BFLASE;
#endif
}

/*---------------------------------------------------------------------*/
/*    open_input_file ...                                              */
/*    -------------------------------------------------------------    */
/*    !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!!  */
/*    We fill up its associated buffer when opening an input port.     */
/*    -------------------------------------------------------------    */
/*    This function open two kind of files. Regular file and Unix      */
/*    like pipes when the file name is something like "| ...".         */
/*---------------------------------------------------------------------*/
obj_t
open_input_file( obj_t name, obj_t bbufsiz ) { 
   FILE *file;
   char *cname = (char *)BSTRING_TO_STRING( name );
   
#if( HAVE_PIPE )
   if( pipe_name_p( cname ) ) {
      if( !(file = popen( pipe_name( cname ), "r" )) )
	 return BFALSE;

      return make_input_port( (char *)cname, file, KINDOF_PIPE, CINT( bbufsiz ) );
   } else
#endif
   {
      if( !(file = fopen( (const char *)cname, "rb" )) )
	 return BFALSE;

      return make_input_port( cname, file, KINDOF_FILE, CINT( bbufsiz ) );
   }
}

/*---------------------------------------------------------------------*/
/*    open_input_console ...                                           */
/*---------------------------------------------------------------------*/
obj_t
open_input_console() {
   return make_input_port( "[stdin]",
			   stdin,
			   KINDOF_CONSOLE,
			   default_io_bufsiz );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    file_to_buffered_input_port ...                                  */
/*---------------------------------------------------------------------*/
obj_t
file_to_buffered_input_port( FILE *file, long bufsize ) {
   if( file == stdin )
      return open_input_console();
   else {
      long bsize = bufsize <= 0 ? default_io_bufsiz : bufsize;
      
      return make_input_port( "[file]", file, KINDOF_FILE, bsize );
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    file_to_input_port ...                                           */
/*---------------------------------------------------------------------*/
obj_t
file_to_input_port( FILE *file ) {
   return file_to_buffered_input_port( file, -1 );
}

/*---------------------------------------------------------------------*/
/*    open_input_string ...                                            */
/*---------------------------------------------------------------------*/
obj_t
open_input_string( obj_t string ) {
   obj_t port;
   long  bufsiz = STRING_LENGTH( string );

   port = make_input_port( "[string]", 0L, KINDOF_STRING, bufsiz + 1 );
   
   CREF( port )->input_port_t.eof     = 1;
   CREF( port )->input_port_t.abufsiz = bufsiz + 1;
   memcpy( &RGC_BUFFER( port )[ 0 ],
	   (void *)BSTRING_TO_STRING( string ),
	   bufsiz );
   RGC_BUFFER( port )[ bufsiz ] = '\0';
   
   return port;   
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_input_c_string ...                                          */
/*---------------------------------------------------------------------*/
obj_t
open_input_c_string( char *c_string ) {
   obj_t port;
   long  bufsiz = strlen( c_string );
   char *new_string = 0;

   if( bufsiz > 0 )
      new_string = BGL_HEAP_DEBUG_MARK_STR(
	 (char *)GC_MALLOC_ATOMIC( bufsiz + 1 ) );

   strcpy( new_string, c_string );

   port = make_input_port( "[c_string]", 0L, KINDOF_STRING, 0 );
   
   CREF( port )->input_port_t.eof     = 1;
   CREF( port )->input_port_t.bufsiz  = bufsiz + 1;
   CREF( port )->input_port_t.abufsiz = bufsiz + 1;
   CREF( port )->input_port_t.buffer  = (unsigned char *)new_string;
   
   return port;   
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    reopen_input_c_string ...                                        */
/*    -------------------------------------------------------------    */
/*    Simply changes the input buffer of an input string. Does not     */
/*    re-allocate a brand new input-port.                              */
/*---------------------------------------------------------------------*/
obj_t
reopen_input_c_string( obj_t port, char *c_string ) {
   long bufsiz = strlen( c_string );

   if( CREF( port )->input_port_t.bufsiz < (bufsiz + 1) ) {
      CREF( port )->input_port_t.bufsiz = bufsiz + 1;
      CREF( port )->input_port_t.buffer =
	 (unsigned char *)BGL_HEAP_DEBUG_MARK_STR(
	    (char *)GC_MALLOC_ATOMIC( bufsiz + 1 ) );
   }

   CREF( port )->input_port_t.abufsiz     = bufsiz + 1;
   CREF( port )->input_port_t.matchstart  = 0;
   CREF( port )->input_port_t.matchstop   = 0;
   CREF( port )->input_port_t.forward     = 0;
   CREF( port )->input_port_t.lastchar    = '\n';
   strcpy( CREF( port )->input_port_t.buffer, (unsigned char *)c_string );

   return port;
}

/*---------------------------------------------------------------------*/
/*    close_input_port ...                                             */
/*---------------------------------------------------------------------*/
obj_t
close_input_port( obj_t port ) {
   if( INPUT_PORTP( port ) ) {
      /* We do not close the console port */
      switch( (long)(INPUT_PORT( port ).kindof) ) {
	 case (long)KINDOF_FILE:
	    INPUT_PORT( port ).kindof = KINDOF_CLOSED;
	    INPUT_PORT( port ).eof = 1;
	    RGC_BUFFER( port ) = 0L;
	    fclose( INPUT_PORT( port ).file );
	    break;
	    
#if( HAVE_PIPE ) 
	 case (long)KINDOF_PROCPIPE:
	    INPUT_PORT( port ).kindof = KINDOF_CLOSED;
	    INPUT_PORT( port ).eof = 1;
	    RGC_BUFFER( port ) = 0L;
	    fclose( INPUT_PORT( port ).file );
	    break;
	    
	 case (long)KINDOF_PIPE:
	    INPUT_PORT( port ).kindof = KINDOF_CLOSED;
	    INPUT_PORT( port ).eof = 1;
	    RGC_BUFFER( port ) = 0L;
	    pclose( INPUT_PORT( port ).file );
	    break;
#endif
	    
	 case (long)KINDOF_STRING:
	    INPUT_PORT( port ).kindof = KINDOF_CLOSED;
	    break;
	    
	 case (long)KINDOF_CLOSED:
	 case (long)KINDOF_CONSOLE:
	    break;
 
	 case (long)KINDOF_SOCKET:
	    INPUT_PORT( port ).kindof = KINDOF_CLOSED;
	    INPUT_PORT( port ).eof = 1;
	    RGC_BUFFER( port ) = 0L;
	    fclose( INPUT_PORT( port ).file );
	    break;
 
	 default:
	    C_FAILURE( "close-input-port", "unknown input-port type", port );
	    break;
      }
   } 

   return port;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_input_port_seek ...                                          */
/*---------------------------------------------------------------------*/
obj_t
bgl_input_port_seek( obj_t port, long pos ) {
   if( !INPUT_PORT_ON_FILEP( port ) )
      return BFALSE;

   fseek( INPUT_PORT( port ).file, pos, SEEK_SET );
	   
   INPUT_PORT( port ).filepos     = pos;
   INPUT_PORT( port ).eof         = 0;
   INPUT_PORT( port ).matchstart  = 0;
   INPUT_PORT( port ).matchstop   = 0;
   INPUT_PORT( port ).forward     = 0;
   INPUT_PORT( port ).abufsiz     = 1;
   INPUT_PORT( port ).lastchar    = '\n';
   INPUT_PORT( port ).buffer[ 0 ] = '\0';
   
   return BTRUE;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_input_port_reopen ...                                        */
/*---------------------------------------------------------------------*/
obj_t
bgl_input_port_reopen( obj_t port ) {
   if( !INPUT_PORT_ON_FILEP( port ) )
      return BFALSE;

   freopen( INPUT_PORT( port ).name, "r", INPUT_PORT( port ).file );
	   
   INPUT_PORT( port ).filepos     = 0;
   INPUT_PORT( port ).eof         = 0;
   INPUT_PORT( port ).matchstart  = 0;
   INPUT_PORT( port ).matchstop   = 0;
   INPUT_PORT( port ).forward     = 0;
   INPUT_PORT( port ).abufsiz     = 1;
   INPUT_PORT( port ).lastchar    = '\n';
   INPUT_PORT( port ).buffer[ 0 ] = '\0';
   
   return BTRUE;
}

/*---------------------------------------------------------------------*/
/*    obj                                                              */
/*    reset_console ...                                                */
/*    -------------------------------------------------------------    */
/*    We flush input port, for ^C to work correctly within the         */
/*    interpreter. The only place this function is called is in the    */
/*    REPL (see Eva/eval.scm).                                         */
/*---------------------------------------------------------------------*/
obj_t
reset_console( obj_t port ) {
   if( INPUT_PORT( port ).kindof == KINDOF_CONSOLE ) {
      INPUT_PORT( port ).matchstart  = 0;
      INPUT_PORT( port ).matchstop   = 0;
      INPUT_PORT( port ).abufsiz     = 1;
      INPUT_PORT( port ).buffer[ 0 ] = '\0';
      INPUT_PORT( port ).lastchar    = '\n';
   }

   return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*     init_io ...                                                     */
/*---------------------------------------------------------------------*/
void init_io() {
#if( !defined( _SBFSIZ ) )
#   define _SBFSIZ 1
#endif

   default_io_bufsiz = BUFSIZ * _SBFSIZ;
   
   current_output_port = make_output_port( "stdout", stdout, KINDOF_CONSOLE );
   current_error_port  = make_output_port( "stderr", stderr, KINDOF_CONSOLE );
   current_input_port  = open_input_console();
}

/*---------------------------------------------------------------------*/
/*    fexists ...                                                      */
/*---------------------------------------------------------------------*/
bool_t
fexists( char *name ) {
#if( HAVE_PIPE )
   if( pipe_name_p( name ) ) {
      return 1;
   }
#else
   if( pipe_name_p( name ) ) {
      return 0;
   }
#endif   
   return !access( name, F_OK );
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    reset_eof ...                                                    */
/*    -------------------------------------------------------------    */
/*    The function erase the end-of-file of input console port.        */
/*    This allows, restart reading after a ^D.                         */
/*---------------------------------------------------------------------*/
bool_t
reset_eof( obj_t port ) {
   if( INPUT_PORT( port ).kindof == KINDOF_CONSOLE ) {
      /* we forget about EOF */
      INPUT_PORT( port ).eof = 0;
      
      /* we cleanup buffer   */
      reset_console( port );

      /* we clear errors.    */
      clearerr( stdin );

      return 1;
   }
   else
      return 0;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strport_flush ...                                                */
/*    -------------------------------------------------------------    */
/*    On flush un string port.                                         */
/*---------------------------------------------------------------------*/
obj_t
strport_flush( obj_t port ) {
   obj_t res;

   res = string_to_bstring( OUTPUT_STRING_PORT( port ).buffer );

   OUTPUT_STRING_PORT( port ).offset      = 0;
   OUTPUT_STRING_PORT( port ).buffer[ 0 ] = 0;

   return res;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strport_grow ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
strport_grow( obj_t p ) {
   long  old_size, new_size;
   char *old_buffer, *new_buffer;
	
   old_buffer = OUTPUT_STRING_PORT( p ).buffer;
   old_size   = OUTPUT_STRING_PORT( p ).size;

   if( old_size == 0 ) {
      C_FAILURE( "string-port", "close port", p );
      
      return p;
   } else {
      new_size   = old_size * 2;
      new_buffer = BGL_HEAP_DEBUG_MARK_STR(
	 (char *)( GC_MALLOC_ATOMIC( new_size + 1 ) ) );
      bzero( &new_buffer[ old_size + 1 ], old_size );
      strcpy( new_buffer, old_buffer );

      OUTPUT_STRING_PORT( p ).buffer = new_buffer;
      OUTPUT_STRING_PORT( p ).size   = new_size;

      return p;
   }
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    pipe_name_p ...                                                  */
/*    -------------------------------------------------------------    */
/*    Is a file name a pipe name ? A pipe name start by the            */
/*    sequence "| ".                                                   */
/*---------------------------------------------------------------------*/
static bool_t
pipe_name_p( char *name ) {
   return ((name[ 0 ] == '|') && (name[ 1 ] == ' '));
}

/*---------------------------------------------------------------------*/
/*    char *                                                           */
/*    pipe_name ...                                                    */
/*    -------------------------------------------------------------    */
/*    Pipe name to name translation.                                   */
/*---------------------------------------------------------------------*/
static char *
pipe_name( char *pipe_name ) {
   return (pipe_name + 1);
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    directoryp ...                                                   */
/*    -------------------------------------------------------------    */
/*    Is a file a directory?                                           */
/*---------------------------------------------------------------------*/
bool_t directoryp( char *name ) {
   struct stat buf;

   if( stat( name, &buf ) == -1 )
      return 0;

   return S_ISDIR( buf.st_mode & S_IFMT );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    directory_to_list ...                                            */
/*---------------------------------------------------------------------*/
obj_t directory_to_list( char *name ) {
   DIR *dir;
   struct dirent *dirent;
   obj_t res = BNIL;
   
   if( (dir = opendir( name )) ) {
      while( (dirent = readdir( dir )) ) {
	 char *fname = dirent->d_name;
	 
	 if( strcmp( fname, ".") && strcmp( fname, ".." ) )
	    res = MAKE_PAIR( string_to_bstring( fname ), res );
      }
      closedir( dir );
   }

   return res;
}

