/** 
 * @namespace   biew_plugins_I
 * @file        plugins/hexmode.c
 * @brief       This file contains implementation of hexadecimal mode viewers.
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 *
 * @author      Nick Kurshev
 * @since       1995
 * @note        Development, fixes and improvements
**/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "plugins/hexmode.h"
#include "colorset.h"
#include "bconsole.h"
#include "biewutil.h"
#include "biewhelp.h"
#include "bmfile.h"
#include "reg_form.h"
#include "codeguid.h"
#include "editor.h"
#include "biewlib/file_ini.h"
#include "biewlib/biewlib.h"
#include "biewlib/kbd_code.h"

typedef char *(__NEAR__ __FASTCALL__ *hexFunc)(unsigned long);
typedef unsigned char (__NEAR__ __FASTCALL__ *sizeFunc) ( void );

typedef struct tag_hexView
{
  const char *  name;
  hexFunc       func;
  sizeFunc      width;
  unsigned char size;
  unsigned char hardlen;
}hexView;

static char * __NEAR__ __FASTCALL__ GetB(unsigned long val) { return GetBinary(BMReadByteEx(val,BM_SEEK_SET)); }
static char * __NEAR__ __FASTCALL__ Get2D(unsigned long val) { return Get2Digit(BMReadByteEx(val,BM_SEEK_SET)); }
static char * __NEAR__ __FASTCALL__ Get4D(unsigned long val) { return Get4Digit(BMReadWordEx(val,BM_SEEK_SET)); }
static char * __NEAR__ __FASTCALL__ Get8D(unsigned long val) { return Get8Digit(BMReadDWordEx(val,BM_SEEK_SET)); }

static unsigned char __NEAR__ __FASTCALL__ sizeBit( void )  { return (tvioWidth-10)/(8+1+1); }
static unsigned char __NEAR__ __FASTCALL__ sizeByte( void ) { return (((tvioWidth-10)/(2+1+1)-(tvioWidth-10)%((2+1+1)-1))/4)*4; }
static unsigned char __NEAR__ __FASTCALL__ sizeWord( void ) { return (tvioWidth-10)/(4+1+2); }
static unsigned char __NEAR__ __FASTCALL__ sizeDWord( void ){ return (tvioWidth-10)/(8+1+4); }

hexView hexViewer[] =
{
  { "B~it",         GetB,   sizeBit,   1, 8 },
  { "~Byte",        Get2D,  sizeByte,  1, 2 },
  { "~Word",        Get4D,  sizeWord,  2, 4 },
  { "~Double word", Get8D,  sizeDWord, 4, 8 }
};


static unsigned hmode = 1;

tBool hexAddressResolv = False;

static unsigned __FASTCALL__ drawHex( unsigned keycode,unsigned textshift )
{
 int i,I,Limit,dir;
 char outstr[__TVIO_MAXSCREENWIDTH+1];
 unsigned char HWidth;
 unsigned long sindex,cpos,flen,lindex,SIndex;
 static unsigned long hmocpos = 0L;
 int __inc,dlen;
 cpos = BMGetCurrFilePos();
 if(hmocpos != cpos || keycode == KE_SUPERKEY)
 {
   HWidth = hexViewer[hmode].width();
   if(!(hmocpos == cpos + HWidth || hmocpos == cpos - HWidth)) keycode = KE_SUPERKEY;
   hmocpos = cpos;
   __inc = hexViewer[hmode].size;
   dlen = hexViewer[hmode].hardlen;
   flen = BMGetFLength();
   if(flen < HWidth) HWidth = flen;
   if(keycode == KE_UPARROW) 
   {
       I = Height-1; 
       dir = -1;       
       Limit = -1; 
       if(cpos >= HWidth)
       {
         twScrollWinDn(MainWnd,1,1);
         I = 0;
       }
       else keycode = KE_SUPERKEY;
   }
   else 
   { 
       I = 0; 
       dir = 1; 
       Limit = Height; 
   }
   if(keycode == KE_DOWNARROW && flen >= HWidth)
   {
     twScrollWinUp(MainWnd,Height-1,1);
     I = Height-1;
   }
   SIndex = cpos + HWidth*I;
   lindex = flen - SIndex;
   /* This loop is called only when line or screen is repainting */
   for(i = I,sindex = SIndex;i != Limit;i += 1*dir,sindex += HWidth*dir*__inc)
   {
     memset(outstr,TWC_DEF_FILLER,Width);
     if(sindex < flen)
     {
       int freq,j,rwidth,xmin,len;
       lindex = (flen - sindex)/__inc;
       rwidth = lindex > HWidth ? HWidth : (int)lindex;
       memcpy(outstr,GidEncodeAddress(sindex,hexAddressResolv),10);
       len = 10;
       for(j = 0,freq = 0,lindex = sindex;j < rwidth;j++,lindex += __inc,freq++)
       {
          memcpy(&outstr[len],hexViewer[hmode].func(lindex),dlen);
          len += dlen + 1;
          if(hmode == 1) if(freq == 3) { freq = -1; len++; }
       }
       BMReadBufferEx((void *)&outstr[Width - HWidth*__inc],rwidth*__inc,sindex,BM_SEEK_SET);
       xmin = tvioWidth-HWidth;
       twDirectWrite(1,i + 1,outstr,xmin);
       if(isHOnLine(sindex,HWidth))  
       {
          HLInfo hli;
          hli.text = &outstr[xmin];
          HiLightSearch(MainWnd,sindex,xmin,Width,i,&hli,HLS_NORMAL);
       }
       else  twDirectWrite(xmin + 1,i + 1,&outstr[xmin],Width - xmin);
     }
     else twDirectWrite(1,i + 1,outstr,Width);
   }
   lastbyte = lindex + __inc;
 }
 return textshift;
}

static void __FASTCALL__ HelpHex( void )
{
   hlpDisplay(1002);
}

static unsigned long __FASTCALL__ hexPrevPageSize( void ) { return hexViewer[hmode].width()*hexViewer[hmode].size*Height; }
static unsigned long __FASTCALL__ hexCurrPageSize( void ) { return hexViewer[hmode].width()*hexViewer[hmode].size*Height; }
static unsigned long __FASTCALL__ hexPrevLineWidth( void ) { return hexViewer[hmode].width()*hexViewer[hmode].size; }
static unsigned long __FASTCALL__ hexCurrLineWidth( void ) { return hexViewer[hmode].width()*hexViewer[hmode].size; }
static const char *  __FASTCALL__ hexMiscKeyName( void ) { return hmode == 1 ? "Modify" : "      "; }

static tBool __FASTCALL__ hexSelectMode( void )
{
  const char *names[sizeof(hexViewer)/sizeof(hexView)];
  size_t i,nModes;
  int retval;
  nModes = sizeof(hexViewer)/sizeof(hexView);
  for(i = 0;i < nModes;i++) names[i] = hexViewer[i].name;
  retval = SelBoxA(names,nModes," Select hexadecimal mode: ",hmode);
  if(retval != -1)
  {
    hmode = retval;
    return True;
  }
  return False;
}

static const char *aresolv[] =
{
  "~Global (global file offset)",
  "~Local (local offset within blocks and virtual addresses)"
};

tBool __FASTCALL__ hexAddressResolution( void )
{
  unsigned nModes;
  int i;
  nModes = sizeof(aresolv)/sizeof(char *);
  i = SelBoxA(aresolv,nModes," Select address resolving: ",(unsigned)hexAddressResolv);
  if(i != -1)
  {
    hexAddressResolv = i ? True : False;
    return True;
  }
  return False;
}

static tBool __FASTCALL__ hexDetect( void ) { return True; }

static int __NEAR__ __FASTCALL__ FullHexEdit(TWindow * txtwnd)
{
 size_t i,j;
 unsigned mlen;
 unsigned int _lastbyte;
 unsigned flags;
 char work[__TVIO_MAXSCREENWIDTH],owork[__TVIO_MAXSCREENWIDTH];
 tBool redraw;
 {
   TWindow * using = twUsedWin();
   twUseWin(txtwnd);
   twSetColorAttr(browser_cset.main);
   twFreezeWin(txtwnd);
   for(i = 0;i < Height;i++)
   {
     twDirectWrite(Width - EditorMem.width + 1,i + 1,&EditorMem.buff[i*EditorMem.width],EditorMem.alen[i]);
     if((unsigned)EditorMem.alen[i] + 1 < EditorMem.width)
     {
        twGotoXY(Width - EditorMem.width + EditorMem.alen[i] + 2,i + 1); twClrEOL();
     }
   }
   twRefreshWin(txtwnd);
   twUseWin(using);
 }
 twSetColorAttr(browser_cset.edit.main);
 for(i = 0;i < Height;i++)
 {
   unsigned eidx;
   eidx = i*EditorMem.width;
   ExpandHex(work,&EditorMem.buff[eidx],EditorMem.alen[i],1);
   mlen = ExpandHex(owork,&EditorMem.save[eidx],EditorMem.alen[i],1);
   for(j = 0;j < mlen;j++)
   {
     twSetColorAttr(work[j] == owork[j] ? browser_cset.edit.main : browser_cset.edit.change);
     twDirectWrite(j + 1,i + 1,&work[j],1);
   }
   if(mlen + 1 < EditorMem.width)
   {
     twGotoXY(mlen + 1,i + 1); 
     twClrEOL();
   }
 }
 redraw = True;
 PaintETitle(edit_y*EditorMem.width + edit_x,0);
 twShowWin(twUsedWin());
 twSetCursorType(TW_CUR_NORM);
 while(1)
 {
   unsigned eidx;
   mlen = EditorMem.alen[edit_y];
   eidx = edit_y*EditorMem.width;
   ExpandHex(work,&EditorMem.buff[eidx],mlen,1);
   mlen = ExpandHex(owork,&EditorMem.save[eidx],mlen,1);
   edit_x*=3;
   flags = __ESS_WANTRETURN | __ESS_HARDEDIT | __ESS_ASHEX;
   if(!redraw) flags |= __ESS_NOREDRAW;
   _lastbyte = eeditstring(work,&legalchars[2],&mlen,(unsigned)(edit_y + 1),(unsigned *)&edit_x,
                          flags,owork);
   edit_x/=3;
   CompressHex(&EditorMem.buff[eidx],work,mlen/3,True);
   switch(_lastbyte)
   {
     case KE_F(1)   : ExtHelp(); continue;
     case KE_F(2)   : editSaveContest();
     case KE_F(10)  :
     case KE_ESCAPE :
     case KE_TAB : goto bye;
     default     : redraw = editDefAction(_lastbyte); break;
   }
   CheckBounds();
   if(redraw)
   {
      TWindow * using = twUsedWin();
      twUseWin(txtwnd);
      twDirectWrite(Width - EditorMem.width + 1,edit_y + 1,&EditorMem.buff[edit_y*EditorMem.width],mlen/3);
      twUseWin(using);
   }
   PaintETitle(edit_y*EditorMem.width + edit_x,0);
 }
 bye:
 twSetCursorType(TW_CUR_OFF);
 return _lastbyte;
}

static void __FASTCALL__ EditHex( void )
{
 TWindow * ewnd[2];
 tBool has_show[2];
 int active = 0,oactive = 0;
 unsigned bound;
 if(hmode != 1) return;
 bound = Width-hexViewer[hmode].width();
 ewnd[0] = WindowOpen(11,2,bound,tvioHeight-1,TWS_CURSORABLE);
 twSetColorAttr(browser_cset.edit.main); twClearWin();
 ewnd[1] = WindowOpen(bound+1,2,Width,tvioHeight-1,TWS_CURSORABLE);
 twSetColorAttr(browser_cset.edit.main); twClearWin();
 drawEditPrompt();
 has_show[0] = has_show[1] = False;
 if(editInitBuffs(hexViewer[hmode].width()))
 {
   edit_x = edit_y = 0;
   while(1)
   {
     int _lastbyte;
     if(active != oactive)
     {
       twHideWin(ewnd[oactive]);
       if(has_show[active]) twShowWin(ewnd[active]);
       oactive = active;
     }
     twUseWin(ewnd[active]);
     if(!active) _lastbyte = FullHexEdit(MainWnd);
     else        _lastbyte = FullEdit(MainWnd);
     has_show[active] = True;
     if(_lastbyte == KE_TAB) active = active ? 0 : 1;
     else break;
   }
   editDestroyBuffs();
 }
 CloseWnd(ewnd[0]);
 CloseWnd(ewnd[1]);
 PaintTitle();
}

void __FASTCALL__ ReadIniAResolv( hIniProfile *ini )
{
  char tmps[10];
  biewReadProfileString(ini,"Biew","Browser","SubSubMode6","0",tmps,sizeof(tmps));
  hexAddressResolv = (unsigned)strtoul(tmps,NULL,10);
  if(hexAddressResolv > 1) hexAddressResolv = 0;
}

void __FASTCALL__ WriteIniAResolv( hIniProfile *ini )
{
  char tmps[10];
  sprintf(tmps,"%i",hexAddressResolv);
  biewWriteProfileString(ini,"Biew","Browser","SubSubMode6",tmps);
}

static void __FASTCALL__ hexReadIni( hIniProfile *ini )
{
  char tmps[10];
  if(isValidIniArgs())
  {
    biewReadProfileString(ini,"Biew","Browser","LastSubMode","2",tmps,sizeof(tmps));
    hmode = (unsigned)strtoul(tmps,NULL,10);
    if(hmode > 3) hmode = 1;
    ReadIniAResolv(ini);
  }
}

static void __FASTCALL__ hexSaveIni( hIniProfile * ini)
{
  char tmps[10];
  sprintf(tmps,"%i",hmode);
  biewWriteProfileString(ini,"Biew","Browser","LastSubMode",tmps);
  WriteIniAResolv(ini);
}

static unsigned __FASTCALL__ hexCharSize( void ) { return hexViewer[hmode].size; }

REGISTRY_MODE hexMode =
{
  "~Hexadecimal mode",
  { NULL, "HexMod", NULL, NULL, NULL, "AResol", NULL, NULL, NULL, NULL },
  { NULL, hexSelectMode, NULL, NULL, NULL, hexAddressResolution, NULL, NULL, NULL, NULL },
  hexDetect,
  __MF_NONE,
  drawHex,
  NULL,
  hexCharSize,
  hexMiscKeyName,
  EditHex,
  hexPrevPageSize,
  hexCurrPageSize,
  hexPrevLineWidth,
  hexCurrLineWidth,
  HelpHex,
  hexReadIni,
  hexSaveIni,
  NULL,
  NULL
};








