/** 
 * @namespace   biew_plugins_auto
 * @file        plugins/bin/elf386.c
 * @brief       This file contains implementation of ELF-32 file format decoder.
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 *
 * @author      Nick Kurshev
 * @since       1999
 * @note        Development, fixes and improvements
**/
#include <limits.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "colorset.h"
#include "plugins/bin/elf386.h"
#include "plugins/disasm.h"
#include "bin_util.h"
#include "codeguid.h"
#include "bmfile.h"
#include "biewhelp.h"
#include "tstrings.h"
#include "biewutil.h"
#include "bconsole.h"
#include "reg_form.h"
#include "biewlib/biewlib.h"
#include "biewlib/pmalloc.h"
#include "biewlib/kbd_code.h"

static char is_big;
#define ELF_WORD(cval)  FMT_WORD(cval,is_big)
#define ELF_DWORD(cval) FMT_DWORD(cval,is_big)
static Elf386_External_Ehdr elf;
static unsigned active_shtbl = 0;
static unsigned long elf_min_va = ULONG_MAX;
static unsigned      __elfNumSymTab = 0;
static unsigned      __elfSymShTbl = 0;
static unsigned long __elfSymEntSize = 0L;
static unsigned long __elfSymPtr = 0L;

struct tag_elfVAMap
{
  unsigned long va;
  unsigned long size;
  unsigned long foff;
  unsigned long nameoff;
  unsigned long flags;
};

static tBool __NEAR__ __FASTCALL__ FindPubName(char *buff,unsigned cb_buff,unsigned long pa);
static void __FASTCALL__ elf_ReadPubNameList(BGLOBAL handle,void (__FASTCALL__ *mem_out)(const char *));

static BGLOBAL namecache = &bNull;
static BGLOBAL namecache2 = &bNull;
static BGLOBAL elfcache = &bNull;

static linearArray *va_map_phys,* va_map_virt;

static tCompare __FASTCALL__ vamap_comp_virt(const void __HUGE__ *v1,const void __HUGE__ *v2)
{
  const struct tag_elfVAMap __HUGE__ *pnam1,__HUGE__ *pnam2;
  pnam1 = (const struct tag_elfVAMap __HUGE__ *)v1;
  pnam2 = (const struct tag_elfVAMap __HUGE__ *)v2;
  if(pnam1->va < pnam2->va) return -1;
  else
    if(pnam1->va > pnam2->va) return 1;
    else                      return 0;
}

static tCompare __FASTCALL__ vamap_comp_phys(const void __HUGE__ *v1,const void __HUGE__ *v2)
{
  const struct tag_elfVAMap __HUGE__ *pnam1,__HUGE__ *pnam2;
  pnam1 = (const struct tag_elfVAMap __HUGE__ *)v1;
  pnam2 = (const struct tag_elfVAMap __HUGE__ *)v2;
  if(pnam1->foff < pnam2->foff) return -1;
  else
    if(pnam1->foff > pnam2->foff) return 1;
    else                          return 0;
}

static unsigned long __FASTCALL__ elfVA2PA(unsigned long va)
{
  unsigned long i;
  for(i = 0; i < va_map_virt->nItems;i++)
  {
    struct tag_elfVAMap __HUGE__ *evm;
    evm = &((struct tag_elfVAMap __HUGE__ *)va_map_virt->data)[i];
    if(va >= evm->va && va < evm->va + evm->size) return evm->foff + (va - evm->va);
  }
  return 0L;
}

static unsigned long __FASTCALL__ elfPA2VA(unsigned long pa)
{
  unsigned long i,ret;
  ret = 0L;
  for(i = 0; i < va_map_phys->nItems;i++)
  {
    struct tag_elfVAMap __HUGE__ *evm;
    evm = &((struct tag_elfVAMap __HUGE__ *)va_map_phys->data)[i];
    if(pa >= evm->foff && pa < evm->foff + evm->size)
    {
      ret = evm->va + (pa - evm->foff);
      break;
    }
  }
  return ret;
}

static tBool __FASTCALL__ elfLowMemFunc( unsigned long need_mem )
{
  UNUSED(need_mem);
  if(!fmtActiveState)
  {
    if(PubNames)
    {
       la_Destroy(PubNames);
       PubNames = NULL;
       return True;
    }
  }
  return False;
}

static void __NEAR__ __FASTCALL__ elf386_readnametable(unsigned long off,char *buf,unsigned blen)
{
  unsigned long foff;
  Elf386_External_Shdr sh;
  unsigned char ch;
  unsigned freq;
  BGLOBAL b_cache,b_cache2;
  b_cache = namecache;
  b_cache2 = namecache2;
  foff = ELF_DWORD(elf.e_shoff)+(ELF_WORD(elf.e_shstrndx)*sizeof(Elf386_External_Shdr));
  bioSeek(b_cache2,foff,SEEKF_START);
  bioReadBuffer(b_cache2,&sh,sizeof(Elf386_External_Shdr));
  foff = ELF_DWORD(sh.sh_offset) + off;
  freq = 0;
  while(1)
  {
     bioSeek(b_cache,foff++,SEEKF_START);
     ch = bioReadByte(b_cache);
     buf[freq++] = ch;
     if(!ch || freq >= blen || bioEOF(b_cache)) break;
  }
}

static void __NEAR__ __FASTCALL__ elf386_readnametableex(unsigned long off,char *buf,unsigned blen)
{
  unsigned long foff;
  Elf386_External_Shdr sh;
  unsigned char ch;
  unsigned freq;
  BGLOBAL b_cache,b_cache2;
  b_cache = namecache;
  b_cache2 = namecache2;
  foff = ELF_DWORD(&elf.e_shoff)+active_shtbl*sizeof(Elf386_External_Shdr);
  bioSeek(b_cache2,foff,SEEKF_START);
  bioReadBuffer(b_cache2,&sh,sizeof(Elf386_External_Shdr));
  foff = ELF_DWORD(sh.sh_offset) + off;
  freq = 0;
  while(1)
  {
     bioSeek(b_cache,foff++,SEEKF_START);
     ch = bioReadByte(b_cache);
     buf[freq++] = ch;
     if(!ch || freq >= blen || bioEOF(b_cache)) break;
  }
}

static const char * __NEAR__ __FASTCALL__ elf_class(char id)
{
  switch(id)
  {
    default: return "Unk.obj/";
    case ELFCLASSNONE: return "Err.obj/";
    case ELFCLASS32: return "32bit/";
  }
}

static const char * __NEAR__ __FASTCALL__ elf_data(char id)
{
  switch(id)
  {
    default: return "Unk.data/";
    case ELFDATANONE: return "Err.data/";
    case ELFDATA2LSB: return "little endian/";
    case ELFDATA2MSB: return "big endian/";
  }
}

static const char *__NEAR__ __FASTCALL__ elf_otype(unsigned short id)
{
  switch(id)
  {
    case ET_NONE: return "none";
    case ET_REL: return "relocatable";
    case ET_EXEC: return "executable";
    case ET_DYN: return "shared object";
    case ET_CORE: return "core";
    case ET_NUM: return "number of defined types";
    case ET_LOPROC: return "low processor-specific";
    case ET_HIPROC: return "high processor-specific";
    default: return "Unknown";
  }
}

static const char *__NEAR__ __FASTCALL__ elf_machine(unsigned short id)
{
  switch(id)
  {
    case EM_NONE: return "none";
    case EM_M32: return "AT&T WE32100";
    case EM_SPARC: return "Sun Sparc";
    case EM_386: return "Intel 386";
    case EM_68K: return "Motorola m68k family";
    case EM_88K: return "Motorola m88k family";
    case EM_860: return "Intel 80860";
    case EM_MIPS: return "MIPS R3000";
    case EM_MIPS_RS4_BE: return "MIPS R4000";
    case EM_SPARC64: return "Sparc v9 64-bit";
    case EM_PARISC: return "HPPA";
    case EM_SPARC32PLUS: return "Sun's \"v8plus\"";
    case EM_PPC: return "Power PC";
    case EM_SH: return "Hitachi SH";
    default: return "Unknown";
  }
}

static const char *__NEAR__ __FASTCALL__ elf_ver(long id)
{
  switch(id)
  {
    case EV_NONE:    return "Invalid ELF version";
    case EV_CURRENT: return "Current version";
    default:         return "Unknown";
  }
}

static unsigned long __FASTCALL__ ShowELFHeader( void )
{
  unsigned long fpos;
  TWindow *w;
  char hdr[81];
  unsigned keycode;
  unsigned long entrya;
  fpos = BMGetCurrFilePos();
  entrya = elfVA2PA(ELF_DWORD(elf.e_entry));
  sprintf(hdr," Executable and Linkable File (ELF) %s%sver.%c "
             ,elf_class(elf.e_ident[4])
             ,elf_data(elf.e_ident[5])
             ,elf.e_ident[6]+'0');
  w = CrtDlgWndnls(hdr,74,13);
  twGotoXY(1,1);
  twPrintF("Object file type                  = %s\n"
           "Machine architecture              = %s\n"
           "Object file version               = %s\n"
           ,elf_otype(ELF_WORD(&elf.e_type))
           ,elf_machine(ELF_WORD(&elf.e_machine))
           ,elf_ver(ELF_DWORD(elf.e_version)));
  twSetColorAttr(dialog_cset.entry);
  twPrintF("Entry point VA                    = %08lXH (Offset: %08lXH)"
           ,ELF_DWORD(elf.e_entry),entrya);
  twClrEOL(); twPrintF("\n");
  twSetColorAttr(dialog_cset.main);
  twPrintF("Program header table offset       = %08lXH\n"
           "Section header table offset       = %08lXH\n"
           "Processor specific flag           = %08lXH\n"
           "ELF header size (bytes)           = %04XH\n"
           "Program header table entry size   = %04XH\n"
           "Program header table entry count  = %04XH\n"
           "Section header table entry size   = %04XH\n"
           "Section header table entry count  = %04XH\n"
           "Section header string table index = %04XH"
           ,ELF_DWORD(elf.e_phoff)
           ,ELF_DWORD(elf.e_shoff)
           ,ELF_DWORD(elf.e_flags)
           ,ELF_WORD(elf.e_ehsize)
           ,ELF_WORD(elf.e_phentsize)
           ,ELF_WORD(elf.e_phnum)
           ,ELF_WORD(elf.e_shentsize)
           ,ELF_WORD(elf.e_shnum)
           ,ELF_WORD(elf.e_shstrndx));
  while(1)
  {
    keycode = GetEvent(drawEmptyPrompt,w);
    if(keycode == KE_ENTER) { fpos = entrya; break; }
    else
      if(keycode == KE_ESCAPE || keycode == KE_F(10)) break;
  }
  CloseWnd(w);
  return fpos;
}

static const char * __NEAR__ __FASTCALL__ elf_encode_p_type(long p_type)
{
   switch(p_type)
   {
      case PT_NULL: return "Unusable";
      case PT_LOAD: return "Loadable";
      case PT_DYNAMIC: return "Dynalinking";
      case PT_INTERP: return "Interpreter";
      case PT_NOTE:  return "Auxiliary";
      case PT_SHLIB: return "Unspecified";
      case PT_PHDR:  return "header itself";
      case PT_NUM: return "Number of types";
      case PT_LOPROC: return "Low processor";
      case PT_HIPROC: return "High processor";
      default:  return "Unknown";
   }
}

static tBool __FASTCALL__ __elfReadPrgHdr(BGLOBAL handle,memArray *obj,unsigned nnames)
{
 size_t i;
  bioSeek(handle,ELF_DWORD(elf.e_phoff),SEEKF_START);
  for(i = 0;i < nnames;i++)
  {
   unsigned long fp;
   char stmp[80];
   Elf386_External_Phdr phdr;
   if(IsKbdTerminate() || bioEOF(handle)) break;
   fp = bioTell(handle);
   bioReadBuffer(handle,&phdr,sizeof(Elf386_External_Phdr));
   bioSeek(handle,fp+ELF_WORD(elf.e_phentsize),SEEKF_START);
   sprintf(stmp,"%-15s %08lX %08lX %08lX %08lX %08lX %c%c%c %08lX",
                elf_encode_p_type(ELF_DWORD(phdr.p_type)),
                ELF_DWORD(phdr.p_offset),
                ELF_DWORD(phdr.p_vaddr),
                ELF_DWORD(phdr.p_paddr),
                ELF_DWORD(phdr.p_filesz),
                ELF_DWORD(phdr.p_memsz),
                (ELF_DWORD(phdr.p_flags) & PF_X) == PF_X ? 'X' : ' ',
                (ELF_DWORD(phdr.p_flags) & PF_W) == PF_W ? 'W' : ' ',
                (ELF_DWORD(phdr.p_flags) & PF_R) == PF_R ? 'R' : ' ',
                ELF_DWORD(phdr.p_align)
          );
   if(!ma_AddString(obj,stmp,True)) break;
  }
  return True;
}

static const char * __NEAR__ __FASTCALL__ elf_encode_sh_type(long p_type)
{
   switch(p_type)
   {
      case SHT_NULL: return "NULL";
      case SHT_PROGBITS: return "PRGBTS";
      case SHT_SYMTAB: return "SYMTAB";
      case SHT_STRTAB: return "STRTAB";
      case SHT_RELA:  return "RELA";
      case SHT_HASH: return "HSHTAB";
      case SHT_DYNAMIC:  return "DYNLNK";
      case SHT_NOTE: return "NOTES";
      case SHT_NOBITS: return "NOBITS";
      case SHT_REL:  return "REL";
      case SHT_SHLIB: return "UNSPEC";
      case SHT_DYNSYM: return "DYNSYM";
      case SHT_NUM: return "NTYPES";
      case SHT_GNU_verdef: return "VERDEF";
      case SHT_GNU_verneed: return "VERNED";
      case SHT_GNU_versym: return "SYMVER";
      case SHT_LOPROC: return "LOPROC";
      case SHT_HIPROC: return "HIPROC";
      case SHT_LOUSER: return "LOUSER";
      case SHT_HIUSER: return "HIUSER";
      default:  return "UNK";
   }
}

static tBool __FASTCALL__ __elfReadSecHdr(BGLOBAL handle,memArray *obj,unsigned nnames)
{
 size_t i;
  bioSeek(handle,ELF_DWORD(elf.e_shoff),SEEKF_START);
  for(i = 0;i < nnames;i++)
  {
   Elf386_External_Shdr shdr;
   char tmp[17];
   unsigned long fp;
   char stmp[80];
   if(IsKbdTerminate() || bioEOF(handle)) break;
   fp = bioTell(handle);
   bioReadBuffer(handle,&shdr,sizeof(Elf386_External_Shdr));
   elf386_readnametable(ELF_DWORD(shdr.sh_name),tmp,sizeof(tmp));
   bioSeek(handle,fp+ELF_WORD(elf.e_shentsize),SEEKF_START);
   tmp[16] = 0;
   sprintf(stmp,"%-16s %-6s %c%c%c %08lX %08lX %08lX %04hX %04hX %04hX %04hX",
                tmp,
                elf_encode_sh_type(ELF_DWORD(shdr.sh_type)),
                (ELF_DWORD(shdr.sh_flags) & SHF_WRITE) == SHF_WRITE ? 'W' : ' ',
                (ELF_DWORD(shdr.sh_flags) & SHF_ALLOC) == SHF_ALLOC ? 'A' : ' ',
                (ELF_DWORD(shdr.sh_flags) & SHF_EXECINSTR) == SHF_EXECINSTR ? 'X' : ' ',
                ELF_DWORD(shdr.sh_addr),
                ELF_DWORD(shdr.sh_offset),
                ELF_DWORD(shdr.sh_size),
                (short)ELF_DWORD(shdr.sh_link),
                (short)ELF_DWORD(shdr.sh_info),
                (short)ELF_DWORD(shdr.sh_addralign),
                (short)ELF_DWORD(shdr.sh_entsize)
          );
    if(!ma_AddString(obj,stmp,True)) break;
  }
  return True;
}

static const char * __NEAR__ __FASTCALL__ elf_SymTabType(char type)
{
  switch(ELF_ST_TYPE(type))
  {
    case STT_NOTYPE:  return "NoType";
    case STT_OBJECT:  return "Object";
    case STT_FUNC:    return "Func. ";
    case STT_SECTION: return "Sect. ";
    case STT_FILE:    return "File  ";
    case STT_NUM:     return "Number";
    case STT_LOPROC:  return "LoProc";
    case STT_HIPROC:  return "HiProc";
    default: return "Unknwn";
  }
}

static const char * __NEAR__ __FASTCALL__ elf_SymTabBind(char type)
{
  switch(ELF_ST_BIND(type))
  {
    case STB_LOCAL:  return "Local ";
    case STB_GLOBAL: return "Global";
    case STB_WEAK:   return "Weak  ";
    case STB_NUM:    return "Number";
    case STB_LOPROC: return "LoProc";
    case STB_HIPROC: return "HiProc";
    default: return "Unknwn";
  }
}

static const char * __NEAR__ __FASTCALL__ elf_SymTabShNdx(unsigned idx)
{
  static char ret[10];
  switch(idx)
  {
    case SHN_UNDEF:  return "Undef ";
    case SHN_LOPROC: return "LoProc";
    case SHN_HIPROC: return "HiProc";
    case SHN_ABS:    return "Abs.  ";
    case SHN_COMMON: return "Common";
    case SHN_HIRESERVE: return "HiRes.";
    default: sprintf(ret,"%04XH ",idx); return ret;
  }
}

static tBool __NEAR__ __FASTCALL__ ELF_IS_SECTION_PHYSICAL(unsigned short sec_num)
{
  return !(sec_num == SHN_UNDEF || sec_num == SHN_LOPROC ||
           sec_num == SHN_HIPROC || sec_num == SHN_ABS ||
           sec_num == SHN_COMMON || sec_num == SHN_HIRESERVE);
}

static unsigned __FASTCALL__ __elfGetNumSymTab( BGLOBAL handle )
{
  UNUSED(handle);
  return __elfNumSymTab;
}

static tBool __FASTCALL__ __elfReadSymTab(BGLOBAL handle,memArray *obj,unsigned nsym)
{
 size_t i;
 char text[31];
  bioSeek(handle,__elfSymPtr,SEEK_SET);
  for(i = 0;i < nsym;i++)
  {
   unsigned long fp;
   char stmp[80];
   Elf386_External_Sym sym;
   if(IsKbdTerminate() || bioEOF(handle)) break;
   fp = bioTell(handle);
   bioReadBuffer(handle,&sym,sizeof(Elf386_External_Sym));
   bioSeek(handle,fp+__elfSymEntSize,SEEKF_START);
   elf386_readnametableex(ELF_DWORD(sym.st_name),text,sizeof(text));
   text[sizeof(text)-1] = 0;
   sprintf(stmp,"%-31s %08lX %08lX %04hX %s %s %s"
               ,text
               ,ELF_DWORD(sym.st_value)
               ,ELF_DWORD(sym.st_size)
               ,ELF_WORD(sym.st_other)
               ,elf_SymTabType(sym.st_info[0])
               ,elf_SymTabBind(sym.st_info[0])
               ,elf_SymTabShNdx(ELF_WORD(sym.st_shndx))
               );
   if(!ma_AddString(obj,stmp,True)) break;
  }
  return True;
}

static tBool __NEAR__ __FASTCALL__ __elfReadDynTab(BGLOBAL handle,memArray *obj, unsigned ntbl,unsigned long entsize)
{
 size_t i;
 char sout[80];
 unsigned len,rlen;
  for(i = 0;i < ntbl;i++)
  {
   unsigned long fp;
   char stmp[80];
   Elf386_External_Dyn pdyn;
   fp = bioTell(handle);
   bioReadBuffer(handle,&pdyn,sizeof(Elf386_External_Dyn));
   bioSeek(handle,fp+entsize,SEEKF_START);
   fp = bioTell(handle);
   elf386_readnametableex(ELF_DWORD(pdyn.d_tag),sout,sizeof(sout));
   len = strlen(sout);
   if(len > 56) len = 53;
   if(IsKbdTerminate() || bioEOF(handle)) break;
   strncpy(stmp,sout,len);
   stmp[len] = 0;
   if(len > 56) strcat(stmp,"...");
   rlen = strlen(stmp);
   if(rlen < 60) { memset(&stmp[rlen],' ',60-rlen); stmp[60] = 0; }
   sprintf(&stmp[strlen(stmp)]," vma=%08lXH",ELF_DWORD(pdyn.d_un.d_val));
   if(!ma_AddString(obj,stmp,True)) break;
   bioSeek(handle,fp,SEEKF_START);
  }
  return True;
}

static unsigned __FASTCALL__ Elf386PrgHdrNumItems(BGLOBAL handle)
{
   UNUSED(handle);
   return ELF_WORD(elf.e_phnum);
}

static unsigned long __FASTCALL__ ShowPrgHdrElf(void)
{
  unsigned long fpos;
  int ret;
  fpos = BMGetCurrFilePos();
  ret = fmtShowList(Elf386PrgHdrNumItems,__elfReadPrgHdr,
                    " type            fileoffs virtaddr physaddr filesize memsize  flg align   ",
                    LB_SELECTIVE,NULL);
  if(ret != -1)
  {
    Elf386_External_Phdr it;
    bmSeek(ELF_DWORD(elf.e_phoff)+sizeof(Elf386_External_Phdr)*ret,SEEKF_START);
    bmReadBuffer(&it,sizeof(Elf386_External_Phdr));
    fpos = ELF_DWORD(it.p_offset);
  }
  return fpos;
}

static unsigned __FASTCALL__ Elf386SecHdrNumItems(BGLOBAL handle)
{
  UNUSED(handle);
  return ELF_WORD(elf.e_shnum);
}

static unsigned long __FASTCALL__ ShowSecHdrElf(void)
{
  unsigned long fpos;
  int ret;
  fpos = BMGetCurrFilePos();
  ret = fmtShowList(Elf386SecHdrNumItems,__elfReadSecHdr,
                    " name             type   flg virtaddr fileoffs   size   link info algn esiz",
                   LB_SELECTIVE,NULL);
  if(ret != -1)
  {
    Elf386_External_Shdr it;
    bmSeek(ELF_DWORD(elf.e_shoff)+sizeof(Elf386_External_Shdr)*ret,SEEKF_START);
    bmReadBuffer(&it,sizeof(Elf386_External_Shdr));
    fpos = ELF_DWORD(it.sh_offset);
  }
  return fpos;
}

static unsigned long __calcSymEntry(BGLOBAL handle,unsigned long num,tBool display_msg)
{
   Elf386_External_Sym it;
   Elf386_External_Shdr sec;
   unsigned long ffpos,fpos = 0L;
   ffpos = bioTell(handle);
   bioSeek(handle,__elfSymPtr+__elfSymEntSize*num,SEEKF_START);
   bioReadBuffer(handle,&it,sizeof(Elf386_External_Sym));
   bioSeek(handle,ELF_DWORD(elf.e_shoff)+sizeof(Elf386_External_Shdr)*ELF_WORD(it.st_shndx),SEEKF_START);
   bioReadBuffer(handle,&sec,sizeof(Elf386_External_Shdr));
   bioSeek(handle,ffpos,SEEKF_START);
   if(ELF_IS_SECTION_PHYSICAL(ELF_WORD(it.st_shndx)))
        fpos = ELF_DWORD(sec.sh_offset) + ELF_DWORD(it.st_value);
   else
     if(display_msg) ErrMessageBox(NO_ENTRY,BAD_ENTRY);
   return fpos;
}

static unsigned long __NEAR__ __FASTCALL__ displayELFsymtab( void )
{
  unsigned long fpos;
  int ret;
  fpos = BMGetCurrFilePos();
  ret = fmtShowList(__elfGetNumSymTab,__elfReadSymTab,
                    " Name                            Value    Size     Oth. Type   Bind   Sec# ",
                    LB_SELECTIVE,NULL);
  if(ret != -1)
  {
    unsigned long ea;
    ea = __calcSymEntry(bmbioHandle(),ret,True);
    fpos = ea ? ea : fpos;
  }
  return fpos;
}

static unsigned long __NEAR__ __FASTCALL__ displayELFdyntab(unsigned long dynptr,unsigned long nitem,long entsize)
{
  unsigned long fpos;
  memArray *obj;
  BGLOBAL handle;
  unsigned ndyn;
  fpos = BMGetCurrFilePos();
  ndyn = (unsigned)nitem;
  if(!(obj = ma_Build(ndyn,True))) return fpos;
  handle = elfcache;
  bioSeek(handle,dynptr,SEEK_SET);
  if(__elfReadDynTab(handle,obj,ndyn,entsize))
  {
    int ret;
    ret = ma_Display(obj," Dynamic section ",LB_SELECTIVE | LB_SORTABLE,-1);
    if(ret != -1)
    {
       char *addr;
       addr = strstr(obj->data[ret],"vma=");
       if(addr)
       {
         unsigned long addr_probe;
         addr_probe = strtoul(&addr[4],NULL,16);
         if(addr_probe && addr_probe >= elf_min_va)
         {
           addr_probe = elfVA2PA(addr_probe);
           if(addr_probe && addr_probe < bmGetFLength()) fpos = addr_probe;
           else goto not_entry;
         }
         else goto not_entry;
       }
       else
       {
         not_entry:
         ErrMessageBox(NOT_ENTRY,NULL);
       }
    }
  }
  ma_Destroy(obj);
  return fpos;
}

static unsigned long __FASTCALL__ ShowELFSymTab( void )
{
  unsigned long fpos;
  fpos = BMGetCurrFilePos();
  if(!__elfSymPtr) { NotifyBox(NOT_ENTRY," ELF symbol table "); return fpos; }
  active_shtbl = __elfSymShTbl;
  return displayELFsymtab();
}

static unsigned long __FASTCALL__ ShowELFDynSec( void )
{
  unsigned long fpos,dynptr,number;
  Elf386_External_Shdr shdr;
  unsigned i;
  fpos = BMGetCurrFilePos();
  dynptr = 0;
  number = 0;
  for(i = 0;i < ELF_WORD(elf.e_shnum);i++)
  {
   bmReadBufferEx(&shdr,sizeof(Elf386_External_Shdr),ELF_DWORD(elf.e_shoff) + i*ELF_WORD(elf.e_shentsize),SEEKF_START);
   if(bmEOF()) break;
   if(ELF_DWORD(shdr.sh_type) == SHT_DYNSYM)
   {
     dynptr = ELF_DWORD(shdr.sh_offset);
     number = ELF_DWORD(shdr.sh_size)/ELF_DWORD(shdr.sh_entsize);
     active_shtbl = ELF_WORD(shdr.sh_link);
     break;
   }
  }
  if(!dynptr) { NotifyBox(NOT_ENTRY," ELF dynamic symbol table "); return fpos; }
  return displayELFdyntab(dynptr,number,ELF_DWORD(shdr.sh_entsize));
}

#define S_INTERPRETER "Interpreter : "

static void __NEAR__ __FASTCALL__ displayELFdyninfo(unsigned long f_off,unsigned nitems)
{
  Elf386_External_Dyn dyntab;
  Elf386_External_Phdr phdr;
  unsigned long curroff,stroff;
  unsigned i;
  tBool is_add;
  memArray * obj;
  char stmp[80];
  curroff = f_off;
  stroff = 0;
  bmSeek(curroff,SEEK_SET);
  for(i = 0;i < nitems;i++)
  {
    bmReadBuffer(&dyntab,sizeof(dyntab));
    if(bmEOF()) break;
    if(ELF_DWORD(dyntab.d_tag) == DT_STRTAB)
    {
       stroff = ELF_DWORD(dyntab.d_un.d_ptr);
       break;
    }
  }
  if(!stroff) { NotifyBox(" String information not found!",NULL); return; }
  stroff = elfVA2PA(stroff);
  f_off = curroff;
  bmSeek(f_off,SEEKF_START);
  if(!(obj = ma_Build(0,True))) return;
  strcpy(stmp,S_INTERPRETER);
  for(i = 0;i < ELF_WORD(elf.e_phnum);i++)
  {
   bmReadBufferEx(&phdr,sizeof(Elf386_External_Phdr),ELF_DWORD(elf.e_phoff) + i*ELF_WORD(elf.e_phentsize),SEEKF_START);
   if(bmEOF()) break;
   if(ELF_DWORD(phdr.p_type) == PT_INTERP)
   {
     bmReadBufferEx(&stmp[sizeof(S_INTERPRETER) - 1],sizeof(stmp)-sizeof(S_INTERPRETER)-1,
                    ELF_DWORD(phdr.p_offset),SEEKF_START);
     break;
   }
  }
  if(!ma_AddString(obj,stmp,True)) goto dyn_end;
  bmSeek(f_off,SEEKF_START);
  for(i = 0;i < nitems;i++)
  {
    bmReadBufferEx(&dyntab,sizeof(dyntab),f_off,SEEKF_START);
    if(bmEOF()) break;
    f_off += sizeof(dyntab);
    is_add = True;
    switch(ELF_DWORD(dyntab.d_tag))
    {
      case DT_NULL: goto dyn_end;
      case DT_NEEDED:
                    {
                      strcpy(stmp,"Needed : ");
                      bmReadBufferEx(&stmp[strlen(stmp)],70,ELF_DWORD(dyntab.d_un.d_ptr) + stroff,SEEKF_START);
                    }
                    break;
      case DT_SONAME:
                    {
                      strcpy(stmp,"SO name: ");
                      bmReadBufferEx(&stmp[strlen(stmp)],70,ELF_DWORD(dyntab.d_un.d_ptr) + stroff,SEEKF_START);
                    }
                    break;
      case DT_RPATH:
                    {
                      strcpy(stmp,"LibPath: ");
                      bmReadBufferEx(&stmp[strlen(stmp)],70,ELF_DWORD(dyntab.d_un.d_ptr) + stroff,SEEKF_START);
                    }
                    break;
       default:     is_add = False; break;
    }
    if(is_add) if(!ma_AddString(obj,stmp,True)) break;
  }
  dyn_end:
  ma_Display(obj," Dynamic linking information ",LB_SORTABLE,-1);
  ma_Destroy(obj);
}

static unsigned long __FASTCALL__ ShowELFDynInfo( void )
{
  unsigned long fpos,dynptr,number;
  Elf386_External_Phdr phdr;
  unsigned i;
  fpos = BMGetCurrFilePos();
  dynptr = 0;
  number = 0;
  for(i = 0;i < ELF_WORD(elf.e_phnum);i++)
  {
   bmReadBufferEx(&phdr,sizeof(Elf386_External_Phdr),ELF_DWORD(elf.e_phoff) + i*ELF_WORD(elf.e_phentsize),SEEKF_START);
   if(bmEOF()) break;
   if(ELF_DWORD(phdr.p_type) == PT_DYNAMIC)
   {
     dynptr = ELF_DWORD(phdr.p_offset);
     number = ELF_DWORD(phdr.p_filesz)/sizeof(Elf386_External_Dyn);
     break;
   }
  }
  bmSeek(fpos,SEEKF_START);
  if(!dynptr) { NotifyBox(NOT_ENTRY," ELF dynamic linking information "); return fpos; }
  displayELFdyninfo(dynptr,(unsigned)number);
  return fpos;
}

static int __FASTCALL__ AppendELFRef(char *str,unsigned long ulShift,int flags,int codelen,unsigned long r_sh)
{
  char buff[400];
  tBool ret = False;
  UNUSED(ulShift);
  UNUSED(codelen);
  memset(buff,-1,sizeof(buff));
  if(flags & APREF_TRY_LABEL)
  {
     if(!PubNames) elf_ReadPubNameList(bmbioHandle(),MemOutBox);
     if(FindPubName(buff,sizeof(buff),r_sh))
     {
       strcat(str,buff);
       if(!DumpMode && !EditMode) GidAddGoAddress(str,r_sh);
       ret = True;
     }
  }
  return ret ? RAPREF_DONE : RAPREF_NONE;
}

static tBool __FASTCALL__ IsELF32( void )
{
  char id[4];
  bmReadBufferEx(id,sizeof(id),0,SEEKF_START);
  return id[0] == 0x7F && id[1] == 'E' && id[2] == 'L' && id[3] == 'F'
         &&  bmReadByteEx(4,SEEKF_START) <= ELFCLASS32;
}

static void __FASTCALL__ __elfReadSegments(linearArray **to, tBool is_virt )
{
 Elf386_External_Phdr phdr;
 Elf386_External_Shdr shdr;
 struct tag_elfVAMap vamap;
 unsigned long fp;
 unsigned va_map_count;
 tBool test;
 size_t i;
   /* We'll try to build section headers first 
      since they is used in calculations of objects.
      For translation of virtual address to physical and vise versa
      more preferred to use program headers. But they often are not presented
      and often unordered by file offsets. */
   if((va_map_count = ELF_WORD(elf.e_shnum)) != 0)
   {  /* Sections headers are present */
     va_map_count = ELF_WORD(elf.e_shnum);
     if(!(*to = la_Build(0,sizeof(struct tag_elfVAMap),MemOutBox)))
     {
       exit(EXIT_FAILURE);
     }
     bmSeek(ELF_DWORD(elf.e_shoff),SEEKF_START);
     for(i = 0;i < va_map_count;i++)
     {
       unsigned long flg,x_flags;
       fp = bmGetCurrFilePos();
       bmReadBuffer(&shdr,sizeof(Elf386_External_Shdr));
       bmSeek(fp+ELF_WORD(elf.e_shentsize),SEEKF_START);
       vamap.va = ELF_DWORD(shdr.sh_addr);
       vamap.size = ELF_DWORD(shdr.sh_size);
       vamap.foff = ELF_DWORD(shdr.sh_offset);
       vamap.nameoff = ELF_DWORD(shdr.sh_name);
       flg = ELF_DWORD(shdr.sh_flags);
       x_flags = 0;
       if(ELF_ST_TYPE(ELF_DWORD(shdr.sh_type)) == SHT_PROGBITS) 
       {
         x_flags |= PF_R; /* Ugle: means flags is not empty */
         if(flg & SHF_WRITE)     x_flags |= PF_W;
         if(flg & SHF_EXECINSTR) x_flags |= PF_X;
       }
       vamap.flags = x_flags;
       test = is_virt ? elfVA2PA(vamap.va) != 0 : elfPA2VA(vamap.foff) != 0;
       if(!test)
       {
         if(!la_AddData(*to,&vamap,MemOutBox))
         {
           exit(EXIT_FAILURE);
         }
         /** We must sort va_map after adding of each element because ELF section
             header has unsorted and has nested elements */
         la_Sort(*to,is_virt ? vamap_comp_virt : vamap_comp_phys);
       }
     }
   }
   else /* Try to build program headers map */
   {
     if(!(*to = la_Build(va_map_count,sizeof(struct tag_elfVAMap),MemOutBox)))
     {
       exit(EXIT_FAILURE);
     }
     bmSeek(ELF_DWORD(elf.e_phoff),SEEKF_START);
     for(i = 0;i < va_map_count;i++)
     {
       fp = bmGetCurrFilePos();
       bmReadBuffer(&phdr,sizeof(Elf386_External_Phdr));
       bmSeek(fp+ELF_WORD(elf.e_phentsize),SEEKF_START);
       vamap.va = ELF_DWORD(phdr.p_vaddr);
       vamap.size = ELF_DWORD(phdr.p_filesz);
       vamap.foff = ELF_DWORD(phdr.p_offset);
       vamap.nameoff = ELF_DWORD(phdr.p_type) & 0x000000FFUL ? ~ELF_DWORD(phdr.p_type) : 0xFFFFFFFFUL;
       vamap.flags = ELF_DWORD(phdr.p_flags);
       test = is_virt ? elfVA2PA(vamap.va) != 0 : elfPA2VA(vamap.foff) != 0;
       if(!test)
       {
         if(!la_AddData(*to,&vamap,MemOutBox))
         {
           exit(EXIT_FAILURE);
         }
         /** We must sort va_map after adding of each element because ELF program
             header has unsorted and has nested elements */
         la_Sort(*to,is_virt ? vamap_comp_virt : vamap_comp_phys);
       }
     }
   }
}

static void __FASTCALL__ ELFinit( void )
{
 BGLOBAL main_handle;
 unsigned long symptr,number;
 size_t i;
 Elf386_External_Shdr shdr;
   PMRegLowMemCallBack(elfLowMemFunc);
   bmReadBufferEx(&elf,sizeof(Elf386_External_Ehdr),0,SEEKF_START);
   is_big = elf.e_ident[5] == ELFDATA2MSB;
   __elfReadSegments(&va_map_virt,True);
   __elfReadSegments(&va_map_phys,False);
   /** Find min value of virtual address */
   for(i = 0; i < va_map_virt->nItems;i++)
   {
     struct tag_elfVAMap __HUGE__ *evm;
     evm = &((struct tag_elfVAMap __HUGE__ *)va_map_virt->data)[i];
     if(evm->va < elf_min_va) elf_min_va = evm->va;
   }
   main_handle = bmbioHandle();
   if((namecache = bioDupEx(main_handle,BBIO_SMALL_CACHE_SIZE)) == &bNull) namecache = main_handle;
   if((namecache2 = bioDupEx(main_handle,BBIO_SMALL_CACHE_SIZE)) == &bNull) namecache2 = main_handle;
   if((elfcache = bioDupEx(main_handle,BBIO_SMALL_CACHE_SIZE)) == &bNull) elfcache = main_handle;
   /** Computing symbol table entry */
   symptr = 0;
   number = 0;
   for(i = 0;i < ELF_WORD(elf.e_shnum);i++)
   {
     bmReadBufferEx(&shdr,sizeof(Elf386_External_Shdr),ELF_DWORD(elf.e_shoff) + i*ELF_WORD(elf.e_shentsize),SEEKF_START);
     if(bmEOF()) break;
     if(ELF_DWORD(shdr.sh_type) == SHT_SYMTAB)
     {
       symptr = ELF_DWORD(shdr.sh_offset);
       number = ELF_DWORD(shdr.sh_size)/ELF_DWORD(shdr.sh_entsize);
       active_shtbl = ELF_WORD(shdr.sh_link);
       break;
     }
    }
    __elfSymShTbl = active_shtbl;
    __elfSymPtr = symptr;
    __elfNumSymTab = (unsigned)number;
    __elfSymEntSize = ELF_DWORD(shdr.sh_entsize);
}

static void __FASTCALL__ ELFdestroy( void )
{
   BGLOBAL main_handle;
   main_handle = bmbioHandle();
   if(namecache != &bNull && namecache != main_handle) bioClose(namecache);
   if(namecache2 != &bNull && namecache2 != main_handle) bioClose(namecache2);
   if(elfcache != &bNull && elfcache != main_handle) bioClose(elfcache);
   if(PubNames) { la_Destroy(PubNames); PubNames = 0; }
   PMUnregLowMemCallBack(elfLowMemFunc);
   la_Destroy(va_map_virt);
   la_Destroy(va_map_phys);
}

static int __FASTCALL__ ELFbitness(unsigned long off)
{
  UNUSED(off);
  return DAB_USE32;
}

static unsigned long __FASTCALL__ ELFHelp( void )
{
  hlpDisplay(10003);
  return BMGetCurrFilePos();
}

static tBool __FASTCALL__ ELFAddrResolv(char *addr,unsigned long cfpos)
{
 /* Since this function is used in references resolving of disassembler
    it must be seriously optimized for speed. */
  tBool bret = True;
  uint32_t res;
  if(cfpos < sizeof(Elf386_External_Ehdr))
  {
    strcpy(addr,"ELFhdr:");
    strcpy(&addr[7],Get2Digit(cfpos));
  }
  else 
    if((res=elfPA2VA(cfpos))!=0)
    {
      addr[0] = '.';
      strcpy(&addr[1],Get8Digit(res));
    }
    else bret = False;
  return bret;
}

static tCompare __FASTCALL__ compare_pubnames(const void __HUGE__ *v1,const void __HUGE__ *v2)
{
  const struct PubName __HUGE__ *pnam1,__HUGE__ *pnam2;
  tCompare ret;
  pnam1 = (const struct PubName __HUGE__ *)v1;
  pnam2 = (const struct PubName __HUGE__ *)v2;
  if(pnam1->pa < pnam2->pa) ret = -1;
  else
    if(pnam1->pa > pnam2->pa) ret = 1;
    else                      ret = 0;
  return ret;
}

static tBool __NEAR__ __FASTCALL__ FindPubName(char *buff,unsigned cb_buff,unsigned long pa)
{
  struct PubName *ret,key;
  key.pa = pa;
  ret = la_Find(PubNames,&key,compare_pubnames);
  if(ret)
  {
    active_shtbl = ret->addinfo;
    elf386_readnametableex(ret->nameoff,buff,cb_buff);
    buff[cb_buff-1] = 0;
    return True;
  }
  return False;
}

static void __FASTCALL__ elf_ReadPubNameList(BGLOBAL handle,void (__FASTCALL__ *mem_out)(const char *))
{
  unsigned long i,fpos,fp,tableptr,number;
  unsigned short pubname_shtbl;
  Elf386_External_Shdr shdr;
  struct PubName epn;
  BGLOBAL b_cache;
  b_cache = handle;
  if(!(PubNames = la_Build(0,sizeof(struct PubName),mem_out))) return;
  fpos = bioTell(b_cache);
  pubname_shtbl = 0;
  tableptr = 0L;
  number = 0L;
  for(i = 0;i < ELF_WORD(elf.e_shnum);i++)
  {
   bioSeek(b_cache,ELF_DWORD(elf.e_shoff) + i*ELF_WORD(elf.e_shentsize),SEEKF_START);
   bioReadBuffer(b_cache,&shdr,sizeof(Elf386_External_Shdr));
   if(bioEOF(b_cache)) break;
   if(ELF_DWORD(shdr.sh_type) == SHT_DYNSYM)
   {
     tableptr = ELF_DWORD(shdr.sh_offset);
     number = ELF_DWORD(shdr.sh_size)/ELF_DWORD(shdr.sh_entsize);
     pubname_shtbl = ELF_WORD(shdr.sh_link);
     break;
   }
  }
  if(!(PubNames = la_Build(0,sizeof(struct PubName),mem_out))) return;
  if(tableptr)
  {
    bioSeek(b_cache,tableptr,SEEK_SET);
    for(i = 0;i < number;i++)
    {
     Elf386_External_Dyn pdyn;
     fp = bioTell(b_cache);
     bioReadBuffer(b_cache,&pdyn,sizeof(Elf386_External_Dyn));
     if(bioEOF(b_cache)) break;
     bioSeek(b_cache,fp+ELF_DWORD(shdr.sh_entsize),SEEKF_START);
     epn.nameoff = ELF_DWORD(pdyn.d_tag);
     epn.pa = elfVA2PA(ELF_DWORD(pdyn.d_un.d_val));
     epn.addinfo = pubname_shtbl;
     epn.attr = ELF_ST_INFO(STB_GLOBAL,STT_NOTYPE);
     if(!la_AddData(PubNames,&epn,mem_out)) break;
    }
  }
  /** If present symbolic information we must read it */

  if(__elfNumSymTab)
  {
    bioSeek(handle,__elfSymPtr,SEEK_SET);
    for(i = 0;i < __elfNumSymTab;i++)
    {
      Elf386_External_Sym sym;
      fp = bioTell(handle);
      bioReadBuffer(handle,&sym,sizeof(Elf386_External_Sym));
      if(bioEOF(handle) || IsKbdTerminate()) break;
      bioSeek(handle,fp+__elfSymEntSize,SEEKF_START);
      if(ELF_IS_SECTION_PHYSICAL(ELF_WORD(sym.st_shndx)) &&
         ELF_ST_TYPE(sym.st_info[0]) != STT_SECTION)
      {
        epn.pa = __calcSymEntry(handle,i,False);
        epn.nameoff = ELF_DWORD(sym.st_name);
        epn.addinfo = __elfSymShTbl;
        epn.attr = sym.st_info[0];
        if(!la_AddData(PubNames,&epn,MemOutBox)) break;
      }
    }
  }
  la_Sort(PubNames,compare_pubnames);
  bioSeek(b_cache,fpos,SEEK_SET);
}

static void __FASTCALL__ elf_ReadPubName(BGLOBAL b_cache,const struct PubName *it,
                            char *buff,unsigned cb_buff)
{
   UNUSED(b_cache);
   active_shtbl = it->addinfo;
   elf386_readnametableex(it->nameoff,buff,cb_buff);
}

static unsigned long __FASTCALL__ elfGetPubSym(char *str,unsigned cb_str,unsigned *func_class,
                           unsigned long pa,tBool as_prev)
{
   return fmtGetPubSym(elfcache,str,cb_str,func_class,pa,as_prev,
                       elf_ReadPubNameList,
                       elf_ReadPubName);
}

static unsigned __FASTCALL__ elfGetObjAttr(unsigned long pa,char *name,unsigned cb_name,
                       unsigned long *start,unsigned long *end,int *class,int *bitness)
{
  unsigned i,ret;
  struct tag_elfVAMap *evam;
  *start = 0;
  *end = bmGetFLength();
  *class = OC_NOOBJECT;
  *bitness = ELFbitness(pa);
  name[0] = 0;
  ret = 0;
  evam = va_map_phys->data;
  for(i = 0;i < va_map_phys->nItems;i++)
  {
    if(!(evam[i].foff && evam[i].size)) continue;
    if(pa >= *start && pa < evam[i].foff)
    {
      /** means between two objects */
      *end = evam[i].foff;
      ret = 0;
      break;
    }
    if(pa >= evam[i].foff &&
       pa < evam[i].foff + evam[i].size)
    {
      *start = evam[i].foff;
      *end = *start + evam[i].size;
      if(evam[i].flags)
      {
        if(evam[i].flags & PF_X) *class = OC_CODE;
        else                     *class = OC_DATA;
      }
      else  *class = OC_NOOBJECT;
      elf386_readnametable(evam[i].nameoff,name,cb_name);
      ret = i+1;
      break;
    }
    *start = evam[i].foff + evam[i].size;
  }
  return ret;
}

static int __FASTCALL__ ELFplatform( void ) { return DISASM_CPU_IX86; }

REGISTRY_BIN elf386Table =
{
  "ELF-i386 (Executable and Linkable Format)",
  { "ELFhlp", "DynInf", "DynSec", NULL, NULL, NULL, "SymTab", NULL, "SecHdr", "PrgDef" },
  { ELFHelp, ShowELFDynInfo, ShowELFDynSec, NULL, NULL, NULL, ShowELFSymTab, NULL, ShowSecHdrElf, ShowPrgHdrElf },
  IsELF32, ELFinit, ELFdestroy,
  ShowELFHeader,
  AppendELFRef,
  fmtSetState,
  ELFplatform,
  ELFbitness,
  ELFAddrResolv,
  elfVA2PA,
  elfPA2VA,
  elfGetPubSym,
  elfGetObjAttr,
  NULL,
  NULL
};
