/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "projectconfig.h"
#include "directories.h"
#include "gwrappers.h"
#include "sqlite_reader.h"
#include "utilities.h"
#include "vacuum.h"
#include "date_time_utils.h"
#include "generalconfig.h"
#include "stylesheetutils.h"


void project_configuration_vacuum (const ustring& project, unsigned int starttime)
{
  vacuum_database (project_configuration_filename (project), starttime);
}


ustring project_configuration_filename (const ustring& project)
{
  return gw_build_filename (directories_get_projects (), project, "configuration.sqlite.1");
}


void upgrade_project_configuration (const ustring& project)
{
  // No upgrade at present.
}


ProjectConfiguration::ProjectConfiguration (ustring project) :
  genconfig (0)
{
  // If no project given, take one currently opened.
  if (project.empty()) {
    project = genconfig.project();
  }
  error = NULL;
  ustring filename = project_configuration_filename (project);
  try
  {
    if (!g_file_test (filename.c_str(), G_FILE_TEST_IS_REGULAR))
      create (filename);
    rc = sqlite3_open (filename.c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


ProjectConfiguration::~ProjectConfiguration ()
{
  // Close connection.
  sqlite3_close (db);
}


void ProjectConfiguration::create (const ustring& filename)
// Creates a new table for storing the configuration.
// One tables is created for boolean, string and integer values.
// This is for easier management.
// A sequence field is added for storing lists of things.
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open (filename.c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    char * sql;
    sql = g_strdup_printf ("create table data (key text, value text, sequence integer);");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    g_free (sql);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);
}


ustring ProjectConfiguration::value_get (char * key, const ustring& standard)
{
  ustring value = standard;
  try
  {
    char * sql;
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    sql = g_strdup_printf ("select value from data where key = '%s';", key);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if ((sqlitereader.ustring0.size() > 0)) {
      value = sqlitereader.ustring0[0];
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  return value;
}


void ProjectConfiguration::value_set (char * key, const ustring& value)
{
  try
  {
    char * sql;
    sql = g_strdup_printf ("delete from data where key = '%s';", key);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    sql = g_strdup_printf ("insert into data values ('%s', '%s', 0)", key, value.c_str());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    g_free (sql);
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


vector<ustring> ProjectConfiguration::valuelist_get (char * key)
{
  vector<ustring> value;
  vector<unsigned int> sequence;
  try
  {
    char * sql;
    sqlite3_busy_timeout (db, 1000);
    SqliteReader sqlitereader (0);
    sql = g_strdup_printf ("select value, sequence from data where key = '%s';", key);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    value.clear ();
    value.assign (sqlitereader.ustring0.begin(), sqlitereader.ustring0.end());
    for (unsigned int i = 0; i < sqlitereader.ustring1.size(); i++) {
      sequence.push_back (convert_to_int (sqlitereader.ustring1[i]));
    }
    quick_sort (sequence, value, 0, sequence.size());
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  return value;
}


void ProjectConfiguration::valuelist_set (char * key, const vector<ustring>& value)
{
  try
  {
    char * sql;
    sql = g_strdup_printf ("delete from data where key = '%s';", key);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    for (unsigned int i = 0; i < value.size(); i++) {
      sql = g_strdup_printf ("insert into data values ('%s', '%s', %d)", key, value[i].c_str(), i);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
      if (rc) {
        throw runtime_error (sqlite3_errmsg(db));
      }
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
}


bool ProjectConfiguration::bool_get (char * id, bool standard)
{
  return convert_to_bool (value_get (id, convert_to_string (standard)));
}


void ProjectConfiguration::bool_set (char * id, bool value)
{
  value_set (id, convert_to_string (value));
}


int ProjectConfiguration::int_get (char * id, int standard)
{
  return convert_to_int (value_get (id, convert_to_string (standard)));
}


void ProjectConfiguration::int_set (char * id, int value)
{
  value_set (id, convert_to_string (value));
}


ustring ProjectConfiguration::string_get (char * id, const ustring& standard)
{
  return value_get (id, standard);
}


void ProjectConfiguration::string_set (char * id, const ustring& value)
{
  value_set (id, value);
}


double ProjectConfiguration::double_get (char * id, double standard)
{
  return convert_to_double (value_get (id, convert_to_string (standard)));  
}


void ProjectConfiguration::double_set (char * id, double value)
{
  value_set (id, convert_to_string (value));
}


vector<bool> ProjectConfiguration::vector_bool_get (char * id)
{
  vector<ustring> value = valuelist_get (id);
  vector<bool> value2;
  for (unsigned int i = 0; i < value.size(); i++)
    value2.push_back (convert_to_bool (value[i]));
  return value2;
}


void ProjectConfiguration::vector_bool_set (char * id, const vector<bool>& value)
{
  vector<ustring> value2;
  for (unsigned int i = 0; i < value.size(); i++)
    value2.push_back (convert_to_string (value[i]));
  valuelist_set (id, value2);
}


vector<ustring> ProjectConfiguration::vector_string_get (char * id)
{
  return valuelist_get (id);
}


void ProjectConfiguration::vector_string_set (char * id, const vector<ustring>& value)
{
  valuelist_set (id, value);
}


vector<unsigned int> ProjectConfiguration::vector_unsigned_int_get (char * id)
{
  vector<ustring> value = valuelist_get (id);
  vector<unsigned int> value2;
  for (unsigned int i = 0; i < value.size(); i++)
    value2.push_back (convert_to_int (value[i]));
  return value2;
}


void ProjectConfiguration::vector_unsigned_int_set (char * id, const vector<unsigned int>& value)
{
  vector<ustring> value2;
  for (unsigned int i = 0; i < value.size(); i++)
    value2.push_back (convert_to_string (value[i]));
  valuelist_set (id, value2);
}


char * ProjectConfiguration::editor_font_key ()
/*
The font that Bibledit uses to display the project.
*/
{
  return "editor_font";
}

ustring ProjectConfiguration::editor_font ()
{
  return string_get (editor_font_key(), genconfig.editor_font_name());
}

void ProjectConfiguration::editor_font_set (ustring value)
{
  string_set (editor_font_key(), value);
}


char * ProjectConfiguration::stylesheet_key ()
/*
The stylesheet attached to the project.
*/
{
  return "stylesheet";
}

ustring ProjectConfiguration::stylesheet ()
{
  ustring checked_sheet (string_get (stylesheet_key(), STANDARDSHEET));
  // See whether our stylesheet exists.
  vector<ustring> stylesheets;
  stylesheet_get_ones_available (stylesheets);
  set<ustring> sheets (stylesheets.begin(), stylesheets.end());
  // Sheet is there? Fine.
  if (sheets.find (checked_sheet) != sheets.end());
  // Sheets is not there - take Standard, if it's around.
  else if (sheets.find (STANDARDSHEET) != sheets.end())
    checked_sheet = STANDARDSHEET;
  // Else take first sheet in the list.
  else
    checked_sheet = stylesheets[0];
  // Return sheet.
  return checked_sheet;  
}

void ProjectConfiguration::stylesheet_set (ustring value)
{
  string_set (stylesheet_key(), value);
}


char * ProjectConfiguration::versification_key ()
/*
The project's versification system.
*/
{
  return "versification";
}
  
ustring ProjectConfiguration::versification ()
{
  return string_get (versification_key(), "English");
}

void ProjectConfiguration::versification_set (ustring value)
{
  string_set (versification_key(), value);
}


char * ProjectConfiguration::printing_fonts_key ()
/*
List of fonts to be used during printing.
*/
{
  return "print_fonts2";
}

vector<ustring> ProjectConfiguration::printing_fonts ()
{
  return vector_string_get (printing_fonts_key());
}

void ProjectConfiguration::printing_fonts_set (vector<ustring> value)
{
  vector_string_set (printing_fonts_key(), value);
}


char * ProjectConfiguration::printing_font_size_key ()
/*
The size of the fonts for printing.
*/
{
  return "print_font_size";
}

int ProjectConfiguration::printing_font_size ()
{
  return int_get (printing_font_size_key(), 12);
}

void ProjectConfiguration::printing_font_size_set (int value)
{
  int_set (printing_font_size_key(), value);
}


char * ProjectConfiguration::printing_line_height_key ()
/*
The line height for printing, in percentage.
*/
{
  return "print_line_height";
}

int ProjectConfiguration::printing_line_height ()
{
  return int_get (printing_line_height_key(), 100);
}

void ProjectConfiguration::printing_line_height_set (int value)
{
  int_set (printing_line_height_key(), value);
}


char * ProjectConfiguration::sword_name_key ()
/*
Export to SWORD: Name of module, excluding project name.
*/
{
  return "sword_name";
}


ustring ProjectConfiguration::sword_name ()
{
  return string_get (sword_name_key (), "");
}


void ProjectConfiguration::sword_name_set (ustring value)
{
  string_set (sword_name_key (), value);
}


char * ProjectConfiguration::sword_description_key ()
/*
Export to SWORD: Description of module, excluding project name.
*/
{
  return "sword_desc";
}


ustring ProjectConfiguration::sword_description ()
{
  return string_get (sword_description_key (), " Bibledit project");
}


void ProjectConfiguration::sword_description_set (ustring value)
{
  string_set (sword_description_key (), value);
}


char * ProjectConfiguration::sword_about_key ()
/*
Export to SWORD: Description of module, excluding project name.
*/
{
  return "sword_about";
}


ustring ProjectConfiguration::sword_about ()
{
  return string_get (sword_about_key (), " Bibledit project");
}


void ProjectConfiguration::sword_about_set (ustring value)
{
  string_set (sword_about_key (), value);
}


char * ProjectConfiguration::sword_lcsh_key ()
/*
Export to SWORD: Library of Congress Subject Heading of module.
*/
{
  return "sword_lcsh";
}


ustring ProjectConfiguration::sword_lcsh ()
{
  return string_get (sword_lcsh_key (), "Bible--Translation");
}


void ProjectConfiguration::sword_lcsh_set (ustring value)
{
  string_set (sword_lcsh_key (), value);
}


char * ProjectConfiguration::sword_license_key ()
/*
Export to SWORD: License of module.
*/
{
  return "sword_license";
}


ustring ProjectConfiguration::sword_license ()
{
  return string_get (sword_license_key (), "GNU General Public License");
}


void ProjectConfiguration::sword_license_set (ustring value)
{
  string_set (sword_license_key (), value);
}


char * ProjectConfiguration::sword_version_key ()
/*
Export to SWORD: Version of module.
*/
{
  return "sword_version";
}


ustring ProjectConfiguration::sword_version ()
{
  return string_get (sword_version_key (), "1.0");
}


void ProjectConfiguration::sword_version_set (ustring value)
{
  string_set (sword_version_key (), value);
}


char * ProjectConfiguration::sword_language_key ()
/*
Export to SWORD: Language of module.
*/
{
  return "sword_lang";
}


ustring ProjectConfiguration::sword_language ()
{
  return string_get (sword_language_key (), "English");
}


void ProjectConfiguration::sword_language_set (ustring value)
{
  string_set (sword_language_key (), value);
}


char * ProjectConfiguration::reordered_books_key ()
/*
Books in another order.
*/
{
  return "reorder_books";
}

vector<ustring> ProjectConfiguration::reordered_books ()
{
  return vector_string_get (reordered_books_key());
}

void ProjectConfiguration::reordered_books_set (vector<ustring> value)
{
  vector_string_set (reordered_books_key(), value);
}


char * ProjectConfiguration::reordered_includes_key ()
/*
Which books to include.
*/
{
  return "books_incl";
}

vector<bool> ProjectConfiguration::reordered_includes ()
{
  return vector_bool_get (reordered_includes_key());
}

void ProjectConfiguration::reordered_includes_set (vector<bool> value)
{
  vector_bool_set (reordered_includes_key(), value);
}


char * ProjectConfiguration::reordered_portions_key ()
/*
Which portion of books to print.
*/
{
  return "books_portion";
}

vector<ustring> ProjectConfiguration::reordered_portions ()
{
  return vector_string_get (reordered_portions_key());
}

void ProjectConfiguration::reordered_portions_set (vector<ustring> value)
{
  vector_string_set (reordered_portions_key(), value);
}


char * ProjectConfiguration::language_key ()
/*
The project's language system.
*/
{
  return "language";
}
  
ustring ProjectConfiguration::language ()
{
  return string_get (language_key(), "English");
}

void ProjectConfiguration::language_set (ustring value)
{
  string_set (language_key(), value);
}


char * ProjectConfiguration::book_order_key ()
/*
Order of the books.
*/
{
  return "book_order";
}

vector<unsigned int> ProjectConfiguration::book_order ()
{
  return vector_unsigned_int_get (book_order_key());
}

void ProjectConfiguration::book_order_set (vector<unsigned int> value)
{
  vector_unsigned_int_set (book_order_key(), value);
}


char * ProjectConfiguration::editable_key ()
/*
Whether project is editable.
*/
{
  return "editable";
}

bool ProjectConfiguration::editable ()
{
  return bool_get (editable_key(), true);
}

void ProjectConfiguration::editable_set (bool value)
{
  bool_set (editable_key(), value);
}
