/* BEAST - Bedevilled Audio System
 * Copyright (C) 1998, 1999, 2000 Olaf Hoehmann and Tim Janik
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include        "bstdefs.h"

#include        "bstapp.h"
#include        "bstsamplerepo.h"
#include	"bstxkb.h"
#include	"bstkeytables.h"
#include	"bstgconfig.h"
#include	"bstpreferences.h"
#include	"../PKG_config.h"




#define	PROGRAM	"BEAST"
#define	TITLE	"Beast"
#define	VERSION	"Pre-Alpha"


/* --- prototypes --- */
static void			bst_parse_args		(gint        *argc_p,
							 gchar     ***argv_p);
static BstKeyTablePatch*	bst_key_table_from_xkb	(const gchar *display);
static void			bst_print_blurb		(FILE	     *fout,
							 gboolean     print_help);


/* --- variables --- */
static guint        args_changed_signal_id = 0;
BstDebugFlags       bst_debug_flags = 0;
static GDebugKey    bst_debug_keys[] = { /* keep in sync with bstdefs.h */
  { "keytable",		BST_DEBUG_KEYTABLE, },
  { "samples",		BST_DEBUG_SAMPLES, },
};
static const guint  bst_n_debug_keys = sizeof (bst_debug_keys) / sizeof (bst_debug_keys[0]);
static gboolean     arg_force_xkb = FALSE;
static const gchar *bst_rc_string =
( "style'BstTooltips-style'"
  "{"
  "bg[NORMAL]={.94,.88,0.}"
  "}"
  "widget'gtk-tooltips'style'BstTooltips-style'"
  "\n");


/* --- functions --- */
int
main (int   argc,
      char *argv[])
{
  BsePcmDevice *pdev = NULL;
  BstApp *app = NULL;
  guint i;
  
  /* initialize BSE and preferences
   */
  bse_init (&argc, &argv);
  bst_globals_init ();
  
  /* initialize GUI libraries and pre-parse args
   */
  bst_parse_args (&argc, &argv);

  g_message ("BEAST: pid = %u", getpid ());
  
  gtk_init (&argc, &argv);
  gtk_rc_parse_string (bst_rc_string);
  gdk_rgb_init ();
  gnome_type_init ();
  // gnome_init (PROGRAM, VERSION, argc, argv);
  bst_free_radio_button_get_type ();
  
  /* parse rc file */
  if (1)
    {
      BseGConfig *gconf;
      BseErrorType error;
      gchar *file_name;
      
      file_name = BST_STRDUP_RC_FILE ();
      gconf = bse_object_new (BST_TYPE_GCONFIG, NULL);
      bse_gconfig_revert (gconf);
      error = bst_rc_parse (file_name, gconf);
      if (error != BSE_ERROR_FILE_NOT_FOUND)
	{
	  bse_gconfig_apply (gconf);
	  if (error)
	    g_warning ("error parsing rc-file \"%s\": %s", file_name, bse_error_blurb (error));
	}
      bse_object_unref (BSE_OBJECT (gconf));
      g_free (file_name);
    }
  
  /* setup default keytable for pattern editor class
   */
  bst_key_table_install_patch (bst_key_table_from_xkb (gdk_get_display ()));
  
  
  /* check load BSE plugins to register types
   */
  if (1)
    {
      GList *free_list, *list;
      
      free_list = bse_plugin_dir_list_files (BSE_PATH_PLUGINS);
      for (list = free_list; list; list = list->next)
	{
	  gchar *error, *string = list->data;
	  
	  g_message ("loading plugin \"%s\"...", string);
	  error = bse_plugin_check_load (string);
	  if (error)
	    g_warning ("error encountered loading plugin \"%s\": %s", string, error);
	  g_free (string);
	}
      g_list_free (free_list);
      if (!free_list)
	g_warning ("strange, can't find any plugins, please check %s", BSE_PATH_PLUGINS);
    }
  
  
  /* hackery rulez!
   */
  args_changed_signal_id =
    gtk_object_class_user_signal_new (gtk_type_class (GTK_TYPE_OBJECT),
				      "args_changed",
				      GTK_RUN_ACTION,
				      gtk_signal_default_marshaller,
				      GTK_TYPE_NONE, 0);
  
  
  /* setup PCM Devices
   */
  if (!pdev && BSE_TYPE_ID (BsePcmDeviceAlsa) && !BST_DISABLE_ALSA)
    {
      BseErrorType error;
      
      pdev = (BsePcmDevice*) bse_object_new (BSE_TYPE_ID (BsePcmDeviceAlsa), NULL); /* FIXME: TYPE_ID */
      error = bse_pcm_device_update_caps (pdev);
      if (error && error != BSE_ERROR_DEVICE_BUSY)
	{
	  bse_object_unref (BSE_OBJECT (pdev));
	  pdev = NULL;
	}
    }
  if (!pdev && BSE_TYPE_ID (BsePcmDeviceOSS))
    {
      BseErrorType error;
      
      pdev = (BsePcmDevice*) bse_object_new (BSE_TYPE_ID (BsePcmDeviceOSS), NULL); /* FIXME: TYPE_ID */
      error = bse_pcm_device_update_caps (pdev);
      if (error && error != BSE_ERROR_DEVICE_BUSY)
	{
	  bse_object_unref (BSE_OBJECT (pdev));
	  pdev = NULL;
	}
    }
  if (!pdev)
    g_error ("No PCM device driver known");
  bse_heart_register_device ("Master", pdev);
  bse_object_unref (BSE_OBJECT (pdev));
  bse_heart_set_default_odevice ("Master");
  bse_heart_set_default_idevice ("Master");
  
  
  /* register sample repositories
   */
  bst_sample_repo_init ();
  
  
  /* open files given on command line
   */
  for (i = 1; i < argc; i++)
    {
      BseStorage *storage = bse_storage_new ();
      BseErrorType error;
      
      error = bse_storage_input_file (storage, argv[i]);
      
      if (!error)
	{
	  BseProject *project = bse_project_new (argv[i]);
	  
	  bse_storage_set_path_resolver (storage, bse_project_path_resolver, project);
	  error = bse_project_restore (project, storage);
	  if (!error)
	    {
	      app = bst_app_new (project);
	      gtk_idle_show_widget (GTK_WIDGET (app));
	    }
	  bse_object_unref (BSE_OBJECT (project));
	}
      bse_storage_destroy (storage);
      if (error)
	g_warning ("failed to load project `%s': %s", /* FIXME */
		   argv[i],
		   bse_error_blurb (error));
    }
  
  /* open default app window
   */
  if (!app)
    {
      BseProject *project = bse_project_new ("Untitled.bse");
      
      app = bst_app_new (project);
      bse_object_unref (BSE_OBJECT (project));
      /* bst_app_operate (app, BST_OP_PROJECT_NEW_SONG); */
      gtk_idle_show_widget (GTK_WIDGET (app));
    }
  
  
  
  /* and away into the main loop
   */
  gtk_main ();
  
  /* stop everything playing */
  bse_heart_reset_all_attach ();
  
  /* FXME: wrt cleanup cycles */
  
  /* perform necessary cleanup cycles */
  while (g_main_iteration (FALSE))
    ;
  
  /* save rc file */
  if (1)
    {
      BseGConfig *gconf;
      BseErrorType error;
      gchar *file_name;
      
      gconf = bse_object_new (BST_TYPE_GCONFIG, NULL);
      bse_gconfig_revert (gconf);
      file_name = BST_STRDUP_RC_FILE ();
      error = bst_rc_dump (file_name, gconf);
      bse_object_unref (BSE_OBJECT (gconf));
      if (error)
	g_warning ("error saving rc-file \"%s\": %s", file_name, bse_error_blurb (error));
      g_free (file_name);
    }
  
  /* remove pcm devices */
  bse_heart_unregister_all_devices ();
  
  /* perform necessary cleanup cycles */
  while (g_main_iteration (FALSE))
    ;
  
  return 0;
}

static void
bst_parse_args (int    *argc_p,
		char ***argv_p)
{
  guint argc = *argc_p;
  gchar **argv = *argv_p;
  gchar *envar;
  guint i, e;
  
  envar = getenv ("BEAST_DEBUG");
  if (envar)
    bst_debug_flags |= g_parse_debug_string (envar, bst_debug_keys, bst_n_debug_keys);
  envar = getenv ("BST_DEBUG");
  if (envar)
    bst_debug_flags |= g_parse_debug_string (envar, bst_debug_keys, bst_n_debug_keys);
  envar = getenv ("BEAST_NO_DEBUG");
  if (envar)
    bst_debug_flags &= ~g_parse_debug_string (envar, bst_debug_keys, bst_n_debug_keys);
  envar = getenv ("BST_NO_DEBUG");
  if (envar)
    bst_debug_flags &= ~g_parse_debug_string (envar, bst_debug_keys, bst_n_debug_keys);
  
  for (i = 1; i < argc; i++)
    {
      if (strcmp ("--beast-debug", argv[i]) == 0 ||
	  strncmp ("--beast-debug=", argv[i], 14) == 0)
	{
	  gchar *equal = argv[i] + 13;
	  
	  if (*equal == '=')
	    bst_debug_flags |= g_parse_debug_string (equal + 1, bst_debug_keys, bst_n_debug_keys);
	  else if (i + 1 < argc)
	    {
	      bst_debug_flags |= g_parse_debug_string (argv[i + 1],
						       bst_debug_keys,
						       bst_n_debug_keys);
	      argv[i] = NULL;
	      i += 1;
	    }
	  argv[i] = NULL;
	}
      else if (strcmp ("--bst-debug", argv[i]) == 0 ||
	       strncmp ("--bst-debug=", argv[i], 12) == 0)
	{
	  gchar *equal = argv[i] + 11;
	  
	  if (*equal == '=')
	    bst_debug_flags |= g_parse_debug_string (equal + 1, bst_debug_keys, bst_n_debug_keys);
	  else if (i + 1 < argc)
	    {
	      bst_debug_flags |= g_parse_debug_string (argv[i + 1],
						       bst_debug_keys,
						       bst_n_debug_keys);
	      argv[i] = NULL;
	      i += 1;
	    }
	  argv[i] = NULL;
	}
      else if (strcmp ("--beast-no-debug", argv[i]) == 0 ||
	       strncmp ("--beast-no-debug=", argv[i], 17) == 0)
	{
	  gchar *equal = argv[i] + 16;
	  
	  if (*equal == '=')
	    bst_debug_flags &= ~g_parse_debug_string (equal + 1, bst_debug_keys, bst_n_debug_keys);
	  else if (i + 1 < argc)
	    {
	      bst_debug_flags &= ~g_parse_debug_string (argv[i + 1],
							bst_debug_keys,
							bst_n_debug_keys);
	      argv[i] = NULL;
	      i += 1;
	    }
	  argv[i] = NULL;
	}
      else if (strcmp ("--bst-no-debug", argv[i]) == 0 ||
	       strncmp ("--bst-no-debug=", argv[i], 15) == 0)
	{
	  gchar *equal = argv[i] + 14;
	  
	  if (*equal == '=')
	    bst_debug_flags &= ~g_parse_debug_string (equal + 1, bst_debug_keys, bst_n_debug_keys);
	  else if (i + 1 < argc)
	    {
	      bst_debug_flags &= ~g_parse_debug_string (argv[i + 1],
							bst_debug_keys,
							bst_n_debug_keys);
	      argv[i] = NULL;
	      i += 1;
	    }
	  argv[i] = NULL;
	}
      else if (strcmp ("--force-xkb", argv[i]) == 0)
	{
	  arg_force_xkb = TRUE;
          argv[i] = NULL;
	}
      else if (strcmp ("-h", argv[i]) == 0 ||
	       strcmp ("--help", argv[i]) == 0)
	{
	  bst_print_blurb (stderr, TRUE);
          argv[i] = NULL;
	  exit (0);
	}
      else if (strcmp ("-v", argv[i]) == 0 ||
	       strcmp ("--version", argv[i]) == 0)
	{
	  bst_print_blurb (stderr, FALSE);
	  argv[i] = NULL;
	  exit (0);
	}
    }
  
  e = 0;
  for (i = 1; i < argc; i++)
    {
      if (e)
	{
	  if (argv[i])
	    {
	      argv[e++] = argv[i];
	      argv[i] = NULL;
	    }
	}
      else if (!argv[i])
	e = i;
    }
  if (e)
    *argc_p = e;
}

static void
bst_print_blurb (FILE    *fout,
		 gboolean print_help)
{
  if (!print_help)
    {
      fprintf (fout, "BEAST version %s (%s)\n", BST_VERSION, BST_VERSION_HINT);
      fprintf (fout, "Libraries: ");
      fprintf (fout, "BSE %u.%u.%u", bse_major_version, bse_minor_version, bse_micro_version);
      fprintf (fout, ", GTK+ %u.%u.%u", gtk_major_version, gtk_minor_version, gtk_micro_version);
      fprintf (fout, ", GLib %u.%u.%u", glib_major_version, glib_minor_version, glib_micro_version);
#ifdef BST_WITH_GDK_PIXBUF
      fprintf (fout, ", GdkPixbuf");
#endif
#ifdef BST_WITH_XKB
      fprintf (fout, ", XKBlib");
#endif
      fprintf (fout, "\n");
      fprintf (fout, "Compiled for: %s\n", BST_ARCH_NAME);
      fprintf (fout, "\n");
      fprintf (fout, "Doc-path:    %s\n", BST_PATH_DOCS);
      fprintf (fout, "Plugin-path: %s\n", BSE_PATH_PLUGINS);
      fprintf (fout, "Sample-path: %s\n", BST_PATH_DATA_SAMPLES);
      fprintf (fout, "\n");
      fprintf (fout, "BEAST comes with ABSOLUTELY NO WARRANTY.\n");
      fprintf (fout, "You may redistribute copies of BEAST under the terms of\n");
      fprintf (fout, "the GNU General Public License which can be found in the\n");
      fprintf (fout, "BEAST source package. Sources, examples and contact\n");
      fprintf (fout, "information are available at http://beast.gtk.org\n");
    }
  else
    {
      fprintf (fout, "Usage: beast [options] [files...]\n");
      fprintf (fout, "  --beast-debug=keys		enable certain BEAST debug stages\n");
      fprintf (fout, "  --beast-no-debug=keys		disable certain BEAST debug stages\n");
      fprintf (fout, "  --force-xkb			force XKB keytable queries\n");
      fprintf (fout, "  --bse-debug=keys		enable certain BSE debug stages\n");
      fprintf (fout, "  --bse-no-debug=keys		disable certain BSE debug stages\n");
      fprintf (fout, "  -h, --help			show this help message\n");
      fprintf (fout, "  -v, --version			print version informations\n");
      fprintf (fout, "  --display=DISPLAY		X server to contact; see X(1)\n");
      fprintf (fout, "  --no-xshm			disable use of X shared memory extension\n");
      fprintf (fout, "  --gtk-debug=FLAGS		Gtk+ debugging flags to enable\n");
      fprintf (fout, "  --gtk-no-debug=FLAGS		Gtk+ debugging flags to disable\n");
      fprintf (fout, "  --gtk-module=MODULE		load additional Gtk+ modules\n");
      fprintf (fout, "  --gdk-debug=FLAGS		Gdk debugging flags to enable\n");
      fprintf (fout, "  --gdk-no-debug=FLAGS		Gdk debugging flags to disable\n");
      fprintf (fout, "  --g-fatal-warnings		make warnings fatal (abort)\n");
      fprintf (fout, "  --sync   			do all X calls synchronously\n");
    }
}

static BstKeyTablePatch*
bst_key_table_from_xkb_symbol (const gchar *xkb_symbol)
{
  gchar *encoding, *layout, *model, *variant;
  GSList *slist, *name_list = NULL;
  BstKeyTablePatch *patch = NULL;
  
  bst_xkb_parse_symbol (xkb_symbol, &encoding, &layout, &model, &variant);
  BST_IF_DEBUG (KEYTABLE)
    g_message ("keytable %s: encoding(%s) layout(%s) model(%s) variant(%s)",
	       xkb_symbol, encoding, layout, model, variant);
  
  /* strip number of keys (if present) */
  if (layout)
    {
      gchar *n, *l = layout;
      
      while (*l && (*l < '0' || *l > '9'))
	l++;
      n = l;
      while (*n >= '0' && *n <= '9')
	n++;
      *n = 0;
      n = layout;
      layout = *l ? g_strdup (l) : NULL;
      g_free (n);
    }
  
  /* list guesses */
  if (encoding)
    {
      name_list = g_slist_prepend (name_list, g_strdup (encoding));
      if (layout)
	name_list = g_slist_prepend (name_list,
				     g_strdup_printf ("%s-%s",
						      encoding,
						      layout));
    }
  if (model)
    {
      name_list = g_slist_prepend (name_list, g_strdup (model));
      if (layout)
	name_list = g_slist_prepend (name_list,
				     g_strdup_printf ("%s-%s",
						      model,
						      layout));
    }
  g_free (encoding);
  g_free (layout);
  g_free (model);
  g_free (variant);
  
  for (slist = name_list; slist; slist = slist->next)
    {
      gchar *name = slist->data;
      
      if (!patch)
	{
	  patch = bst_key_table_patch_find (name);
	  BST_IF_DEBUG (KEYTABLE)
	    g_message ("Guessing keytable, %s \"%s\"",
		       patch ? "found" : "failed to get",
		       name);
	}
      else
	BST_IF_DEBUG (KEYTABLE)
	  g_message ("Guessing keytable, discarding \"%s\"", name);
      g_free (name);
    }
  g_slist_free (name_list);
  
  return patch;
}

static BstKeyTablePatch*
bst_key_table_from_xkb (const gchar *display)
{
  BstKeyTablePatch *patch = NULL;
  
  if (!BST_XKB_FORCE_QUERY && !arg_force_xkb && BST_XKB_SYMBOL)
    patch = bst_key_table_patch_find (BST_XKB_SYMBOL);
  
  if (!patch && !BST_XKB_FORCE_QUERY && !arg_force_xkb && BST_XKB_SYMBOL)
    {
      BST_IF_DEBUG (KEYTABLE)
	g_message ("Failed to find keytable \"%s\"", BST_XKB_SYMBOL);
    }
  
  if (!patch)
    {
      gchar *xkb_symbol = NULL;
      
      BST_IF_DEBUG (KEYTABLE)
	g_message ("Querying keytable layout from X-Server...");
      
      if (bst_xkb_open (display, TRUE))
	{
	  xkb_symbol = g_strdup (bst_xkb_get_symbol (TRUE));
	  if (!xkb_symbol)
	    xkb_symbol = g_strdup (bst_xkb_get_symbol (FALSE));
	  bst_xkb_close ();
	}
      
      patch = bst_key_table_from_xkb_symbol (xkb_symbol);
      g_free (xkb_symbol);
    }
  
  if (patch)
    {
      BST_IF_DEBUG (KEYTABLE)
	g_message ("Using keytable \"%s\"", patch->identifier);
    }
  else
    {
      gchar *name = BST_DFL_KEYTABLE;
      
      BST_IF_DEBUG (KEYTABLE)
	g_message ("Guessing keytable failed, reverting to \"%s\"", name);
      patch = bst_key_table_patch_find (name);
    }
  
  bst_globals_set_xkb_symbol (patch->identifier);
  
  return patch;
}

/* read bstdefs.h on this */
void
bst_update_can_operate (GtkWidget *widget)
{
  g_return_if_fail (GTK_IS_WIDGET (widget));
  
  /* FIXME, we need to queue a high prioritized idle here as
   * this function can be called multiple times in a row
   */
  
  /* figure toplevel app, and update it
   */
  widget = gtk_widget_get_ancestor (widget, BST_TYPE_APP);
  g_return_if_fail (BST_IS_APP (widget));
  
  bst_app_update_can_operate (BST_APP (widget));
}

/* read bstdefs.h on this */
extern void
bst_object_set (gpointer     object,
		const gchar *first_arg_name,
		...)
{
  va_list args;
  
  g_return_if_fail (GTK_IS_OBJECT (object));
  
  gtk_object_ref (object);
  
  va_start (args, first_arg_name);
  
  if (GNOME_IS_CANVAS_ITEM (object))
    gnome_canvas_item_set_valist (object, first_arg_name, args);
  else
    {
      GSList *arg_list = NULL;
      GSList *info_list = NULL;
      gchar *error;
      
      error = gtk_object_args_collect (GTK_OBJECT_TYPE (object),
				       &arg_list,
				       &info_list,
				       first_arg_name,
				       args);
      
      if (error)
	{
	  g_warning ("bst_object_set(): %s", error);
	  g_free (error);
	}
      else if (arg_list)
	{
	  GSList *arg;
	  GSList *info;
	  
	  for (arg = arg_list, info = info_list; arg; arg = arg->next, info = info->next)
	    gtk_object_arg_set (object, arg->data, info->data);
	  
	  gtk_args_collect_cleanup (arg_list, info_list);
	}
    }
  va_end (args);
  
  BST_OBJECT_ARGS_CHANGED (object);
  
  gtk_object_unref (object);
}
