
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.component;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.text.*;

/**
 * Component that provides a text I/O console with advanced line editing
 * and various other intelligent features.
 * The user enters input one line at a time.  The console will process
 * each line as it is received and produce any corresponding output if
 * required.  Thus interaction with the console runs on a line-in,
 * line-out cycle.  If multiple lines are entered at once (such as if a
 * paste operation is performed), the input will be separated into
 * individual lines and processed and displayed as if the individual
 * lines had been entered in turn.
 * <p>
 * The actual details of the processing required are left to subclasses
 * of <tt>ConsolePane</tt>, which are required to implement the abstract
 * method <tt>processInput()</tt>.  Processing will be done in a
 * separate thread, so if processing is slow then any intermediate output
 * will be displayed to the console as it is produced.  Routine
 * <tt>preProcess()</tt> can also be overridden to do preliminary
 * processing when the console is first started.
 * <p>
 * Before the console is offered to the user for interaction,
 * <tt>startConsole()</tt> <b>must</b> be called.
 * When the console is ready to receive input from the user, it is said
 * to be in <i>input mode</i>.  When it is processing a line of input and
 * producing the corresponding output it is said to be not in input
 * mode.  Once <tt>startConsole()</tt> has been called, the console will be
 * in input mode ready for its first input line.
 * <p>
 * Note that output may be sent to the console before it is started.
 * <p>
 * The console should <b>never</b> be closed while processing is still
 * taking place!  The recommended solution is to catch the frame close
 * event and notify the user with an error if <tt>isProcessing()</tt>
 * indicates that processing is still happening.
 * <p>
 * When processing starts and stops, a property change event will be
 * fired for property <tt>"processing"</tt>.
 *
 * @see #processInput
 * @see #preProcess
 * @see #startConsole()
 * @see #startConsole(java.lang.String)
 * @see #isProcessing
 */
public abstract class ConsolePane extends JTextPane {
    /**
     * The default prompt to present to the user each time input is
     * required.
     */
    public static final String DEFAULT_PROMPT = "> ";

    /**
     * The string of spaces that a tab expands to.
     */
    public static final String TAB_EXPANSION = "    ";

    /**
     * The prompt to present to the user each time input is required.
     */
    private String prompt;

    /**
     * Is input currently enabled?
     */
    private boolean inputEnabled;
    /**
     * Are we currently in input mode?
     */
    private boolean inputting;

    /**
     * The position in the underlying document at which the current
     * input line begins.  This input line does not include any prompt
     * that has been displayed.
     */
    private int inputLinePosition;

    /**
     * The document containing the entire history of this console
     * including any current partial input.
     */
    private ConsoleDocument doc;

    /**
     * A history of all input lines that have been entered, including the
     * input line currently being edited.
     * The earliest input line is at index 0.
     * The input line currently being edited will be copied at various
     * occasions into this history list, so what is stored in this list
     * may not always be up to date with what has been typed.  This is
     * only true of the final entry in the history list.
     */
    private Vector inputHistory;

    /**
     * The line in the input history list upon which the the current working
     * input line has been based.
     */
    private int inputHistoryPosition;

    /**
     * The attributes to be used for displaying user input.
     */
    private AttributeSet inputAttributes;
    /**
     * The attributes to be used for displaying console output.
     */
    private AttributeSet outputAttributes;
    /**
     * The attributes to be used for displaying the console prompt.
     */
    private AttributeSet promptAttributes;

    /**
     * An output stream that writes output to this console.
     */
    private OutputStream outputStream;

    /**
     * The thread currently doing input processing, or <tt>null</tt> if
     * no processing is being done.
     */
    private Thread processor;
    /**
     * A lock to ensure only one thread is doing processing at a time.
     */
    private Object processingLock = new Object();
    /**
     * Has input processing been cancelled by the user?
     */
    private boolean processingCancelled;
    /**
     * A lock to control cancellation of input processing.
     */
    private Object cancelLock = new Object();

    /**
     * Creates a new console.  Note that you <b>must</b> call
     * <tt>startConsole()</tt> before the console is offered to the
     * user for interaction!
     *
     * @see #startConsole()
     * @see #startConsole(java.lang.String)
     */
    public ConsolePane() {
        super();
        this.prompt = DEFAULT_PROMPT;
        inputting = false;
        inputEnabled = false;
        inputHistory = new Vector();
        processor = null;
        processingCancelled = false;

        initAttributes();

        doc = new ConsoleDocument();
        setDocument(doc);
        outputStream = new ConsoleOutputStream();

        Keymap keymap = addKeymap("BTools Console", getKeymap());
        keymap.addActionForKeyStroke(KeyStroke.getKeyStroke(
            KeyEvent.VK_DOWN, 0), new InputHistoryAction(true));
        keymap.addActionForKeyStroke(KeyStroke.getKeyStroke(
            KeyEvent.VK_UP, 0), new InputHistoryAction(false));
        keymap.addActionForKeyStroke(KeyStroke.getKeyStroke(
            KeyEvent.VK_LEFT, 0), new MoveAction(MoveAction.LEFT));
        keymap.addActionForKeyStroke(KeyStroke.getKeyStroke(
            KeyEvent.VK_RIGHT, 0), new MoveAction(MoveAction.RIGHT));
        keymap.addActionForKeyStroke(KeyStroke.getKeyStroke(
            KeyEvent.VK_HOME, 0), new MoveAction(MoveAction.HOME));
        keymap.addActionForKeyStroke(KeyStroke.getKeyStroke(
            KeyEvent.VK_END, 0), new MoveAction(MoveAction.END));
        setKeymap(keymap);
    }

    /**
     * Readies the console for action.  The first prompt will be
     * displayed and the console will be enabled for input and put in input
     * mode.
     * <p>
     * Before any of this is done, <tt>preProcess()</tt> will
     * be called to do any required preliminary processing.
     *
     * @see #preProcess
     */
    public void startConsole() {
        startConsole(null);
    }

    /**
     * Readies the console for action.  The given greeting will be
     * displayed followed by the first prompt.  The console will be
     * enabled for input and put in input mode.  If the given greeting
     * is <tt>null</tt>, no greeting will be displayed.
     * <p>
     * Before any of this is done, <tt>preProcess()</tt> will
     * be called to do any required preliminary processing.
     *
     * @param greeting the greeting to present; this should include a
     * final newline.
     *
     * @see #preProcess
     */
    public void startConsole(String greeting) {
        synchronized(processingLock) {
            processor = new PreProcessingThread(greeting);
            processor.start();
            firePropertyChange("processing", false, true);
        }
    }

    /**
     * Returns the prompt to be presented to the user each time input is
     * required.
     *
     * @return the current prompt.
     */
    public String getPrompt() {
        return prompt;
    }

    /**
     * Sets the prompt to be presented to the user each time input is
     * required.  This will only affect future inputs; prompts that have
     * already been displayed will not be changed.
     * If this routine is not called, a reasonable default will be
     * provided.
     *
     * @param prompt the new prompt to use.
     */
    public void setPrompt(String prompt) {
        this.prompt = prompt;
    }

    /**
     * Returns whether the console is currently enabled for user input.
     * Input will be enabled by default.
     * Disabling input allows the user to still browse through the
     * console history, copy to the clipboard and so on, but no new
     * input can be given.
     *
     * @return <tt>true</tt> if and only if input is enabled.
     */
    public boolean isInputEnabled() {
        return inputEnabled;
    }

    /**
     * Enables or disables input to the console.
     * Disabling input allows the user to still browse through the
     * console history, copy to the clipboard and so on, but no new
     * input can be given.
     *
     * @param enabled <tt>true</tt> to enable input or <tt>false</tt> to
     * disable input.
     */
    public void setInputEnabled(boolean enabled) {
        inputEnabled = enabled;
    }

    /**
     * Returns the attributes used for displaying user input.
     *
     * @return the attributes used to display user input.
     */
    public AttributeSet getConsoleInputAttributes() {
        return inputAttributes;
    }

    /**
     * Sets the attributes to be used for displaying user input.
     * Text that has already been displayed will not be changed.
     * If this routine is not called, a reasonable default is provided.
     *
     * @param set the new attribute set to use.
     */
    public void setConsoleInputAttributes(AttributeSet set) {
        inputAttributes = set;
    }

    /**
     * Returns the attributes used for displaying console output.
     *
     * @return the attributes used to display console output.
     */
    public AttributeSet getConsoleOutputAttributes() {
        return outputAttributes;
    }

    /**
     * Sets the attributes to be used for displaying console output.
     * Text that has already been displayed will not be changed.
     * If this routine is not called, a reasonable default is provided.
     *
     * @param set the new attribute set to use.
     */
    public void setConsoleOutputAttributes(AttributeSet set) {
        outputAttributes = set;
    }

    /**
     * Returns the attributes used for displaying the console prompt.
     *
     * @return the attributes used to display the console prompt.
     */
    public AttributeSet getConsolePromptAttributes() {
        return promptAttributes;
    }

    /**
     * Sets the attributes to be used for displaying the console prompt.
     * Text that has already been displayed will not be changed.
     * If this routine is not called, a reasonable default is provided.
     *
     * @param set the new attribute set to use.
     */
    public void setConsolePromptAttributes(AttributeSet set) {
        promptAttributes = set;
    }

    /**
     * Outputs a blank line to the console.
     * <p>
     * <b>Precondition:</b> The console is not currently in input mode.
     */
    public void outputLine() {
        outputMessage("\n");
    }

    /**
     * Outputs the given line to the console.
     * <p>
     * <b>Precondition:</b> The console is not currently in input mode.
     *
     * @param line the line to output; this should not include a final
     * newline.
     */
    public void outputLine(String line) {
        outputMessage(line + "\n");
    }

    /**
     * Outputs the given message to the console.
     * No addition newline will be appended to the given message.  The
     * caller of this routine is advised to ensure a newline is already
     * present.
     * <p>
     * This is the routine that actually does the work of outputting a
     * message; all other outputting routines end up calling this routine.
     * <p>
     * <b>Precondition:</b> The console is not currently in input mode.
     *
     * @param message the message to output.
     */
    public void outputMessage(String message) {
        try {
            doc.sysInsertString(doc.getLength(), message, outputAttributes);
        } catch (BadLocationException e) {}

        setCaretPosition(doc.getLength());
    }

    /**
     * Erases any text currently on the input line, places the given
     * string there instead and then processes the input as if return
     * had been pressed.
     * <p>
     * <b>Precondition:</b> The console is currently in input mode.<br>
     * <b>Precondition:</b> The given string contains no newlines.
     *
     * @param line the line of input to display and process.
     */
    public void sendInputLine(String line) {
        try {
            if (doc.getLength() > inputLinePosition)
                doc.remove(inputLinePosition,
                    doc.getLength() - inputLinePosition);
            doc.insertString(inputLinePosition, line + "\n",
                inputAttributes);
        } catch (BadLocationException e) {}
    }
    
    /**
     * Processes the given line of input and produces whatever
     * corresponding output is required.
     * Output can be sent to the output stream returned by
     * <tt>getOutputStream()</tt>, in which case it will be sent
     * directly to the console output.  Furthermore, any string returned by
     * this routine will also be sent to the console output.
     * This returned string may be <tt>null</tt>, in which case no extra
     * string will be output.
     * <p>
     * Processing will be done in a separate thread, so if processing is
     * slow then any intermediate output will be displayed to the console
     * as it is produced.
     * <p>
     * If processing is time consuming, this routine should regularly
     * check <tt>isProcessingCancelled()</tt> to see if the user has
     * cancelled the processing.  If so, this routine should exit as
     * soon as possible, possibly with an appropriate error message.
     * If <tt>isProcessingCancelled()</tt> cannot be regularly checked
     * (such as when processing is done in exteral libraries or native
     * routines), see <tt>internalCancelProcessing()</tt> for details on
     * what else can be done to ensure prompt cancellation.
     * <p>
     * This routine is advised to ensure that a final newline is output.
     * <p>
     * <b>Precondition:</b> The console is not currently in input mode.
     *
     * @param line the single line of input to process.
     * @return any additional text to output that has not already been
     * written to the console during this routine.
     *
     * @see #isProcessingCancelled
     * @see #internalCancelProcessing
     * @see #processInput
     */
    protected abstract String processInput(String line);

    /**
     * Does any preliminary processing required before the console is
     * started.  This routine should behave similarly to
     * <tt>processInput()</tt> and will also be run in a separate thread.
     * <p>
     * The default implementation does nothing; subclasses may override
     * this routine to do whatever preliminary processing is necessary.
     * <p>
     * <b>Precondition:</b> The console has not yet been started.
     *
     * @see #processInput
     */
    protected void preProcess() {
    }

    /**
     * Initialises the default attributes used for displaying different
     * types of text.
     */
    private void initAttributes() {
        inputAttributes = new SimpleAttributeSet();
        StyleConstants.setFontFamily((SimpleAttributeSet)inputAttributes,
            "Monospaced");

        outputAttributes = new SimpleAttributeSet(inputAttributes);
        promptAttributes = new SimpleAttributeSet(inputAttributes);

        StyleConstants.setBold((SimpleAttributeSet)outputAttributes, true);
        StyleConstants.setBold((SimpleAttributeSet)promptAttributes, true);
        StyleConstants.setItalic((SimpleAttributeSet)promptAttributes, true);
    }

    /**
     * Puts the console into input mode and puts the given string onto the
     * input line.  The cursor will be placed at the end of this string.
     * The prompt will be displayed at the beginning of the input line.
     * <p>
     * <b>Precondition:</b> The console is not already in input
     * mode.<br>
     * <b>Precondition:</b> The given string contains no newlines.
     *
     * @param preInput the string to place on the input line as if it
     * were input that had already been entered.  This may be
     * <tt>null</tt>, in which case nothing extra will be placed on the
     * input line.
     */
    private void awaitInput(String preInput) {
        try {
            doc.sysInsertString(doc.getLength(), prompt, promptAttributes);
        } catch (BadLocationException e) {}
        
        inputLinePosition = doc.getLength();

        if (preInput != null) {
            try {
                doc.sysInsertString(doc.getLength(), preInput, inputAttributes);
            } catch (BadLocationException e) {}
        }

        inputHistoryPosition = inputHistory.size();
        if (preInput == null)
            inputHistory.addElement("");
        else
            inputHistory.addElement(preInput);

        setCaretPosition(doc.getLength());
        inputting = true;
    }

    /**
     * Returns an output stream that writes output to the console.
     * Calling this routine multiple times will return the same output
     * stream.
     * <p>
     * <b>Precondition:</b> This output stream is not used while the
     * console is in input mode.
     *
     * @return an output stream associated with this console.
     */
    public OutputStream getOutputStream() {
        return outputStream;
    }

    /**
     * Notifies the user that an illegal action has been attempted.
     */
    public void oops() {
        Toolkit.getDefaultToolkit().beep();
    }

    /**
     * Writes the entire contents of this console, up to but not
     * including the current input prompt, to the given file.
     *
     * @param file the file to which to write.
     * @throws IOException thrown if an I/O error occurs, in which case
     * this routine will attempt to close the file before throwing the
     * exception.
     */
    public void writeContentsToFile(File file) throws IOException {
        FileWriter w = new FileWriter(file);

        try {
            if (inputting)
                w.write(getText(), 0, inputLinePosition - prompt.length());
            else
                w.write(getText(), 0, doc.getLength());
        } catch (IOException exc) {
            try {
                w.close();
            } catch (Throwable th) {}
            throw exc;
        }

        w.close();
    }

    /**
     * Is this console currently processing user input?
     * When processing starts and stops, a property change event will be
     * fired for property <tt>"processing"</tt>.
     *
     * @return <tt>true</tt> if and only if this console is currently
     * processing user input.
     */
    public final boolean isProcessing() {
        return (processor != null);
    }

    /**
     * Causes this thread to wait until input processing is finished.
     * If this console is not currently processing user input, this
     * routine returns immediately.
     */
    public final void waitForProcessing() {
        synchronized(processingLock) {
            return;
        }
    }

    /**
     * Cancel any current input processing that is taking place.
     * If no processing is taking place, this routine will do nothing.
     * Otherwise the processing will be cancelled and a new input prompt
     * will be presented.
     *
     * #see isProcessingCancelled
     */
    public final void cancelProcessing() {
        synchronized(cancelLock) {
            if (processor != null) {
                processingCancelled = true;
                internalCancelProcessing(processor);
            }
        }
    }

    /**
     * Has the input processing been cancelled?  Note that processing is
     * cancelled through the use of <tt>cancelProcessing()</tt>.
     *
     * @return <tt>true</tt> if and only if input processing is still
     * taking place and the user has requested that the processing be
     * cancelled.
     *
     * @see #cancelProcessing
     */
    public final boolean isProcessingCancelled() {
        return processingCancelled;
    }

    /**
     * Calls any custom code needed to cancel input processing.
     * This will be called from inside <tt>cancelProcessing()</tt> if
     * there is any processing actually taking place.
     * <p>
     * Generally this routine should do nothing; the processing routine
     * should instead call <tt>isProcessingCancelled()</tt> regularly
     * and exit if appropriate.  However, if large amounts of processing
     * are done in external libraries or native routines, you will need
     * some way to ask these external routines to stop.  This can be done
     * by overriding <tt>internalCancelProcessing()</tt> to take whatever
     * additional action is necessary.
     * <p>
     * The default implementation of this routine does nothing, and in
     * most cases that should be sufficient.
     *
     * @param t the thread in which input processing is taking place.
     * This thread should <b>not</b> simply be killed; it does
     * important cleaning up for the console after processing is done.
     */
    protected void internalCancelProcessing(Thread t) {
    }

    /**
     * A thread class that does preprocessing before the console is
     * started.
     */
    private class PreProcessingThread extends Thread {
        /**
         * The greeting to display immediately before the first prompt.
         */
        private String greeting;

        /**
         * Creates a new preprocessing thread.
         *
         * @param greeting the greeting to display immediately before
         * the first prompt; this should include a final newline.  If
         * nothing should be displayed then this may be <tt>null</tt>.
         */
        public PreProcessingThread(String greeting) {
            super("Console preprocessing");
            this.greeting = greeting;
        }

        /**
         * Does preliminary processing for the console.  At the end of
         * this routine the given greeting will be presented to the user
         * and the console will be placed in input mode.
         * <p>
         * <b>Precondition:</b> The console is currently in the process
         * of being started.
         */
        public void run() {
            synchronized(processingLock) {
                try {
                    preProcess();
                } catch (Throwable th) {}
                
                if (greeting != null)
                    outputMessage(greeting);

                inputEnabled = true;
                awaitInput(null);

                synchronized(cancelLock) {
                    processingCancelled = false;
                    processor = null;
                }

                firePropertyChange("processing", true, false);
            }
        }
    }

    /**
     * A thread class that processes a set of input lines.
     */
    private class ProcessingThread extends Thread {
        /**
         * The lines of input to process.
         */
        private Vector lines;
        /**
         * The string to offer the user at the subsequent input prompt.
         */
        private String extra;

        /**
         * Creates a new processing thread.
         *
         * @param lines a vector containing the lines of input to
         * process.  These should not contain newlines.  If there is
         * text already on the input line, this will be prepended to the
         * first line of input before it is processed.
         * @param extra a string to offer the user at the subsequent
         * input prompt after all lines have been processed.  If nothing
         * should be offered then this may be <tt>null</tt>.
         */
        public ProcessingThread(Vector lines, String extra) {
            super("Console input processing");
            this.lines = lines;
            this.extra = extra;
        }

        /**
         * Processes the lines of input.  At the end of this routine the
         * given extra string will be offered to the user at the
         * subsequent input prompt and the console will be placed in
         * input mode.
         * <p>
         * <b>Precondition:</b> The console is currently not in input
         * mode.
         */
        public void run() {
            synchronized(processingLock) {
                // Delete the working line from input history; we'll put
                // this back when we're finished.
                inputHistory.removeElementAt(inputHistory.size() - 1);

                Enumeration e = lines.elements();
                String line, prefix, ans;
                boolean first = true;
                while (e.hasMoreElements() && ! processingCancelled) {
                    line = (String)e.nextElement();
                    prefix = null;
                    if (first) {
                        if (inputLinePosition < doc.getLength()) {
                            try {
                                prefix = doc.getText(inputLinePosition,
                                    doc.getLength() - inputLinePosition);
                            } catch (BadLocationException exc) {
                                prefix = null;
                            }
                        }
                        first = false;
                    } else {
                        try {
                            doc.sysInsertString(doc.getLength(), prompt,
                                promptAttributes);
                        } catch (BadLocationException exc) {}
                    }

                    // Insert required text.
                    try {
                        doc.sysInsertString(doc.getLength(), line + '\n',
                            inputAttributes);
                    } catch (BadLocationException exc) {}
                    setCaretPosition(doc.getLength());

                    // Add the command to history and process it.
                    if (prefix != null)
                        line = prefix + line;
                    inputHistory.addElement(line);

                    ans = null;
                    try {
                        ans = processInput(line);
                    } catch (Throwable th) {}
                    if (ans != null)
                        outputMessage(ans);
                }

                if (processingCancelled)
                    extra = "";
                awaitInput(extra);

                synchronized(cancelLock) {
                    processingCancelled = false;
                    processor = null;
                }

                firePropertyChange("processing", true, false);
            }
        }
    }

    /**
     * An output stream that writes directly to this console's output.
     * Such a stream should only be used when the console is not in input
     * mode.
     */
    private class ConsoleOutputStream extends OutputStream {
        /**
         * Creates a new output stream linked to this console.
         */
        public ConsoleOutputStream() {
            super();
        }
        /**
         * Writes the given character to this output stream.
         */
        public void write(int b) {
            outputMessage(String.valueOf((char)b));
        }
        /**
         * Writes the given array of characters to this output stream.
         */
        public void write(byte[] b) {
            outputMessage(new String(b));
        }
        /**
         * Writes the given subarray of the given array of characters to
         * this output stream.
         */
        public void write(byte[] b, int offset, int length) {
            outputMessage(new String(b, offset, length));
        }
    }

    /**
     * A document storing the entire console history as well as any
     * current partial input line.  This document censors any editing
     * actions to ensure editing only takes place when and where it is
     * permissible.
     */
    private class ConsoleDocument extends DefaultStyledDocument {
        /**
         * Creates a new document.
         */
        public ConsoleDocument() {
            super();
        }
        /**
         * Inserts the given string into this document, assuming this
         * string has been supplied by the console, not the user.
         */
        public void sysInsertString(int offset, String string,
                AttributeSet attr) throws BadLocationException {
            super.insertString(offset, string, attr);
        }
        /**
         * Inserts the given string into this document, treating the
         * string as if it were input supplied by the user.
         */
        public void insertString(int offset, String string, AttributeSet attr)
                throws BadLocationException {
            if (! (inputEnabled && inputting)) {
                oops();
                return;
            }

            synchronized(processingLock) {
                // This must be an input string so make sure it's inserted
                // on the input line.
                if (offset < inputLinePosition) {
                    offset = getLength();
                    setCaretPosition(getLength());
                }

                // Replace all the tabs with strings of spaces.
                int pos;
                int used = 0;
                StringBuffer realString = new StringBuffer();
                pos = string.indexOf('\t');
                while (pos >= 0) {
                    realString.append(string.substring(used, pos - used));
                    realString.append(TAB_EXPANSION);
                    used = pos + 1;
                    pos = string.indexOf('\t', used);
                }
                realString.append(string.substring(used));
                string = realString.toString();

                // If there's no newline, there's nothing to process.
                if (string.indexOf('\n') < 0) {
                    super.insertString(offset, string, inputAttributes);
                    return;
                }

                // We have one or more complete commands.
                inputting = false;

                // If we're in the middle of the input line, whatever comes
                // after offset must be presented again on the input line after
                // processing is done.
                String epilogue;
                if (offset == getLength())
                    epilogue = null;
                else {
                    epilogue = getText(offset, doc.getLength() - offset);
                    remove(offset, epilogue.length());
                }

                // Build up a list of given commands.
                Vector commands = new Vector();
                while ((pos = string.indexOf('\n')) >= 0) {
                    commands.addElement(string.substring(0, pos));
                    string = string.substring(pos + 1);
                }

                // Reappend the epilogue if necessary.
                if (epilogue != null)
                    string = string + epilogue;

                // Process these lines of input.
                processor = new ProcessingThread(commands, string);
                processor.start();
                firePropertyChange("processing", false, true);
            }
        }
        /**
         * Removes the given portion of this document.
         */
        public void remove(int offset, int length)
                throws BadLocationException {
            if (! inputEnabled)
                oops();
            else if (offset >= inputLinePosition)
                super.remove(offset, length);
            else if (length > inputLinePosition - offset)
                super.remove(inputLinePosition, length -
                    (inputLinePosition - offset));
            else
                oops();
        }
        /**
         * Sets the attributes of the given portion of this document.
         */
        public void setCharacterAttributes(int offset, int length,
                AttributeSet attr, boolean replace) {
            try {
                if (inputting) {
                    oops(); return;
                }
            } catch (Throwable t) {}
            super.setCharacterAttributes(offset, length, attr, replace);
        }
        /**
         * Sets the logical style at the given position.
         */
        public void setLogicalStyle(int pos, Style s) {
            try {
                if (inputting) {
                    oops(); return;
                }
            } catch (Throwable t) {}
            super.setLogicalStyle(pos, s);
        }
        /**
         * Sets the paragraph attributes of the given portion of this
         * document.
         */
        public void setParagraphAttributes(int offset, int length,
                AttributeSet attr, boolean replace) {
            try {
                if (inputting) {
                    oops(); return;
                }
            } catch (Throwable t) {}
            super.setParagraphAttributes(offset, length, attr, replace);
        }
    }

    /**
     * An action that represents a cursor movement.
     */
    private class MoveAction extends AbstractAction {
        /**
         * A specific type of cursor movement.
         */
        public static final int LEFT = 1;
        /**
         * A specific type of cursor movement.
         */
        public static final int RIGHT = 2;
        /**
         * A specific type of cursor movement.
         */
        public static final int HOME = 3;
        /**
         * A specific type of cursor movement.
         */
        public static final int END = 4;

        /**
         * Which type of movement does this action represent?
         */
        private int type;

        /**
         * Creates a new movement action.
         *
         * @param type the type of movement this action represents; this
         * must be one of the movement type constants defined in this
         * class.
         */
        public MoveAction(int type) {
            super("Move cursor " + 
                (type == LEFT ? "left" : type == RIGHT ? "right" :
                type == HOME ? "home" : type == END ? "end" : ""));
            this.type = type;
        }

        /**
         * Perform this action.
         */
        public void actionPerformed(ActionEvent e) {
            int pos;
            switch (type) {
                case LEFT:
                    pos = getCaretPosition();
                    if (pos == inputLinePosition || pos == 0)
                        oops();
                    else
                        setCaretPosition(pos - 1);
                    return;
                case RIGHT:
                    pos = getCaretPosition();
                    if (pos == doc.getLength())
                        oops();
                    else
                        setCaretPosition(pos + 1);
                    return;
                case HOME:
                    setCaretPosition(inputLinePosition);
                    return;
                case END:
                    setCaretPosition(doc.getLength());
                    return;
            }
        }
    }

    /**
     * An action that represents browsing through the input history
     * list.
     */
    private class InputHistoryAction extends AbstractAction {
        /**
         * Does this action browse forwards or backwards through input
         * history?
         */
        private boolean forwards;

        /**
         * Creates a new input history action.
         *
         * @param forwards <tt>true</tt> to browse forwards through
         * input history or <tt>false</tt> to browse backwards.
         */
        public InputHistoryAction(boolean forwards) {
            super("Browse input history " +
                (forwards ? "forwards" : "backwards"));
            this.forwards = forwards;
        }

        /**
         * Perform this action.
         */
        public void actionPerformed(ActionEvent e) {
            if (! (inputting && inputEnabled)) {
                oops();
                return;
            }

            // Check that we can actually browse further in the
            // requested direction.
            if (forwards) {
                if (inputHistoryPosition == inputHistory.size() - 1) {
                    oops();
                    return;
                }
            } else {
                if (inputHistoryPosition == 0) {
                    oops();
                    return;
                }
            }

            // Store the current input line in the highest vector
            // position if appropriate.
            if (inputHistoryPosition == inputHistory.size() - 1)
                try {
                    inputHistory.setElementAt(doc.getText(inputLinePosition,
                        doc.getLength() - inputLinePosition),
                        inputHistoryPosition);
                } catch (BadLocationException exc) {}

            // Offer the appropriate line from history.
            if (forwards)
                inputHistoryPosition++;
            else
                inputHistoryPosition--;

            try {
                if (doc.getLength() > inputLinePosition)
                    doc.remove(inputLinePosition, doc.getLength() -
                        inputLinePosition);
                doc.insertString(inputLinePosition,
                    (String)inputHistory.elementAt(inputHistoryPosition),
                    inputAttributes);
            } catch (BadLocationException exc) {}

            // Set the new cursor position.
            setCaretPosition(doc.getLength());
        }
    }
}

