/*
 *
 * (c) Vladi Belperchinov-Shabanski "Cade" <cade@biscom.net> 1998-1999
 *
 * SEE `README',LICENSE' OR COPYING' FILE FOR LICENSE AND OTHER DETAILS!
 *
 */
 
/*

  This is general purpose array classes.
  `CLUSTERS' name replaces boring `Arrays' name :)
  It really doesn't make much difference however...

*/

#ifndef _CLUSTERS_H_
#define _CLUSTERS_H_

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <assert.h>
#ifndef ASSERT
#define ASSERT assert
#endif

#ifndef int32
#define int32 int
#endif

#define CE_OK    0   // ok -- no error
#define CE_MEM   1   // low memory
#define CE_OVR   128 // overflow
#define CE_ERR   255 // error

//////////////////////////////////////////////////////////////////////////
//
// class FREE-LIST cluster
//

  class FLCluster
  {
    int32 es;      // element size
    int32 ds;      // data size
    int32 size;    // elements allocated
    int32 used;    // elements used == count
    int32 ff;      // first free element
    int32 growby;  // `grow by' elements
    char *data;    // actual data

    public:

    int base;
    int null;

    FLCluster();
    ~FLCluster();

    int create( int32 pcnt, int32 pgrow, int32 pes ); // pes=element size, initial count & `grow by'
    void done();

    int32 add( void* pe ); // insert element, return handle or -1
    int del( int32 pn ); // delete element (memset(`+') & mark it `free'), ret E_OK or error
    int get( int32 pn, void* pe ); // get element, return E_OK or error
    char* get( int32 pn ); // get element pointer or NULL if error

    int is_used( int32 pn ); // 1 if pn element is used

    void dump(); // only for debugging

    protected:
    int Realloc( int32 pn ); // expand/shrink data list, return CE_OK, or error

  };

//////////////////////////////////////////////////////////////////////////
//
// Base Cluster prototype
//

class BaseCluster
  {
    protected:

    int es;
    int size;
    int cnt;
    int bsize;
    char *data;

    int status;

    void (*destructor_func)( void* );

    public:

    BaseCluster() { data = NULL; es = 0; size = 0; cnt = 0; bsize = 0; destructor_func = NULL; };
    virtual ~BaseCluster() { if (data) done(); };

    int create( int p_size, int p_bsize, int p_es );
    void done();

    void del( int pn );
    void delall();
    void free( int pn );
    void freeall();

    int count() { return cnt; };

    void shrink() { Realloc( cnt ); };

    void set_destructor( void (*dfunc)(void*) ) { destructor_func = dfunc; };

    protected:

    virtual void destroy_element( void* pe )
      { if ( destructor_func != NULL ) destructor_func( pe ); };
    int Realloc( int pn );

  };

////////////////////////////////////////////////////////////////////////////
//
// Data Cluster
//

class DCluster : public BaseCluster
  {
    public:
    int add( void* pe );
    int ins( int pn, void* pe );
    void put( int pn, void* pe );
    void get( int pn, void* pe );

    const void* operator [] ( int pn ) { ASSERT( pn >= 0 && pn < cnt  ); return data + (pn*es); };

  };

////////////////////////////////////////////////////////////////////////////
//
// Template Data Cluster
//

template< class T >
class TDCluster : public DCluster
  {
    public:

    const T* operator [] ( int pn ) const { ASSERT( pn >= 0 && pn < cnt  ); return (T*)(data + (pn*es)); };
    T operator [] ( int pn ) { ASSERT( pn >= 0 && pn < cnt  ); return (*((T*)(data + (pn*es)))); };

  };

////////////////////////////////////////////////////////////////////////////
//
// Pointer Cluster
//

class PCluster : public BaseCluster
  {
    public:
    int create( int p_size, int p_bsize )
      { return BaseCluster::create( p_size, p_bsize, sizeof(void*) ); };

    int add( void* pe );
    int ins( int pn, void* pe );
    void put( int pn, void* pe );
    void* get( int pn );

    void* operator [] ( int pn ) { ASSERT( pn >= 0 && pn < cnt  ); return ((void**)data)[pn]; };

    protected:
    virtual void destroy_element( void* pe )
      { if ( destructor_func != NULL ) destructor_func( ((void**)pe)[0] ); };
  };

////////////////////////////////////////////////////////////////////////////
//
// Template Pointer Cluster
//

template< class T >
class TPCluster : PCluster
  {
    public:
    int create( int p_size, int p_bsize )
      { return BaseCluster::create( p_size, p_bsize, sizeof(void*) ); };

    T* get( int pn ) {};
    T* operator [] ( int pn ) { ASSERT( pn >= 0 && pn < cnt  ); return ((T**)data)[pn]; };

    protected:
    virtual void destroy_element( void* pe )
      { delete ((T**)pe)[0]; };
  };

////////////////////////////////////////////////////////////////////////////
//
//  Template Cluster
//

template< class T >
class TCluster
  {
    int cnt;
    int size;
    int bsize;
    T   *data;

    public:

    TCluster() { data = NULL; cnt = 0; size = 0; bsize = 0; };
    ~TCluster() { if (data) done(); };

    void create( int p_size, int p_bsize ) { bsize = p_bsize; Realloc( p_size ); };
    void done() { if (data) delete [] data; data = NULL; cnt = 0; size = 0; bsize = 0; };

    long count() const { return cnt; }
    void Realloc( int pn )
      {
        if( pn == 0 )
          {
          if( data ) delete [] data;
          size = 0;
          cnt = 0;
          data = NULL;
          return;
          }

        T *newdata = new T[pn];

        int less =  pn < cnt ? pn : cnt;
        int z;
        for( z = 0; z < less; z++ )  newdata[z] = data[z];
        if( data )  delete [] data;

        cnt = less;
        size = pn;
        data   = newdata;
      };
    void add( T &e )
      {
      if ( cnt == size ) Realloc( size + bsize );
      data[cnt] = e;
      cnt++;
      };
    T& operator [] ( int pn )
      { ASSERT( pn >= 0 && pn < cnt  ); return data[pn]; };
    const T& operator [] ( int pn ) const
      { ASSERT( pn >= 0 && pn < cnt  ); return data[pn]; };
};

////////////////////////////////////////////////////////////////////////////
//
// BSCluster -- bit set cluster
//

class BSCluster
  {
    int size;  // size (in bits)
    int datasize; // size (in bytes)
    char *data;

    public:

    BSCluster() { data = NULL; size = 0; datasize = 0; };
    ~BSCluster() { if (data) done(); };

    int create( int p_size ) { return Resize( p_size ); };
    void done() { if (data) free( data ); data = NULL; size = 0; };

    void set1( int pn );
    void set0( int pn );
    void set( int pn, int val );
    int get( int pn );
    void setall1() { if (data) memset( data, 0xff, datasize ); };
    void setall0() { if (data) memset( data, 0x00, datasize ); };
    void reverse();

    const int operator [] ( int pn ) const { ASSERT( pn >= 0 && pn < size  ); return (data[pn / 8] & (1 << (pn % 8))) != 0; };

    BSCluster& operator  = ( const BSCluster &b1 );
    BSCluster& operator &= ( const BSCluster &b1 );
    BSCluster& operator |= ( const BSCluster &b1 );
    BSCluster  operator ~  ();

    friend BSCluster operator & ( const BSCluster &b1, const BSCluster &b2 );
    friend BSCluster operator | ( const BSCluster &b1, const BSCluster &b2 );

    int Resize( int p_size );

  };

////////////////////////////////////////////////////////////////////////////
//
//
//
  #ifndef SIZE_T_MAX
  #define SIZE_T_MAX    INT_MAX
  #endif

  class PSZCluster : public PCluster
    {

    size_t max_str_len;
    size_t min_str_len;

    public:
    PSZCluster() : PCluster() { max_str_len = 0; min_str_len = SIZE_T_MAX; };

    size_t maxlen() { return max_str_len; };

    int add( const char* pe );
    int ins( int pn, const char* pe );
    void put( int pn, const char* pe );
    void get( int pn, char* pe );
    char* get( int pn );

    void freeall() { max_str_len = min_str_len = 0; BaseCluster::freeall(); };

    char* operator [] ( int pn ) { ASSERT( pn >= 0 && pn < cnt  ); return (char*)((void**)data)[pn]; };

    protected:
    virtual void destroy_element( void* pe )
      { delete (char*)(((void**)pe)[0]); };

    };

////////////////////////////////////////////////////////////////////////////
//
//
//

////////////////////////////////////////////////////////////////////////////
//
// misc support functions
//

int SaveToFile( FILE *f, PSZCluster *szc );
int SaveToFile( const char *fname, PSZCluster *szc );
int LoadFromFile( FILE *f, PSZCluster *szc, size_t maxlen );
int LoadFromFile( const char *fname, PSZCluster *szc, size_t maxlen );


////////////////////////////////////////////////////////////////////////////

#endif //_CLUSTERS_H_
