/* # skkinput (Simple Kana-Kanji Input)
 * skkwin.c --- Draw skkinput window.
 * This file is part of skkinput.
 * Copyright (C) 1997
 * Takashi SAKAMOTO (sakamoto@yajima.kuis.kyoto-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

enum {
  MODELINE_NOTHING = 0, MODELINE_HANKAKU, MODELINE_ZENKAKU,
  MODELINE_KATAKANA, MODELINE_KANA,
} ;

static void skkwin_jdiffputstring
( Widget gw, int x, int y, int width, char *text, char *ptext,
  int roman_width, int kana_width ) ;
static void skkinput_FullDrawWindow
( Widget gw, struct SKKInputNode *node, char *text, 
  int start_pos, int x, int y, int end_x, int end_y ) ;

/*
 * skkinput λꤵ줿˥饯ɽؿ
 *----
 * flag ϽŤ͹碌Ԥɤɽ˴ط롣
 * rf ȿžɽ򤹤뤫ɤǤ롣
 */
static void skkwin_jputchar
( Widget gw, int x, int y, int chara, int flag, int rf )
{
  char buffer[ 3 ] ;
  SkkInputWidget w = ( SkkInputWidget )gw ;

  if( chara > 0x100 ){
    buffer[ 0 ] = ( chara & 0x07F00 ) >> 8 ;
    buffer[ 1 ] = ( chara & 0x0007F ) ;
    buffer[ 2 ] = '\0' ;
    /* ʸɽ*/
    if( flag ){
      if( rf ){
	XDrawString16
	  ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_rgc,
	    x, y, ( XChar2b *)buffer, 1 ) ;
      } else {
	XDrawString16
	  ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_ngc,
	    x, y, ( XChar2b *)buffer, 1 ) ;
      }
    } else {
      if( rf ){
	XDrawImageString16
	  ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_rgc,    
	    x, y, ( XChar2b *)buffer, 1 ) ;
      } else {
	XDrawImageString16
	  ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_ngc,
	    x, y, ( XChar2b *)buffer, 1 ) ;
      }
    }
  } else {
    buffer[ 0 ] = chara ;
    buffer[ 1 ] = '\0' ;
    if( flag ){
      if( rf ){
	XDrawString
	  ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
	    x, y, buffer, 1 ) ;
      } else {
	XDrawString
	  ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
	    x, y, buffer, 1 ) ;
      }
    } else {
      if( rf ){
	XDrawImageString
	  ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
	    x, y, buffer, 1 ) ;
      } else {
	XDrawImageString
	  ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
	    x, y, buffer, 1 ) ;
      }
    }
  }
  return ;
}

/*
 * ɽؿ
 *----
 * int kf : ե饰EUC ɤʤ True ˤʤ롣
 */
static void skkinputDrawCursor
( Widget gw, int x, int y, int kf )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;

  /* ɽ롣*/
  if( kf ){
    if( w->skkinput.is_focus ){
      XFillRectangle
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_ngc,
	  x, y - w->skkinput.font_ascent,
	  w->skkinput.kanji_width - 1, w->skkinput.font_height ) ;
    } else {
      XFillRectangle
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_rgc,
	  x, y - w->skkinput.font_ascent,
	  w->skkinput.kanji_width - 1, w->skkinput.font_height ) ;
      XDrawRectangle
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_ngc,
	  x, y - w->skkinput.font_ascent,
	  w->skkinput.kanji_width - 1, w->skkinput.font_height - 1 ) ;
    }
  } else {
    if( w->skkinput.is_focus ){
      XFillRectangle
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
	  x, y - w->skkinput.font_ascent,
	  w->skkinput.roman_width - 1, w->skkinput.font_height ) ;
    } else {
      XFillRectangle
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
	  x, y - w->skkinput.font_ascent,
	  w->skkinput.roman_width - 1, w->skkinput.font_height ) ;
      XDrawRectangle
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
	  x, y - w->skkinput.font_ascent,
	  w->skkinput.roman_width - 1, w->skkinput.font_height - 1 ) ;
    }
  }
  return ;
}

/*
 * skk window ʸõؿ
 *----
 * x, y Ϻɸkf ϴե٥åȤɤȽ̤Ѥ롣
 */
void skkwin_jerase( Widget gw, int x, int y, int kf )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  if( kf ){
    XFillRectangle
      ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
       x, y - w->skkinput.font_ascent,
       w->skkinput.kanji_width, w->skkinput.font_height ) ;
  } else {
    XFillRectangle
      ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
       x, y - w->skkinput.font_ascent,
       w->skkinput.roman_width, w->skkinput.font_height ) ;
  }
  return ;
}

/*
 * skkinput λꤵ줿˥饯ɽؿ
 *----
 * flag ϽŤ͹碌Ԥɤɽ˴ط롣
 */
static void skkwin_jputchar2
( Widget gw, int x, int y, int chara, int flag, int wflag )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  char buffer[ 3 ] ;
  int kanji_flag ;

  if( chara > 0x100 ){
    buffer[ 0 ] = ( chara & 0x07F00 ) >> 8 ;
    buffer[ 1 ] = ( chara & 0x0007F ) ;
    buffer[ 2 ] = '\0' ;
    kanji_flag  = True ;
  } else {
    buffer[ 0 ] = chara ;
    buffer[ 1 ] = '\0' ;
    kanji_flag  = False ;
  }
  if( flag ){
    if( kanji_flag ){
      XDrawString16
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_rgc,
	  x, y, ( XChar2b *)buffer, 1 ) ;
    } else {
      XDrawString
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
	  x, y, buffer, 1 ) ;
    }
  } else {
    /* ʸɽ*/
    skkwin_jerase( gw, x, y, kanji_flag ) ;
    if( wflag )
      return ;
    if( kanji_flag ){
      XDrawImageString16
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.kanji_ngc,
	  x, y, ( XChar2b *)buffer, 1 ) ;
    } else {
      XDrawImageString
	( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
	  x, y, buffer, 1 ) ;
    }
  }
  return ;
}

static char *skkinput_WinGetNextLineTop
( char *ptr, int win_width, int roman_width, int kana_width )
{
  int kana_flag = 0, x = 0 ;

  while( *ptr != '\0' ){
    if( 0x80 & *ptr ){
      kana_flag = kana_flag % 2 + 1 ;
    } else {
      kana_flag = 0 ;
    }
    switch( kana_flag ){
      /* ɤν*/
    case 0 :
      x += roman_width ;
      if( x >= win_width )
	return ptr ;
      break ;
      /* ɤΰХܤν*/
    case 1 :
      x += kana_width ;
      if( x >= win_width )
	return ptr ;
      break ;
      /* ɤХܤν*/
    case 2 :
    default :
      break ;
    }
    ptr ++ ;
  }
  return ptr ;
}

/*
 * ʬԤäƤʬΥᥤʬĤδؿ
 *---------------
 * ɽ줿ʸȺɽʤȤʤʸ򤸤äȸ٤
 * ɤʡ ʸƱɽʤǺѤ͡ ȤԤ
 * ä顢ѹϤ͡ Ȥġ
 */
static char *skkinput_DiffDrawWindowSub1
( Widget gw, struct SKKInputNode *node, char *text, char *ptext,
  int start_pos, int *retx, int *rety, int end_x, int end_y,
  int *rcx, int *rcy, int *retpos )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  int x, y, pos, cx, cy, chara ;
  int charas_are_different, dkana_flag, skana_flag, diff_flag ;
  char *dptr, *sptr, *linetop ;

  /* ե饰ν*/
  diff_flag = skana_flag = dkana_flag = False ;

  /* ɽ֤ꡣ*/
  pos = start_pos ;
  x = *retx ;
  y = *rety ;
  chara = 0 ;
  cx = cy = -1 ;

  /* ʸꡣ*/
  linetop = sptr = ptext ;
  dptr = text ;

  /* ĤʸӤ롣*/
  while( *dptr != '\0' ){
    /* ɤɤɤʤ餽βХܤ*/
    if( 0x80 & *dptr ){
      dkana_flag = dkana_flag % 2 + 1 ;
    } else {
      dkana_flag = 0 ;
    }
    /* Ԥ뤫ɤȽꡣ*/
    switch( dkana_flag ){
      /* ɤν*/
    case 0 :
      /* Ԥޤ֤Υå*/
      if( ( x + w->skkinput.roman_width ) >= end_x ){
	/* 뤬ɽƤ顢äѤõ*/
	if( x == node->prev_cx && y == node->prev_cy )
	  skkwin_jerase( gw, x, y, True ) ;
	/* ޤ֤򼨤 backslash ɽ롣*/
	skkwin_jputchar( gw, x, y, '\\', False, False ) ;
	x = 0 ;
	y += w->skkinput.font_height ;
	linetop = sptr = skkinput_WinGetNextLineTop
	  ( linetop, end_x, w->skkinput.roman_width,
	    w->skkinput.kanji_width ) ;
	/* ǲԤ뤳ȤϤʤΤǡǤ롣*/
	skana_flag = 0 ;
	/* ιԿĶƤޤäʤ顢ȴ롣*/
	if( y >= end_y )
	  goto exit_loop ;
      }
      break ;
      /* ɤΰХܤν*/
    case 1 :
      if( ( x + w->skkinput.kanji_width ) >= end_x ){
	/* 뤬ɽƤ顢äѤõ*/
	if( x >= node->prev_cx &&
	    ( node->prev_cx <= ( x + w->skkinput.kanji_width ) ) &&
	    y == node->prev_cy )
	  skkwin_jerase( gw, x, y, True ) ;
	skkwin_jputchar
	  ( gw, x, y, '\\', False, False ) ;
	skkwin_jputchar
	  ( gw, x + w->skkinput.roman_width, y, '\\', False, False ) ;
#if 0
	skkwin_jerase( gw, x, y, True ) ;
#endif
	x = 0 ;
	y += w->skkinput.font_height ;
	/* ӤƤɽ줿ʸԤ줿֤õơ*
	 * إݥ󥿤ư롣*/
	linetop = sptr = skkinput_WinGetNextLineTop
	  ( linetop, end_x, w->skkinput.roman_width,
	    w->skkinput.kanji_width ) ;
	/* ǲԤ뤳ȤϤʤΤǡǤ롣*/
	skana_flag = 0 ;
	/* ιԿĶƤޤäʤ顢ȴ롣*/
	if( y >= end_y )
	  goto exit_loop ;
      }
      break ;
      /* ɤХܤν*/
    case 2 :
    default :
      break ;
    }
    /* ɽ줿ȽԤ*/
    if( 0x80 & *sptr ){
      skana_flag = skana_flag % 2 + 1 ;
    } else {
      skana_flag = 0 ;
    }
    /* ơ饯ϰۤʤΤǤ礦 */
    if( *sptr != *dptr || pos == node->cur_pos ||
	( x == node->prev_cx && y == node->prev_cy ) ||
	skana_flag != dkana_flag ){
      /* ࡣ֤ϥǥեȤǰۤʤΤʡ */
      charas_are_different = True ;
    } else {
      charas_are_different = False ;
    }
    switch( dkana_flag ){
      /* ɤν*/
    case 0 :
      /* ꤬˲ԤƤޤäƤˤ̾ɽ*/
      /* ǤϤʤơ饯Ӥưۤʤɽ*/
      if( charas_are_different ){
	if( pos == node->cur_pos ){
	  if( node->cur_exist ){
	    skkinputDrawCursor( gw, x, y, False ) ;
	    skkwin_jputchar
	      ( gw, x, y, *dptr, True, w->skkinput.is_focus ) ;
	  } else {
	    skkwin_jputchar2( gw, x, y, *dptr, False, False ) ;
	  }
	  cx = x ; cy = y ;
	} else {
	  skkwin_jputchar2( gw, x, y, *dptr, False, False ) ;
	}
      }
      /* ɽ֤ΰư*/
      x += w->skkinput.roman_width ;
      break ;
      /* ɤΰХܤν*/
    case 1 :
      chara = ( unsigned char )*dptr ;
      /* ä˲ԤƤޤäƤˤԤġ*/
      /* ɤΰХܤʸ󤬰ۤʤäƤ硣*/
      if( charas_are_different )
	diff_flag = True ;
      break ;
      /* ɤХܤν*/
    case 2 :
      chara = ( chara << 8 ) | ( ( unsigned char )*dptr ) ;
      /* ɤХܤʸ󤬰ۤʤäƤڤӡХ *
       * ܤǰۤʤäƤˤϽľκȤ롣                 */
      if( charas_are_different || diff_flag ){
	if( pos == ( node->cur_pos + 1 ) ){
	  if( node->cur_exist ){
	    skkinputDrawCursor( gw, x, y, True ) ;
	    skkwin_jputchar
	      ( gw, x, y, chara, True, w->skkinput.is_focus ) ;
	  } else {
	    skkwin_jputchar2( gw, x, y, chara, False, False ) ;
	  }
	  cx = x ; cy = y ;
	} else {
	  skkwin_jputchar2( gw, x, y, chara, False, False ) ;
	}
	diff_flag = False ;
      }
      /* ɽ֤ΰư*/
      x += w->skkinput.kanji_width ;
      break ;
    default :
      dkana_flag = 0 ;
      break ;
    }
    dptr ++ ;
    if( *sptr != '\0' )
      sptr ++ ;
    pos  ++ ;
  }
exit_loop:
  /* ɽη̥뤬ޤǰưΤ֤*/
  *retx   = x ;
  *rety   = y ;
  *rcx    = cx ;
  *rcy    = cy ;
  *retpos = pos ;
  return sptr ;
}

/*
 * ʬɽԤؿ
 *----
 * ޤޤưʤǤɤƤϡ档
 */
static void skkinput_DiffDrawWindow
( Widget gw, struct SKKInputNode *node, char *text, char *ptext,
  int start_pos, int x, int y, int end_x, int end_y )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  int pos, cx, cy, kx, ky ;
  char *sptr ;

  /* ξԤɽƤʬˤĤƺʬԤ*/
  sptr = skkinput_DiffDrawWindowSub1
    ( gw, node, text, ptext, start_pos, &x, &y, end_x, end_y,
      &cx, &cy, &pos ) ;

  /* ޤ̰դˤϽ񤤤ƤʤΤǡõȤĤäƤ뤫⤷ʤ*/
  if( y < end_y ){
    /* õԤ˸ߤΥΰ֤ФƤ*/
    kx = x ;
    ky = y ;
    while( *sptr != '\0' ){
      if( ( ( x + w->skkinput.kanji_width ) >= end_x &&
	    ( 0x80 & *sptr ) ) ||
	  ( x + w->skkinput.roman_width ) >= end_x ){
	skkwin_jerase( gw, x, y, True ) ;
	x = 0 ;
	y += w->skkinput.font_height ;
      }
      if( y >= end_y )
	break ;
      skkwin_jerase( gw, x, y, False ) ;
      x += w->skkinput.roman_width ;
      sptr ++ ;
    }

    /* 뤬ɽƤ֤ޤãƤޤ... *
     * ˤϡõޤ*/
    if( y < node->prev_cy ||
	( x <= node->prev_cx && y == node->prev_cy ) ){
      skkwin_jerase( gw, node->prev_cx, node->prev_cy, True ) ;
    }
    x = kx ;
    y = ky ;

    /* ޤ뤬ɽƤʤȤȤϡǸΰ֤ˤ *
     * Τʤ*/
    if( cy < 0 ){
      if( ( x + w->skkinput.roman_width ) > end_x ){
	x = 0 ;
	y += w->skkinput.font_height ;
      }
      if( node->cur_exist )
	skkinputDrawCursor( gw, x, y, False ) ;
      cx = x ; cy = y ;
    }
  }
  /* 󥫡ɽ֤򵭲Ƥ*/
  node->prev_cx = cx ; 
  node->prev_cy = cy ;
  /* ɽʸ򵭲Ƥ*/
  return ;
}

/*
 * cur_line != NULL ʤСΤä꤬ܤǤ뤫֤
 */
static int skkinput_GetLineNum
( SkkInputWidget w, unsigned char *text, unsigned int width,
  int cur_pos, int *cur_line )
{
  int num, cur_num, x, y, pos, kana_flag ;
  char *ptr ;

  /* ɽѿν*/
  x = y = pos = kana_flag = cur_num = num = 0 ;
  /* 롼ס*/
  for( ptr = text ; *ptr != '\0' ; ptr ++, pos ++ ){
    /* ɤ򸫤롣*/
    if( 0x80 & *ptr ){
      kana_flag = kana_flag % 2 + 1 ;
    } else {
      kana_flag = 0 ;
    }
    /* ɤν*/
    switch( kana_flag ){
      /* ɤξν*/
    case 0 :
      x += w->skkinput.roman_width ;
      if( x >= width ){
	x = w->skkinput.roman_width ;
	num ++ ;
      }
      break ;
      /* ɤΰХܤν*/
    case 1 :
      x += w->skkinput.kanji_width ;
      if( x >= width ){
	x = w->skkinput.kanji_width ;
	num ++ ;
      }
      break ;
      /* ɤХܤν*/
    case 2 :
      break ;
      /* ʳΥɤʤ̵롣*/
    default :
      break ;
    }
    /* ֤򵭲롣*/
    if( pos == cur_pos )
      cur_num = num ;
  }

  /* ̤˥ȯƤʤ硣*/
  if( pos <= cur_pos ){
    if( ( x + w->skkinput.kanji_width ) > width )
      num ++ ;
    cur_num = num ;
  }

  /* ΰ֤֤Τʡ */
  if( cur_line != NULL )
    *cur_line = cur_num ;
  return num ;
}

static int skkinput_FindStartLine
( SkkInputWidget w, unsigned char *text, int startline, unsigned int width )
{
  int x, kana_flag, pos ;
  char *ptr ;

  /* ɽѿν*/
  x = pos = kana_flag = 0 ;
  /* 롼ס*/
  for( ptr = text ; *ptr != '\0' ; ptr ++, pos ++ ){
    /* ɤ򸫤롣*/
    if( 0x80 & *ptr ){
      kana_flag = kana_flag % 2 + 1 ;
    } else {
      kana_flag = 0 ;
    }
    /* ɤν*/
    switch( kana_flag ){
      /* ɤξν*/
    case 0 :
      x += w->skkinput.roman_width ;
      if( x >= width ){
	x = w->skkinput.roman_width ;
	startline -- ;
	if( startline <= 0 )
	  goto exit_loop ;
      }
      break ;
      /* ɤΰХܤν*/
    case 1 :
      x += w->skkinput.kanji_width ;
      if( x >= width ){
	x = w->skkinput.kanji_width ;
	startline -- ;
	if( startline <= 0 )
	  goto exit_loop ;
      }
      break ;
      /* ɤХܤν*/
    case 2 :
      break ;
      /* ʳΥɤʤ̵롣*/
    default :
      break ;
    }
  }
exit_loop:
  return pos ;
}

static void skkwin_jputstring
( Widget gw, int x, int y, char *text,
  int roman_width, int kana_width, int rf )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  int kanji_flag = False ;
  char buffer[ 3 ] ;

  buffer[ 2 ] = '\0' ;

  for( ; *text != '\0' ; text ++ ){
    if( ( 0x80 & *text ) && !kanji_flag ){
      kanji_flag = True ;
      buffer[ 0 ] = ( *text ) & 0x7F ;
    } else {
      if( 0x80 & *text ){
	buffer[ 1 ] = 0x7F & ( *text ) ;
	if( rf ){
	  XDrawString16( XtDisplay( gw ), XtWindow( gw ),
			 w->skkinput.kanji_rgc,
			 x, y, ( XChar2b *)buffer, 1 ) ;
	} else {
	  XDrawString16( XtDisplay( gw ), XtWindow( gw ),
			 w->skkinput.kanji_ngc,
			 x, y, ( XChar2b *)buffer, 1 ) ;
	}
	x += w->skkinput.kanji_width ;
	kanji_flag = False ;
      } else {
	buffer[ 0 ] = *text ;
	buffer[ 1 ] = '\0' ;
	if( rf ){
	  XDrawString( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
		     x, y, buffer, 1 ) ;
	} else {
	  XDrawString( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
		     x, y, buffer, 1 ) ;
	}
	x += w->skkinput.roman_width ;
#if 0
	kanji_flag = False ;
#endif
      }
    }
  }
  return ;
}

/*
 * Ϥ줿 skkinput Υ⡼ɥ饤󤬲ǤΤĴ٤ؿ
 */
static int skkinput_GetModelineNumber( struct SKKInputNode *node )
{
  int modeline ;
  /* ̾ϥ⡼ɤʤΡ ǤϤʤΡ */
  if( !node->j_mode ){
    /* ϥ⡼ɤʤΡ Ⱦϥ⡼ɤʤΡ */
    if( node->j_zenkaku ){
      modeline = MODELINE_ZENKAKU ;
    } else {
      modeline = MODELINE_HANKAKU ;
    }
  } else {
    /* Ҳ̾ϥ⡼ɤʤΡ ʿ̾ϥ⡼ɤʤΡ */
    if( node->j_katakana_mode ){
      modeline = MODELINE_KATAKANA ;
    } else {
      modeline = MODELINE_KANA ;
    }
  }
  return modeline ;
}

/*
 * skkinput Υ⡼ɥ饤ʬ褹ؿ
 */
static void skkinput_DiffDrawModeLine
( Widget gw, struct SKKInputNode *node, int x, int y, unsigned int width )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  static char *modelineStrings[] = {
    "SKK:", "", "", "",
  } ;
  static char *modelineStrings2[] = {
    "--", "**",
  } ;
  static char *chatAdapterStrings[] = {
    "(Fundamental",
    "(ChatAdapter",
  } ;
  int modeline = skkinput_GetModelineNumber( node ) ;

  /* ⡼ɥ饤ɽ줿ΤƱǤ뤫ɬפ *
   * ʤ*/
  if( w->skkinput.prev_modeline != modeline ){
    /* ѹоݤȤʤʬõ롣*/
    XFillRectangle
      ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
	x + w->skkinput.roman_width * 2,
	y - w->skkinput.font_ascent,
	w->skkinput.kanji_width * 2,
	w->skkinput.font_height ) ;
    /* ⡼ɥ饤ʬ褹롣*/
    skkwin_jputstring
      ( gw, x + w->skkinput.roman_width * 2, y,
	modelineStrings[ modeline - 1 ],
	w->skkinput.roman_width,
	w->skkinput.kanji_width, True ) ;
    /* ɽФ롣*/
    w->skkinput.prev_modeline = modeline ;
  }
  if( w->skkinput.prev_jisyo_dirty != w->skkinput.jisyo_dirty ){
    /* ѹоݤȤʤʬõ롣*/
    XFillRectangle
      ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
       x + w->skkinput.roman_width * 11,
       y - w->skkinput.font_ascent,
       w->skkinput.roman_width * 2,
       w->skkinput.font_height ) ;
    /* ⡼ɥ饤ʬ褹롣*/
    skkwin_jputstring
      ( gw, x + w->skkinput.roman_width * 11, y,
       modelineStrings2[ w->skkinput.jisyo_dirty ],
       w->skkinput.roman_width,
       w->skkinput.kanji_width, True ) ;
    w->skkinput.prev_jisyo_dirty = w->skkinput.jisyo_dirty ;
  }
  if( w->skkinput.prev_chat_adapter != w->skkinput.chat_adapter ){
    /* ѹоݤȤʤʬõ롣*/
    XFillRectangle
      ( XtDisplay( gw ), XtWindow( gw ),
       w->skkinput.roman_ngc,
       x + w->skkinput.roman_width * 35,
       y - w->skkinput.font_ascent,
       w->skkinput.roman_width * 12,
       w->skkinput.font_height ) ;
    /* ⡼ɥ饤ʬ褹롣*/
    skkwin_jputstring
      ( gw, x + w->skkinput.roman_width * 35, y,
       chatAdapterStrings[ w->skkinput.chat_adapter ],
       w->skkinput.roman_width,
       w->skkinput.kanji_width, True ) ;
    w->skkinput.prev_chat_adapter = w->skkinput.chat_adapter ;
  }
  if( w->skkinput.prev_eggnl != w->skkinput.egg_like_newline ){
    /* ѹоݤȤʤʬõ롣*/
    XFillRectangle
      ( XtDisplay( gw ), XtWindow( gw ),
       w->skkinput.roman_ngc,
       x + w->skkinput.roman_width * 47,
       y - w->skkinput.font_ascent,
       w->skkinput.roman_width * 13,
       w->skkinput.font_height ) ;
    /* ⡼ɥ饤ʬ褹롣*/
    if( w->skkinput.egg_like_newline ){
      skkwin_jputstring
	( gw, x + w->skkinput.roman_width * 47, y,
	 " egg-nl)--All",
	 w->skkinput.roman_width,
	 w->skkinput.kanji_width, True ) ;
    } else {
      skkwin_jputstring
	( gw, x + w->skkinput.roman_width * 47, y,
	 ")--All-------",
	 w->skkinput.roman_width,
	 w->skkinput.kanji_width, True ) ;
    }
    w->skkinput.prev_eggnl = w->skkinput.egg_like_newline ;
  }
  return ;
}

/*
 * skkinput β̤ʬɽѤɽؿ
 *-----
 * ̾β̤ν(ѹ)ϤδؿˤäƹԤ롣
 */
static void skkinput_DrawScreen( Widget gw )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  int x, y, cur_line, linenum, maxline, spos, sty, edy ;
  Window root ;
  unsigned int width, height, border, depth ;
  char *sptr, *dptr ;
  struct SKKInputNode *node = w->skkinput.topbuffer, *mNode ;

  /* 뤬¸ߤƤꡣ*/

  /* Window  size 롣*/
  XGetGeometry( XtDisplay( gw ), XtWindow( gw ),
	        &root, &x, &y, &width, &height, &border, &depth ) ;

  /* ɽǽԿ롣*/
  maxline = height / w->skkinput.font_height ;
  /* ߡƥȤϲԤޤɽʤФʤʤΤ򻻽Ф롣*/
  linenum = skkinput_GetLineNum
    ( w, node->textbuffer, width, node->cur_pos, &cur_line ) ;

  /* ⡼ɥ饤 minibuffer ΤɬפǤ뤫롣*/
  maxline = ( maxline < 3 )? 1 : maxline - 2 ;

#ifdef DEBUG_LV1
  fprintf( stderr, "cursor_line = %d\n", cur_line ) ;
#endif
  /* ɽǽԿĶƤν֤ϰϤ *
   * ǰֺǽιԤֹ椬Ǥ⾮ʤ褦ꤹ롣*/
  if( linenum >= maxline && cur_line >= maxline ){
    int startline = cur_line - maxline + 1 ;
    /* ꤵ줿꤫ɽʤФʤʤΤǡΰ֤򻻽Ф롣*/
    spos = skkinput_FindStartLine( w, node->textbuffer, startline,
				   width ) ;
    dptr = node->textbuffer + spos ;
    sptr = w->skkinput.pwrite_string + node->prev_start_pos ;
  }  else {
    spos = 0 ;
    /* ˰ֺǽʸɽǽǤ롣*/
    dptr = node->textbuffer ;
    sptr = w->skkinput.pwrite_string + node->prev_start_pos ;
  }    
  sty = w->skkinput.font_ascent ;
  edy = sty + maxline * w->skkinput.font_height ;
  /* Window  buffer ʸɽ롣*/
  skkinput_DiffDrawWindow( gw, node,
			   dptr, sptr, spos, 0, sty, width, edy ) ;
  strcpy( w->skkinput.pwrite_string, node->textbuffer ) ;
  node->prev_start_pos = spos ;

  /* ⡼ɥ饤ɽ롣ʬɽľбͽǤϤ롣*/
  sty = edy ;
  edy = sty + w->skkinput.font_height ;
  skkinput_DiffDrawModeLine( gw, node, 0, sty, width ) ;
			 
  /* mini-buffer ɽ롣*/
  mNode = w->skkinput.lastbuffer ;

  sty = edy ;
  edy = sty + w->skkinput.font_height ;

  /* for debugging */ 
  if( mNode == node ){
    skkwin_jdiffputstring
      ( gw, 0, sty, width, node->mtextbuffer,
	w->skkinput.pmtextbuffer, w->skkinput.roman_width,
	w->skkinput.kanji_width ) ;
    strcpy( w->skkinput.pmtextbuffer, node->mtextbuffer ) ;
  } else {
    if( mNode->mtextbuffer[ 0 ] != '\0' ){
      skkwin_jdiffputstring
	( gw, 0, sty, width, mNode->mtextbuffer,
	  w->skkinput.pmtextbuffer, w->skkinput.roman_width,
	  w->skkinput.kanji_width ) ;
      strcpy( w->skkinput.pmtextbuffer, mNode->mtextbuffer ) ;
    } else {
      /* Window  buffer ʸɽ롣*/
      skkinput_DiffDrawWindow
	( gw, mNode, mNode->textbuffer, w->skkinput.pmtextbuffer,
	  0, 0, sty, width, edy ) ;
      strcpy( w->skkinput.pmtextbuffer, mNode->textbuffer ) ;
    }
  }
  return ;
}

/*
 * skkinput ɽؿ
 *-----
 * ϺʬԤʤ
 */
static void skkinput_FullDrawWindow
( Widget gw, struct SKKInputNode *node, char *text, 
  int start_pos, int x, int y, int end_x, int end_y )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  int pos, chara, kana_flag, kanji_check, cursor_flag ;
  int cx, cy ;

  pos         = start_pos ;
  kana_flag   = False ;
  cursor_flag = False ;

  chara = cx = cy = 0 ;

  while( *text != '\0' ){
    /* ɤȽԤ*/
    kanji_check = 0x80 & *text ;
    /* ɽʤΤɤȽꤹ롣*/
    if( pos == node->cur_pos ){
      if( node->cur_exist ){
	skkinputDrawCursor( gw, x, y, kanji_check ) ;
	cursor_flag = True ;
      }
      cx = x ;
      cy = y ;
    }
    /* ɤΰХܤʤΤǤ礦 */
    if( kanji_check && kana_flag == False ){
      kana_flag = True ;
      chara = ( unsigned char )*text ;
      /* ̤αüޤ褷ΤɤȽǤ롣*/
      if( ( x + w->skkinput.kanji_width ) >= end_x ){
	skkwin_jputchar
	  ( gw, x, y, '\\', False, False ) ;
	skkwin_jputchar
	  ( gw, x + w->skkinput.roman_width, y, '\\', False, False ) ;
	x = 0 ;
	y += w->skkinput.font_height ;
	/* ̤βޤǽڤäΤɤȽǤ롣*/
	if( y >= end_y )
	  goto exit_loop ;
      }
    } else {
      /* ɤХܤʤΤǤ礦 */
      if( kana_flag ){
	chara = ( chara << 8 ) | ( ( unsigned char )*text ) ;
	/* ξ˥ɽƤΤǤ */
	if( cursor_flag ){
	  skkwin_jputchar( gw, x, y, chara, False, True ) ;
	  cursor_flag = False ;
	} else {
	  skkwin_jputchar( gw, x, y, chara, True, False ) ;
	}
	x += w->skkinput.kanji_width ;
	kana_flag = False ;
      } else {
	/* ɤäνǤ*/
	if( ( x + w->skkinput.roman_width ) >= end_x ){
	  skkwin_jputchar
	    ( gw, x, y, '\\', False, False ) ;
	  x =  0 ;
	  y += w->skkinput.font_height ;
	  /* ̤βޤǽڤäΤɤȽǤ롣*/
	  if( y >= end_y )
	    goto exit_loop ;
	}
	/* ΰ֤˥뤬ɽƤΤǤСʸȿž *
	 * 롣*/
	if( cursor_flag ){
	  skkwin_jputchar( gw, x, y, *text, False, True ) ;
	  cursor_flag = False ;
	} else {
	  skkwin_jputchar( gw, x, y, *text, True, False ) ;
	}
	x += w->skkinput.roman_width ;
      }
    }
    text ++ ;
    pos ++ ;
  }
  /* Ǹޤǥ뤬ɽƤʤäΤʤ顢ǸιԤ˥ *
   * ɽ롣*/
  if( pos == node->cur_pos && node->cur_exist ){
    /* Ԥɬפꡩ */
    if( ( x + w->skkinput.roman_width ) > end_x ){
      skkwin_jputchar
	( gw, x, y, '\\', False, False ) ;
      x  = 0 ;
      y += w->skkinput.font_height ;
    }
    skkinputDrawCursor( gw, x, y, False ) ;
    cx = x ;
    cy = y ;
  }

exit_loop:
  node->prev_cx = cx ; 
  node->prev_cy = cy ;
}

static void skkinput_FullDrawModeLine
( Widget gw, struct SKKInputNode *node, int x, int y, unsigned int width )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  int modeline = skkinput_GetModelineNumber( node ) ;
  static char *modelineStrings[] = {
    "--SKK:E.:--", "--E.:--", "--E.:--", "--E.:--",
  } ;
  /* νˤäư㤦⡼ɥ饤*/
  static char *modelineStrings2[] = {
    "---Skkinput:", "**-Skkinput:",
  } ;
  static char *chatadapterString[] = {
    "(Fundamental", "(ChatAdapter",
  } ;
  /* ޤ⡼ɥ饤ɽƤõ롣*/
  XFillRectangle
    ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_ngc,
     x, y - w->skkinput.font_ascent,
     width, w->skkinput.font_height ) ;
  /* ⡼ɥ饤ɽ( SKKΥ⡼ɽ )*/
  skkwin_jputstring
    ( gw, x, y, modelineStrings[ modeline - 1 ],
     w->skkinput.roman_width, w->skkinput.kanji_width, True ) ;
  /* ֤ι*/
  x += w->skkinput.roman_width * 11 ;
  /* ⡼ɥ饤ɽ( ξɽ )*/
  skkwin_jputstring
    ( gw, x, y, modelineStrings2[ w->skkinput.jisyo_dirty ],
     w->skkinput.roman_width,
     w->skkinput.kanji_width, True ) ;
  /* ֤ι*/
  x += w->skkinput.roman_width * 24 ;
  /* ᡼⡼ɤɽ*/
  XDrawString
    ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
     x, y, chatadapterString[ w->skkinput.chat_adapter % 2 ], 12 ) ;
  /* ֤ι*/
  x += w->skkinput.roman_width * 12 ;
  /* ޥʡ⡼ɤɽ(...ʣä餳դƿϤʥɤ *
   * 줽Ǥ) *
   *     ֤ʤǤ
   * ե꡼֤󤿾ʬäƤΡ
   *     ֿޤ󡣷ʿ˸ȤƤäˤĤƤä
   *           ޤǤ*/
  if( w->skkinput.egg_like_newline ){
    XDrawString
      ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
       x, y, " egg-nl", 7 ) ;
    x += w->skkinput.roman_width * 7 ;
  } ;
  XDrawString
    ( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
     x, y, ")", 1 ) ;
  x += w->skkinput.roman_width ;
  /* ³*/
  XDrawString( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
	       x, y, "--All", 5 ) ;
  x += w->skkinput.roman_width * 5 ;
  for( ; x < width ; x += w->skkinput.roman_width )
    XDrawString( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
		 x, y, "-", 1 ) ;
  /* ɽ⡼ɥ饤򵭲Ƥ*/
  w->skkinput.prev_modeline     = modeline ;
  w->skkinput.prev_jisyo_dirty  = w->skkinput.jisyo_dirty ;
  w->skkinput.prev_eggnl        = w->skkinput.egg_like_newline ;
  w->skkinput.prev_chat_adapter = w->skkinput.chat_adapter ;
  return ;
}

/*
 * skkinput β̤򤤤ľؿ
 *-----
 * ̤礭ѹʤɤԤ줿ˤϤδؿƤФ뤳Ȥ
 * ʤ롣
 */
static void skkinput_RedrawScreen( Widget gw )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  int x, y, cur_line, linenum, maxline, spos, sty, edy ;
  Window root ;
  unsigned int width, height, border, depth ;
  char *ptr ;
  struct SKKInputNode *mNode, *node = w->skkinput.topbuffer ;

  /* Window Ȥõ롣*/
  XClearWindow( XtDisplay( gw ), XtWindow( gw ) ) ;

  /* Window  size 롣*/
  XGetGeometry( XtDisplay( gw ), XtWindow( gw ),
	        &root, &x, &y, &width, &height, &border, &depth ) ;

  /* ɽǽԿ롣*/
  maxline = height / w->skkinput.font_height ;
  /* ߡƥȤϲԤޤɽʤФʤʤΤ򻻽Ф롣*/
  linenum = skkinput_GetLineNum( w, node->textbuffer, width,
				 node->cur_pos, &cur_line ) ;
  /* ⡼ɥ饤 minibuffer ΤɬפǤ뤫롣*/
  maxline = ( maxline < 3 )? 1 : maxline - 2 ;
  /* ɽǽԿĶƤν֤ϰϤ *
   * ǰֺǽιԤֹ椬Ǥ⾮ʤ褦ꤹ롣*/
  if( linenum >= maxline && cur_line >= maxline ){
    int startline = cur_line - maxline + 1 ;
    /* ꤵ줿꤫ɽʤФʤʤΤǡΰ֤򻻽Ф롣*/
    spos = skkinput_FindStartLine( w, node->textbuffer, startline,
				   width ) ;
    ptr = node->textbuffer + spos ;
  }  else {
    spos = 0 ;
    /* ˰ֺǽʸɽǽǤ롣*/
    ptr = node->textbuffer ;
  }    
  sty = w->skkinput.font_ascent ;
  edy = sty + maxline * w->skkinput.font_height ;
  /* Window  buffer ʸɽ롣*/
  skkinput_FullDrawWindow( gw, node, ptr, spos, 0, sty, width, edy ) ;
  strcpy( w->skkinput.pwrite_string, node->textbuffer ) ;

  /* ⡼ɥ饤ɽ롣ʬɽľбͽǤϤ롣*/
  sty = edy ;
  edy = sty + w->skkinput.font_height ;
  skkinput_FullDrawModeLine( gw, node, 0, sty, width ) ;
			 
  /* mini-buffer ɽ롣*/
  mNode = w->skkinput.lastbuffer ;
  sty = edy ;
  edy = sty + w->skkinput.font_height ;

  if( mNode == node ){
    skkwin_jputstring( gw, 0, sty, node->mtextbuffer,
		       w->skkinput.roman_width,
		       w->skkinput.kanji_width, False ) ;
    strcpy( w->skkinput.pmtextbuffer, node->mtextbuffer ) ;
  } else {
    if( mNode->mtextbuffer[ 0 ] != '\0' ){
      skkwin_jputstring( gw, 0, sty, mNode->mtextbuffer,
			 w->skkinput.roman_width,
			 w->skkinput.kanji_width, False ) ;
      strcpy( w->skkinput.pmtextbuffer, mNode->mtextbuffer ) ;
    } else {
      /* Window  buffer ʸɽ롣*/
      skkinput_FullDrawWindow( gw, mNode,
			       mNode->textbuffer, 0, 0, sty,
			       width, edy ) ;
      w->skkinput.pmtextbuffer[ 0 ] = '\0' ;
    }
  }
  return ;
}

/*
 * ʬؿɽطνʤԵǽ̵
 *-----
 * Ϥ minibuffer ѤȤäޤ
 */
static void skkwin_jdiffputstring( Widget gw,
				   int x, int y, int width,
				   char *text, char *ptext,
				   int roman_width, int kana_width )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  int dkana_flag, skana_flag, charas_are_different, diff_flag, chara ;

  charas_are_different = diff_flag = dkana_flag = skana_flag = False ;
  chara = 0 ;

  /* ĤʸӤ롣*/
  while( *text != '\0' && x < width ){
    /* ɤɤɤʤ餽βХܤ*/
    if( 0x80 & *text ){
      dkana_flag = dkana_flag % 2 + 1 ;
    } else {
      dkana_flag = 0 ;
    }
    /* ɽ줿ȽԤ*/
    if( 0x80 & *ptext ){
      skana_flag = skana_flag % 2 + 1 ;
    } else {
      skana_flag = 0 ;
    }
    /* ơ饯ϰۤʤΤǤ礦 */
    if( *ptext != *text || skana_flag != dkana_flag || *ptext == '\0' ){
      charas_are_different = True ;
    } else {
      charas_are_different = False ;
    }
    switch( dkana_flag ){
      /* ɤν*/
    case 0 :
      /* ꤬˲ԤƤޤäƤˤ̾ɽ*/
      /* ǤϤʤơ饯Ӥưۤʤɽ*/
      if( charas_are_different )
	skkwin_jputchar2( gw, x, y, *text, False, False ) ;
      /* ɽ֤ΰư*/
      x += w->skkinput.roman_width ;
      break ;
      /* ɤΰХܤν*/
    case 1 :
      chara = ( unsigned char )*text ;
      /* ä˲ԤƤޤäƤˤԤġ*/
      /* ɤΰХܤʸ󤬰ۤʤäƤ硣*/
      if( charas_are_different )
	diff_flag = True ;
      break ;
      /* ɤХܤν*/
    case 2 :
      chara = ( chara << 8 ) | ( ( unsigned char )*text ) ;
      /* ɤХܤʸ󤬰ۤʤäƤڤӡХ *
       * ܤǰۤʤäƤˤϽľκȤ롣                 */
      if( charas_are_different || diff_flag ){
	skkwin_jputchar2( gw, x, y, chara, False, False ) ;
	diff_flag = False ;
      }
      /* ɽ֤ΰư*/
      x += w->skkinput.kanji_width ;
      break ;
    default :
      dkana_flag = 0 ;
      break ;
    }
    text ++ ;
    if( *ptext != '\0' )
      ptext ++ ;
  }
  /* ̰դޤɽƤΤǾõȤɬ̵*/
  /* ⤷ɽ˻ȤäʬĶɽƤΤǡ*/
  if( x >= width || *ptext == '\0' )
    return ;
  /* 񤭤Ƥʤʬõľ*/
  while( *ptext != '\0' ){
    skkwin_jerase( gw, x, y, False ) ;
    x += roman_width ;
    ptext ++ ;
  }
  return ;
}

/*
 * ߥ˥ХåեɽƤΤõؿ
 */
static void skkinput_ClearMinibuffer( Widget gw )
{
  SkkInputWidget w = ( SkkInputWidget )gw ;
  Window root ;
  int x, y, maxline ;
  unsigned int width, height, border, depth ;

  /* Window  size 롣*/
  XGetGeometry( XtDisplay( gw ), XtWindow( gw ),
	        &root, &x, &y, &width, &height, &border, &depth ) ;

  /* ɽǽԿ롣*/
  maxline = height / w->skkinput.font_height ;
  if( maxline < 3 ){
    y = 2 * w->skkinput.font_height ;
  } else {
    y = ( maxline - 1 ) * w->skkinput.font_height ;
  }
  w->skkinput.pmtextbuffer[ 0 ] = '\0' ;
  XFillRectangle( XtDisplay( gw ), XtWindow( gw ), w->skkinput.roman_rgc,
		  0, y, width, w->skkinput.font_height ) ;
  return ;
}
