/********************************************************************\
 * guile-util.c -- utility functions for using guile for GnuCash    *
 * Copyright (C) 1999 Linas Vepstas                                 *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
\********************************************************************/

#include "top-level.h"

#include "guile-util.h"
#include "util.h"

/* This static indicates the debugging module this .o belongs to.  */
static short module = MOD_GUILE;


/********************************************************************\
 * gnc_register_c_side_scheme_ptr                                   *
 *   register a SCM handle that the C code will be keeping around   *
 *   this prevents guile from garbage collecting it                 *
 *                                                                  *
 * Args: scm_obj - the SCM object to register                       *
 * Returns: SCM opaque handle to be used to unregister, or          *
 *          SCM_UNDEFINED if there was a problem.                   *
\********************************************************************/
SCM
gnc_register_c_side_scheme_ptr(SCM scm_obj)
{
  SCM func = gh_eval_str("gnc:register-c-side-scheme-ptr");
  SCM result = SCM_UNDEFINED;
  
  if (gh_procedure_p(func))
    result = gh_call1(func, scm_obj);
  else
  {
    PERR("gnc_register_c_side_scheme_ptr: not a function\n");
  }

  return result;
}


/********************************************************************\
 * gnc_unregister_c_side_scheme_ptr                                 *
 *   unregister a SCM handle that the C code is not using anymore   *
 *   so the guile side can garbage collect it                       *
 *                                                                  *
 * Args: scm_obj_id - the SCM handle returned by the register       *
 *                    function above.                               *
 * Returns: true if successful                                      *
\********************************************************************/
int
gnc_unregister_c_side_scheme_ptr_id(SCM scm_obj_id)
{
  SCM func = gh_eval_str("gnc:unregister-c-side-scheme-ptr-id");
  SCM result = SCM_BOOL_F;

  if (gh_procedure_p(func))
    result = gh_call1(func, scm_obj_id);
  else
  {
    PERR("gnc_unregister_c_side_scheme_ptr_id: not a function\n");
  }

  return gh_scm2bool(result);
}


/********************************************************************\
 * gnc_guile_call1_to_string                                        *
 *   returns the malloc'ed string returned by the guile function    *
 *   or NULL if it can't be retrieved                               *
 *                                                                  *
 * Args: func - the guile function to call                          *
 *       arg  - the single function argument                        *
 * Returns: malloc'ed char * or NULL                                *
\********************************************************************/
char *
gnc_guile_call1_to_string(SCM func, SCM arg)
{
  SCM value;

  if (gh_procedure_p(func))
  {
    value = gh_call1(func, arg);

    if (gh_string_p(value))
      return gh_scm2newstr(value, NULL);
    else
    {
      PERR("gnc_guile_call1_to_string: bad value\n");
    }
  }
  else
  {
    PERR("gnc_guile_call1_to_string: not a procedure\n");
  }

  return NULL;
}


/********************************************************************\
 * gnc_guile_call1_symbol_to_string                                 *
 *   returns the malloc'ed string returned by the guile function    *
 *   or NULL if it can't be retrieved. The return value of the      *
 *   function should be a symbol.                                   *
 *                                                                  *
 * Args: func - the guile function to call                          *
 *       arg  - the single function argument                        *
 * Returns: malloc'ed char * or NULL                                *
\********************************************************************/
char *
gnc_guile_call1_symbol_to_string(SCM func, SCM arg)
{
  SCM value;

  if (gh_procedure_p(func))
  {
    value = gh_call1(func, arg);

    if (gh_symbol_p(value))
      return gh_symbol2newstr(value, NULL);
    else
    {
      PERR("gnc_guile_call1_symbol_to_string: bad value\n");
    }
  }
  else
  {
    PERR("gnc_guile_call1_symbol_to_string: not a procedure\n");
  }

  return NULL;
}


/********************************************************************\
 * gnc_guile_call1_to_procedure                                     *
 *   returns the SCM handle to the procedure returned by the guile  *
 *   function, or SCM_UNDEFINED if it couldn't be retrieved.        *
 *                                                                  *
 * Args: func - the guile function to call                          *
 *       arg  - the single function argument                        *
 * Returns: SCM function handle or SCM_UNDEFINED                    *
\********************************************************************/
SCM
gnc_guile_call1_to_procedure(SCM func, SCM arg)
{
  SCM value;

  if (gh_procedure_p(func))
  {
    value = gh_call1(func, arg);

    if (gh_procedure_p(value))
      return value;
    else
    {
      PERR("gnc_guile_call1_to_procedure: bad value\n");
    }
  }
  else
  {
    PERR("gnc_guile_call1_to_procedure: not a procedure\n");
  }

  return SCM_UNDEFINED;
}


/********************************************************************\
 * gnc_guile_call1_to_list                                          *
 *   returns the SCM handle to the list returned by the guile       *
 *   function, or SCM_UNDEFINED if it couldn't be retrieved.        *
 *                                                                  *
 * Args: func - the guile function to call                          *
 *       arg  - the single function argument                        *
 * Returns: SCM list handle or SCM_UNDEFINED                        *
\********************************************************************/
SCM
gnc_guile_call1_to_list(SCM func, SCM arg)
{
  SCM value;

  if (gh_procedure_p(func))
  {
    value = gh_call1(func, arg);

    if (gh_list_p(value))
      return value;
    else
    {
      PERR("gnc_guile_call1_to_list: bad value\n");
    }
  }
  else
  {
    PERR("gnc_guile_call1_to_list: not a procedure\n");
  }

  return SCM_UNDEFINED;
}


/********************************************************************\
 * gnc_guile_call1_to_vector                                        *
 *   returns the SCM handle to the vector returned by the guile     *
 *   function, or SCM_UNDEFINED if it couldn't be retrieved.        *
 *                                                                  *
 * Args: func - the guile function to call                          *
 *       arg  - the single function argument                        *
 * Returns: SCM vector handle or SCM_UNDEFINED                      *
\********************************************************************/
SCM
gnc_guile_call1_to_vector(SCM func, SCM arg)
{
  SCM value;

  if (gh_procedure_p(func))
  {
    value = gh_call1(func, arg);

    if (gh_vector_p(value))
      return value;
    else
    {
      PERR("gnc_guile_call1_to_vector: bad value\n");
    }
  }
  else
  {
    PERR("gnc_guile_call1_to_vector: not a procedure\n");
  }

  return SCM_UNDEFINED;
}
