/* This software is copyrighted (c) 1998 and may only be used under the *****/
/* terms of the GNU Public License ******************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <sysexits.h>

#include <sys/stat.h>

/****************************************************************************/

#define SINCE_DONTUPDATE 0  /* don't modify .since file this time */
#define SINCE_DOUPDATE   1  /* write back if things have changed */

#define SINCE_LBUF     4096 /* large buffer (for displaying file) */
#define SINCE_SBUF      256 /* small buffer (for reading bits of .since) */

char SINCE_FORMAT[SINCE_SBUF];

/****************************************************************************/

/* BUGS: performance is O(nm) where n=files to be displayed, m=all files    */
/*       previously displayed. With more effort it could be O(log(m))       */

/* SINCEFILE format: each line contains stat information about a file       */
/*       device+inode=lookup key, stored value=old file size                */ 
/*       st_dev:st_ino:st_size\n                                            */ 

/****************************************************************************/
/* Does       : Ugly runtime guess at the size a type, return the format    */
/*              modifier                                                    */
/* Returns    : format modifier as string                                   */
/* Parameters : sizeof type to be guessed                                   */
/* Errors     : will exit() if no type matches.                             */
/* Others     : could be done at compile-time, overwrites data on repeat    */

char *guessformat(int len){
  static char result[2];

  if(sizeof(long int)==len){
    result[0]='l';
  }
  else if(sizeof(int)==len){
    result[0]='\0';
  }
  else if(sizeof(char)==len||sizeof(short int)==len){
    result[0]='h';
  }
  else if(sizeof(long long int)==len){
    /* this is confusing - should it not be L ? */
    result[0]='q';
  }
  else{
    fprintf(stderr,"Eeek, can't figure out types in stat(2) field\n");
    exit(EX_DATAERR);
  }

  result[1]='\0';

#ifdef TRACE
fprintf(stderr,"guessformat(): <%d>bytes is type<%s>\n",len,result);
#endif

  return result;
}

/****************************************************************************/
/* Does       : Ugly runtime guess at the size of the types of the stat     */
/*              field                                                       */
/* Returns    : nothing, but sets up the global SINCE_FORMAT string         */
/* Others     : could be done at compile-time                               */

void prepsinceformat(){
  struct       stat st;
  char tmp[SINCE_SBUF];

  sprintf(SINCE_FORMAT, "%%0%d%s",2*sizeof(st.st_dev), guessformat(sizeof(st.st_dev)));
  sprintf(tmp,        "x:%%0%d%s",2*sizeof(st.st_ino), guessformat(sizeof(st.st_ino)));
  strcat(SINCE_FORMAT,tmp);
  sprintf(tmp,        "x:%%0%d%s",2*sizeof(st.st_size),guessformat(sizeof(st.st_size)));
  strcat(SINCE_FORMAT,tmp);
  strcat(SINCE_FORMAT,"x\n");

#ifdef TRACE
fprintf(stderr,"prepsinceformat(): SINCE format is <%s>\n",SINCE_FORMAT);
#endif
}

/****************************************************************************/
/* Does       : Gets the last displayed offset of file statted (via st)     */
/*              from the database (from file argument)                      */
/* Parameters : file - database, st - statted file to be found, update -    */
/*              option to control if changes get written                    */
/* Returns    : recorded offset or 0 on failure                             */
/* Errors     : will exit() on errors                                       */

off_t sincefile(FILE *file, struct stat *st, int update){
  char have[SINCE_SBUF],got[SINCE_SBUF];
  int     match,matchlen,havelen,gotlen;
  char                          *endptr;
  off_t                        result=0;

  havelen=sprintf(have,SINCE_FORMAT,st->st_dev,st->st_ino,st->st_size);
  match=0;
  matchlen=0;
  while(have[matchlen]!=':')matchlen++;
  matchlen++;
  while(have[matchlen]!=':')matchlen++;

#ifdef TRACE
fprintf(stderr,"sincefile(): st->st_ino=%lu, havelen=%d, matchlen=%d, havestring=%s",st->st_ino,havelen,matchlen,have);
#endif

  rewind(file);
  do{
    gotlen=fread(got,sizeof(char),havelen,file);

#ifdef TRACE
fprintf(stderr,"sincefile(): Got bytes=%d\n",gotlen);
#endif

    if(gotlen==havelen){
      if(got[gotlen-1]=='\n'){
        if(!strncmp(got,have,matchlen)){
#ifdef TRACE
fprintf(stderr,"sincefile(): match\n");
#endif
          match=1;
          gotlen=0;
        }
      }
      else{
        gotlen=(-1);
      }
    }
  }while(gotlen==havelen);

  if(gotlen){
    perror("Could not read .since information");
    exit(EX_DATAERR);
  }
  else{

    if(match){
#ifdef TRACE
fprintf(stderr,"sincefile(): got[matchlen]=%c\n",got[matchlen]);
#endif
      got[havelen]='\0';
      result=strtol(got+matchlen+1,&endptr,16);
      if(result>st->st_size){
#ifdef TRACE
fprintf(stderr,"sincefile(): Stored information is larger than current, assuming a truncation\n");
#endif
        result=0;
      }
#ifdef TRACE
fprintf(stderr,"sincefile(): Return is <%lx>\n",result);
#endif
    }

    if(update==SINCE_DOUPDATE){
      if(match){
#ifdef TRACE
fprintf(stderr,"sincefile(): matched - updating an existing record\n");
#endif
        if(strncmp(have,got,havelen)){
#ifdef TRACE
fprintf(stderr,"sincefile(): real update, position before seek=<%ld>\n",ftell(file));
#endif
          if(fseek(file,ftell(file)-havelen,SEEK_SET)){
            perror("Could not seek .since database");
            exit(EX_IOERR);
          }
#ifdef TRACE
fprintf(stderr,"sincefile(): position after seek=<%ld>\n",ftell(file));
#endif
          if(fwrite(have,sizeof(char),havelen,file)!=havelen){
            perror("Could not write update to .since database");
            exit(EX_IOERR);
          }
          fflush(file);
        }
      }
      else{
#ifdef TRACE
fprintf(stderr,"sincefile(): Unmatched - generating a new record\n");
#endif
        if(fwrite(have,sizeof(char),havelen,file)!=havelen){
          perror("Could not write update to .since database");
          exit(EX_IOERR);
        }
        fflush(file);
      }
    }
  }
  
  return result;
}

/****************************************************************************/
/* Does       : dump bytes beyond offset of file fname to stdout            */
/* Parameters : fname - file name, offset - start of dump                   */
/* Returns    : always zero, though it probably shouldn't                   */
/* Errors     : IO errors                                                   */

int dispfile(int fd, off_t offset){
  char buf[SINCE_LBUF];
  int            wl,rl;

  if(lseek(fd,offset,SEEK_SET)!=offset){
    perror("Could not seek to desired location");
  }
  else{
    do{
      rl=read(fd,buf,SINCE_LBUF);
      wl=fwrite(buf,sizeof(char),rl,stdout);
    }while(rl==wl&&rl>0);
    if(rl<0||wl<0){
      perror("Could not display file");
    }
  }

  return 0;
}

int main(int argc, char **argv){

  /* since operations */
  FILE           *since;
  char sbuf[SINCE_SBUF];
  char        *sinceptr;
  struct stat        st;
  off_t        seekdest;

  /* vars for parsing cmdline */
  char             *ptr;
  int               i,j;
  int          dashfile;

  /* number of files to be displayed */
  int             files;
  /* descriptor for the current file */
  int                fd;

  /* options */
  int            update;
  int           verbose;
  
  update=SINCE_DOUPDATE;
  verbose=0;

  /* process options */
  i=1;
  files=0;
  dashfile=0;
  while(i<argc){
    ptr=argv[i];
    if(ptr[0]=='-'&&!dashfile){
      j=1;
      while(ptr[j]!='\0'){
        switch(ptr[j]){
          case '?' :
          case 'H' :
          case 'h' :
printf("since   : A tail(1) which remembers its last invocation\nUsage   : since [options] file ...\nOptions :\n");
printf("-h        This help\n");
printf("-n        Do not write updates to .since file\n");
printf("-v        More verbose output\n");
printf(".since  : Location determined by $SINCE environment variable,\n          else $HOME/.since, if neither are set then /tmp/since\n");
            exit(EX_OK);
          break;
          case 'c' :
printf("since   : Copyright (c) 1998 by mgw. May only be used in terms of the GNU Public License as published by the Free Software Foundation\n");
            exit(EX_OK);
          break;
          case 'n' :
            update=SINCE_DONTUPDATE;
          break;
          case 'v' :
            verbose++;
          break;
          case '-' :
            dashfile=1;
          break;
          default  :
fprintf(stderr,"%s: Unrecognized option -%c, try -h for help\n",argv[0],ptr[j]);
            exit(EX_USAGE);
          break;
        }
        j++;
      }
    }
    else{
      files++;
    }
    i++;
  }

  if(files==0){
fprintf(stderr,"%s: Insufficient arguments, try -h for help\n",argv[0]);
    exit(EX_USAGE);
  }

  /* get hold of .since file */
  sinceptr=getenv("SINCE");
  if(sinceptr){
    close(open(sinceptr,O_WRONLY|O_CREAT,S_IRUSR|S_IWUSR));
    since=fopen(sinceptr,"r+");
  }
  else{
    sinceptr=getenv("HOME");
    if(sinceptr){
      snprintf(sbuf,SINCE_SBUF,"%s/.since",sinceptr);
      close(open(sbuf,O_WRONLY|O_CREAT,S_IRUSR|S_IWUSR));
      since=fopen(sbuf,"r+");
    }
    else{
      close(open("/tmp/since",O_WRONLY|O_CREAT,S_IRUSR|S_IWUSR));
      since=fopen("/tmp/since","r+");
    }
  }

  if(since==NULL){
perror("Could not open .since file");
    exit(EX_DATAERR);
  }

  prepsinceformat();

  signal(SIGPIPE,SIG_IGN);

  /* now process files */
  i=1;
  dashfile=0;
  while(i<argc){
    ptr=argv[i];
    if(ptr[0]=='-'&&!dashfile){
      if(ptr[1]=='-'){
        dashfile=1;
      }
    }
    else{
      if(files>1||verbose){
printf("==> %s ",argv[i]); /* I don't like the header, but tail uses it... */
      }
      fd=open(ptr,O_RDONLY);
      if(fd==(-1)||fstat(fd,&st)){
perror("Can not open or stat file");
      }
      else{
#ifdef TRACE
fprintf(stderr,"main(): About to sincefile size=%08lx, ino=%08lx, dev=%04Lx \n",st.st_size,st.st_ino,st.st_dev);
#endif
        /* find the offset which we last looked at */
        seekdest=sincefile(since,&st,update); 
        if(verbose){
          printf(SINCE_FORMAT,st.st_dev,st.st_ino,seekdest);
          printf("    ");
        }
        /* any changes ? */
        if(seekdest==st.st_size){
          if(files>1||verbose){
printf("[nothing new] <==\n");
          }
        }
        else{
          if(files>1||verbose){
printf("<==\n");
          }
          /* dump the selected file to stdout */
          dispfile(fd,seekdest);
        }
        close(fd);
      }
    }
    i++;
  }

  fclose(since);

  exit(EX_OK);

  /* keep gcc -Wall happy */
  return 0;
}

