/*
 * Copyright (c) 1997, 1998  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This program requires the following Autoconf macros:
 *   AC_TYPE_UID_T
 *   AC_TYPE_GID_T
 *   AC_CHECK_HEADERS(unistd.h)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <syslog.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef USE_FAKELOG
#include "fakelog.h"
#endif


/*
 * Set UID and GID of the current process.
 * To set a UID, the current UID of the process must be root.
 *
 * If succeeded, 0 is returned.  Otherwise -1 is returned.
 */
int
set_privilege(newuid, newgid)
    uid_t newuid;
    gid_t newgid;
{
    struct passwd *pw;
    uid_t curuid;
    gid_t curgid;
    int gidok = 0;

    /*
     * Set a GID.
     */
    curgid = getgid();
    if (newgid != curgid) {
	pw = getpwuid(newuid);
	if (pw == NULL) {
	    syslog(LOG_ERR, "getpwuid() failed, %m: uid=%d", (int)newuid);
	    goto failed;
	}
	if (initgroups(pw->pw_name, newgid) < 0) {
	    syslog(LOG_ERR, "initgroups() failed, %m: gid=%d", (int)newgid);
	    goto failed;
	}
	if (setgid(newgid) < 0) {
	    syslog(LOG_ERR, "setgid() fialed, %m: gid=%d", (int)newgid);
	    goto failed;
	}
    }
    gidok = 1;

    /*
     * Set an UID only when the UID of the current process is root.
     */
    curuid = getuid();
    if (curuid != newuid) {
	if (curuid != 0)
	    goto failed;
	if (setuid(newuid) < 0) {
	    syslog(LOG_ERR, "setuid() fialed, %m: uid=%d", (int)newuid);
	    goto failed;
	}
    }

    syslog(LOG_DEBUG, "debug: set privilege: uid=%d, gid=%d", newuid, newgid);
    return 0;

    /*
     * An error occurs...
     */
  failed:
    if (gidok)
	syslog(LOG_ERR, "failed to set owner of the process");
    else
	syslog(LOG_ERR, "failed to set owner and group of the process");
    return -1;
}
