/*
 * GUI directory view module
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <gnome.h>
#include "gdiff.h"
#include "misc.h"
#include "gui.h"
#include "dirview.h"
#include "guiwin.h"
#include "fileview.h"
#include "guimisc.h"
#include "hide.h"

/* Constant numbers */
/* Column definitions */
enum {
	COL_FILENAME1 = 0,
	COL_FILENAME2 = 1,
	COL_TYPE = 2,
	COL_NLINES1 = 3,
	COL_NLINES2 = 4,
};

/* Private data structure definitions */
/* used for CList sort */
typedef struct {
	char *title;
	GtkCListCompareFunc cmp_func;
} DirViewCList;

/* Private function declarations */
static void dirv_create_widgets(GDiffDirViews *gdirviews);
static void dirv_create_clist(GDiffDirViews *gdirviews, GtkWidget *hbox);
static void dirv_draw_clist(GDiffDirViews *gdirviews);
static gboolean check_row_hide(GDiffDirViews *gdirviews, const char *fname);
static void click_dirview(GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer data);
static void click_column(GtkCList *clist, gint column, gpointer data);
static gint number_compare(GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2);

/* Private variables */
static const DirViewCList dview_clist[] = {
	{N_("file name(from)"), NULL},
	{N_("file name(to)"), NULL},
	{N_("diff type"), NULL},
	{N_("lines(from)"), number_compare},
	{N_("lines(to)"), number_compare},
};
/* gcc extension: to use this value as char *titles[DVIEW_COLUMNS].
   static const int DVIEW_COLUMNS = sizeof(dview_clist) / sizeof(dview_clist[0]);*/
#define DVIEW_COLUMNS	(sizeof(dview_clist) / sizeof(dview_clist[0]))



/**
 * gdiff_dirviews_new:
 * Allocate GDiffDirView, intialize it and return its pointer.
 * Internally, it calls back-end routine(diff_dir_new()).
 * And it also creates widgets for displaying the result.
 * If comparing two files instead of comparing two directories,
 * it internally allocates GDiffFileViews and shows its result.
 * But the GDiffFileViews will be finalized via gdiff_dirviews_delete().
 * Input:
 * GDiffWindow *gdwin; kept in GDiffDirView for reference.
 * const char *fname1; first file name.
 * const char *fname2; second file name.
 * gboolean isdir; TRUE if they are directories.
 * Output:
 * GDiffWindow *gdwin; GList *gdirv_list is updated.
 * Return value; Allocated GDiffDirViews.
 **/
GDiffDirViews*
gdiff_dirviews_new(GDiffWindow *gdwin, const char *fname1, const char *fname2, gboolean isdir)
{
	GDiffDirViews *gdirviews;

	gdirviews = g_new(GDiffDirViews, 1);

	/* Initialize the preference */
	gdirviews->pref = g_pref.dvpref;/* copy */
	gdirviews->pref.row_hide_stat_mask = 0;
	gdirviews->pref.row_hide_func_list = NULL;

	/* Back-end data */
	gdirviews->diffdir = diff_dir_new(fname1, fname2, gdirviews->pref.diff_args);
	gdirviews->isdir = isdir;
	gdirviews->gdwin = gdwin;
	gdirviews->gdfile_list = NULL;

	/* Initialize dirty-bit */
	gdirviews->b_dirty = FALSE;
	
	/* Can be refererd to from GDiffWindow. */
	gdwin->gdirv_list = g_list_prepend(gdwin->gdirv_list, gdirviews);

	if (isdir) {
		gdirviews->dirname1 = g_strdup(fname1);
		gdirviews->dirname2 = g_strdup(fname2);

		/* Create widgets for display file list */
		dirv_create_widgets(gdirviews);
	} else {
		/* compare two files */
		DiffFiles *dfiles;
		GDiffFileViews *gfileviews;

		gdirviews->dirname1 = NULL;
		gdirviews->dirname2 = NULL;
		gdirviews->base = NULL;
		gdirviews->hbox = NULL;
		gdirviews->clist = NULL;
		dfiles = g_slist_nth_data(gdirviews->diffdir->dfiles_list, 0);
		gfileviews = gdiff_fileviews_new(gdwin, gdirviews, dfiles, g_pref.fvpref.pane_mode);
	}

	return gdirviews;
}

/**
 * gdiff_dirviews_delete:
 * Finalize GDiffDirViews structure, and free its memory.
 * It also destroys the related widgets.
 * Every related GDiffFileViews and their related widgets will be finalized, too.
 * Input:
 * GDiffDirViews *gdirviews;
 * Output:
 * None;
 **/
void
gdiff_dirviews_delete(GDiffDirViews *gdirviews)
{
	GDiffWindow *gdwin;
	GSList *list;/* Node of GDiffFileViews */

	g_return_if_fail(gdirviews != NULL);

	gdwin = gdirviews->gdwin;
	
	for (list = gdirviews->gdfile_list; list; list = list->next) {
		gdiff_fileviews_delete(list->data);
	}
	g_slist_free(gdirviews->gdfile_list);
	
	/* Front-end data */
	if (gdirviews->base) {
		gint page;
		page = gtk_notebook_page_num(gdirviews->gdwin->notebook, GTK_WIDGET(gdirviews->base));
		gtk_notebook_remove_page(gdirviews->gdwin->notebook, page);
		/*gtk_widget_destroy(GTK_WIDGET(gdirviews->base));*/
	}

	/* Row hide function list */
	if (gdirviews->pref.row_hide_func_list)
		g_slist_free(gdirviews->pref.row_hide_func_list);

	/* Back-end data */
	diff_dir_delete(gdirviews->diffdir);
	if (gdirviews->dirname1)
		g_free(gdirviews->dirname1);
	if (gdirviews->dirname2)
		g_free(gdirviews->dirname2);

	/* Remove own node from the list managed by GDiffWindow */
	gdwin->gdirv_list = g_list_remove(gdwin->gdirv_list, gdirviews);

	/* Free its own memory */
	g_free(gdirviews);
}


/**
 * gdiff_current_views:
 * Search the current view in the notebook widget,
 * and return its related data strucure(GDiffDirViews or GDiffFileViews).
 * This doesn't return a proper type.
 * It is up to the caller to cast it to the proper type depending on ret_isdir.
 * Input:
 * const GDiffWindow *gdwin;
 * gboolean *ret_isdir;
 * Output:
 * gboolean *ret_isdir; return TRUE if it is directory.
 * Return value; pointer to GDiffDirViews or GDiffFileViews. In no view case,  NULL.
 **/
void*
gdiff_current_views(const GDiffWindow *gdwin, gboolean *ret_isdir)
{
	GList *list;
	GtkNotebook *notebook = gdwin->notebook;
	GtkWidget *w;
	gint page;

	page = gtk_notebook_current_page(notebook);
	if (page < 0)
		return NULL;
	w = gtk_notebook_get_nth_page(notebook, page);

	for (list = gdwin->gdirv_list; list; list = list->next) {
		GDiffDirViews *gdviews = list->data;
		if (gdviews->base == w) {
			*ret_isdir = TRUE;
			return gdviews;
		}
	}
	for (list = gdwin->gfilev_list; list; list = list->next) {
		GDiffFileViews *gfviews = list->data;
		if (gfviews->base == w) {
			*ret_isdir = FALSE;
			return gfviews;
		}
	}
	
	return NULL;
}


/**
 * dirview_display:
 * Display diff directory, i.e. file names list, on GtkCList.
 * If comparing two files, call a function to display diff files.
 * (See "gdiff.h", I deal with comparing two files as one ot the cases.)
 * Input:
 * GDiffDirViews *gdirviews;
 * Outupt:
 * None;
 **/
void
dirview_display(GDiffDirViews *gdirviews)
{
	if (gdirviews->isdir) {
		GtkCList *clist = gdirviews->clist;

		dirv_draw_clist(gdirviews);
		gtk_signal_connect(GTK_OBJECT(clist), "select_row",
						   GTK_SIGNAL_FUNC(click_dirview), gdirviews);
		gtk_signal_connect(GTK_OBJECT(clist), "click_column",
						   GTK_SIGNAL_FUNC(click_column), gdirviews);
	} else {
		GDiffFileViews *gfviews;

		gfviews = g_slist_nth_data(gdirviews->gdfile_list, 0);
		if (gfviews->pref.pane_mode == ONE_PANE)
			onepane_display(gfviews);
		else
			twopane_display(gfviews);
	}
}

/**
 * dirview_redisplay:
 **/
void
dirview_redisplay(GDiffDirViews *gdirviews)
{
	GtkCList *clist = gdirviews->clist;

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);

	dirv_draw_clist(gdirviews);
}


/* ---The followings are private functions--- */
/**
 * dirv_create_widgets:
 * Called from gdiff_dirviews_new().
 * Create widgets and show them.
 * The contents(diff results) are not shown here, which is up to display_dirs().
 * Input:
 * GDiffDirViews *gdirviews;
 * Output:
 * GDiffDirViews *gdirviews; updated.
 **/
static void
dirv_create_widgets(GDiffDirViews *gdirviews)
{
	GDiffWindow *gdwin = gdirviews->gdwin;
	GtkWidget *hbox;
	GtkWidget *label;
	gint page;

	/* hbox */
	hbox = gtk_hbox_new(FALSE, 0);
	gdirviews->hbox = GTK_HBOX(hbox);
	gdirviews->base = GTK_WIDGET(gdirviews->hbox);/* Virtual base widget. */
	
	/* Add to the notebook */
	label = make_label_for_notebook(gdirviews->dirname1, gdirviews->dirname2, DIR_VIEW);
	gdirviews->label = GTK_LABEL(label);
	gtk_notebook_append_page(gdwin->notebook, hbox, label);

	/* clist area */
	dirv_create_clist(gdirviews, hbox);

	gtk_widget_show(hbox);
	/* Is this a proper way ? */
	page = gtk_notebook_page_num(gdwin->notebook, hbox);
	gtk_notebook_set_page(gdwin->notebook, page);
}

/**
 * dirv_create_clist:
 * Create GtkCList widget for directory view.
 * Input:
 * GDiffDirViews *gdirviews;
 * GtkWidget *hbox; Parent of GtkCList.
 * Output:
 * None;
 **/
static void
dirv_create_clist(GDiffDirViews *gdirviews, GtkWidget *hbox)
{
	GtkWidget *clist;
	GtkWidget *scrolled_window;
	int n;
	char *titles[DVIEW_COLUMNS];
	gint total_width;

	for (n = 0; n < DVIEW_COLUMNS; n++) {
		titles[n] = _(dview_clist[n].title);
	}
	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_box_pack_start(GTK_BOX(hbox),
					   scrolled_window, TRUE, TRUE, 0);
	gtk_widget_show(scrolled_window);

	clist = gtk_clist_new_with_titles(DVIEW_COLUMNS, titles);
	gtk_clist_column_titles_active(GTK_CLIST(clist));

	/* XXX: a very heuristic way to calculate columns size.
	   Try to assign wider areas to file name columns. */
	total_width = GTK_WIDGET(gdirviews->gdwin->notebook)->allocation.width;
	gtk_clist_set_column_width(GTK_CLIST(clist), COL_FILENAME1, total_width/3);
	gtk_clist_set_column_width(GTK_CLIST(clist), COL_FILENAME2, total_width/3);
	gtk_clist_set_column_width(GTK_CLIST(clist), COL_NLINES1, 30);
	gtk_clist_set_column_width(GTK_CLIST(clist), COL_NLINES2, 30);

	gtk_container_add(GTK_CONTAINER(scrolled_window), clist);
	gtk_widget_grab_focus(clist);
	gtk_widget_show(clist);

	gdirviews->clist = GTK_CLIST(clist);
}


/**
 * dirv_draw_clist:
 * Draw directory view's CList,
 * getting the information from back-end data(DiffDir *diffdir).
 * Input:
 * GDiffDirViews *gdirviews;
 * Output:
 * None;
 **/
static void
dirv_draw_clist(GDiffDirViews *gdirviews)
{
	static const char CAPTION_DIFFERENT[] = N_("Different");
	static const char CAPTION_ONLYONE[]	= N_("Only");
	static const char CAPTION_BINARY[] = N_("Binary(different)");
	DiffDir *dir = gdirviews->diffdir;
	GSList *list;
	GtkCList *clist = gdirviews->clist;
	int key = 1;/* when a row is clicked, this is used to access the internal data. */
	int row = 0;
		
	gtk_clist_freeze(clist);

	/* ignore the first node, which has directory names. */
	for (list = dir->dfiles_list->next; list; list = list->next, key++) {
		DiffFiles *files = list->data;
		char *rowtext[DVIEW_COLUMNS];
		FilesSpecialStatus fsst;
		int n;
		char buf1[32];
		char buf2[32];
		const FileInfo *fi1 = dfiles_get_fileinfo(files, FIRST_FILE, FALSE);
		const FileInfo *fi2 = dfiles_get_fileinfo(files, SECOND_FILE, FALSE);
		char fname1[PATH_MAX+1];
		char fname2[PATH_MAX+1];

		fsst = dfiles_get_special_status(files);
		/* Filter to hide the row */
		if (fsst & gdirviews->pref.row_hide_stat_mask)
			continue;
		if (gdirviews->pref.row_hide_func_list && fi1->fname && fi1->fname[0]) {
			if (check_row_hide(gdirviews, fi1->fname) == TRUE)
				continue;
		}
		if (gdirviews->pref.row_hide_func_list && fi2->fname && fi2->fname[0]) {
			if (check_row_hide(gdirviews, fi2->fname) == TRUE)
				continue;
		}

		switch (fsst) {
		case IDENTICAL_FILES:
		case DIRECTORIES:
			g_warning("XXX: weird in display_dirs %d\n", fsst);
			continue;
		case BINARY_FILES:
			rowtext[COL_TYPE] = _(CAPTION_BINARY);
			break;
		case ONLY_FILE1_EXISTS:
		case ONLY_FILE2_EXISTS:
			rowtext[COL_TYPE] = _(CAPTION_ONLYONE);
			break;
		case DIFFERENT_FILES:
			rowtext[COL_TYPE] = _(CAPTION_DIFFERENT);
			break;
		}

		if (gdirviews->pref.show_path == TRUE) {
			rowtext[COL_FILENAME1] = fi1->fname;
			rowtext[COL_FILENAME2] = fi2->fname;
		} else {
			get_file_name(fname1, fi1->fname);
			rowtext[COL_FILENAME1] = fname1;
			get_file_name(fname2, fi2->fname);
			rowtext[COL_FILENAME2] = fname2;
		}

		n = dfiles_calc_total_nlines(files, FIRST_FILE);
		sprintf(buf1, "%d", n);
		rowtext[COL_NLINES1] = buf1;
		n = dfiles_calc_total_nlines(files, SECOND_FILE);
		sprintf(buf2, "%d", n);
		rowtext[COL_NLINES2] = buf2;

		gtk_clist_append(clist, rowtext);

		gtk_clist_set_row_data(GTK_CLIST(clist), row, (gpointer)key);
		row++;
	}
	gtk_clist_thaw(clist);
}

/**
 * check_row_hide:
 * Using RowHideFunc funtions, check whether the row should be hidden.
 * See hide.[ch] about the details.
 * Input:
 * GDiffDirViews *gdirviews;
 * const char *fname; file name of the row.
 * Output:
 * Return value; TRUE if it should be hidden.
 **/
static gboolean
check_row_hide(GDiffDirViews *gdirviews, const char *fname)
{
	GSList *list;

	for (list = gdirviews->pref.row_hide_func_list; list; list = list->next) {
		RowHideFunc rh_func = (RowHideFunc)list->data;
		
		if (rh_func(fname) == TRUE)
			return TRUE;
	}
	return FALSE;
}

/**
 * click_dirview:
 * Called when a user click a row on directory view.
 * Open the files view(diff results) for it.
 * At first, look for the file view among all opened views.
 * If it is found, and it is the same mode as the current favorite mode(g_pref.fvpref.pane_mode), just focus it.
 * If it doesn't exist, open a new view in the current favorite mode.
 **/
static void
click_dirview(GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer data)
{
	GDiffDirViews *gdviews = data;
	GDiffFileViews *gfviews;
	DiffDir *diffdir = gdviews->diffdir;
	DiffFiles *dfiles;
	GSList *list;
	int key;

	
	key = (int)gtk_clist_get_row_data(clist, row);
	dfiles = g_slist_nth_data(diffdir->dfiles_list, key);
	if (is_files_different(dfiles, TRUE) == FALSE)
		return;

	/* Look for the file view among all opened views */
	for (list = gdviews->gdfile_list; list; list = list->next) {
		gfviews = list->data;
		if (gfviews->dfiles == dfiles) {/* Opened */
			gint page;
			page = gtk_notebook_page_num(gfviews->gdwin->notebook, gfviews->base);
			/* Same mode as the current favorite mode */
			if (gfviews->pref.pane_mode == g_pref.fvpref.pane_mode) {
				if (!GTK_WIDGET_VISIBLE(gfviews->base)) {
					gtk_widget_show(gfviews->base);
				}
				gtk_notebook_set_page(gfviews->gdwin->notebook, page);
				return;
			}
		}
	}

	/* When not opened, make it open */	
	gfviews = gdiff_fileviews_new(gdviews->gdwin, gdviews, dfiles, g_pref.fvpref.pane_mode);
	if (gfviews->pref.pane_mode == ONE_PANE)
		onepane_display(gfviews);
	else
		twopane_display(gfviews);
}


/**
 * click_column:
 * Called when a user clicks a title to sort CList.
 **/
static void
click_column(GtkCList *clist, gint column, gpointer data)
{
	if (column == clist->sort_column) {
		if (clist->sort_type == GTK_SORT_ASCENDING)
			clist->sort_type = GTK_SORT_DESCENDING;
		else
			clist->sort_type = GTK_SORT_ASCENDING;
    } else {
		gtk_clist_set_sort_column(clist, column);
		gtk_clist_set_compare_func(clist, dview_clist[column].cmp_func);
	}
	gtk_clist_sort (clist);
}     

/**
 * number_compare:
 * Compare function to sort according to arithmetic value.
 * This is derived from default_compare() in gtkclist.c(GTK+1.2.0).
 **/
static gint
number_compare(GtkCList *clist, gconstpointer ptr1, gconstpointer ptr2)
{
	int num1;
	int num2;
	char *text1 = NULL;
	char *text2 = NULL;
	GtkCListRow *row1 = (GtkCListRow*)ptr1;
	GtkCListRow *row2 = (GtkCListRow*)ptr2;

	switch (row1->cell[clist->sort_column].type) {
    case GTK_CELL_TEXT:       
		text1 = GTK_CELL_TEXT(row1->cell[clist->sort_column])->text;
		break;
    case GTK_CELL_PIXTEXT:    
		text1 = GTK_CELL_PIXTEXT(row1->cell[clist->sort_column])->text; 
		break;
    default:
		break;
    }

	switch (row2->cell[clist->sort_column].type) {
    case GTK_CELL_TEXT:       
		text2 = GTK_CELL_TEXT(row2->cell[clist->sort_column])->text;
		break;
    case GTK_CELL_PIXTEXT:    
		text2 = GTK_CELL_PIXTEXT(row2->cell[clist->sort_column])->text; 
		break;
    default:
		break;
    }

	if (!text2)
		return (text1 != NULL);   
	if (!text1)
		return -1;

	num1 = atoi(text1);
	num2 = atoi(text2);

	return num1 - num2;
}
