// Aspell main C++ include file
// Copyright 1998 by Kevin Atkinson under the terms of the LGPL.

#ifndef __aspell_manager__
#define __aspell_manager__

#include <string>
#include "exception.hh"
#include "emulation.hh"
#include "copy_ptr.hh"
#include "clone_ptr.hh"

namespace autil {
  class StringMap;
}

// The manager class is responsible for keeping track of the
// dictionaries coming up with suggestions and the like Its methods
// are NOT meant to be used my multiple threads and/or documents.

namespace aspell {

  using namespace autil;

  class Language;
  class Config;
  class Suggest;
  class SuggestionList;

  class DataSet;
  class BasicWordSet;
  class WritableWordSet;
  class WritableReplacementSet;

  class Manager {
  public:
    Manager(const string & override);
    Manager(const Config & override);

    Manager(); // does not set anything up. 
    ~Manager();

    void setup(const string & override);
    void setup(const Config & override);

    // use with extreme caution
    Config & mutable_config() {return *config_;}

    const Config & config () const {return *config_;}

    //
    // Low level Word List Management methods
    // (implementation in spell_data.cc)
    //

  public:

    enum SpecialId {main_id, personal_id, session_id, personal_repl_id, none_id};

    typedef Emulation<DataSet *> WordLists;

    WordLists wordlists() const;
    int  num_wordlists() const;

    //these functions are undefined if the requesed wl doesn't exist
    BasicWordSet    &        main_wl();
    WritableWordSet &        personal_wl();
    WritableWordSet &        session_wl();
    WritableReplacementSet & personal_repl();

    const BasicWordSet    &        main_wl() const;
    const WritableWordSet &        personal_wl() const;
    const WritableWordSet &        session_wl() const;
    const WritableReplacementSet & personal_repl() const;

    bool have(const DataSet *) const;
    bool have(SpecialId) const;
    bool attach(DataSet *);
    bool steal(DataSet *);
    bool detach(const DataSet *);
    bool destroy(const DataSet *);
  
    SpecialId check_id(const DataSet *) const;
    void change_id(const DataSet *, SpecialId);
  
    bool use_to_check(const DataSet *) const;
    void use_to_check(const DataSet *, bool);
    bool use_to_suggest(const DataSet *) const;
    void use_to_suggest(const DataSet *, bool);
    bool save_on_saveall(const DataSet *) const;
    void save_on_saveall(const DataSet *, bool);
    bool own(const DataSet *) const;
    void own(const DataSet *, bool);

    bool check_raw (const char *w) const;
    bool check_raw (const string &w) const;

    //
    // Language methods
    //

    const char * lang_name() const;

    const Language & lang() const {return *lang_;}

    //
    // Spelling methods
    //  

    bool check(const string &word) const;
    bool check(const char *word) const;
    bool check_notrim(const string &word) const;
    bool check_notrim(const char *word) const;

    //
    // High level Word List management methods
    //

    void add_to_personal(const string & word);
    void add_to_session(const string & word);

    void save_all_wls();

    void clear_session();

    SuggestionList & suggest(const char * word);
    SuggestionList & suggest(const string & word) {
      return suggest(word.c_str());
    }
    // the suggestion list and the elements in it are only 
    // valid until the next call to suggest.

    bool ignore_repls() const;
    bool ignore_repls(bool);
  
    void store_repl(const string& mis, const string &cor,
		    bool memory = true);

    //
    // Private Stuff (from here to the end of the class)
    //

  private:

    class DataSetCollection;

    void setup();
    void populate();

    CopyPtr<const Language>     lang_;
    CopyPtr<Config>             config_;
    CopyPtr<DataSetCollection>  wls_;
    ClonePtr<Suggest>       suggest_;
    ClonePtr<Suggest>       intr_suggest_;
    unsigned int            ignore_count;
    bool                    ignore_repl;
    string                  prev_mis_repl_;
    string                  prev_cor_repl_;

    void operator= (const Manager &other);
    Manager(const Manager &other);

  public:
    // these are public so that other classes and functions can use them, 
    // DO NOT USE

    const DataSetCollection & word_set_collection() const {return *wls_;}

    void set_check_lang(const string& lang, const string &lang_dir="");
  
    double distance (const char *, const char *, 
		     const char *, const char *) const;

  private:
    void destroy();
  };

  struct DontOwnWordlist : public Exception {
    const DataSet * word_set;
    DontOwnWordlist(const DataSet *wl);
  };
  
  struct ChangeIdException : public Exception {
    Manager::SpecialId try_id;
    DataSet *          word_set;
    ChangeIdException(Manager::SpecialId id, DataSet * ws, const string & m)
      : Exception(m), try_id(id), word_set(ws) {}
  };
  
  struct AlreadyHaveType : public ChangeIdException {
    AlreadyHaveType(Manager::SpecialId id, DataSet * ws);
  };
  
  struct DataSetNotCompatible : public ChangeIdException {
    DataSetNotCompatible(Manager::SpecialId id, DataSet * ws);
  };
}

#endif
