
/* Copyright (C) 1996-1997  Janos Farkas

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */


#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_LASTLOG_H
#include <lastlog.h>
#endif
#include <utmp.h>
#ifdef HAVE_UTMPX_H
#include <utmpx.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#include <sys/syslog.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "finger.h"
#include "util.h"
#include "broken.h"
   
/* Some medium-weight trick */

static sigjmp_buf alarmjmp;
static struct sigaction alarmact;

static void
alarmhandler (int sig)
{
  siglongjmp (alarmjmp, 1);
}

static int
bad_uid(uid_t u)
{
  if (u != 0 && /* root has a chance to have .nofinger */
      (u < (uid_t)FINGER_MINUID || u > (uid_t)FINGER_MAXUID))
    return 1;

  return 0;
}

/* the main finger function, be prepared to nasty things */

void
finger_name (struct fng_info *fi, struct fng_config *fg,
	     const char *name)
{
  const char *p, *sane;
  struct passwd *pwdent;

  if (is_empty (name)) {

    /* Loosen the paranoia, show them the world */
    if (fg->userlist) {
      FILE *f;
      char buf[256];
      struct sigaction oldact;

      /* If it takes longer than 60 secs, it probably
         hung, so don't bother.
         XXX does this really work? (i.e. fgets) */
      alarmact.sa_handler = alarmhandler;
      sigaction (SIGALRM, &alarmact, &oldact);
      if (!sigsetjmp (alarmjmp, 1)) {
        alarm (60);
        f = popen(fg->userlist, "r");
        if (f) {
          while (!feof (f)) {
            if (fgets (buf, sizeof (buf), f))
              fputs (buf, stdout);
          }
          pclose(f);
          alarm (0);
          return;
        }
      } else {
        alarm (0);
        sigaction (SIGALRM, &oldact, NULL);
      }
    }

    /* If we are paranoid, and accept only strict identifiers */
    printf ("finger: %s\n", fg->nouser);
    return;
  }

  p = normalize (name);
  sane = sanitize (p);
  if (is_empty (p) || is_illegal (sane)) {
    printf ("finger: %s\n", fg->nouser);
    return;
  }

  /* first try the login name */
  if ((pwdent = getpwnam (p))) {
    if (!bad_uid(pwdent->pw_uid)) {
      show_user (fi, fg, pwdent);
      return;
    }
  }

  /* try to match gecos field */
  if (fg->match_gecos && (pwdent = scan_gecos (sane))) {
    syslog (LOG_WARNING, "[`%s' -> `%s']", name, pwdent->pw_name);
    show_user (fi, fg, pwdent);
    return;
  }

  printf ("finger: %s\n", fg->nouser);
  syslog (LOG_WARNING, "[`%s' does not exist]", name);
}

/* this function matches two strings for gecos match */

extern int
name_match (const char *p, const char *q)
{
  char *source, *big, *saved, *next;
  const char *found, *last;
  int matched;

  /* Try to convert both string to a too sane format */
  source = sanitize (p);
  big = sanitize (q);
  matched = 0;

  next = source;
  while (next && *next) {
    saved = next;
    next = index (saved, '.');
    if (next) {
      *next = 0;
      next++;
    }

    if (*saved == 0)
      continue;

    /* Try to find the first substring we match */
    found = strstr (big, saved);
    if (found) {
      last = found + strlen (saved);
      if ((found == big || found[-1] == '.') && (*last == '.' || *last == 0)) {
        matched++;			/* correct */
      } else {
        free (source);
        free (big);
        return 1;			/* some part did not match */
      }
    } else {
      free (source);
      free (big);
      return 1;
    }
  }
  free (source);
  free (big);
  return !matched;			/* all nonempty part matched, ok */
}

/* scan the passwd file for a matching name or gecos...
   try to be forgiving, but allow only a single match
   NAME should be normalized */
/* XXX this is ugly */

struct passwd *
scan_gecos (const char *name)
{
  struct passwd *pwdent;
  static struct passwd saved;
  char *lwname, *lwpwname, *gname, *p;
  int matched;

  lwname = xstrdup (name);
  strlwr (lwname);

  matched = 0;
  setpwent ();
  while ((pwdent = getpwent())) {
    if (bad_uid(pwdent->pw_uid))
      continue;

    lwpwname = xstrdup (pwdent->pw_name);

    /* Check qmail user names */
    if (!index (lwpwname, '-') && (p = index (lwname, '-'))) {
      *p = 0;
      if (!strcmp (lwname, lwpwname))
        goto match;
      *p = '-';
    }

    /* One more chance for login name mismatch */
    strlwr (lwpwname);
    if (!strcmp (lwname, lwpwname))
      goto match;

    free (lwpwname);
    gname = xstrdup (pwdent->pw_gecos);
    if ((p = index (gname, ',')))
      *p = 0;

    strlwr (gname);
    lwpwname = normalize (gname);
    free (gname);

    if (!name_match (lwname, lwpwname))
      goto match;

    goto next;

match:
    if (++matched <= 1) {
      saved.pw_name = xstrdup (pwdent->pw_name);
      saved.pw_passwd = NULL;
      saved.pw_uid = pwdent->pw_uid;
      saved.pw_gid = pwdent->pw_gid;
      saved.pw_gecos = xstrdup (pwdent->pw_gecos);
      saved.pw_dir = xstrdup (pwdent->pw_dir);
      saved.pw_shell = xstrdup (pwdent->pw_shell);
    }

next:
    free (lwpwname);
    if (matched > 1)
      break;
  }
  endpwent ();
  free (lwname);

  if (matched > 1) {
    free (saved.pw_name);
    free (saved.pw_gecos);
    free (saved.pw_dir);
    free (saved.pw_shell);
    /* XXX free saved misc */
  }

  return matched == 1 ? &saved : NULL;
}

/* show long info for the specified user */

void
show_user (struct fng_info *fi, struct fng_config *fg,
	   struct passwd *pwdent)
{
  char *gecos;
  char *info[GECOS_MAX];
  struct stat sb;
  int i, pos;
  char *s, *next;
  const char *name;

  name = pwdent->pw_name;

  /* parse the gecos field; note that this may not be standard
     also note that we cannot use strtok, since that skips empty
     fields, and strsep is not portable... fortunately there's
     index () :) */
  gecos = xstrdup (pwdent->pw_gecos);
  for (i = 0, s = gecos; s && *s && i < GECOS_MAX; i++) {
    next = index (s, ',');
    if (next)
      *next++ = 0;
    info[i] = normalize (s);	/* skip blanks */
    s = next;
  }
  while (i < GECOS_MAX)
    info[i++] = xstrdup ("");	/* clear remaining fields */
  free (gecos);

  /* Ok, user has a chance to reject any information */
  if (fi->nofinger) {
    char *fn;
    int shown;

    start_block (0);
    fn = xmakepath (pwdent->pw_dir, fg->nofinger_file);
    shown = display_file (NULL, fn, pwdent->pw_uid);
    free (fn);
    end_block ();
    if (shown)
      return;
  }

  /* start dumping normal info */
  start_block (0);
  fill_block ();
  pos = display_start ();

  if (fi->logname)
    pos = display_field (pos, "Login: ", name);

  if (fi->realname && !is_empty (info[GECOS_NAME]))
    pos = display_field (pos, "Name: ", info[GECOS_NAME]);

  if (fi->homedir && !is_empty (pwdent->pw_dir))
    pos = display_field (pos, "Directory: ", pwdent->pw_dir);

  if (fi->shell)
    pos = display_field (pos, "Shell: ", pwdent->pw_shell);

  if (fi->room && !is_empty (info[GECOS_ROOM]))
    pos = display_field (pos, "Room: ", info[GECOS_ROOM]);

  if (fi->workphone && !is_empty (info[GECOS_WORK]))
    pos = display_field (pos, "Office phone: ", info[GECOS_WORK]);

  if (fi->homephone && !is_empty (info[GECOS_HOME]))
    pos = display_field (pos, "Home phone: ", info[GECOS_HOME]);

  if (fi->other && !is_empty (info[GECOS_OTHER]))
    pos = display_field (pos, "Other: ", info[GECOS_OTHER]);

  pos = display_end (pos);

  /* free space allocated for infos */
  for (i=0; i<GECOS_MAX; i++)
    free (info[i]);

  end_block ();

  if (fi->laston || fi->ifon) {
#ifdef HAVE_UTMPX_H
    struct utmpx *utent;
#else
    struct utmp *utent;
#endif
    char username[UT_NAMESIZE+1];
    int times_in;
    char ttyline[UT_LINESIZE+5+1];	/* "/dev/" */
    char origin[UT_HOSTSIZE+1];
    char act_from[UT_HOSTSIZE+1];
    char log_from[UT_HOSTSIZE+1];
    time_t line_time, act_last, log_last;
    int talkable;

    username[UT_NAMESIZE] = 0;
    origin[UT_HOSTSIZE] = 0;
    act_last = log_last = 0;
    talkable = 0;
    strcpy (ttyline, "/dev/"); ttyline[UT_LINESIZE+5] = 0;

    times_in = 0;
#ifdef HAVE_UTMPX_H
    for (utent=NULL, setutxent (); (utent = getutxent ());)
#else
    for (utent=NULL, setutent (); (utent = getutent ());)
#endif
    {
      memcpy (username, utent->ut_user, UT_NAMESIZE);

      if (utent->ut_type == USER_PROCESS && !strcmp (username, name)) {
#ifdef HAVE_UTMPX_H
	line_time = (time_t)utent->ut_tv.tv_sec;
#else
        line_time = utent->ut_time;
#endif

	/* record last login time & location*/
	if (line_time > log_last ) {
	  memcpy (origin, utent->ut_host, UT_HOSTSIZE);
	  strcpy (log_from, origin);
	  log_last = line_time;
	}

        memcpy (ttyline+5, utent->ut_line, UT_LINESIZE);
	if (stat (ttyline, &sb) == 0) {
	  if (sb.st_atime > line_time)
            line_time = sb.st_atime;	/* could be broken? */
	  if ((sb.st_mode & M_TALKABLE) == M_TALKABLE)
	    talkable++;
	}

	/* record last active time */
        if (line_time > act_last) {
          memcpy (origin, utent->ut_host, UT_HOSTSIZE);
          strcpy (act_from, origin);
          act_last = line_time;
        }
        times_in++;
      }
    }
#ifdef HAVE_UTMPX_H
    endutxent ();
#else
    endutent ();
#endif

#ifdef HAVE_LASTLOG_H
    /* check lastlog file for lastlog information */
    if (!log_last) {
      struct lastlog llent = {0, {0}, {0}};

      if ((freopen (_PATH_LASTLOG, "r", stdin))) {
        fseek (stdin, ((unsigned) pwdent->pw_uid)*sizeof(llent), SEEK_SET);
        fread (&llent, sizeof (llent), 1, stdin);
      }
      if (llent.ll_time > log_last) {
        log_last = llent.ll_time;
        memcpy (origin, llent.ll_host, UT_HOSTSIZE);
        strcpy (log_from, origin);
      }
    }
#else
#warn no lastlog.h
#endif

    start_block (1);
    /* display collected information */
    if (fi->laston) {
      if (log_last == 0) {
        fill_block ();
        printf ("Never logged in.\n");
      } else {
	if (fi->ifon) {
	  if (times_in > 0) {
            fill_block ();
	    printf ("On since %s%s (on %d %s%s)\n",
                    nicedate (&log_last), if_origin (fi, log_from),
		    times_in, times_in == 1 ? "tty" : "ttys",
		    talkable ? "" : ", messages off");
          } else {
            fill_block ();
	    printf ("Last login at %s%s\n", nicedate (&log_last), if_origin (fi, log_from));
	  }
	} else {
          fill_block ();
	  printf ("Last seen on %s%s\n", nicedate (&log_last), if_origin (fi, log_from));
	}
      }
    } else {
      if (fi->ifon) {
	if (times_in > 0) {
          fill_block ();
	  printf ("Presently logged in%s%s.\n", if_origin (fi, log_from),
		  talkable ? "" : " (messages off)");
	} else {
          fill_block ();
	  printf ("Not presently logged in.\n");
	}
      } else {
	/* !laston && !ifon -- also skip it.. */
      }
    }
  }
  end_block ();

  start_block (0);
  if (fi->nmdate || fi->mrdate) {
    char *mailbox;
    int maybebad, statresult;

    maybebad = 0;
    mailbox = xmakepath (_PATH_MAILDIR, name);
    statresult = stat (mailbox, &sb);
    if (statresult != 0) {
      if (errno != ENOENT && errno != ENOTDIR)
        maybebad = 1;
      free(mailbox);
      /* Ok, no go.  Check qmail mailboxes */
      mailbox = xmakepath (pwdent->pw_dir, "Mailbox");
      statresult = stat (mailbox, &sb);
      if (statresult != 0) {
        if (errno != ENOENT && errno != ENOTDIR)
          maybebad = 1;
        free(mailbox);
        /* The /. is important */
        mailbox = xmakepath (pwdent->pw_dir, "Maildir/.");
        statresult = stat (mailbox, &sb);
        if (statresult != 0 && errno != ENOENT && errno != ENOTDIR)
          maybebad = 1;
      }
    }
      
    if (statresult != 0 || sb.st_size <= 0) {
      fill_block ();
      if (maybebad)
        printf ("Unable to check mail status.\n");
      else
        printf ("No unread mail.\n");
    } else {
      if (fi->nmdate && sb.st_mtime > sb.st_atime) {
        fill_block ();
        printf ("New mail received on %s\n", nicedate (&sb.st_mtime));
        if (fi->mrdate) {
  	  printf ("        unread since %s\n", nicedate (&sb.st_atime));
        }
      } else if (fi->mrdate) {
        fill_block ();
        printf ("Last read mail on %s\n", nicedate (&sb.st_atime));
      }
    }

    free (mailbox);
  }
  end_block ();

  start_block (1);
  /* XXX uid swap? (note, we are nobody!) */
  if (fi->plan) {
    char *fn;
    fn = xmakepath (pwdent->pw_dir, fg->plan_file);
    display_file ("Plan:\n", fn, pwdent->pw_uid);
    free (fn);
  }
  end_block ();

  start_block (1);
  if (fi->project) {
    char *fn;
    fn = xmakepath (pwdent->pw_dir, fg->project_file);
    display_file ("Project:\n", fn, pwdent->pw_uid);
    free (fn);
  }
  end_block ();

  start_block (1);
  if (fi->pgp) {
    char *fn;
    fn = xmakepath (pwdent->pw_dir, fg->pgpkey_file);
    display_file ("PGP public key:\n", fn, pwdent->pw_uid);
    free(fn);
  }
  end_block ();
}
