/*
 * Copyright (c) 1997 Loughborough University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the LUTCHI Research
 *      Centre at Loughborough University.
 * 4. Neither the name of the University nor of the Centre may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


/*
 *  File: comms.c
 *
 *  Description: Whiteboard multicast communications.
 *
 *  J.C.Highfield, 3/97.
 *
 */

#define MULTICAST
#define LEN	256

#ifdef SGI_NAP
#define _BSD_TIME
#endif
#ifndef WIN32
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/stat.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_SYS_TIME_H
    #include <sys/time.h>
#else
    #include <time.h>
#endif
#include <pwd.h>
#include <unistd.h>
#include <limits.h>
#include <fcntl.h>
#include <ctype.h>
#include <math.h>

#else

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#undef WIN32_LEAN_AND_MEAN
#include <math.h>
#include <assert.h>
#include <io.h>
#include <process.h>
#include <fcntl.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <winsock.h>
#include <time.h>

#endif

#include <tcl.h>
#include <tk.h>

#include "wb.h"
#include "db.h"
#include "proto.h"

extern Tcl_Interp *tcl;

/* Conference bus (-I 5) address. */
#define CONFBUS_ADDR 0xe0ffdeef
#define CONFBUS_PORT (5 + 0xdeaf)

/* Globals for the conference bus socket. */
int                 confbus_sock = -1;
int	noloopback_broken=0;
uint32	local_address;
unsigned short localport=0;
struct sockaddr_in  confbus_address;

/* Globals for the multicast sockets. */
extern int rsock;
extern int tsock;
extern int maxhops;

/* This host's IP address. */
uint32 ThisIPAddress = 0;

extern int debugmode;
extern unsigned long repairTime;
/* The scaling struct. */
struct scale
{
    float width;
    float height;
    float average;
    int   orient;
} page_scale = {(float)PAGE_WIDTH, (float)PAGE_HEIGHT, 1.0, 1};



/* The font tables. Full of guesses! */

/* Adobe's 15 standard faces? */
#define FFMAX	15
char *FontFamily[FFMAX] = 
{
    "helvetica",
    "courier",
    "times",
    "symbol",
    "new century schoolbook",
    "lucida",
    "lucidabright",
    "lucidatypewriter",
    "palatino",
    "utopia",
    "itc zapf chancery",
    "itc zapf dingbats",
    "itc bookman",
    "itc avant garde gothic",
    "utopia"
};

/* Medium and bold are certein. */
#define FWMAX	12
char *FontWeight[FWMAX] = 
{
    "medium",
    "ultralight",
    "extralight",
    "light",
    "book",
    "demibold",
    "bold",
    "extrabold",
    "ultrabold",
    "semibold",
    "demi",
    "regular",
};

/* First two certain. */
#define FSMAX	3
char *FontSlant[FSMAX] = 
{
    "r",
    "o",
    "i"
};


/*************************************************************/
/*                                                           */
/* Assorted routines.                                        */
/*                                                           */
/*************************************************************/

#ifdef WIN32
int gettimeofday(struct timeval *p, struct timezone *z)
{
    if (p) {
	extern void TclpGetTime(Tcl_Time*);
	Tcl_Time tt;

	TclpGetTime(&tt);
        p->tv_sec = tt.sec;
	p->tv_usec = tt.usec;
    }
    if (z) {
	TIME_ZONE_INFORMATION tz;
	GetTimeZoneInformation(&tz);
	z->tz_minuteswest = tz.Bias ;
	z->tz_dsttime = tz.StandardBias != tz.Bias;
    }
    return 0;
}
#endif

void nonblock(int fd)
{       
#ifdef WIN32
	u_long flag = 1;
	if (ioctlsocket(fd, FIONBIO, &flag) == -1) {
		fprintf(stderr, "ioctlsocket: FIONBIO: %lu\n", GetLastError());
		exit(1);
	}
#else
        int flags = fcntl(fd, F_GETFL, 0);
#if defined(hpux) || defined(__hpux)
        flags |= O_NONBLOCK;
#else
        flags |= O_NONBLOCK|O_NDELAY;
#endif
        if (fcntl(fd, F_SETFL, flags) == -1) {
                perror("fcntl: F_SETFL");
                exit(1);
        }
#endif
}

/* Currently returns the address in host byte order. */
uint32 this_ip (void)
{
    uint32 numLocalhost;
    struct hostent *who;
    char localhost[LEN] = "";
	char *index;
	char *hostaddr="";
	int place;

    if (ThisIPAddress != 0)
        return ThisIPAddress;

    gethostname (localhost, 255);
    who = (struct hostent *) gethostbyname (localhost);
    memcpy (&numLocalhost, who->h_addr_list[0], who->h_length);
    ThisIPAddress = ntohl (numLocalhost);
	local_address=numLocalhost;

    return ThisIPAddress;
}


/* Return a pointer to a static string of the form user@host. */
char *ascii_source (char *newname)
{
    static char result[LEN] = "";
    char localhost[LEN]     = "";
    struct passwd  *pwd;
	char           *uname;
    struct hostent *h;
    int len1, len2;

    /* Are we setting the name? */
    if (newname != NULL)
    {
        strcpy (result, newname);
        return result;
    }

    /* Is this already filled out? */
    if (result[0] != '\0')
        return result;

    /* Find the username. */
#ifdef WIN32
	if ((uname = getenv("USER")) == NULL) {
		uname = "unknown";
	}
        strcpy (result, uname);
#else
    pwd = getpwuid (getuid ());
    if (pwd == NULL)
        strcpy (result, "unknown");
    else
        strcpy (result, pwd->pw_name);
    strcat (result, "@");
#endif
    /* Find the hostname. */
    gethostname (localhost, LEN);
    h =gethostbyname (localhost);

    len1 = strlen (result);
    len2 = strlen (h->h_name);
    if ((len1 + len2 + 2) > LEN)
        len2 = LEN - len1 - 2;
    strncat (result, h->h_name, (1+len2));
    result[LEN - 1] = '\0';

    return result;
}


/* Return a pointer to a static string of the form user:page. */
char *ascii_page (uint32 page_num)
{
    static char result[LEN] = "\0";
    char page[LEN]   = "\0";
    int len1, len2;

    /* Find the username and hostname. */
    strcpy (result, ascii_source(NULL));
    strcat (result, ":");

    /* And the page number string. */
    sprintf (page, "%ld", page_num);

    len1 = strlen (result);
    len2 = strlen (page);
    if ((len1 + len2 + 1) > LEN)
        len2 = LEN - len1 - 1;
    strncat (result, page, (1+len2));
    result[LEN - 1] = '\0';

    return result;
}


/* In hundredths of a second since Jan 1 01:00:00 1970 ?! */
uint32 time_stamp (void)
{
    struct timeval gmt;
    uint32 stamp;

    /* Find the current time... */
    if (gettimeofday (&gmt, NULL) < 0)
    {
        perror ("gettimeofday failed");
        return 0;
    }

    /* gmt now holds time since Jan 1 00:00:00 1970. */

    stamp = (gmt.tv_sec * 100) + (gmt.tv_usec / 10000);

    return stamp;
}

/* Build a tag. Returns an empty list for nonlocal commands. */
char *build_tag (uint32 cmd_source, uint32 cmd_id)
{
    static char result[32];

    if (cmd_source != this_ip())
        sprintf (result, "seqno:%ld:%08X", cmd_id, cmd_source);
    else
        sprintf (result, "local:%ld", cmd_id);

    return result;
}

/* Byte reverse a real32. */
#if BYTE_ORDER == LITTLE_ENDIAN

real32 htonf (real32 x)
{
    real32 r;
    uint8 *s1 = (uint8 *) &x;
    uint8 *s2 = (uint8 *) &r;

    s2[0] = s1[3];
    s2[1] = s1[2];
    s2[2] = s1[1];
    s2[3] = s1[0];

    return r;
}

real32 ntohf (real32 x)
{
    real32 r;
    uint8 *s1 = (uint8 *) &x;
    uint8 *s2 = (uint8 *) &r;

    s2[0] = s1[3];
    s2[1] = s1[2];
    s2[2] = s1[1];
    s2[3] = s1[0];

    return r;
}

#else

#define htonf(x)	x
#define ntohf(x)	x

#endif

unsigned short getlocalport(int tx_fd)
{
    struct sockaddr_in local;
    int len;

    memset((char *) &local, 0, sizeof(local));
    local.sin_family = AF_INET;
    len = sizeof(local);
    if (getsockname(tx_fd, (struct sockaddr *) &local, &len) < 0) {
	perror("getsockname");
    }
    return local.sin_port;
}

/*************************************************************/
/*                                                           */
/* Network I/O routines.                                     */
/*                                                           */
/*************************************************************/



int
recv_sock_init(struct sockaddr_in *address)
{
#ifndef WIN32
	char	ttl = (char)maxhops;
#else
	int	ttl = maxhops;
#endif
	int	tmp_fd;
	int	multi = 0;
	int	reuse = 1;
	int	bsize = 65536;
	uint32	inaddr;
	struct hostent *addr;

	/*
	 * Must examine inaddr to see if this is a multicast address and set
	 * multi to TRUE
	 */
	inaddr= address->sin_addr.s_addr;
	multi = IN_MULTICAST(ntohl(inaddr));

	if ((tmp_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		exit(-1);
	}

	if (setsockopt(tmp_fd, SOL_SOCKET, SO_REUSEADDR, (char *) &reuse, sizeof(reuse)) < 0) {
		perror("setsockopt SO_REUSEADDR");
	}

	/* Bind the socket to the address. */
    if (bind(tmp_fd, (struct sockaddr *)address, sizeof (struct sockaddr_in)) < 0)
    {
        /* Some systems need this instead. */
        address->sin_addr.s_addr = INADDR_ANY;
        if (bind(tmp_fd, (struct sockaddr *)address, sizeof (struct sockaddr_in)) < 0)
        {
            perror("Binding datagram socket");
            return -1;
        }
	}

	address->sin_addr.s_addr=inaddr;

	if (multi) {
		char            loop = 0;
		struct ip_mreq  imr;

	    memcpy((char *) &imr.imr_multiaddr.s_addr, (char *) &inaddr, sizeof(inaddr));
		imr.imr_interface.s_addr = htonl(INADDR_ANY);

		if (setsockopt(tmp_fd, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *) &imr, sizeof(struct ip_mreq)) == -1)
			fprintf(stderr, "IP multicast join failed!\n");
	}
	if (setsockopt (tmp_fd, SOL_SOCKET, SO_RCVBUF, (void *)&bsize, sizeof (bsize)) == -1) {
		perror("Failed to set SO_RCVBUF socket option"); 
		return -1;
	}
	nonblock(tmp_fd);

	return tmp_fd;
}

int
send_sock_init(struct sockaddr_in *address)
{
#ifndef WIN32
	char	ttl = (char)maxhops;
#else
	int		ttl = maxhops;
#endif
	int		tmp_fd;
	int		multi = 0;
	int		reuse = 1;
	uint32	inaddr;
	int		bsize = 65536;
	struct	hostent *addr;

	/*
	 * Must examine inaddr to see if this is a multicast address and set
	 * multi to TRUE
	 */
	inaddr= address->sin_addr.s_addr;
	multi = IN_MULTICAST(ntohl(inaddr));

	if ((tmp_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		exit(-1);
	}

	/* Connect to the group. */
    if (connect(tmp_fd, (struct sockaddr *)address, sizeof (struct sockaddr_in)) < 0)
    {
        perror("Connecting datagram transmit socket");
        return -1;
    }

	if (multi) {
		char            loop = 0;
		struct ip_mreq  imr;

	    imr.imr_multiaddr.s_addr = address->sin_addr.s_addr;
		imr.imr_interface.s_addr = htonl(INADDR_ANY);

		if (setsockopt(tmp_fd, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char *) &imr, sizeof(struct ip_mreq)) == -1)
			fprintf(stderr, "IP multicast join failed!\n");

		setsockopt(tmp_fd, IPPROTO_IP, IP_MULTICAST_LOOP, &loop, sizeof(loop));
#ifdef WIN32
		noloopback_broken = 1;
		localport = getlocalport(tmp_fd);
#endif
		if (setsockopt(tmp_fd, IPPROTO_IP, IP_MULTICAST_TTL, &ttl, sizeof(ttl)) < 0)
			perror("setsockopt IP_MULTICAST_TTL");
	}

	/* setup send buffer*/
	if (setsockopt (tmp_fd, SOL_SOCKET, SO_SNDBUF, (void *)&bsize, sizeof (bsize)) == -1) {
        perror("Failed to set SO_SNDBUF socket option"); 
        return -1;
    }

	return tmp_fd;
}

int NetFetch (int sock, struct sockaddr_in *address, char *buf, int buflen)
{
	int len, rlen;
	int remlength;
	int error;
	struct sockaddr_in from;
	int fromlen;
	struct timeval  timeout, *tvp;
	fd_set          rfds;

	/* Read the packet. */
	fromlen=sizeof(struct sockaddr);

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);
	timeout.tv_sec  = 0;
	timeout.tv_usec = 40000;
	tvp = &timeout;
	select(1, &rfds, NULL, NULL, tvp);
	rlen = recvfrom (sock, buf, buflen, 0, address, &fromlen);

	if (rlen < 0)
	{
#ifdef WIN32
		error=WSAGetLastError();
#endif
		return -1;
	}
	else if (rlen == 0)
        		return 0;

	/* noloopback broken on windows */
#ifdef WIN32
	if (noloopback_broken && address->sin_addr.s_addr == local_address && address->sin_port == localport) return 0;
#endif

	buf[buflen-1] = 0;
	return rlen;
}

/* Send a multicast message. */
int NetSend (char *message, int messlen)
{
    int result;
    if ((message == NULL) || (messlen <= 0))
        return 0;

    result = send (tsock, message, messlen, 0);

    if (result < 0)
    {
        perror("wbd: Sending message");
        return TCL_ERROR;
    }

    return 0;
}

/* Print out a message as hexadecimal. */
int NetDump (char *buf, int mesglen)
{
    int i;
    unsigned char *d = (unsigned char *) buf;

    printf ("Dump: ");
    for (i = 0; i < mesglen; i++)
        printf ("%02X ", d[i]);
    printf ("\n");

    return 0;
}


/*************************************************************/
/*                                                           */
/* Routines for parsing incoming messages.                   */
/*                                                           */
/*************************************************************/


/* Parse a whiteboard Session packet. */
int NetParseSession (char *buf, int mesglen)
{
    struct pk_session  *Session = NULL;
    struct writer_desc *wd      = NULL;
    struct writer      *writer  = NULL;
    int offset, i;
    char *str;

    Session = (struct pk_session *)buf;

    if (mesglen < sizeof(struct pk_session))
        return 0;

    /*printf ("PK_SESSION %8X ", Session->pk_type);*/
    /*printf ("=> %d pages\n", Session->page_count);*/

    /* Update the database with this knowledge. */
    SessionUpdate (Session);

    if (PageFind (ntohl(Session->last_page_num), ntohl(Session->last_page_ip)) == NULL)
        PageAdd (ntohl(Session->last_page_num), ntohl(Session->last_page_ip));
    if (PageFind (ntohl(Session->current_page_num), ntohl(Session->current_page_ip)) == NULL)
        PageAdd (ntohl(Session->current_page_num), ntohl(Session->current_page_ip));

    if (PageCurrent() == NULL)
        PageSet (ntohl(Session->current_page_num), ntohl(Session->current_page_ip));

    /* Update the database. */
    offset = sizeof (struct pk_session);
    for (i = 0; i < ntohs(Session->active); i++)
    {
        wd = (struct writer_desc *) (offset + buf);
        offset += sizeof (struct writer_desc);

        writer = WriterAdd (ntohl(Session->current_page_num),
                            ntohl(Session->current_page_ip),
                            ntohl(wd->writer_ip));
        if (writer != NULL)
        {
            /*printf ("Session: %8X %8X %ld\n", ntohl(Session->sender), 
                            ntohl(wd->writer_ip), ntohl(wd->write_count));*/

            if (ntohl(wd->write_count) != 0)
                if ((ntohl(wd->write_count) - 1) > writer->seq_num_max)
                    writer->seq_num_max = ntohl(wd->write_count) - 1;
        }
    }

    /* Print the ID string. */
    str = (char *) ((ntohs(Session->active) * sizeof (struct writer_desc)) +
          + sizeof (struct pk_session) + (char *)Session);

    /* Update the sender name. */
    SenderAdd (ntohl(Session->sender), 
               ntohl(Session->timestamp), 
               time_stamp(), 
               PK_SESSION, str);

    return 0;
}

/* Parse a whiteboard Request Draw (request repair) packet. */
int NetParseReqDraw (char *buf, int mesglen, int type)
{
    struct pk_req_draw *ReqDraw = NULL;
    struct pk_rpy_draw *RpyDraw = NULL;
    struct mqueue      *q;
    uint32 size;
    int done;

    ReqDraw = (struct pk_req_draw *)buf;

    if (mesglen < sizeof(struct pk_req_draw))
        return 0;

    /*printf ("PK_REQ_DRAW (%8X) ", ntohl(ReqDraw->pk_type));
    printf ("=> From=%08X Writer=%08X Page=%08X:%ld C1=%ld C2=%ld\n", 
            ntohl(ReqDraw->sender), ntohl(ReqDraw->drawer_ip),
            ntohl(ReqDraw->page_ip), ntohl(ReqDraw->page_num),
            ntohl(ReqDraw->first_cmd), ntohl(ReqDraw->last_cmd));*/

    /* NetDump (buf, mesglen); */

    /* Check to see if we queued an identical request. */
    if ((q = QueueCheckRequest (ReqDraw)) != NULL)
        QueueRemove (q);

    /* Queue the requested list of draw commands. */
    done = 0;
    while (!done)
    {
        RpyDraw = build_reply_draw (ntohl(ReqDraw->page_ip), 
                    ntohl(ReqDraw->page_num), ntohl(ReqDraw->drawer_ip),
                    ntohl(ReqDraw->first_cmd), ntohl(ReqDraw->last_cmd),
                    &size);
        if (RpyDraw == NULL)
            return -1;

        /*if (ntohl(ReqDraw->first_cmd) == ntohl(ReqDraw->last_cmd))
            NetDump ((char *) RpyDraw, size);*/

        /* Check to see if we queued a similar repair */
        /* and if so delete the old repairs.          */
        if ((q = QueueCheckRepair (RpyDraw)) != NULL)
            QueueRemove (q);

        QueueAddTo (RpyDraw, size, Q_REPAIR);

        if (ntohl(RpyDraw->last_cmd) < ntohl(ReqDraw->last_cmd))
            ReqDraw->first_cmd = htonl(ntohl(RpyDraw->last_cmd) + 1);
        else
            done = 1;
    }

    return 0;
}

/* Parse a whiteboard Ask Page packet. */
int NetParseAskPage (char *buf, int mesglen)
{
    struct pk_ask_page *AskPage;
    struct pk_page_rpt *PageRpt;
    struct mqueue      *q;
    uint32 size;

    AskPage = (struct pk_ask_page *)buf;

    if (mesglen < sizeof(struct pk_ask_page))
        return 0;

    /*printf ("PK_ASK_PAGE %8X ", ntohl(AskPage->pk_type));
    printf ("=> From=%08X Page=%08X:%ld D1=%08X D2=%08X\n", 
            ntohl(AskPage->sender),
            ntohl(AskPage->page_ip), ntohl(AskPage->page_num), 
            ntohl(AskPage->page_index), ntohl(AskPage->d2));*/

    /* Check to see if we queued an identical request. */
    if ((q = QueueCheckAskPage (AskPage)) != NULL)
        QueueRemove (q);

    /* Queue a Page Report. */
    PageRpt = build_page_report (ntohl(AskPage->page_num), 
                                 ntohl(AskPage->page_ip),
                                 ntohl(AskPage->page_index),
                                 &size);
    if (PageRpt != NULL)
    {

        /* Check to see if we queued a similar report */
        /* and if so delete the old report.           */
        if ((q = QueueCheckReport (PageRpt)) != NULL)
            QueueRemove (q);

        QueueAddTo (PageRpt, size, Q_REPORT);
    }

    return 0;
}

/* Parse a whiteboard Page Report packet. */
int NetParsePageRpt (char *buf, int mesglen)
{
    struct pk_page_rpt *PageRpt;
    struct page_desc   *pd     = NULL;
    struct writer_desc *wd     = NULL;
    struct page        *page;
    struct writer      *writer;
    struct mqueue      *q;
    int i, j, pages, writers;
    int offset;

    PageRpt = (struct pk_page_rpt *)buf;

    if (mesglen < sizeof(struct pk_page_rpt))
        return 0;

    /*printf ("PK_PAGE_RPT %8X (%ld pages, %d bytes)\n", 
        ntohl(PageRpt->pk_type), ntohl(PageRpt->count), mesglen);*/

    /* Check to see if we queued an identical report. */
    if ((q = QueueCheckReport (PageRpt)) != NULL)
        QueueRemove (q);

    /* Update the database to reflect this information. */
    pages  = ntohl(PageRpt->count);
    offset = sizeof (struct pk_page_rpt);

    /* NetDump (buf, mesglen); */

    /* For each page in the report... */
    for (i = 0; i < pages; i++)
    {
        if (offset > (mesglen - sizeof(struct page_desc)))
            break;

        pd = (struct page_desc *)(offset + 
                                   (char *)PageRpt);
        offset += sizeof (struct page_desc);

        /* Ensure the page is in the database. */
        page = PageAdd (ntohl(pd->page_num), ntohl(pd->page_id));
        if (page != NULL)
        {
            /* Check and update the summary... */
            writers = ntohs(pd->active);
            for (j = 0; j < writers; j++)
            {
                wd = (struct writer_desc *)(offset + 
                                        (char *)PageRpt);
                offset += sizeof (struct writer_desc);
                if (offset > mesglen)
                    break;

                /* Ensure the writer is in the database. */
                writer = WriterAdd (ntohl(pd->page_num), ntohl(pd->page_id),
                                                 ntohl(wd->writer_ip));
                if (writer != NULL)
                {
                    /* Update the writer status. */
                    if (writer->seq_num_max < (ntohl(wd->write_count) - 1))
                        writer->seq_num_max = ntohl(wd->write_count) - 1;
                }
            }
        }
    }

    return 0;
}

/* Parse a whiteboard Reply Draw (repair) packet. */
int NetParseRpyDraw (char *buf, int mesglen)
{
    struct pk_rpy_draw *RpyDraw = NULL;
    struct pk_draw_hdr *DrawHdr = NULL;
    struct pk_draw_msg *mesg    = NULL;
    struct page        *page    = NULL;
    struct mqueue      *q;
    char *DrawMsg = NULL;
    int i, offset;
    int dirty = 0;

    RpyDraw = (struct pk_rpy_draw *)buf;

    if (mesglen < sizeof(struct pk_rpy_draw))
        return 0;

    /*printf ("PK_RPY_DRAW %8X ", ntohl(RpyDraw->pk_type));
    printf ("=> From=%08X Writer=%08X Page=%08X:%ld C1=%ld C2=%ld\n", 
            ntohl(RpyDraw->sender), ntohl(RpyDraw->drawer_ip),
            ntohl(RpyDraw->page_ip), ntohl(RpyDraw->page_num),
            ntohl(RpyDraw->first_cmd), ntohl(RpyDraw->last_cmd));*/
    /* NetDump (buf, mesglen); */

    /* Check to see if we queued an identical reuest. */
    if ((q = QueueCheckRepair (RpyDraw)) != NULL)
        QueueRemove (q);

    /* Add this information into the database. */
    page = PageFind (ntohl(RpyDraw->page_num), ntohl(RpyDraw->page_ip));
    if (page == NULL)
        return 0;

    offset = sizeof (struct pk_rpy_draw);
    for (i = ntohl(RpyDraw->first_cmd); i <= ntohl(RpyDraw->last_cmd); i++)
    {
        if (offset > (mesglen - sizeof(struct pk_draw_hdr)))
        {
            /*printf ("message too short at C=%d\n", i);*/
            break;
        }

        DrawHdr = (struct pk_draw_hdr *) (offset + buf);
        offset += sizeof (struct pk_draw_hdr);

        if (offset > (mesglen - ntohs(DrawHdr->draw_cmd_length)))
            break;

        DrawMsg = offset + buf;
        if (ActionFind(ntohl(RpyDraw->page_num), ntohl(RpyDraw->page_ip),
                             ntohl(RpyDraw->drawer_ip), i) == NULL)
        {
            /* ActionAdd() takes pk_draw_msg, so... */
            mesg = malloc (sizeof(struct pk_draw_msg) +
                           ntohs(DrawHdr->draw_cmd_length));
            if (mesg != NULL)
            {
                /* These are all in network byte order. */
                mesg->sender    = RpyDraw->drawer_ip;
                mesg->timestamp = RpyDraw->timestamp;
                mesg->pk_type   = htonl(PK_DRAW_MSG);
                mesg->page_ip   = RpyDraw->page_ip;
                mesg->page_num  = RpyDraw->page_num;

                mesg->draw_cmds_local = RpyDraw->last_cmd; /* Wrong. */
                mesg->draw_cmd_num    = htonl(i);
                mesg->draw_timestamp  = DrawHdr->draw_timestamp;
                mesg->draw_cmd_length = DrawHdr->draw_cmd_length;
                mesg->draw_cmd        = DrawHdr->draw_cmd;

                memcpy (sizeof(struct pk_draw_msg) + (char *)mesg, 
                        (offset + buf),
                        ntohs(DrawHdr->draw_cmd_length));

                /*printf ("Add action %d.\n", i);*/
                ActionAdd(mesg);
                dirty = 1;
            }
        }
        offset += ntohs(DrawHdr->draw_cmd_length);
        while ((offset % 4) != 0)
            offset++;
    }

    if ( (page == PageCurrent()) && 
         (ntohl(RpyDraw->sender) != this_ip()) &&
         (dirty == 1) )
    {
		/* wait two seconds just incase there is more repairs */
		repairTime = time(NULL)+1;
    }

    return 0;
}

/* Parse a whiteboard Draw Rect packet. */
int ParseDrawRect (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_rect *DrawRect;
    char arg[LEN];
    float x1, y1;
    float x2, y2;
    float w;

    DrawRect = (struct drw_rect *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_rect))
        return 0;

    x1 = ntohf(DrawRect->from_x);
    y1 = ntohf(DrawRect->from_y);
    x2 = ntohf(DrawRect->to_x);
    y2 = ntohf(DrawRect->to_y);
    w  = ntohf(DrawRect->linestyle);
    PageToCanvasXY (&x1, &y1);
    PageToCanvasXY (&x2, &y2);
    PageToCanvasW  (&w);

    sprintf (arg, "draw_rect %f #%06X fill %f %f %f %f %s", 
        w, (ntohl(DrawRect->colour) >> 8), 
        x1, y1, x2, y2, tag );

    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_rect failed: %s\n", tcl->result);
    }

    if (debugmode) fprintf(stdout, "%s\n", arg);
    return 0;
}

/* Parse a whiteboard Draw Line packet. */
int ParseDrawLine (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_line *DrawLine;
    char arg[LEN];
    float x1, y1;
    float x2, y2;
    float w;

    DrawLine = (struct drw_line *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_line))
        return 0;

    x1 = ntohf(DrawLine->from_x);
    y1 = ntohf(DrawLine->from_y);
    x2 = ntohf(DrawLine->to_x);
    y2 = ntohf(DrawLine->to_y);
    w  = ntohf(DrawLine->linestyle);
    PageToCanvasXY (&x1, &y1);
    PageToCanvasXY (&x2, &y2);
    PageToCanvasW  (&w);

    sprintf (arg, "draw_line %f #%06X %f %f %f %f %s", 
        w, (ntohl(DrawLine->colour) >> 8), 
        x1, y1, x2, y2, tag );

    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_line failed: %s\n", tcl->result);
    }
    if (debugmode) fprintf(stdout, "%s\n", arg);
    return 0;
}

/* Parse a whiteboard Draw Group Lines packet. */
int ParseDrawGrpLines (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_grp_lines *DrawGrpLines;
    char arg[LEN];

    DrawGrpLines = (struct drw_grp_lines *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_grp_lines))
        return 0;

    if (ntohl(DrawMsg->sender) != this_ip())
        sprintf (arg, "draw_group_lines %ld %ld seqno: :%08X %ld", 
            ntohl(DrawGrpLines->first_cmd), ntohl(DrawGrpLines->last_cmd),
            ntohl(DrawMsg->sender), ntohl(DrawMsg->draw_cmd_num) );
    else
        sprintf (arg, "draw_group_lines %ld %ld local: \"\" %ld", 
            ntohl(DrawGrpLines->first_cmd), ntohl(DrawGrpLines->last_cmd),
            ntohl(DrawMsg->draw_cmd_num) );

    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_group_lines failed: %s\n", tcl->result);
    }
    if (debugmode) fprintf(stdout, "%s\n", arg);
    return 0;
}

/* Redraws an object, recursively handling grouped objects. */
int ReDrawAction (uint32 page_num, uint32 page_ip, uint32 sender, uint32 seqnum)
{
    struct pk_draw_msg   *DrawMsg      = NULL;
    struct action        *action       = NULL;
    struct drw_grp_lines *DrawGrpLines = NULL;
    struct drw_grp_ps    *DrawGrpPS    = NULL;
    uint32 first, last;
    int i;

    action = ActionFind (page_num, page_ip, sender, seqnum);
    if (action == NULL)
        return -1;

    DrawMsg = action->message;

    switch (DrawMsg->draw_cmd)
    {
        case DRW_GRP_TEXT:
        case DRW_GRP_LINES:
            DrawGrpLines = (struct drw_grp_lines *) &DrawMsg[1];
            first = ntohl(DrawGrpLines->first_cmd);
            last  = ntohl(DrawGrpLines->last_cmd);

            /* First draw the objects to be grouped */
            for (i = first; i <= last; i++)
                ReDrawAction (page_num, page_ip, sender, i);

            break;

        case DRW_GRP_PS:
            DrawGrpPS = (struct drw_grp_ps *) &DrawMsg[1];
            first = ntohl(DrawGrpPS->first_cmd);
            last  = ntohl(DrawGrpPS->last_cmd);

            /* First draw the objects to be grouped */
            for (i = first; i <= last; i++)
                ReDrawAction (page_num, page_ip, sender, i);

            break;

        default:
            break;
    }

    ParseDrawMsg (DrawMsg, size_command (DrawMsg));
}

/* Parse a whiteboard Draw Delete packet. */
int ParseDrawDel (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_del *DrawDel;
    struct action  *action;
    char arg[LEN];
    uint32 first, last;
    uint32 page_num, page_ip;
    uint32 sender, i;

    DrawDel = (struct drw_del *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_del))
        return 0;

    first = ntohl(DrawDel->first_cmd);
    last  = ntohl(DrawDel->last_cmd);
    page_num = ntohl(DrawMsg->page_num);
    page_ip  = ntohl(DrawMsg->page_ip);
    sender   = ntohl(DrawMsg->sender);

    if (first == last)
    {
        action = ActionFind (page_num, page_ip, sender, first);
        if (action != NULL)
        {
            if (action->message->draw_cmd == DRW_DEL)
            {
                /* Deleting a delete...            */
                /*   redraw the deleted object(s). */
                /*if (debugmode) fprintf (stdout, "DRW_DEL \n");*/

                DrawDel = (struct drw_del *)(&action->message[1]);
                first = ntohl(DrawDel->first_cmd);
                last  = ntohl(DrawDel->last_cmd);
                for (i = first; i <= last; i++)
                    ReDrawAction (page_num, page_ip, sender, i);

                return 0;
            }
        }
        else
            /* Give up, the deleted message isn't here! */
            return 0;
    }

    /* Delete the object(s). */
    for (i = first; i <= last; i++)
    {
        sprintf (arg, "draw_delete %s", build_tag (sender, i) );

        if (Tcl_Eval(tcl, arg) != TCL_OK)
        {
            printf ("draw_delete failed: %s\n", tcl->result);
        }
    }
    if (debugmode) fprintf(stdout, "%s\n", arg);
    return 0;
}

/* Parse a whiteboard Draw Copy packet. */
int ParseDrawCopy (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_copy *DrawCopy;
    struct action   *action   = NULL;
    char arg[LEN];
    float x, y;

    DrawCopy = (struct drw_copy *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_copy))
        return 0;

    x = ntohf(DrawCopy->to_x);
    y = ntohf(DrawCopy->to_y);
    PageToCanvasXY (&x, &y);

    /* Make a copy (with the new tag) and then move it. */
    sprintf (arg, "draw_copy %s %s", 
                build_tag (ntohl(DrawMsg->sender), ntohl(DrawCopy->cmd_id)),
                tag );
    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_copy failed: %s\n", tcl->result);
    }
    if (debugmode) fprintf(stdout, "%s\n", arg);

    sprintf (arg, "draw_move %s %f %f", tag, x, y );
    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_move failed: %s\n", tcl->result);
    }
    if (debugmode) fprintf(stdout, "%s\n", arg);
    return 0;
}

/* Parse a whiteboard Draw Oval packet. */
int ParseDrawOval (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_oval *DrawOval;
    char arg[LEN];
    float x1, y1;
    float x2, y2;
    float w;

    DrawOval = (struct drw_oval *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_oval))
        return 0;

    x1 = ntohf(DrawOval->center_x) - ntohf(DrawOval->width);
    y1 = ntohf(DrawOval->center_y) - ntohf(DrawOval->height);
    x2 = ntohf(DrawOval->center_x) + ntohf(DrawOval->width);
    y2 = ntohf(DrawOval->center_y) + ntohf(DrawOval->height);
    w  = ntohf(DrawOval->linestyle);
    PageToCanvasXY (&x1, &y1);
    PageToCanvasXY (&x2, &y2);
    PageToCanvasW  (&w);

    sprintf (arg, "draw_oval %f #%06X fill %f %f %f %f %s", 
        w, (ntohl(DrawOval->colour) >> 8), 
        x1, y1, x2, y2, tag );

    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_oval failed: %s\n", tcl->result);
    }
    if (debugmode) fprintf(stdout, "%s\n", arg);
    return 0;
}

/* Parse a whiteboard Draw PostScript packet. */
int ParseDrawPS (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_ps *DrawPs;
    char arg[LEN];
    int i;

    DrawPs = (struct drw_ps *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_ps))
        return 0;

    /*printf ("DrawPS: ");
    for (i = 0; i < mesglen; i++)
        printf ("%c", ((char *)DrawMsg)[i]);
    printf ("\n");*/

    /* Don't do anything. This is only part of a */
    /* PS document. Wait for a later DRW_GRP_PS. */

    return 0;
}

/* Parse a whiteboard Draw Group PostScript packet. */
int ParseDrawGrpPS (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_grp_ps  *DrawGrpPs;
    struct drw_ps      *DrawPs  = NULL;
    struct action      *action  = NULL;
    struct pk_draw_msg *message = NULL;
    char arg[LEN];
    char filename[L_tmpnam] = "";
    FILE *tmpfile;
    int i, result;

    DrawGrpPs = (struct drw_grp_ps *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_grp_ps))
        return 0;

    /*printf ("DrawGrpPS: from %ld to %ld\n", ntohl(DrawGrpPs->first_cmd),
                                   ntohl(DrawGrpPs->last_cmd));*/

    /*NetDump ((uint8 *) DrawMsg, mesglen);*/

    /* Check that all the pieces are here! */
    for (i = ntohl(DrawGrpPs->first_cmd); i < (ntohl(DrawGrpPs->last_cmd))+1; i++)
    {
        action = ActionFind (ntohl(DrawMsg->page_num), ntohl(DrawMsg->page_ip), 
                           ntohl(DrawMsg->sender), i);
        if (action == NULL)
        {
            /*printf ("failed - PostScript not all here.\n");*/
            return -1;
        }
    }

    /* Dump the PostScript to a temp file. */
	if (tmpnam (filename) == NULL)
    {
        printf ("Failed to create temp file.\n");
        return -1;
    }
    tmpfile = fopen (filename, "wb");
    if (tmpfile == NULL)
    {
        printf ("Failed to create temp file.\n");
        return -1;
    }
    for (i = ntohl(DrawGrpPs->first_cmd); i < (ntohl(DrawGrpPs->last_cmd))+1; i++)
    {
        action = ActionFind (ntohl(DrawMsg->page_num), ntohl(DrawMsg->page_ip), 
                                              ntohl(DrawMsg->sender), i);
        /* DUMP */
        message = action->message;
        result = fwrite ((&message[1]), 1, 
                             ntohs(message->draw_cmd_length), tmpfile);
        if (result <= 0)
        {
            printf ("Failed to write to temp file.\n");
            fclose (tmpfile);
            return -1;
        }
    }
    fclose (tmpfile);

    /* Render the PostScript, using Ghostscript. */
    /* Done in tcl for reconfigurability.        */
    sprintf (arg, "draw_postscript \\%s", filename);
    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_postscript failed: %s\n", tcl->result);
    }
#ifdef WIN32
    remove (filename);
#else
    unlink (filename);
#endif
    return 0;
}

/* Parse a whiteboard Draw Character packet. */
int ParseDrawChr (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_chr *DrawChr;
    char arg[LEN];
    char c;
    float x, y;
    float w;
    long family;
    long weight;
    long slant;

    DrawChr = (struct drw_chr *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_chr))
        return 0;

    x = ntohf(DrawChr->where_x);
    y = ntohf(DrawChr->where_y);
    w = (float) (ntohl(DrawChr->font) & 0xff);
    PageToCanvasXY (&x, &y);
    PageToCanvasW  (&w);

    family = (ntohl(DrawChr->font) >> 8) & 0xff;
    if (family >= FFMAX) family = 0;
    weight = (ntohl(DrawChr->font) >> 16) & 0x0f;
    if (weight >= FWMAX) weight = 0;
    slant  = (ntohl(DrawChr->font) >> 20) & 0x0f;
    if (slant >= FSMAX) slant = 0;

    /* The range of characters is limited by draw_chr. */
    c = ntohs(DrawChr->ascii);
    if ((c == '\\') || (c == '{') || (c == '}') || 
                       (c == '[') || (c == ']'))
        sprintf (arg, "draw_chr {\\%c} %ld #%06X %f %f \"%s\" %s %s %ld %s", 
            ntohs(DrawChr->ascii), ntohs(DrawChr->orientation), 
            (ntohl(DrawChr->colour) >> 8), x, y,
            FontFamily[family], FontWeight[weight], FontSlant[slant],
            (long) w, tag );
    else
        sprintf (arg, "draw_chr {%c} %ld #%06X %f %f \"%s\" %s %s %ld %s", 
            ntohs(DrawChr->ascii), ntohs(DrawChr->orientation), 
            (ntohl(DrawChr->colour) >> 8), x, y,
            FontFamily[family], FontWeight[weight], FontSlant[slant],
            (long) w, tag );

    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_chr failed: %s\n", tcl->result);
    }
    if (debugmode) fprintf(stdout, "%s\n", arg);
    return 0;
}

/* Parse a whiteboard Draw Group Text packet. */
int ParseDrawGrpText (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_grp_text *DrawGrpText;
    char arg[LEN];

    DrawGrpText = (struct drw_grp_text *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_grp_text))
        return 0;

    if (ntohl(DrawMsg->sender) != this_ip())
        sprintf (arg, "draw_group_chrs %ld %ld seqno: :%08X %ld", 
            ntohl(DrawGrpText->first_cmd), ntohl(DrawGrpText->last_cmd),
            ntohl(DrawMsg->sender), ntohl(DrawMsg->draw_cmd_num) );
    else
        sprintf (arg, "draw_group_chrs %ld %ld local: \"\" %ld", 
            ntohl(DrawGrpText->first_cmd), ntohl(DrawGrpText->last_cmd),
            ntohl(DrawMsg->draw_cmd_num) );

    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("draw_group_text failed: %s\n", tcl->result);
    }
    if (debugmode) fprintf(stdout, "%s\n", arg);
    return 0;
}

/* Parse a whiteboard New Page packet. */
int ParseDrawNewPage (struct pk_draw_msg *DrawMsg, int mesglen, char *tag)
{
    struct drw_new_pg  *DrawNewPg;
    char arg[LEN];
    int i;

    DrawNewPg = (struct drw_new_pg *)(&DrawMsg[1]);

    if (mesglen < sizeof(struct drw_new_pg))
        return 0;

    if (debugmode) fprintf (stdout, "Draw DRW_NEW_PG %ld \n", ntohl(DrawMsg->page_num));
    /*for (i = 0; i < ntohs(DrawMsg->draw_cmd_length); i++)
        printf ("%c", DrawNewPg->name[i]);
    printf ("\n");*/

    return 0;
}

/* Parse a whiteboard Draw Message packet. */
int ParseDrawMsg (struct pk_draw_msg *DrawMsg, int mesglen)
{
    int i;
    char  tag[32] = "";
    char *tags = NULL;
    long page_num;
    char arg[LEN];
	struct hostent *who;

    if (mesglen < sizeof(struct pk_draw_msg))
        return 0;

    /* Change to page number, draw message is coming from. */
    if (PageCurrent() == NULL)
        return 0;

    if (ntohl(DrawMsg->page_num) < 1)
        return;

	if ((ntohl(DrawMsg->page_num))!=(PageCurrent()->page_number)) 
		PageSet ((ntohl(DrawMsg->page_num)), (ntohl(DrawMsg->page_ip)));

    /*printf ("Known packet type PK_DRAW_MSG %8X ", 
                            ntohl(DrawMsg->pk_type));
    printf ("=> command %8X\n", ntohs(DrawMsg->draw_cmd));

    NetDump (buf, buflen, mesglen);*/

	if (debugmode) {
		fprintf (stdout, "From: %d.%d.%d.%d - ", (ntohl(DrawMsg->sender) >> 24) & 0xff,
                (ntohl(DrawMsg->sender) >> 16) & 0xff,
                (ntohl(DrawMsg->sender) >> 8) & 0xff,
                (ntohl(DrawMsg->sender) & 0xff) );
	}
    tags = build_tag (ntohl(DrawMsg->sender), ntohl(DrawMsg->draw_cmd_num));
    if (tags == NULL)
    {
        perror ("failed to build canvas tag");
        return 0;
    }

    /* Have to copy it from the build_tag static storage. */
    strcpy (tag, tags);

    switch (ntohs(DrawMsg->draw_cmd))
    {
        case DRW_RECT:
            ParseDrawRect (DrawMsg, mesglen, tag);
            break;

        case DRW_LINE:
            ParseDrawLine (DrawMsg, mesglen, tag);
            break;

        case DRW_GRP_LINES:
            ParseDrawGrpLines (DrawMsg, mesglen, tag);
            break;

        case DRW_DEL:
            ParseDrawDel (DrawMsg, mesglen, tag);
            break;

        case DRW_COPY:
            ParseDrawCopy (DrawMsg, mesglen, tag);
            break;

        case DRW_OVAL:
            ParseDrawOval (DrawMsg, mesglen, tag);
            break;

        case DRW_CHR:
            ParseDrawChr (DrawMsg, mesglen, tag);
            break;

        case DRW_GRP_TEXT:
            ParseDrawGrpText (DrawMsg, mesglen, tag);
            break;

        case DRW_PS:
            ParseDrawPS (DrawMsg, mesglen, tag);
            break;

        case DRW_GRP_PS:
            ParseDrawGrpPS (DrawMsg, mesglen, tag);
            break;

        case DRW_CLONE_GRP:
            printf ("Draw DRW_CLONE_GRP\n");
            break;

        case DRW_NEW_PG:
            ParseDrawNewPage (DrawMsg, mesglen, tag);
            break;


        default:
            printf ("UNKNOWN DrawMsg type %08lX\n", ntohs(DrawMsg->draw_cmd));
            break;
    }

    return 0;
}

/* Parse a whiteboard multicast packet. */
int NetParse (char *buf, int mesglen)
{
    int result;

    struct wb_packet *Packet;

    if (mesglen < 12)
        return ERR_BAD_PACKET;

    Packet = (struct wb_packet *)buf;

    /* Update table of senders. */
    SenderAdd (ntohl(Packet->sender), 
               ntohl(Packet->timestamp), 
               time_stamp(), 
               ntohl(Packet->pk_type), NULL);

    switch (ntohl(Packet->pk_type))
    {
        case PK_SESSION:
            NetParseSession(buf, mesglen);
            result = NO_ERR;
            break;

        case PK_REQ_DRAW:
            NetParseReqDraw(buf, mesglen, PK_REQ_DRAW);
            result = NO_ERR;
            break;

        case PK_REQ_DRAX:
            NetParseReqDraw(buf, mesglen, PK_REQ_DRAX);
            result = NO_ERR;
            break;

        case PK_ASK_PAGE:
            NetParseAskPage(buf, mesglen);
            result = NO_ERR;
            break;

        case PK_PAGE_RPT:
            NetParsePageRpt(buf, mesglen);
            result = NO_ERR;
            break;

        case PK_DRAW_MSG:
            ActionAdd ((struct pk_draw_msg *)buf);
            ParseDrawMsg((struct pk_draw_msg *)buf, mesglen);
            result = NO_ERR;
            break;

        case PK_RPY_DRAW:
            NetParseRpyDraw(buf, mesglen);
            result = NO_ERR;
            break;

        default:            /* Unknown type. */
            printf ("Unknown packet type %8X\n", ntohl(Packet->pk_type));
            result = NO_ERR;
            break;
    }

    return result;
}


/*************************************************************/
/*                                                           */
/* Outgoing message routines.                                */
/*                                                           */
/*************************************************************/


/* Send a packet. */
uint32 size_command (void *cmd)
{
    struct wb_packet   *Packet  = NULL;
    struct pk_draw_msg *Draw    = NULL;
    struct pk_session  *Session = NULL;
    struct pk_page_rpt *PageRpt = NULL;
    uint32 size = 0;

    if (cmd == NULL)
        return 0;

    /* Find out how long the message is. */
    Packet = (struct wb_packet *) cmd;
    switch (ntohl(Packet->pk_type))
    {
        case PK_DRAW_MSG:
                Draw = (struct pk_draw_msg *) cmd;
                size = sizeof (struct pk_draw_msg);
                size += ntohs(Draw->draw_cmd_length);
                break;
        case PK_SESSION:
                Session = (struct pk_session *) cmd;
                size = sizeof (struct pk_session);
                size += ntohs(Session->active) * sizeof (struct writer_desc);
                size += 1 + strlen (size + (char *)cmd);
                break;
        case PK_PAGE_RPT:                            
                PageRpt = (struct pk_page_rpt *) cmd;
                size = sizeof (struct pk_page_rpt);  
        default:
                return 0;
    }

    return size;
}


struct pk_session *build_session (uint32 *size_ptr)
{
    struct pk_session  *session       = NULL;
    struct page        *last_page_ptr = NULL;
    struct page        *this_page_ptr = NULL;
    struct writer      *writer        = NULL;
    struct writer_desc *wd;
    unsigned long       num_writers   = 0;
    long size;

    /* Calculate message size. */
    this_page_ptr = PageCurrent();
    if (this_page_ptr == NULL)
        return NULL;

    writer = this_page_ptr->writers;
    num_writers = 0;
    while (writer != NULL)
    {
        num_writers++;
        writer = writer->next;
    }

    size = sizeof (struct pk_session) 
           + (num_writers * sizeof (struct writer_desc))
           +  strlen (ascii_source(NULL)) + 1;
    session = (struct pk_session *) malloc (size);
    if (session == NULL)
        return NULL;

    session->sender     = htonl(this_ip());
    session->timestamp  = htonl(time_stamp());
    session->pk_type    = htonl(PK_SESSION);

    /* Is this the number of pages or the highest */
    /* page number or the number of local pages?  */
    /* There may be two pages with the same       */
    /* number and different creators?             */
    session->page_count = htonl(PageCount());

    last_page_ptr = PageFind (PageNumMax(), 0);
    if (last_page_ptr != NULL)
    {
        session->last_page_ip     = htonl(last_page_ptr->creator_ip);
        session->last_page_num    = htonl(last_page_ptr->page_number);
    }

    this_page_ptr = PageCurrent();
    session->page_num = htonl(this_page_ptr->page_number);
    if (this_page_ptr != NULL)
    {
        session->current_page_ip  = htonl(this_page_ptr->creator_ip);
        session->current_page_num = htonl(this_page_ptr->page_number);
    }

    /* How many are active on the current page? */
    writer = this_page_ptr->writers;
    session->active = htons(num_writers);
    session->dummy  = htons(0);

    /* Describe each writer... */
    wd = (struct writer_desc *) (sizeof (struct pk_session) 
                                            + (uint8 *) session);
    writer = this_page_ptr->writers;
    while (writer != NULL)
    {
        wd->writer_ip   = htonl(writer->writer_ip);

        /* The number of the next command, or zero! */
        if (writer->seq_num_max == 0)
            wd->write_count = htonl(0);
        else
            wd->write_count = htonl(writer->seq_num_max + 1);

        wd = (struct writer_desc *) (sizeof (struct writer_desc) 
                                      + (uint8 *) wd);
        writer = writer->next;
    }

    /* Append a description string: "user@host". */
    strncpy ((char *)wd, ascii_source(NULL), 1 + strlen(ascii_source(NULL)));

    /* Failure detection. */
    if ((last_page_ptr == NULL) || (this_page_ptr == NULL))
    {
        /* Failed. */
        if (session != NULL)
            free (session);
        session = NULL;
    }

    *size_ptr = size;
    return session;
}


struct pk_req_draw *build_draw_request (uint32 page_num, 
                                        uint32 page_ip,
                                        uint32 writer_ip, 
                                        uint32 first_cmd, 
                                        uint32 last_cmd,
                                        uint32 *size_ptr)
{
    struct pk_req_draw *request = NULL;
    struct page        *page    = NULL;;

    /* Does the whiteboard know about this page? */
    if ((page = PageFind (page_num, page_ip)) == NULL)
        return NULL;

    request = (struct pk_req_draw *) malloc (sizeof (struct pk_req_draw));
    if (request != NULL)
    {
        /* PK_REQ_DRAX is ignored by wb if */
        /* it doesn't end at the maximum.  */
        /* I see no advantage to using it. */
        request->sender    = htonl(this_ip());
        request->timestamp = htonl(time_stamp());
        request->pk_type   = htonl(PK_REQ_DRAW); /*PK_REQ_DRAX;*/
        request->drawer_ip = htonl(writer_ip);
        request->page_ip   = htonl(page_ip);
        request->page_num  = htonl(page_num);
        request->first_cmd = htonl(first_cmd);
        request->last_cmd  = htonl(last_cmd);

        /*if ((first_cmd != 1) || (last_cmd))
            request->pk_type   = htonl(PK_REQ_DRAW);*/

        *size_ptr = sizeof (struct pk_req_draw);
    }

    return request;
}


struct pk_ask_page *build_ask_page (uint32 page_num, 
                                    uint32 page_ip,
                                    uint32 *size_ptr)
{
    struct pk_ask_page *request = NULL;

    request = (struct pk_ask_page *) malloc (sizeof (struct pk_ask_page));
    if (request != NULL)
    {
        request->sender     = htonl(this_ip());
        request->timestamp  = htonl(time_stamp());
        request->pk_type    = htonl(PK_ASK_PAGE);
        request->page_ip    = htonl(page_ip);
        request->page_num   = htonl(page_num);
        request->page_index = htonl(page_num); /* ??? */
        request->d2         = htonl(0);

        *size_ptr = sizeof (struct pk_ask_page);
    }

    return request;
}


struct pk_page_rpt *build_page_report (uint32 page_num, 
                                       uint32 page_ip,
                                       uint32 page_index,
                                       uint32 *size_ptr)
{
    struct pk_page_rpt *report = NULL;
    struct page_desc   *pd     = NULL;
    struct writer_desc *wd     = NULL;
    struct page        *page;
    struct writer      *writer;
    int i, pages, writers, size;
    int first_page, target_page, last_page;
    int offset;

    /*printf ("BuildPageReport: %8X:%ld (%ld)\n", 
                    page_ip, page_num, page_index);*/

    /* Find the requested page. */
    target_page = PageIndex (page_num, page_ip);
    if (target_page == 0)
    {
        if (page_num != 0)
            target_page = page_num;
        else
            target_page = page_index;
    }

    /* Limit packet size by describing only ~ten pages. */
    first_page = target_page - 4;
    if (first_page < 1)
        first_page = 1;
    last_page = PageCount();
    if (last_page > (target_page + 4))
        last_page = target_page + 4;

    /* Find out how big the report is going to be. */
    pages = 0;
    writers = 0;
    size = sizeof (struct pk_page_rpt);
    for (i = first_page; i <= last_page; i++)
    {
        page = PageIndexFind (i);
        if (page != NULL)
        {
            size += sizeof (struct page_desc);
            pages++;

            writer = page->writers;
            while (writer != NULL)
            {
                size += sizeof (struct writer_desc);

                writers++;
                writer = writer->next;
            }
        }

        if (size > PK_TRIGGER_SIZE)
        {
            last_page = i;
            pages = last_page - first_page;
        }
    }

    /* Any pages to describe? */
    if (pages < 1)
        return NULL;

/*printf ("BuildPageReport: %d pages (%d to %d), %d writers, %d bytes.\n",
    pages, first_page, last_page, writers, size);*/

    report = malloc (size);
    if (report == NULL)
        return NULL;

    report->sender    = htonl(this_ip());
    report->timestamp = htonl(time_stamp());
    report->pk_type   = htonl(PK_PAGE_RPT);

    /* How many pages are there? */
    report->count     = htonl(pages);

    offset = sizeof (struct pk_page_rpt);

    /* For each page... */
    for (i = first_page; i <= last_page; i++)
    {
        page = PageIndexFind (i);
        if (page != NULL)
        {
            pd = (struct page_desc *) (offset + 
                                       (uint8 *)report);
            offset += sizeof (struct page_desc);

            pd->page = htonl(page->page_number);
            pd->page_id  = htonl(page->creator_ip);
            pd->page_num = htonl(page->page_number);

            writer = page->writers;
            writers = 0;
            while (writer != NULL)
            {
                wd = (struct writer_desc *) (offset + 
                                        (uint8 *)report);
                offset += sizeof (struct writer_desc);

                wd->writer_ip   = htonl(writer->writer_ip);

                /* The number of the next command, or zero! */
                if (writer->seq_num_max == 0)
                    wd->write_count = htonl(0);
                else
                    wd->write_count = htonl(writer->seq_num_max + 1);

                writers++;
                writer = writer->next;
            }
            pd->active = htons(writers);
            pd->dummy  = htons(0);
        }
    }

    if (offset > size)
        printf ("BuildPageReport error size %d < offset %d\n",
                    size, offset);

    *size_ptr = size;
    return report;
}


struct pk_rpy_draw *build_reply_draw (uint32 page_ip, 
                    uint32 page_num, uint32 drawer_ip, 
                    uint32 first_cmd, uint32 last_cmd,
                    uint32 *size_ptr)
{
    struct pk_rpy_draw *reply = NULL;
    struct page   *page       = NULL;
    struct action *action     = NULL;
    unsigned long i, offset, size, msize;
    int found = 0;

    page = PageFind (page_num, page_ip);
    if (page == NULL)
        return NULL;

    /* Calculate the packet size. */
    size = sizeof (struct pk_rpy_draw);
    for (i = first_cmd; i <= last_cmd; i++)
    {
        action = ActionFind (page_num, page_ip, drawer_ip, i);
        if (action != NULL)
        {
            found = 1;

            msize = (sizeof (uint32) + 2*sizeof(uint16));
            msize += htons(action->message->draw_cmd_length);
            while ((msize % 4) != 0)
                msize++;
            size = size + msize;
        }

        /* Limit the packet size. */
        if (size > PK_TRIGGER_SIZE)
        {
            last_cmd = i;

            break;
        }
    }

    /* Don't know any of the answers. */
    if (found == 0)
        return NULL;

/*printf ("BuildReplyDraw: from %ld to %ld, %ld bytes\n", 
first_cmd, last_cmd, size);*/

    reply = (struct pk_rpy_draw *) malloc (size);
    if (reply == NULL)
        return NULL;

    reply->sender     = htonl(this_ip());
    reply->timestamp  = htonl(time_stamp());
    reply->pk_type    = htonl(PK_RPY_DRAW);

    reply->drawer_ip    = htonl(drawer_ip);
    reply->page_ip      = htonl(page_ip);
    reply->page_num     = htonl(page_num);
    reply->first_cmd    = htonl(first_cmd);
    reply->last_cmd     = htonl(last_cmd);

    offset = sizeof (struct pk_rpy_draw);

    /* Now add all of those commands... */
    for (i = first_cmd; i <= last_cmd; i++)
    {
        action = ActionFind (page_num, page_ip, drawer_ip, i);
        if (action != NULL)
        {
            /* Copy timestamp, length, command type. */
            /* Copy the draw command. */
            msize = (sizeof (uint32) + 2*sizeof(uint16));
            msize += htons(action->message->draw_cmd_length);
            memcpy ( (offset + (char *)reply),
                     &(action->message->draw_timestamp),
                     msize);
            offset += msize;

            /* Pad to a word boundary if necessary. */
            while ((msize %4) != 0)
            {
                *(offset + (char *)reply) = 0;
                offset++;
                msize++;
            }
        }
        /*else
            printf ("didn't find %ld\n", i);*/
    }

    *size_ptr = size;
    return reply;
}


/* Build an outgoing command. */
/*  (Returns NULL on failure) */
struct pk_draw_msg *build_command (int argc, char *argv[],
                                          uint32 *size_ptr)
{
    struct pk_draw_msg   *command  = NULL;
    struct drw_line      *line     = NULL;
    struct drw_rect      *rect     = NULL;
    struct drw_oval      *oval     = NULL;
    struct drw_chr       *chr      = NULL;
    struct drw_ps        *ps       = NULL;
    struct drw_new_pg    *newpage  = NULL;
    struct drw_grp_lines *grplines = NULL;
    struct drw_grp_text  *grptext  = NULL;
    struct drw_grp_ps    *grpps    = NULL;
    struct drw_copy      *copy     = NULL;
    struct drw_del       *delete   = NULL;
    float x1, y1, x2, y2, w;
    int size, len;

    /* Create the first page if it doesn't already exist. */
    if (PageCurrent() == NULL)
    {
        PageAdd (1, this_ip());
        PageSet (1, this_ip());
    }

    /* Build the draw message. */
    if (strcmp (argv[1], "polyline") == 0)
    {
        ;
    }
    else if (strcmp (argv[1], "line") == 0)
    {
        if (argc < 8)
            return NULL;

        size = sizeof (struct pk_draw_msg) + sizeof (struct drw_line);
        command = (struct pk_draw_msg *) malloc (size);
        line    = (struct drw_line *) &command[1];

        command->draw_cmd        = htons(DRW_LINE);
        command->draw_cmd_length = htons(sizeof (struct drw_line));

        w  = (real32)strtoul (argv[2], NULL, 0);
        x1 = (real32)strtol  (argv[4], NULL, 0);
        y1 = (real32)strtol  (argv[5], NULL, 0);
        x2 = (real32)strtol  (argv[6], NULL, 0);
        y2 = (real32)strtol  (argv[7], NULL, 0);
        CanvasToPageW  (&w);
        CanvasToPageXY (&x1, &y1);
        CanvasToPageXY (&x2, &y2);
        line->linestyle = htonf(w); /* float */
        line->colour    = htonl(strtoul (argv[3], NULL, 0)); /* long  */
        line->from_x    = htonf(x1); /* float */
        line->from_y    = htonf(y1); /* float */
        line->to_x      = htonf(x2); /* float */
        line->to_y      = htonf(y2); /* float */
    }
    else if (strcmp (argv[1], "rect") == 0)
    {
        size = sizeof (struct pk_draw_msg) + sizeof (struct drw_rect);
        command = (struct pk_draw_msg *) malloc (size);
        rect    = (struct drw_rect *) &command[1];

        command->draw_cmd = htons(DRW_RECT);
        command->draw_cmd_length = htons(sizeof (struct drw_rect));

        w  = (real32)strtoul (argv[2], NULL, 0);
        x1 = (real32)strtol  (argv[4], NULL, 0);
        y1 = (real32)strtol  (argv[5], NULL, 0);
        x2 = (real32)strtol  (argv[6], NULL, 0);
        y2 = (real32)strtol  (argv[7], NULL, 0);
        CanvasToPageW  (&w);
        CanvasToPageXY (&x1, &y1);
        CanvasToPageXY (&x2, &y2);
        rect->linestyle = htonf(w); /* float */
        rect->colour    = htonl(strtoul (argv[3], NULL, 0)); /* long  */
        rect->from_x    = htonf(x1); /* float */
        rect->from_y    = htonf(y1); /* float */
        rect->to_x      = htonf(x2); /* float */
        rect->to_y      = htonf(y2); /* float */
    }
    else if (strcmp (argv[1], "oval") == 0)
    {
        size = sizeof (struct pk_draw_msg) + sizeof (struct drw_oval);
        command = (struct pk_draw_msg *) malloc (size);
        oval    = (struct drw_oval *) &command[1];

        command->draw_cmd = htons(DRW_OVAL);
        command->draw_cmd_length = htons(sizeof (struct drw_oval));

        w  = (real32)strtoul (argv[2], NULL, 0);
        x1 = (real32)strtol  (argv[4], NULL, 0);
        y1 = (real32)strtol  (argv[5], NULL, 0);
        x2 = (real32)strtol  (argv[6], NULL, 0);
        y2 = (real32)strtol  (argv[7], NULL, 0);
        CanvasToPageW  (&w);
        CanvasToPageXY (&x1, &y1);
        CanvasToPageXY (&x2, &y2);
        oval->linestyle = htonf(w); /* float */
        oval->colour    = htonl(strtoul (argv[3], NULL, 0)); /* long  */
        oval->center_x  = htonf(0.5 * (x1 + x2)); /* float */
        oval->center_y  = htonf(0.5 * (y1 + y2)); /* float */
        oval->width     = htonf(0.5 * fabs (x2 - x1)); /* float */
        oval->height    = htonf(0.5 * fabs (y2 - y1)); /* float */
    }
    else if (strcmp (argv[1], "chr") == 0)
    {
        if (argc < 8)
            return NULL;

        size = sizeof (struct pk_draw_msg) + sizeof (struct drw_chr);
        command = (struct pk_draw_msg *) malloc (size);
        chr     = (struct drw_chr *) &command[1];

        command->draw_cmd        = htons(DRW_CHR);
        command->draw_cmd_length = htons(sizeof (struct drw_line));

        x1 = (real32)strtol  (argv[5], NULL, 0);
        y1 = (real32)strtol  (argv[6], NULL, 0);
        CanvasToPageXY (&x1, &y1);
        chr->ascii       = htons(strtoul (argv[2], NULL, 0)); /* short */
        chr->orientation = htons(strtoul (argv[3], NULL, 0)); /* short */
        chr->colour      = htonl(strtoul (argv[4], NULL, 0)); /* long  */
        chr->where_x     = htonf(x1); /* float */
        chr->where_y     = htonf(y1); /* float */
        chr->font        = htonl( ((strtoul (argv[7], NULL, 0) << 8)
			     + strtoul (argv[8], NULL, 0)) ); /* long  */
    }
    else if (strcmp (argv[1], "new_pg") == 0)
    {
        /* Append something like "user:page_num".             */
        /* PageCurrent should already be set to the new page. */
        size = sizeof (struct pk_draw_msg) + 1 + 
                   strlen (ascii_page(PageCurrent()->page_number));
        command = (struct pk_draw_msg *) malloc (size);
        newpage = (struct drw_new_pg *) &command[1];

        command->draw_cmd = htons(DRW_NEW_PG);
        command->draw_cmd_length = htons(1 + 
                   strlen (ascii_page(PageCurrent()->page_number)));
        strncpy (newpage->name, ascii_page(PageCurrent()->page_number), 
                                          htons(command->draw_cmd_length));
    }
    else if (strcmp (argv[1], "group_lines") == 0)
    {
        if (argc < 4)
            return NULL;

        size = sizeof (struct pk_draw_msg) + 
                         sizeof (struct drw_grp_lines);
        command  = (struct pk_draw_msg *) malloc (size);
        grplines = (struct drw_grp_lines *) &command[1];

        command->draw_cmd = htons(DRW_GRP_LINES);
        command->draw_cmd_length = htons(sizeof (struct drw_grp_lines));

        grplines->first_cmd = htonl(strtoul (argv[2], NULL, 0)); /* long  */
        grplines->last_cmd  = htonl(strtoul (argv[3], NULL, 0)); /* long  */
    }
    else if (strcmp (argv[1], "group_text") == 0)
    {
        if (argc < 4)
            return NULL;

        size = sizeof (struct pk_draw_msg) + 
                         sizeof (struct drw_grp_text);
        command = (struct pk_draw_msg *) malloc (size);
        grptext = (struct drw_grp_text *) &command[1];

        command->draw_cmd = htons(DRW_GRP_TEXT);
        command->draw_cmd_length = htons(sizeof (struct drw_grp_text));

        grptext->first_cmd = htonl(strtoul (argv[2], NULL, 0)); /* long  */
        grptext->last_cmd  = htonl(strtoul (argv[3], NULL, 0)); /* long  */
    }
    else if (strcmp (argv[1], "postscript") == 0)
    {
        if (argc < 4)
            return NULL;

        len = strtoul (argv[2], NULL, 0);
        size = sizeof (struct pk_draw_msg) + 
                         /*sizeof (struct drw_ps) +*/ len;
        command = (struct pk_draw_msg *) malloc (size);
        ps      = (struct drw_ps *) &command[1];

        command->draw_cmd = htons(DRW_PS);
        command->draw_cmd_length = htons(/*(sizeof (struct drw_ps)) +*/ len);

        memcpy ((char *)ps, argv[3], len);
    }
    else if (strcmp (argv[1], "group_postscript") == 0)
    {
        if (argc < 4)
            return NULL;

        size = sizeof (struct pk_draw_msg) + 
                         sizeof (struct drw_grp_ps);
        command = (struct pk_draw_msg *) malloc (size);
        grpps   = (struct drw_grp_ps *) &command[1];

        command->draw_cmd = htons(DRW_GRP_PS);
        command->draw_cmd_length = htons(sizeof (struct drw_grp_ps));

        grpps->d1        = htonl(1); /* page number? */
        grpps->first_cmd = htonl(strtoul (argv[2], NULL, 0)); /* long  */
        grpps->last_cmd  = htonl(strtoul (argv[3], NULL, 0)); /* long  */
    }
    else if (strcmp (argv[1], "copy") == 0)
    {
        if (argc < 4)
            return NULL;

        if (strlen(argv[2]) <= 6)
            return NULL;

        size = sizeof (struct pk_draw_msg) + 
                         sizeof (struct drw_copy);
        command = (struct pk_draw_msg *) malloc (size);
        copy    = (struct drw_copy *)    &command[1];

        command->draw_cmd = htons(DRW_COPY);
        command->draw_cmd_length = htons(sizeof (struct drw_copy));

        x1 = (real32)strtol  (argv[3], NULL, 0);
        y1 = (real32)strtol  (argv[4], NULL, 0);
        CanvasToPageXY (&x1, &y1);
        copy->cmd_id    = htonl(strtoul (6+argv[2], NULL, 0)); /* long  */
        copy->d1        = htonl(1.0);
        copy->d2        = htonl(0.0);
        copy->d3        = htonl(0.0);
        copy->d4        = htonl(1.0);
        copy->to_x      = htonf(x1); /* float */
        copy->to_y      = htonf(y1); /* float */
    }
    else if (strcmp (argv[1], "delete") == 0)
    {
        if (argc < 3)
            return NULL;

        if (strlen(argv[2]) <= 6)
            return NULL;

        size = sizeof (struct pk_draw_msg) + 
                         sizeof (struct drw_del);
        command = (struct pk_draw_msg *) malloc (size);
        delete  = (struct drw_del *)     &command[1];

        command->draw_cmd = htons(DRW_DEL);
        command->draw_cmd_length = htons(sizeof (struct drw_del));

        delete->first_cmd = htonl(strtoul (6+argv[2], NULL, 0)); /* long  */
        delete->last_cmd  = htonl(strtoul (6+argv[2], NULL, 0)); /* long  */
    }

    /* Fill in the common fields. */
    if (command != NULL)
    {
        command->sender    = htonl(this_ip());
        command->timestamp = htonl(time_stamp());
        command->pk_type   = htonl(PK_DRAW_MSG);

        /* These refer to the current page. */
        command->page_ip   = htonl(PageCurrent()->creator_ip);
        command->page_num  = htonl(PageCurrent()->page_number);

        /* These also refer to the current page. */
        command->draw_cmds_local = /* Continued below... */
        command->draw_cmd_num    = htonl(1 + 
              SeqNumMax(PageCurrent()->page_number, 
                     PageCurrent()->creator_ip, this_ip()));

        command->draw_timestamp  = command->timestamp;
    }

    *size_ptr = size;
    return command;
}

/* Look for a delete to undelete. */
int UnDelete (ClientData clientData, 
            Tcl_Interp *tcl, int argc, char *argv[])
{
    struct pk_draw_msg *DrawMsg = NULL;
    struct drw_del     *DrawDel = NULL;
    struct action      *action;
    uint32 page_num, page_ip;
    uint32 first, last;
    int found = 0;
    int i, max;

    if (PageCurrent() == NULL)
    {
        sprintf (tcl->result, "0");
        return TCL_OK;
    }

    page_num = PageCurrent()->page_number;
    page_ip  = PageCurrent()->creator_ip;

    /* Look for delete messages, last to first. */
    max = SeqNumMax (page_num, page_ip, this_ip());
    for (i = max; i > 0; i--)
    {
        action = ActionFind (page_num, page_ip, this_ip(), i);
        if (action != NULL)
        {
            DrawMsg = action->message;

            if (DrawMsg->draw_cmd == DRW_DEL)
            {
                /* Is it deleting a delete? */
                DrawDel = (struct drw_del *) &DrawMsg[1];

                first = ntohl(DrawDel->first_cmd);
                last  = ntohl(DrawDel->last_cmd);

                if (first == last)
                {
                    action = ActionFind (page_num, page_ip, this_ip(), first);
                    if (action != NULL)
                        if (action->message->draw_cmd != DRW_DEL)
                        {
                            /* Not deleting a delete. */
                            /* Can undelete it.       */
                            found = i;
                            break;
                        }
                        else
                        {
                            /* Delete & Undelete pair.      */
                            /* Move to before the delete.   */
                            /* (Decremented at end of loop) */
                            i = first;
                        }
                }
                else
                {
                    /* Not deleting a delete. Can undelete it. */
                    found = i;
                    break;
                }
            }
        }
    }

    sprintf (tcl->result, "%d", found);

    return TCL_OK;
}

/* Send a PostScript file. */
int SendPS (ClientData clientData, 
            Tcl_Interp *tcl, int argc, char *argv[])
{
    struct pk_draw_msg *command = NULL;
    FILE *infile = NULL;
    char arg[LEN];
    char buf[1024];
    char args[4][20] = {"DrawPS", "", "", ""};
    char *argp[6];
    int  done, len;
    uint32 size;
    uint32 first_command = 0;
    uint32 last_command  = 0;

    if (argc < 2)
    {
        sprintf (tcl->result, "SendPS <filename>");
        return TCL_ERROR;
    }

    /* Check the file size. */

    infile = fopen (argv[1], "r");
    if (infile == NULL)
    {
        sprintf (tcl->result, "failed to open postscript file %s", argv[1]);
        return TCL_ERROR;
    }

    /* Read in pieces and send them. */
    done = 0;
    while (!done)
    {
        len = fread (buf, 1, 1024, infile);
        if (len < 1024)
            done = 1;

        if (len > 0)
        {
            argp[0] = (char *) &args[0];
            argp[1] = (char *) &args[1];
            sprintf (args[1], "postscript");
            argp[2] = (char *) &args[2];
            sprintf (args[2], "%d", len);
            argp[3] = (char *) &buf; /* Hack! */
            argp[4] = (char *) NULL;
            command = build_command (5, argp, &size);
            if (command == NULL)
            {
                sprintf (tcl->result, "failed to build message");
                return TCL_ERROR;
            }

            if (first_command == 0)
                first_command = ntohl(command->draw_cmd_num);
            last_command = ntohl(command->draw_cmd_num);

            /* Add the new command into the system... */
            NetParse ((char *) command, size);

            /* Send it. */
            QueueAddTo ((char *)command, size, Q_DRAW_MSG);
        }
    }

    /* Send the Group Postscript command. */
    argp[0] = (char *) &args[0];
    argp[1] = (char *) &args[1];
    sprintf (args[1], "group_postscript");
    argp[2] = (char *) &args[2];
    sprintf (args[2], "%ld", first_command);
    argp[3] = (char *) &args[3];
    sprintf (args[3], "%ld", last_command);
    argp[4] = (char *) NULL;
    command = build_command (4, argp, &size);
    if (command == NULL)
    {
        sprintf (tcl->result, "failed to build group message");
        return TCL_ERROR;
    }

    /* Add the Group Postscript command into the system... */
    NetParse ((char *) command, size);

    /* Send it. */
    QueueAddTo ((char *)command, size, Q_DRAW_MSG);

    return TCL_OK;
}

/* XXX - send is ok unless we're importing text */
int sendok=1;

int SetSend (ClientData clientData, 
            Tcl_Interp *tcl, int argc, char *argv[])
{
	if (strcmp (argv[1], "1") == 0) sendok=1; else sendok=0;
	return TCL_OK;
}

/* Draw command arriving from the User Interface. */
int Draw (ClientData clientData, 
            Tcl_Interp *tcl, int argc, char *argv[])
{
    int res, len, i;
    char arg[LEN];
    struct pk_draw_msg *command = NULL;
    uint32 size, seqno;

    if (argc < 2)
    {
        sprintf (tcl->result, "Draw type ......");
        return TCL_ERROR;
    }

    /* Create the first page if it doesn't already exist. */
    if (PageCurrent() == NULL)
        PageNew (NULL, tcl, 1, NULL);

    command = build_command (argc, argv, &size);

    /* Should do some error recording here? */
    if (command == NULL)
    {
        sprintf (tcl->result, "failed");
        return TCL_OK;
    }

    seqno = ntohl(command->draw_cmd_num);

    /* Add the new command into the system... */
    NetParse ((char *) command, size);

    /* only send it if told to - eg. when importing 
	   text we only want to send the last page. */
    if (sendok) QueueAddTo ((char *)command, size, Q_DRAW_MSG);

    /* Return the command sequence number. */
    sprintf (tcl->result, "%ld", seqno);

    return TCL_OK;
}


/*************************************************************/
/*                                                           */
/* Page scaling.                                             */
/*                                                           */
/*************************************************************/

int PageCanvasSize (ClientData clientData, Tcl_Interp *tcl, 
                              int argc, char *argv[])
{
    char arg[LEN];
    float w, h;
    int orientation;

    if (argc < 3)
    {
        sprintf (tcl->result, "ScaleSet page_width page_height [orientation]");
        return TCL_ERROR;
    }

    w = -1.0;
    h = -1.0;
    sscanf (argv[1], "%f", &w);
    sscanf (argv[2], "%f", &h);

    if ((w == -1.0) || (h == -1.0))
    {
        page_scale.width  = (float) PAGE_WIDTH;
        page_scale.height = (float) PAGE_HEIGHT;
        page_scale.average = 1.0;
        page_scale.orient = 0;

        sprintf (tcl->result, "bad page values %f %f", w, h);
        return TCL_ERROR;
    }

    page_scale.width  = w;
    page_scale.height = h;
    page_scale.average = sqrt ((w / PAGE_WIDTH) * (h / PAGE_HEIGHT));

    if (argc == 4)
    {
        orientation = -1;
        sscanf (argv[3], "%d", &orientation);
        if ((orientation < 0) || (orientation > 3))
        {
            sprintf (tcl->result, "bad page orientation %d", orientation);
            return TCL_OK;
        }
        else
        {
            page_scale.orient = orientation;
        }
    }

    return TCL_OK;
}

/* Scale and offset: used for coordinates. */
void PageToCanvasXY (float *x, float *y)
{
    float canvas_x, scale_w;
    float canvas_y, scale_h;

    scale_w = page_scale.width / PAGE_WIDTH;
    scale_h = page_scale.height / PAGE_HEIGHT;

    switch (page_scale.orient)
    {
        case 0: /* Vertical page. */
                canvas_x = (*x) * scale_w;
                canvas_y = page_scale.height - ((*y) * scale_h);
                break;

        case 1: /* 90-degree clockwise rotated page, (wb's default) */
        default:
                canvas_x = (*y) * scale_h;
                canvas_y = (*x) * scale_w;
                break;

        case 2: /* 180-degree rotated page, (not in wb). */
                canvas_x = page_scale.width  - ((*x) * scale_w);
                canvas_y = ((*y) * scale_h);
                break;

        case 3: /*  270-degree clockwise rotated page. */
                canvas_x = page_scale.height - ((*y) * scale_h);
                canvas_y = page_scale.width  - ((*x) * scale_w);
                break;
    }

    *x = canvas_x;
    *y = canvas_y;
}

/* Scale by the average of x and y scales: used for widths. */
void PageToCanvasW (float *w)
{
    float canvas_w;
#ifdef NDEF
    double temp;
    canvas_w = (*w) * page_scale.average;
    modf( canvas_w, &temp );
    *w = temp;
#endif
}

/* Scale and offset: used for coordinates. */
void CanvasToPageXY (float *x, float *y)
{
    float page_x, scale_w; 
    float page_y, scale_h;

    scale_w = PAGE_WIDTH / page_scale.width;
    scale_h = PAGE_HEIGHT / page_scale.height;

    switch (page_scale.orient)
    {
        case 0: /* Vertical page. */
                page_x = (*x) * scale_w;
                page_y = (page_scale.height - (*y)) * scale_h;
                break;

        case 1: /* 90-degree clockwise rotated page, (wb's default) */
        default:
                page_x = (*y) * scale_w;
                page_y = (*x) * scale_h;
                break;

        case 2: /* 180-degree rotated page, (not in wb). */
                page_x = (page_scale.width - (*x)) * scale_w;
                page_y = (*y) * scale_h;
                break;

        case 3: /*  270-degree clockwise rotated page. */
                page_x = (page_scale.width  - (*y)) * scale_w;
                page_y = (page_scale.height - (*x)) * scale_h;
                break;
    }

    *x = page_x;
    *y = page_y;
}

/* Scale by the average of x and y scales: used for widths. */
void CanvasToPageW (float *w)
{
    float page_w;

#ifdef NDEF
    page_w = (*w) * page_scale.average;

    *w = page_w;
#endif
}



