/***

fltedit.c	- TCP/UDP Filter Editing Facility

Copyright (c) Gerard Paul Java 1999

This software is open-source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <curses.h>
#include <panel.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include "stdwinset.h"
#include "utfdefs.h"
#include "utfilter.h"
#include "dirs.h"
#include "getpath.h"
#include "attrs.h"
#include "deskman.h"
#include "error.h"

void print_hostparam_line(struct filterent *fe, int idx, WINDOW * win)
{
    wattrset(win, STDATTR);
    mvwprintw(win, idx, 2, "%76c", ' ');
    mvwaddnstr(win, idx, 2, fe->hp.s_fqdn, 15);
    wprintw(win, "/");
    waddnstr(win, fe->hp.s_mask, 12);
    wprintw(win, ":%u", fe->hp.sport);
    mvwaddnstr(win, idx, 38, fe->hp.d_fqdn, 15);
    wprintw(win, "/");
    waddnstr(win, fe->hp.d_mask, 12);
    wprintw(win, ":%u", fe->hp.dport);
    mvwprintw(win, idx, 76, "%c", toupper(fe->hp.reverse));
}

void update_hp_screen(struct filterlist *fl, struct filterent *fe, int idx,
		      WINDOW * win)
{
    wattrset(win, STDATTR);
    if (fe == NULL) {
	mvwprintw(win, 0, 0, "%78c", ' ');
	return;
    }
    while ((fe != NULL) && (idx <= 12)) {
	print_hostparam_line(fe, idx, win);
	idx++;
	fe = fe->next_entry;
    }

    if (idx <= 12) {
        scrollok(win, 0);
	mvwprintw(win, idx, 0, "%78c", ' ');
	scrollok(win, 1);
    }
}

int new_hp_entry(struct filterent **ftemp)
{
    int resp;
    *ftemp = malloc(sizeof(struct filterent));
    if (*ftemp == NULL) {
	errbox("No memory for new filter entry", ANYKEY_MSG, &resp);
	return 0;
    }
    return 1;
}

void select_host_parameters(unsigned int protocol,
                            struct filterlist *fl, struct filterent **rfe,
			    int *aborted, int *command)
{
    WINDOW *bwin;
    PANEL *bpanel;
    WINDOW *win;
    PANEL *panel;
    struct filterent *fe;
    struct filterent *ftemp;
    unsigned int idx = 0;
    int endloop_local = 0;
    int ch;

    bwin = newwin(15, 80, (LINES - 15) / 2, (COLS - 80) / 2);
    bpanel = new_panel(bwin);
    win = newwin(13, 78, (LINES - 13) / 2, (COLS - 78) / 2);
    panel = new_panel(win);

    wattrset(bwin, BOXATTR);
    box(bwin, ACS_VLINE, ACS_HLINE);
    mvwprintw(bwin, 0, 2, " First Host/Mask:Port ");
    mvwprintw(bwin, 0, 38, " Second Host/Mask:Port ");
    mvwprintw(bwin, 0, 74, " I/E ");
    wmove(bwin, 14, 1);
    
    if (protocol == F_TCP)
        wprintw(bwin, " TCP");
    else
        wprintw(bwin, " UDP");
        
    wprintw(bwin, " Filter Data ");
    stdwinset(win);
    wattrset(win, STDATTR);
    colorwin(win);
    
    move(LINES - 1, 1);
    printkeyhelp("Up/Down", "-move ptr ", stdscr);
    printkeyhelp("I", "-insert entry ", stdscr);
    printkeyhelp("A", "-append to list ", stdscr);
    printkeyhelp("D", "-delete entry ", stdscr);
    printkeyhelp("X/Ctrl+X", "-exit", stdscr);

    update_panels();
    doupdate();

    fe = fl->head;

    update_hp_screen(fl, fl->head, 0, win);

    idx = 0;
    fe = fl->head;

    do {
	wattrset(win, PTRATTR);
	if (fe != NULL)
	    mvwaddch(win, idx, 1, ACS_RARROW);
	ch = wgetch(win);
	mvwprintw(win, idx, 1, " ");

	switch (ch) {
	case KEY_UP:
	    if (fl->head != NULL) {
		if (fe->prev_entry != NULL) {
		    if (idx > 0)
			idx--;
		    else {
			wscrl(win, -1);
			mvwprintw(win, 0, 0, "%78c", ' ');
			print_hostparam_line(fe->prev_entry, idx, win);
		    }
		    fe = fe->prev_entry;
		}
	    }
	    break;
	case KEY_DOWN:
	    if (fl->head != NULL) {
		if (fe->next_entry != NULL) {
		    if (idx < 12)
			idx++;
		    else {
			wscrl(win, 1);
			scrollok(win, 0);
			mvwprintw(win, 12, 0, "%78c", ' ');
			scrollok(win, 1);
			print_hostparam_line(fe->next_entry, idx, win);
		    }
		    fe = fe->next_entry;
		}
	    }
	    break;
	case 'i':
	case 'I':
	case KEY_IC:
	    if (!new_hp_entry(&ftemp))
		break;

	    if (fl->head == NULL) {
		ftemp->next_entry = ftemp->prev_entry = NULL;
		fl->head = ftemp;
	    } else {
		ftemp->next_entry = fe;
		ftemp->prev_entry = fe->prev_entry;

		if (ftemp->prev_entry == NULL)
		    fl->head = ftemp;
		else
		    fe->prev_entry->next_entry = ftemp;

		fe->prev_entry = ftemp;
	    }

	    if (ftemp->next_entry == NULL)
		fl->tail = ftemp;

	    fe = ftemp;
	    endloop_local = 1;
	    *aborted = 0;
	    ch = 'I';		/* To tell the caller what this was */
	    break;
	case 'a':
	case 'A':
	case 1:
	    if (!new_hp_entry(&ftemp))
		break;

	    if (fl->tail != NULL) {
		fl->tail->next_entry = ftemp;
		ftemp->prev_entry = fl->tail;
	    } else {
		fl->head = ftemp;
		ftemp->prev_entry = NULL;
	    }

	    ftemp->next_entry = NULL;
	    fl->tail = ftemp;
	    fe = ftemp;
	    endloop_local = 1;
	    *aborted = 0;
	    ch = 'I';
	    break;
	case 'd':
	case 'D':
	case KEY_DC:
	    if (fl->head != NULL) {
		if (fe->next_entry != NULL)
		    fe->next_entry->prev_entry = fe->prev_entry;
		else
		    fl->tail = fe->prev_entry;

		if (fe->prev_entry != NULL)
		    fe->prev_entry->next_entry = fe->next_entry;
		else
		    fl->head = fe->next_entry;

		if (fe->prev_entry == fl->tail) {
		    ftemp = fe->prev_entry;
		    idx--;
		} else
		    ftemp = fe->next_entry;

		free(fe);
		fe = ftemp;
		update_hp_screen(fl, fe, idx, win);
	    }
	    break;
	case 13:
	    endloop_local = 1;
	    *aborted = 0;
	    break;
	case 'x':
	case 'X':
	case 'q':
	case 'Q':
	case 27:
	case 24:
	    endloop_local = 1;
	    *aborted = 1;
	    break;
	case 'l':
	case 'L':
	    refresh_screen();
	    break;
	}
	update_panels();
	doupdate();
    } while (!endloop_local);

    *rfe = fe;

    del_panel(panel);
    delwin(win);
    del_panel(bpanel);
    delwin(bwin);
    update_panels();
    doupdate();

    *command = ch;
}

void editfilter(unsigned int protocol, int *aborted)
{
    char filename[FLT_FILENAME_MAX];
    struct hostparams hptemp;
    struct filterlist fl;
    struct filterent *fe;
    struct ffnode *flist;
    struct ffnode *ffile;
    struct filterfileent *ffe;
    char initsrc[30];
    char initdst[30];
    char initsmask[20];
    char initdmask[20];
    char s_portstr[6];
    char d_portstr[6];
    char inexstr[2];
    int gethp_aborted;
    int keycommand;

    if (!mark_filter_change(protocol))
	return;

    if (loadfilterlist(protocol, &flist) == 1) {
	listfileerr(1);
	destroyfilterlist(flist);
	clear_flt_tag(protocol);
	return;
    }
    pickafilter(flist, &ffile, aborted);

    clear_flt_tag(protocol);
    if ((*aborted)) {
	destroyfilterlist(flist);
	clear_flt_tag(protocol);
	return;
    }
    ffe = &(ffile->ffe);

    get_filter_description(ffe->desc, aborted, ffe->desc);

    if (*aborted) {
	destroyfilterlist(flist);
	clear_flt_tag(protocol);
	return;
    }
    strncpy(filename, get_path(T_WORKDIR, ffe->filename), FLT_FILENAME_MAX - 1);

    if (loadfilter(filename, &fl, FLT_DONTRESOLVE))
        return;

    do {
	select_host_parameters(protocol, &fl, &fe, aborted, &keycommand);

	if (!(*aborted)) {
	    if (keycommand == 'I') {
		strcpy(initsrc, "");
		strcpy(initsmask, "");
		strcpy(initdst, "0.0.0.0");
		strcpy(initdmask, "0.0.0.0");
		strcpy(s_portstr, "");
		strcpy(d_portstr, "0");
		strcpy(inexstr, "I");
	    } else {
		strcpy(initsrc, fe->hp.s_fqdn);
		strcpy(initsmask, fe->hp.s_mask);
		strcpy(initdst, fe->hp.d_fqdn);
		strcpy(initdmask, fe->hp.d_mask);
		sprintf(s_portstr, "%u", fe->hp.sport);
		sprintf(d_portstr, "%u", fe->hp.dport);
		bzero(inexstr, 2);
		inexstr[0] = toupper(fe->hp.reverse);
	    }

	    gethostparams(&hptemp, &gethp_aborted,
			  initsrc, initsmask, s_portstr,
			  initdst, initdmask, d_portstr, inexstr);

	    if (!gethp_aborted)
		memcpy(&(fe->hp), &hptemp, sizeof(struct hostparams));
	}
    } while (!(*aborted));

    save_filterlist(protocol, flist);	/* This also destroys it */
    savefilter(protocol, filename, &fl);
    destroyfilter(&fl);
}
