/*****************************************************************************/
/*  protocols.h                                                              */
/*  Copyright (C) 1999 Brian Masney <masneyb@seul.org>                       */
/*                                                                           */
/*  This library is free software; you can redistribute it and/or            */
/*  modify it under the terms of the GNU Library General Public              */
/*  License as published by the Free Software Foundation; either             */
/*  version 2 of the License, or (at your option) any later version.         */
/*                                                                           */
/*  This library is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU        */
/*  Library General Public License for more details.                         */
/*                                                                           */
/*  You should have received a copy of the GNU Library General Public        */
/*  License along with this library; if not, write to the Free Software      */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#ifndef __PROTOCOLS_H
#define __PROTOCOLS_H

#define _REENTRANT
#include <glib.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <stdio.h>
#include <time.h>
#include "config.h"

#if defined(HAVE_GETTEXT)
 #if !defined(_)
  #define _(String) gettext (String)
 #endif
 #include <libintl.h>
 #include <locale.h>
#else 
 #if !defined(_)
  #define _(String) String
 #endif
#endif

#define GFTP_TYPE_BINARY	1
#define GFTP_TYPE_ASCII		2   

typedef enum gftp_transfer_type_tag {
   gftp_transfer_passive,
   gftp_transfer_active
} gftp_transfer_type;

typedef enum gftp_logging_type_tag {
   gftp_logging_send,
   gftp_logging_recv,
   gftp_logging_error,
   gftp_logging_misc
} gftp_logging_type;

typedef struct gftp_file_tag {
   char *file;
   char *user;
   char *group;
   char *attribs;
   time_t datetime;
   unsigned long size;
} gftp_file;

typedef void (*gftp_logging_func)(gftp_logging_type level, void *ptr, const char *string, ...);

struct _gftp_proxy_hosts {
   gint32 network_address,
          netmask;
   char *domain;
};

typedef struct _gftp_proxy_hosts gftp_proxy_hosts;

typedef struct _gftp_request gftp_request;

struct _gftp_request {
   char *hostname,		/* Hostname we will connect to */
        *username,		/* Username for host*/
        *password,		/* Password for host */
        *account,		/* Account for host (FTP only) */
        *directory,		/* Current working directory */
        *proxy_config,		/* Proxy configuration */
        *proxy_hostname,	/* Proxy hostname */
        *proxy_username,	/* Proxy username */
        *proxy_password,	/* Proxy password */
        *proxy_account,		/* Proxy account (FTP only) */
        *url_prefix,		/* URL Prefix (ex: ftp) */
        *protocol_name,		/* Protocol description */
        *last_ftp_response,	/* Last response from server */
        *last_dir_entry;	/* Last dir entry from server */

   unsigned int port,		/* Port of remote site */
                proxy_port;	/* Port of the proxy server */

   unsigned long read_bytes, 	/* How many bytes we have read so far (HTTP only) */
                 max_bytes;	/* Content length (HTTP only */
   
   FILE *sockfd,		/* Control connection */
        *datafd;		/* Data connection */
   int sockfd_unbuffered,	/* Control connection */
       datafd_unbuffered;	/* Data connection */
        
   struct hostent *host;	/* Remote host we are connected to */

   int data_type;		/* ASCII or BINARY (FTP only) */
   gftp_transfer_type transfer_type;	/* Passive or non-passive (FTP only) */
   unsigned int logging : 1,
                use_proxy : 1,
                resolve_symlinks : 1,
                supports_resume : 1,
                unsafe_symlinks : 1,
                add_onek : 1;
   
   gftp_logging_func logging_function;
   void *user_data;

   GList *proxy_hosts;
      
   void (*init)		(gftp_request *request);
   int (*connect)	(gftp_request *request);
   void (*disconnect) 	(gftp_request *request);
   long (*get_file) (gftp_request *request, const char *filename, long startsize);
   int (*put_file) 	(gftp_request *request, const char *filename, long startsize);
   int (*end_transfer) 	(gftp_request *request);
   int (*list_files) 	(gftp_request *request);
   int (*get_next_file)	(gftp_request *request, gftp_file *fle, FILE *fd);
   int (*set_data_type)	(gftp_request *request, int data_type);
   long (*get_file_size) (gftp_request *request, const char *filename);

   int (*chdir)		(gftp_request *request, const char *directory);
   int (*rmdir)		(gftp_request *request, const char *directory);
   int (*rmfile)	(gftp_request *request, const char *filename);
   int (*mkdir)		(gftp_request *request, const char *directory);
   int (*rename)	(gftp_request *request, const char *oldname, const char *newname);
   int (*chmod)		(gftp_request *request, const char *filename, int mode);
   int (*site)		(gftp_request *request, const char *filename);
};

typedef struct _supported_gftp_protocols supported_gftp_protocols;

struct _supported_gftp_protocols {
   char *name;
   void (*init) (gftp_request *request);
};

extern supported_gftp_protocols gftp_protocols[];

#define GFTP_CONNECTED(request)		(request->sockfd != NULL)
#define GFTP_GET_HOSTNAME(request)	(request->hostname)
#define GFTP_GET_USERNAME(request)	(request->username)
#define GFTP_GET_PASSWORD(request)	(request->password)
#define GFTP_GET_ACCOUNT(request)	(request->account)
#define GFTP_GET_DIRECTORY(request)	(request->directory)
#define GFTP_GET_PORT(request)		(request->port)
#define GFTP_GET_PROXY_CONFIG(request)	(request->proxy_config)
#define GFTP_GET_PROXY_HOSTNAME(request)(request->proxy_hostname)
#define GFTP_GET_PROXY_USERNAME(request)(request->proxy_username)
#define GFTP_GET_PROXY_PASSWORD(request)(request->proxy_password)
#define GFTP_GET_PROXY_ACCOUNT(request)	(request->proxy_account)
#define GFTP_GET_PROXY_PORT(request)	(request->proxy_port)
#define GFTP_GET_URL_PREFIX(request)	(request->url_prefix)
#define GFTP_GET_PROTOCOL_NAME(request)	(request->protocol_name)
#define GFTP_GET_LAST_RESPONSE(request)	(request->last_ftp_response)
#define GFTP_GET_LAST_DIRENT(request)	(request->last_dir_entry)
#define GFTP_GET_CONTROL_FD(request)	(request->sockfd)
#define GFTP_GET_DATA_FD(request)	(request->datafd)
#define GFTP_GET_DATA_TYPE(request)	(request->data_type)
#define GFTP_GET_TRANSFER_TYPE(request)	(request->transfer_type)
#define GFTP_SET_TRANSFER_TYPE(request,val) (request->transfer_type = (val))
#define GFTP_GET_LOGGING(request)	(request->logging)
#define GFTP_SET_LOGGING(request, val)	(request->logging = (val))
#define GFTP_SUPPORTS_RESUME(request)	(request->supports_resume)
#define GFTP_UNSAFE_SYMLINKS(request)	(request->unsafe_symlinks)

void rfc959_init (gftp_request *request);
void rfc2068_init (gftp_request *request);

gftp_request *gftp_request_new (void);
void gftp_request_destroy (gftp_request *request);
void gftp_file_destroy (gftp_file *file);
void gftp_logging (gftp_request *request, int logging, void (*logging_function)(gftp_logging_type level, void *ptr, const char *string, ...), void *ptr);
int gftp_connect (gftp_request *request);
void gftp_disconnect (gftp_request *request);
long gftp_get_file (gftp_request *request, const char *filename, long startsize);
int gftp_put_file (gftp_request *request, const char *filename, long startsize);
int gftp_list_files (gftp_request *request);
int gftp_parse_url (gftp_request *request, const char *url);
int gftp_get_next_file (gftp_request *request, gftp_file *fle, FILE *fd);
int gftp_end_transfer (gftp_request *request);
int gftp_read_response (gftp_request *request);
int gftp_set_data_type (gftp_request *request, int data_type);
void gftp_set_hostname (gftp_request *request, const char *hostname);
void gftp_set_username (gftp_request *request, const char *username);
void gftp_set_password (gftp_request *request, const char *password);
void gftp_set_account (gftp_request *request, const char *account);
int gftp_set_directory (gftp_request *request, const char *directory);
void gftp_set_port (gftp_request *request, unsigned int port);
void gftp_set_proxy_hostname (gftp_request *request, const char *hostname);
void gftp_set_proxy_username (gftp_request *request, const char *username);
void gftp_set_proxy_password (gftp_request *request, const char *password);
void gftp_set_proxy_account (gftp_request *request, const char *account);
void gftp_set_proxy_port (gftp_request *request, unsigned int port);
int gftp_remove_directory (gftp_request *request, const char *directory);
int gftp_remove_file (gftp_request *request, const char *file);
int gftp_make_directory (gftp_request *request, const char *directory);
int gftp_rename_file (gftp_request *request, const char *oldname, const char *newname);
int gftp_chmod (gftp_request *request, const char *file, int mode);
char gftp_site_cmd (gftp_request *request, const char *command);
void gftp_set_proxy_config (gftp_request *request, const char *proxy_config);
long gftp_get_file_size (gftp_request *request, const char *filename);
int gftp_need_proxy (gftp_request *request);

#endif
