/*****************************************************************************/
/*  ftp.h - include file for the whole ftp program                           */
/*  Copyright (C) 1998 Brian Masney <masneyb@seul.org>                       */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                */
/*****************************************************************************/

#ifndef __FTP_H
#define __FTP_H

#define _REENTRANT
#include <pthread.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <utime.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/utsname.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "config.h"

#if defined(HAVE_GETTEXT)
 #if !defined(_)
  #define _(String) gettext (String)
 #endif
 #include <libintl.h>
 #include <locale.h>
#else 
 #if !defined(_)
  #define _(String) String
 #endif
#endif

#if !defined(N_)
 #define N_(String) String
#endif

#include "protocols.h"

/* Some general settings */
#define BASE_CONF_DIR		"~/.gftp"
#define CONFIG_FILE		BASE_CONF_DIR "/gftprc"
#define BOOKMARKS_FILE		BASE_CONF_DIR "/bookmarks"
#define MAXSTR			255
#define MAX_HIST_LEN		10
#define ANON_LOGIN		"anonymous"

struct ftp_file_data {
   /* It is important to keep these structure members in THE SAME ORDER! */
   char *file,			/* The filename */
        *user,			/* User that owns this file */
        *group,			/* Group that owns this file */
        *attribs;		/* File permissions */
   time_t datetime;		/* File date and time */
   unsigned long size;		/* File size */

   /* These structure members can be in any order, and new members can be
      added at the bottom */
   char *remote_file;		/* The full path to the remote directory */
   unsigned long remote_size;	/* The size of the remote filename */
   unsigned int isdir : 1,	/* Is this a directory */
   		isexe : 1,	/* Is this an executable? */
   		islink : 1,	/* Is this a symlink? */
   		possible_dir:1,	/* Is there a possibility this is a directory */
   				/* This gets set when unsafe_symlinks are set */
   		shown : 1,	/* Is this file shown? */
   		selected : 1,	/* Is this file selected? */
   		ascii : 1,	/* Download this as ASCII or BINARY */
   		restart : 1,	/* Restart this file transfer */
   		skip : 1,	/* Skip this file transfer */
   		exists : 1,	/* This file exists locally and remotely */
   		done_view : 1,	/* View this file when it's done */
   		done_edit : 1,	/* Edit this file when it's done */
   		done_rm : 1;	/* Remove this file when it's done */
   GtkCTreeNode *node;		/* Our node in the transfer window */
};

/* This structure will be contained in the ftp_window_data structure below.
   This should contain all of the information that will be needed about a 
   remote site. This structure will also be contained inside the 
   ftp_transfer_data structure for the multiple downloads. */
   
struct ftp_host_data {
   gftp_request *ftpdata; 	/* Information about the current FTP host */
   int totalfiles; 		/* How many files are here */
   unsigned int wait : 1, 	/* Shall we wait for the connection to finish? */
                stopable : 1, 	/* Can we stop? */
                getdir : 1; 	/* Do we want to get the dir listing? */
   unsigned long gotbytes;	/* How many bytes we transfered when we are
   				   getting the directory listing */
   GList *files;		/* All of the files in the current directory */
   struct ftp_window_data *wdata;
   pthread_t tid;
   int connection_number; 	/* The number of times we have tried to connect 
   				   thus far */
};

/* There will be 2 of these created...one for the local side and one for the 
   remote side. When all of the callbacks are created, the proper structure
   will be passed to the function */
   
struct ftp_window_data {
   GtkWidget *combo, 		/* Entry widget/history for the user to enter 
   				   a directory */
             *hoststxt, 	/* Show which directory we're in */
             *listbox; 		/* Our listbox showing the files */
   unsigned int local, 		/* -1 = Not Connected, 0 = Remote, 1 = Local, 2 = Cached */
       numselected, 		/* How many items are selected in the list */
       sortcol, 		/* Which column we are sorting by */
       histlen, 		/* The number of entries in the history list */
       cached : 1, 		/* Is this directory listing cached */
       sorted : 1, 		/* Is this directory listing sorted */
       sortasds : 1; 		/* Sorted ascending or descending */
   char *filespec;		/* Filespec for the listbox */
   struct ftp_host_data *hdata; /* The host that we are connected to */
   GList *history; 		/* The history of the directories in the combo widget */
   GtkItemFactory *ifactory; 	/* The menus that will come up when you right click */
};         

/* This will be the structure used for each thread in transfering files */

struct ftp_transfer_data {
   struct ftp_host_data *hdata;	/* The host that we are connecting to */
   GList *curfle, 		/* The current file that we are transferring */
         *updfle; 		/* The last file that was updated on the screen */
   unsigned long curtrans, 	/* How much of the file we actually transferred */
                 curmax,	/* The size of the file we are transferring */
                 sttrans, 	/* Where we started at on the file */
                 trans_bytes,	/* How many bytes we transfered so far */
                 total_bytes;	/* How many bytes are in all the files to transfer */
                 
   struct timeval inittime,	/* The time that the entire transfer started */
                  starttime, 	/* The time that we started this file transfer */
                  lasttime; 	/* The beginning of the last read() */

   char  **dirs_to_be_made; 	/* Pointer to strings of directories to be made */
   int num_dirs_to_be_made, 	/* Number of directories that need to be made */
       current_file_number, 	/* The current number of the file we are transfering */
       current_file_retries; 	/* The number of times we tried to transfer this file */
       
   unsigned int direction : 1,	/* Are we downloading or uploading */
                ready : 1,	/* Is this transfer ready to go? */
                started : 1,	/* Did we start this transfer */
                cancel : 1,	/* Cancel this transfer */
                done : 1,	/* This transfer is done */
                show : 1,	/* Show this transfer and get ready to start it */
                stalled : 1,	/* Is this transfer stalled? */
                next_file : 1;	/* We're on the next file */
   GtkWidget *clist; 		/* CList widget for the dialog box when a file exists */
   pthread_mutex_t mutex; 	/* Mutex protecting this struct */
   pthread_t tid; 		/* This thread id */
   float kbs, 			/* Current KB/s we are getting */
         maxkbs; 		/* The max KB/s this transfer can get */
   GtkCTreeNode *node;		/* Our node in the transfer window */
};

/* This will be for queueing messages to the logging window. This will be
   used for the threads, since gtk is not thread safe, we cannot update the
   display inside the threads */
   
struct ftp_log_queue {
   char *msg; 			/* Our message */
   int type; 			/* Logging type. LOG_MISC, LOG_SEND, or LOG_RECV */
};

/* This will hold information about currently open files being view or edited */

struct viewedit_data {
   char *filename, 		/* File we are viewing/editing currently */
        *remote_filename; 	/* The filename on the remote computer */
   struct stat st; 		/* Vital file statistics */
   pid_t pid; 			/* Our process id */
   char **argv; 		/* Our arguments we passed to execvp. We will 
   				   free it when the process terminates. This 
   				   is the safest place to free this */
   unsigned int view : 1,	/* View or edit this file */
                rm : 1;		/* Delete this file after we're done with it */
};

/* For registering file formats */

struct pix_ext {
   char *ext, 			/* The file extension to register */
        *filename, 		/* The xpm file to display */
        *view_program, 		/* The program used to view this file */
        *ascii_binary; 		/* Is this a ASCII transfer A or a BINARY transfer B */
   int stlen; 			/* How long is the file extension. */
   GdkPixmap *pixmap; 		/* Our pixmap */
   GdkBitmap *mask; 		/* Mask for the pixmap */
   struct pix_ext *next; 	/* Next file extension */
};

struct conn_categories { 	/* Each category will have one of these nodes */
   char *path; 			/* Path */
   GtkCTreeNode *cnode; 	/* In the edit bookmarks menu, the node */
   char *hostname, 		/* Our actual internet hostname */
   	*protocol,		/* Protocol we will connect through */
        *remote_dir, 		/* Initial directory */
        *local_dir,		/* Init local directory */
        *user, 			/* Username to log in as */
        *pass, 			/* Our password */
        *acct; 			/* Account */

   unsigned int port,		/* The port we will connect to */
      		isfolder : 1, 	/* If this is set, then the children field can
                                   be non-NULL */
                save_password : 1; /* Save this password */
   struct conn_categories *children, /* The children of this node. */
                          *prev, /* The parent of this node */
                          *next; /* The next sibling of this node */
};

/* Passed to callback functions by MakeEditDialog and MakeYesNoDialog */
struct dialog_data {
   GtkWidget *dialog,		/* Pointer to the dialog widget */
             *edit;		/* Pointer to the entry widget */
   GList *all_buttons;		/* A list of all the buttons in the dialog */
   gpointer data;		/* Data pointer we will pass to the function */
};

/* All these variables are defined in options.h */
extern GdkPixmap *dotdot_pixmap, *dir_pixmap, *linkdir_pixmap, *linkfile_pixmap,
   *exe_pixmap, *doc_pixmap, *open_dir_pixmap;
extern GdkBitmap *dotdot_mask, *dir_mask, *linkdir_mask, *linkfile_mask, 
   *exe_mask, *doc_mask, *open_dir_mask;

extern GtkItemFactory *factory;
extern GHashTable *bookmarks_htable;
extern struct pix_ext *registered_exts;
extern struct conn_categories *hostcat;
extern GList *file_transfers;
extern pthread_mutex_t transfer_mutex, log_mutex;
extern GList *viewedit_processes, *file_transfer_logs, *host_history,
	*port_history, *user_history, *proxy_hosts;
extern int host_len, port_len, user_len;
extern GtkWidget *logwdw, *dlwdw, *local_frame, *remote_frame, *log_table, 
   *transfer_scroll, *openurl_btn, *stop_btn, *hostedit, *useredit,
   *passedit, *portedit, *protocol_menu;
extern struct ftp_window_data window1, window2;

extern char *emailaddr, *edit_program, *view_program,
   *firewall_host, *firewall_username, *firewall_password,
   *firewall_account, version[], *proxy_config, *http_proxy_host, 
   *http_proxy_username, *http_proxy_password;
extern int firewall_port, http_proxy_port;

extern int do_one_transfer_at_a_time, start_file_transfers, 
    transfer_in_progress, use_cache, passive_transfer, sort_dirs_first,
    confirm_delete, use_default_dl_types, show_hidden_files,
    reconnect_diag, refresh_files, save_geometry, preserve_permissions,
    listbox_local_width, listbox_remote_width, listbox_file_height, 
    transfer_height, log_height, timeout, retries, sleep_time,
    file_trans_column, local_columns[6], remote_columns[6], resolve_symlinks;
extern float maxkbs;
extern volatile sig_atomic_t viewedit_process_done;

/* bookmarks.c */
void run_bookmark (gpointer data);
void add_bookmark (gpointer data);
void edit_bookmarks (gpointer data);
void build_bookmarks_menu (void);

/* cache.c */
FILE *new_cache_entry (char *description);
FILE *find_cache_entry (char *description);
void clear_cache_files (void);
void delete_cache_entry (char *description);

/* chmod_dialog.c */
void chmod_dialog (gpointer data);

/* config_file.c */
void options_dialog (gpointer data);
void read_config_file (void);
void read_bookmarks (void);
void write_config_file (void);
int parse_args (char *str, int numargs, int lineno, char **first, ...);
GHashTable *build_hash_table (struct conn_categories *entry);

/* delete_dialog.c */
void delete_dialog (gpointer data);

/* dnd.c */
void listbox_drag (GtkWidget *widget, GdkDragContext *context, 
   	GtkSelectionData *selection_data, guint info, guint32 clk_time, gpointer data);
void listbox_get_drag_data (GtkWidget *widget, GdkDragContext *context, gint x, 
	gint y, GtkSelectionData *selection_data, guint info, guint32 clk_time, 
	struct ftp_window_data *wdata);
void openurl_get_drag_data (GtkWidget *widget, GdkDragContext *context, gint x, 
	gint y, GtkSelectionData *selection_data, guint info, guint32 clk_time, 
	struct ftp_window_data *wdata);

/* file_transfer.c */
void retrCB (GtkWidget *widget, gpointer data);
void putCB (GtkWidget *widget, gpointer data);
struct ftp_transfer_data *inittrans(int direction, int add_to_queue, GList *files);
void dotrans (struct ftp_transfer_data *tdata);
void *ftp_get_files (void *ptr);
void *ftp_put_files (void *ptr);

/* ftp.c */
int ftp_list_files (struct ftp_window_data *wdata, int usecache);
GList *get_remote_files (struct ftp_host_data *hdata, char *path, int *total, int load_from_cache, GtkLabel *up_wid);
void add_file_transfer (struct ftp_transfer_data *tdata);
struct ftp_transfer_data *transfer_one_file (char *local_file, char *remote_file, int dl_or_up);
int ftp_connect (struct ftp_host_data *hdata, int getdir);
void stop_button (GtkWidget *widget, struct ftp_window_data *wdata);
void disconnect (struct ftp_window_data *wdata);
int get_file_transfer_mode (char *filename);

/* gftp.c */
int chdiredit (GtkWidget *widget, struct ftp_window_data *wdata);
void sortrows (GtkCList *clist, gint column, gpointer data);
void delete_ftp_file_info (struct ftp_window_data *wdata);
void queue_log (gftp_logging_type level, void *ptr, const char *string, ...);
void ftp_log (gftp_logging_type level, void *ptr, const char *string, ...);
void toolbar_hostedit (GtkWidget *widget, gpointer data);
void update_ftp_info (struct ftp_window_data *wdata);
void refresh (struct ftp_window_data *wdata);
void fix_display (void);

/* misc.c */
char *insert_commas (unsigned long number);
void add_local_files (struct ftp_window_data *win);
GList *get_local_files (char *path, int *total);
GList *parse_local_file (FILE *fd, int local, int *total);
void add_file_listbox (struct ftp_window_data *wdata, struct ftp_file_data *fle);
long file_countlf (int filefd, long endpos);
char *alltrim (char *str);
char *expand_path (char *src);
void remove_double_slashes (char *string);
char *make_temp_filename (char *ext);
void free_file_list (GList *filelist);
void free_fdata (struct ftp_file_data *fle);
void free_hdata (struct ftp_host_data *hdata);
struct ftp_host_data *new_hdata (void);
void free_tdata (struct ftp_transfer_data *tdata);
struct ftp_transfer_data *new_tdata (void);
void copy_hdata_struct (struct ftp_host_data *hdata, struct ftp_host_data *newhdata);
void copy_fdata_struct (struct ftp_file_data *fle, struct ftp_file_data *newfle);
int compare_hdata_structs (gftp_request *request1, gftp_request *request2, int compare_dirs);
GList *get_next_selected_filename (GList *filelist);
void open_xpm (char *filename, GtkWidget *parent, GdkPixmap **pixmap, GdkBitmap **mask, int quit_on_err);
char *get_xpm_path (char *filename, int quit_on_err);
GtkWidget *toolbar_pixmap (GtkWidget *widget, char **xpmdata);
int check_status(char *name, struct ftp_window_data *wdata, int only_one, int at_least_one, int func);
void make_nonnull (char **str);
void create_item_factory (GtkItemFactory *ifactory, guint n_entries, GtkItemFactoryEntry *entries, gpointer callback_data);
void add_history (GtkWidget *widget, GList **history, int *histlen, char *str);

/* misc_dialogs.c */
void about_dialog (gpointer data);
void MakeEditDialog (char *diagtxt, char *infotxt, char *deftext, int edit_shown, int erase,
	char *oktxt, void (*okfunc)(), void *okptr,
	char *canceltxt, void (*cancelfunc)(), void *cancelptr);
void MakeYesNoDialog (char *diagtxt, char *infotxt, int erase, int num, ...);
void destroy_dialog_data (struct dialog_data *data);
void site_dialog (struct ftp_window_data *wdata);
void change_filespec (struct ftp_window_data *wdata);
void tb_openurl_dialog (gpointer data);
void openurl_dialog (gpointer data);
void reconnect_dialog (struct ftp_host_data *hdata);
void viewlog (gpointer data);
void savelog (gpointer data);
void clearlog (gpointer data);

/* mkdir_dialog.c */
void mkdir_dialog (gpointer data);

/* rename_dialog.c */
void rename_dialog (gpointer data);

/* transfer_gui.c */
gint update_downloads (gpointer data);
gint dl_click (GtkWidget *widget, GdkEventButton *event, gpointer data);
void start_transfer (GtkWidget *widget, gpointer data);
void stop_transfer (GtkWidget *widget, gpointer data);

/* view_dialog.c */
void view_dialog (gpointer data);
void edit_dialog (gpointer data);
void view_file (char *filename, int viewedit, int del_file, int start_pos, char *remote_file);

#endif
