/*****************************************************************************/
/*  ftp.c - contains most of the core network routines                       */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

static void *getdir_thread (void *data);
static void try_connect_again (GtkWidget *widget, struct dialog_data *data);
static void dont_connect_again (GtkWidget *widget, struct dialog_data *data);
static void *connect_thread (void *data);
static void sig_quit (int signo);

static sigjmp_buf envir;

/* This struct will be used more in later versions of gftp */
struct task_thread {
   int id; /* What we are doing */
   void *ptr1; /* Pointer to some data */
   void *ptr2;
};

#define ID_GET_DIRECTORY	1

int ftp_list_files (struct ftp_window_data *wdata, int load_from_cache) {
   struct task_thread tthread;
   void *success;

   wdata->hdata->totalfiles = wdata->numselected = 0;
   if (wdata->hdata->files == NULL) {
      if (wdata->local == 2 && !ftp_connect (wdata->hdata, 0)) return (0);
      wdata->local = 0;
      update_ftp_info (wdata);
      wdata->hdata->getdir = 1;
      gtk_widget_set_sensitive (stop_btn, 1);
      gtk_clist_freeze (GTK_CLIST (wdata->listbox));
      wdata->hdata->stopable = 1;
      tthread.id = ID_GET_DIRECTORY;
      tthread.ptr1 = wdata;
      tthread.ptr2 = &load_from_cache;
      pthread_create (&wdata->hdata->tid, NULL, getdir_thread, &tthread);

      while (wdata->hdata->stopable) {
         fix_display();
         sleep (0);
      }

      gtk_clist_thaw (GTK_CLIST (wdata->listbox));
      gtk_widget_set_sensitive (stop_btn, 0);
      pthread_join (wdata->hdata->tid, &success);
      wdata->hdata->tid = 0;
   }
   
   if (wdata->hdata->files == NULL) {
      gftp_disconnect (wdata->hdata->ftpdata);
      return (0);
   }
   wdata->sorted = 0;
   sortrows (GTK_CLIST (wdata->listbox), wdata->sortcol, (gpointer) wdata);
   gtk_clist_select_row (GTK_CLIST (wdata->listbox), 0, 0);
   update_ftp_info (wdata);
   return (1);
}
/*****************************************************************************/
static void *getdir_thread (void *data) {
   struct ftp_window_data *wdata;
   struct task_thread *tthread;
   gftp_logging_func old_func;
   int sj;
   
   tthread = (struct task_thread *) data;
   wdata = (struct ftp_window_data *) tthread->ptr1;
   old_func = wdata->hdata->ftpdata->logging_function;
   wdata->hdata->ftpdata->logging_function = queue_log;

   sj = sigsetjmp (envir, 1);
   signal (SIGINT, sig_quit);
   signal (SIGALRM, sig_quit);
   if ((sj == 0 || sj == 2) && wdata->hdata->getdir && tthread->id == ID_GET_DIRECTORY) {
      wdata->hdata->files = get_remote_files (wdata->hdata, NULL, &wdata->hdata->totalfiles, *(int *) tthread->ptr2, NULL);
   }
   if (sj != 1 && (wdata->hdata->files == NULL && *(int *) tthread->ptr2)) {
      *(int *) tthread->ptr2 = 0;
      siglongjmp (envir, 2);
   }
   wdata->hdata->ftpdata->logging_function = old_func;
   wdata->hdata->stopable = 0;
   return (NULL);
}
/*****************************************************************************/
GList *get_remote_files (struct ftp_host_data *hdata, char *path, int *total, int load_from_cache, GtkLabel *up_wid) {
   struct ftp_file_data *newfle;
   FILE *cachefd, *listfd;
   char *description;
   int got, cached;
   time_t lasttime;
   size_t dltotal;
   gftp_file fle;
   GList *files;
   
   *total = 0;

   if (path != NULL && gftp_set_directory (hdata->ftpdata, path) != 0) {
      return (NULL);
   }

   if (up_wid != NULL) {
      update_ftp_info (hdata->wdata);
      gtk_label_set (up_wid, _("Receiving file names..."));
      fix_display ();
   }

   description = g_strdup_printf ("%s://%s:%s@%s:%d%s", 
   	GFTP_GET_URL_PREFIX (hdata->ftpdata),
   	GFTP_GET_USERNAME (hdata->ftpdata), GFTP_GET_PASSWORD (hdata->ftpdata),
   	GFTP_GET_HOSTNAME (hdata->ftpdata), 
   	GFTP_GET_PORT (hdata->ftpdata) == 0 ? 21 : GFTP_GET_PORT (hdata->ftpdata),
   	path == NULL ? GFTP_GET_DIRECTORY (hdata->ftpdata) == NULL ? "" : GFTP_GET_DIRECTORY (hdata->ftpdata) : path);
   remove_double_slashes (description+6);
  
   /* Check to see if this item is already in the cache */
   listfd = NULL;
   cached = 0;
   cachefd = NULL;
   if (use_cache){
      if (load_from_cache && (listfd = find_cache_entry (description)) != NULL) {
         cached = 1;
      }
      else {
         cachefd = new_cache_entry (description);
      }
   }

   g_free (description);

   /* If we aren't actually connected to the site, reconnect */
   if (!cached && hdata->wdata->hdata == hdata && hdata->wdata->local == 2 && !ftp_connect (hdata, 0)) {
      if (listfd) fclose (listfd);
      return (NULL);
   }

   if (!cached) {
      if (gftp_list_files (hdata->ftpdata) != 0) {
         if (listfd) fclose (listfd);
         if (cachefd) fclose (cachefd);
         return (NULL);
      }
      listfd = GFTP_GET_DATA_FD (hdata->ftpdata);
   }
   
   files = NULL;
   lasttime = time (NULL);
   dltotal = 0;
   hdata->gotbytes = 0;
   while ((got = gftp_get_next_file (hdata->ftpdata, &fle, listfd)) > 0) {
      dltotal += got;
      hdata->gotbytes += got;
      if (fle.file == NULL) break;
      if (strcmp (fle.file, ".") == 0 || strcmp (fle.file, "..") == 0) {
         gftp_file_destroy (&fle);
         continue;
      }

      if (cachefd != NULL) {
         fwrite (GFTP_GET_LAST_DIRENT (hdata->ftpdata), 1, strlen (GFTP_GET_LAST_DIRENT (hdata->ftpdata)), cachefd);
      }
      
      newfle = g_malloc0 (sizeof (struct ftp_file_data));
      memcpy (newfle, &fle, sizeof (gftp_file));
      if (strchr (newfle->attribs, 'd') != NULL) newfle->isdir = 1;
      if (strchr (newfle->attribs, 'l') != NULL) newfle->islink = 1;
      if ((strchr (newfle->attribs, 'x') != NULL) && !newfle->isdir
      		&& !newfle->islink) newfle->isexe = 1;
      newfle->possible_dir = GFTP_UNSAFE_SYMLINKS (hdata->ftpdata);
      files = g_list_append (files, newfle);
      (*total)++;
   }
   
   if (cachefd != NULL) fclose (cachefd);
   if (!cached) gftp_end_transfer (hdata->ftpdata);

   newfle = g_malloc0 (sizeof (struct ftp_file_data));
   newfle->file = g_malloc (3);
   strcpy (newfle->file, "..");
   newfle->user = g_malloc (1);
   *newfle->user = '\0';
   newfle->group = g_malloc (1);
   *newfle->group = '\0';
   newfle->attribs = g_malloc (1);
   *newfle->attribs = '\0';
   newfle->isdir = 1;

   files = g_list_prepend (files, newfle);
   (*total)++;

   if (hdata->wdata->hdata == hdata) hdata->wdata->cached = cached;
   hdata->gotbytes = -1;
   
   return (files);
}
/*****************************************************************************/
int ftp_connect (struct ftp_host_data *hdata, int getdir) {
   int success, use_ftp;
   void *ret;
   
   use_ftp = strcmp (GFTP_GET_PROTOCOL_NAME (hdata->ftpdata), "FTP") == 0 && 
   	strcmp (GFTP_GET_PROXY_CONFIG (hdata->ftpdata), "http") != 0;
   if (use_ftp) {
      gftp_set_proxy_hostname (hdata->ftpdata, firewall_host);
      gftp_set_proxy_port (hdata->ftpdata, firewall_port);
      gftp_set_proxy_username (hdata->ftpdata, firewall_username);
      gftp_set_proxy_password (hdata->ftpdata, firewall_password);
      gftp_set_proxy_account (hdata->ftpdata, firewall_account);
      gftp_set_proxy_config (hdata->ftpdata, proxy_config);
   }
   else {
      gftp_set_proxy_hostname (hdata->ftpdata, http_proxy_host);
      gftp_set_proxy_port (hdata->ftpdata, http_proxy_port);
      gftp_set_proxy_username (hdata->ftpdata, http_proxy_username);
      gftp_set_proxy_password (hdata->ftpdata, http_proxy_password);
      if (strcmp (GFTP_GET_PROTOCOL_NAME (hdata->ftpdata), "FTP") == 0) {
         gftp_protocols[1].init (hdata->ftpdata);
         if (hdata->ftpdata->proxy_config) g_free (hdata->ftpdata->proxy_config);
         hdata->ftpdata->proxy_config = g_malloc (4);
         strcpy (hdata->ftpdata->proxy_config, "ftp");
      }
      else {
         if (hdata->ftpdata->proxy_config) g_free (hdata->ftpdata->proxy_config);
         hdata->ftpdata->proxy_config = g_malloc (5);
         strcpy (hdata->ftpdata->proxy_config, "http");
      }
   }
   
   ret = 0;
   if (hdata->wdata->hdata == hdata) {
      hdata->getdir = getdir;
      gtk_label_set (GTK_LABEL (hdata->wdata->hoststxt), _("Connecting..."));
   }
   if (GFTP_GET_USERNAME (hdata->ftpdata) == NULL || *GFTP_GET_USERNAME (hdata->ftpdata) == '\0') {
      gftp_set_username (hdata->ftpdata, ANON_LOGIN);
   }
   if (GFTP_GET_PASSWORD (hdata->ftpdata) == NULL || *GFTP_GET_PASSWORD (hdata->ftpdata) == '\0') {
      if (strcmp (GFTP_GET_USERNAME (hdata->ftpdata), ANON_LOGIN) == 0) {
         gftp_set_password (hdata->ftpdata, emailaddr);
      }
      else if (hdata->wdata->hdata == hdata) {
         hdata->wait = 1;
         MakeEditDialog (_("Enter Password"), _("Please enter your password for this site"), NULL, 0, 1,
         	_("Connect"), try_connect_again, hdata, _("  Cancel  "), dont_connect_again, hdata);

         while (hdata->wait) {
            fix_display ();
            sleep (0);
         }
         if (GFTP_GET_PASSWORD (hdata->ftpdata) == NULL || *GFTP_GET_PASSWORD (hdata->ftpdata) == '\0') return (0);
      }
      else {
         gftp_set_password (hdata->ftpdata, "");
      }
   }

   if (hdata->wdata->hdata == hdata) {
      hdata->wdata->local = 0;
      gtk_clist_freeze (GTK_CLIST (hdata->wdata->listbox));
      gtk_widget_set_sensitive (stop_btn, 1);
      hdata->stopable = 1;
      pthread_create (&hdata->tid, NULL, connect_thread, hdata);

      while (hdata->stopable) {
         fix_display ();
         sleep (0);
      }

      gtk_widget_set_sensitive (stop_btn, 0);
      gtk_clist_thaw (GTK_CLIST (hdata->wdata->listbox));
      pthread_join (hdata->tid, &ret);
   }
   else ret = connect_thread ((void *) hdata);
   success = (int) ret;
   hdata->tid = 0;

   if (hdata->wdata->hdata == hdata) {
      if (hdata->getdir && (success || hdata->wdata->hdata->files != NULL)) {
         success = ftp_list_files (hdata->wdata, 1);
      }
      else if (!success) {
         hdata->totalfiles = hdata->wdata->numselected = 0;
         hdata->wdata->local = -1;
         if (reconnect_diag) {
            hdata->wait = 1;
            reconnect_dialog (hdata);
         }
      }
      
      while (hdata->wait) {
         fix_display ();
         sleep (0);
      }
   }

   if (GFTP_CONNECTED (hdata->ftpdata)) success = 1;

   if (hdata->wdata->hdata == hdata) {
      update_ftp_info (hdata->wdata);
   }

   return (success);
}
/*****************************************************************************/
static void try_connect_again (GtkWidget *widget, struct dialog_data *data) {
   struct ftp_host_data *hdata;
   
   hdata = (struct ftp_host_data *) data->data;
   gftp_set_password (hdata->ftpdata, gtk_entry_get_text (GTK_ENTRY (data->edit)));
   data->data = NULL;
   hdata->wait = 0;
}
/*****************************************************************************/
static void dont_connect_again (GtkWidget *widget, struct dialog_data *data) {
   struct ftp_host_data *hdata;
   
   hdata = (struct ftp_host_data *) data->data;
   data->data = NULL;
   hdata->wait = 0;
}
/*****************************************************************************/
static void *connect_thread (void *data) {
   struct ftp_host_data *hdata;
   gftp_logging_func old_func;
   int ret;
   int sj;
   
   hdata = (struct ftp_host_data *) data;
   old_func = hdata->ftpdata->logging_function;
   hdata->ftpdata->logging_function = queue_log;

   hdata->connection_number = 0;
   sj = sigsetjmp (envir, 1);
   ret = 0;
   if (hdata->wdata->hdata == hdata) {
      signal (SIGINT, sig_quit);
      signal (SIGALRM, sig_quit);
   }
   if (sj != 0)  {
      ret = 0;
      gftp_disconnect (hdata->ftpdata);
      if (hdata->wdata->hdata->files == NULL) {
         free_file_list (hdata->files);
      }
   }
   while (sj != 1 && (retries == 0 || hdata->connection_number < retries)) {
      hdata->connection_number++;
      if (timeout > 0) alarm (timeout);
      ret = gftp_connect (hdata->ftpdata) == 0;
      alarm (0);

      if (hdata->wdata->hdata == hdata) {
         hdata->totalfiles = hdata->wdata->numselected = 0;
         hdata->wdata->local = 0;
      }
      if (ret) {
         if (hdata->getdir) {
            hdata->wdata->hdata->files = get_remote_files (hdata->wdata->hdata, NULL, &hdata->totalfiles, 1, NULL);
         }
         break;
      }
      else if (retries == 0 || hdata->connection_number < retries) {
         queue_log (gftp_logging_misc, NULL, _("Waiting %d seconds until trying to connect again\n"), sleep_time);
         alarm (sleep_time);
         pause ();
      }
   }
   
   hdata->ftpdata->logging_function = old_func;
   hdata->stopable = 0;
   hdata->wait = 0;
   return ((void *) ret);
}
/*****************************************************************************/
static void sig_quit (int signo) {
   signal (signo, sig_quit);
   siglongjmp (envir, signo == SIGINT ? 1 : 2);
}
/*****************************************************************************/
void stop_button (GtkWidget *widget, struct ftp_window_data *wdata) {
   if (window2.hdata->tid > 0) {
      pthread_kill (window2.hdata->tid, SIGINT);
   }
}
/*****************************************************************************/
void disconnect (struct ftp_window_data *wdata) {
   gftp_disconnect (wdata->hdata->ftpdata);
   gftp_set_username (wdata->hdata->ftpdata, ANON_LOGIN);
   gftp_set_password (wdata->hdata->ftpdata, emailaddr);
   delete_ftp_file_info (wdata);
   wdata->local = -1;
   update_ftp_info (wdata);
}
/*****************************************************************************/
int get_file_transfer_mode (char *filename) {
   struct pix_ext *tempext;
   int stlen, ascii;

   ascii = 0;
   stlen = strlen (filename);
   tempext = registered_exts;
   while (use_default_dl_types && tempext != NULL) {
      if (stlen >= tempext->stlen &&
      		strcmp (&filename[stlen-tempext->stlen], tempext->ext) == 0) {
         if (toupper (*tempext->ascii_binary == 'A')) ascii = 1;
         else if (toupper (*tempext->ascii_binary != 'B')) {
            tempext = NULL;
            break;
         }
         break;
      }
      tempext = tempext->next;
   }
   if (tempext == NULL && GFTP_GET_DATA_TYPE (window2.hdata->ftpdata) == GFTP_TYPE_ASCII) {
      ascii = 1;
   }
   return (ascii);
}
/*****************************************************************************/
void add_file_transfer (struct ftp_transfer_data *tdata) {
   pthread_mutex_lock (&transfer_mutex);
   file_transfers = g_list_append (file_transfers, tdata);
   pthread_mutex_unlock (&transfer_mutex);
}
/*****************************************************************************/
struct ftp_transfer_data *transfer_one_file (char *local_file, char *remote_file, int dl_or_up) {
   struct ftp_transfer_data *newtdata;
   struct ftp_file_data *tempfle;

   newtdata = new_tdata ();
   copy_hdata_struct (window2.hdata, newtdata->hdata);
   tempfle = g_malloc0 (sizeof (struct ftp_file_data));
   tempfle->remote_file = g_malloc (strlen (remote_file) + 1);
   strcpy (tempfle->remote_file, remote_file);
   tempfle->file = g_malloc (strlen (local_file) + 1);
   strcpy (tempfle->file, local_file);
   tempfle->size = 0;
   newtdata->hdata->files = g_list_append (newtdata->hdata->files, tempfle);
   newtdata->curfle = newtdata->hdata->files;
   newtdata->hdata->totalfiles = 1;
   newtdata->show = 1;
   if (dl_or_up) {
      /* We're downloading this file */
      newtdata->direction = 1;
      tempfle->ascii = get_file_transfer_mode (remote_file);
   }
   else {
      /* We're uploading this file */
      tempfle->ascii = get_file_transfer_mode (local_file);
   }
   return (newtdata);
}
/*****************************************************************************/
