/*
   Pathetic Writer
   Copyright (C) 1997-1999  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
 */

/* ---
   Module name:    window.c

   This module handles windows: creating, destroying, printing on windows.
--- */

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xatom.h>
#include <X11/Shell.h>
#include <X11/Xmu/Atoms.h>
#include <X11/Xmu/StdSel.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include "../xcommon/SiagMenuButton.h"
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Repeater.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Paned.h>
#include <X11/xpm.h>

#include "../common/common.h"
#include "../common/cmalloc.h"

#include "../common/bitmaps/rbm.xbm"
#include "../common/bitmaps/pw.xpm"

#include "../xcommon/xcommon.h"
#include "../xcommon/xfonts.h"
#include "../xcommon/embed.h"
#include "../xcommon/Canvas.h"

#include "../xcommon/Richtext.h"

#include "../xcommon/Combo.h"
#include "../xcommon/Frame.h"
#include "../xcommon/Tabbing.h"
#include "../xcommon/Rudegrid.h"
#include "../xcommon/DragAndDrop.h"
#include "../xcommon/plugin.h"
#include "../xcommon/icon.h"
#include "../xcommon/Tooltip.h"
#include "../xcommon/Handle.h"

#include "../pw/pw.h"
#include "xpw.h"

#include "../siod/siod.h"

#define MENUBAR (1)
#define TOOLBAR (2)
#define FORMATBAR (4)

#define APPNAME "Pw"

String fallback_resources[] = {
#include "../xcommon/xcommon-ad.h"
#include "../xcommon/dialogs-ad.h"
#include "../xcommon/filesel-ad.h"
#include "app-defaults.h"
	NULL
};

Widget topLevel, topbox, frame1, frame2, frame3;
Widget menubox, toolbox, formatbox, textbox, statusbox;
Widget form, gridpane, textframe, text1, label1, label2, label3;

/* The toolbar */
Widget tbOpen, tbSave, tbView, tbPrint, tbHelp;

Widget btnFont, mnuFont, btnSize, mnuSize, btnStyle, mnuStyle,
        btnColor, mnuColor;
Widget cmdBold, cmdItalic, cmdUline, cmdHLeft, cmdHCenter, cmdHRight;
Widget cmdVTop, cmdVBottom;
static Widget shortcuts, tooltip;

static int bars = 0;

XIM xim;
XIC xic;

typedef struct {
        Boolean plugin;
} AppData;

static AppData app_data;
static XtAppContext app_context;

#define XtNplugin "plugin"
#define XtCPlugin "Plugin"

static XtResource resources[] = {
        {
                XtNplugin,
                XtCPlugin,
                XtRBoolean,
                sizeof(Boolean),
                XtOffsetOf(AppData, plugin),
                XtRImmediate,
                (XtPointer)False
        }
};

static XrmOptionDescRec options[] = {
        {"-plugin", "*plugin", XrmoptionNoArg, "True"}
};

window *w_list;

int pr_line_flag;
int display_bops = 0;

Display *display;

Window root;
static GC cursor_gc, cell_gc, clear_gc;
static Cursor grid_cursor;
static unsigned long blockbg, noblockbg;

Atom wm_delete_window;	/* Atom sent to destroy a window */
extern Atom target_atom;	/* used for selection */

/* ---
*/
void activate_window(window *w)
{
	char b[256];
	int tab, s;
	buffer *buf;

	w_list = w;
	s = w->sht;

	strcpy(b, "Pathetic Writer: ");
	strncat(b, w->buf->name, 200);
	tab = TabbingTextToPos(w->ui->tab, w->buf->sht[s].name);
	if (tab < 0) {
		w_list->sht = s = tab = 0;
	}
	XtVaSetValues(w->ui->tab, XtNtabbingSelected, tab, (char *)0);
	XtVaSetValues(topLevel, XtNtitle, b, (char *)0);
	XtVaSetValues(w->ui->grid,
		XtNrichtextData, w,
		(char *)0);
	XtSetKeyboardFocus(topLevel, w->ui->grid);
	/* plugins */
	buf = b_list;
	do {
		for (s = 0; s < buf->nsht; s++) {
			int i;
			for (i = 0; i < buf->sht[s].nplugin; i++) {
				if ((w_list->buf != buf || w_list->sht != s)
					&& buf->sht[s].plugin[i].displayed) {
					plugin_hide(buf->sht[s].plugin[i].ph);
					buf->sht[s].plugin[i].displayed = 0;
					pr_scr_flag = 1;
				}
			}
		}
		buf = buf->next;
	} while (buf != b_list);
}

/* ---
*/
void draw_input(Display * display, char *text)
{
	char b[256];
	char b3[256];
	int s = w_list->sht;
	sheet *st = w_list->buf->sht;
	sprintf(b, "[%d,%d - %d,%d]",
		st[s].blku.row, st[s].blku.col,
		st[s].blkl.row, st[s].blkl.col);
	sprintf(b3, "PW %d/%d",
		st[s].point_pos.row,
		row2page(w_list->buf, s, st[s].point_pos.row));
	label_set(label1, b);
	label_set(label3, b3);
}

/* ---
*/
void draw_status(Display * display, char *text)
{
	char b[200];
	strncpy(b, text, 199);
	b[199] = '\0';
	chomp(b);
	label_set(label2, b);
	XFlush(display);
}

/* ---
   void llpr(char *p)
   Prints the string p on the bottom line of the screen.  If p is empty and
   the last string printed was also empty, the string isn't printed.
*/

void llpr(char *p)
{
	static int isclear = FALSE;

	if (isclear && p[0] == '\0')
		return;
	isclear = (p[0] == '\0');

	draw_status(display, p);
}

/* assuming lines are 13 high */
static void draw_vbar(Display * display, window *w)
{
	float pos, size;
	float visible_cells = w->buf->sht[w->sht].alloc_lines+50;
	float wh = height_get(w->ui->grid)/13;

	pos = (get_top(w).row-1)/visible_cells;
	size = wh/visible_cells;

	XawScrollbarSetThumb(w->ui->vscroll, pos, size);
}

static void draw_hbar(Display * display, window *w)
{
	double float_value;

	float_value = w->buf->sht[w->sht].top.col;
	float_value /= BUFFER_COLS;

	XawScrollbarSetThumb(w->ui->hscroll, float_value, 1.0);
}

static void draw_scrollbars(Display * display, window *w)
{
	draw_vbar(display, w);
	draw_hbar(display, w);
}

static int line_start(buffer *buf, int s, int row)
{
	int x = 0;
	int w = buf->paper_width-buf->left_margin-buf->right_margin;
	int hadj;

	hadj = ret_hadj(buf, s, row);

	if (hadj == HADJ_CENTER)
		x = (w-line_width(buf, s, row, line_length(buf, s, row))) / 2;
	else if (hadj == HADJ_RIGHT)
		x = w-line_width(buf, s, row, line_length(buf, s, row));

	return x+buf->left_margin;
}

/* ---
This function is a dog: it copies from the grid which may well be
   partially obscured during the copy. As a result it doesn't work.
   It should use the richtext_pixmap function instead.
*/

Pixmap draw_snapshot(void)
{
	int x, y;
	unsigned int width, height, border_width, depth;
	Window cell_win, root;
	Pixmap bitmap;
	GC gc;
	unsigned long valuemask = 0;
	XGCValues values;

	cell_win = XtWindow(w_list->ui->grid);
	XGetGeometry(display, cell_win, &root,
		&x, &y, &width, &height, &border_width, &depth);
	bitmap = XCreatePixmap(display, cell_win, width, height, 1);
	gc = XCreateGC(display, bitmap, valuemask, &values);
	XCopyPlane(display, cell_win, bitmap, gc, 0, 0, width, height,
			0, 0, 1);
	XFreeGC(display, gc);

	return bitmap;
}

static int trow_height(XtPointer p, int r)
{
	window *w = (window *)p;
	buffer *b = w->buf;
	int s = w->sht;
	if (r > b->sht[s].used_lines) return 20;
	return b->sht[s].text[r].height;
}

static int trow_hadj(XtPointer p, int r)
{
	window *w = (window *)p;
	return ret_hadj(w->buf, w->sht, r);
}

static int trow_style(XtPointer p, int r)
{
	window *w = (window *)p;
	return ret_style(w->buf, w->sht, r);
}

static int trow_bop(XtPointer p, int r)
{
	window *w = (window *)p;
	buffer *b = w->buf;
	int s = w->sht;
	int n;

	if (display_bops) n = ret_bop(b, s, r);
	else n = 0;
	return n;
}

static rich_char *trow_text(XtPointer p, int r)
{
	window *w = (window *)p;
	buffer *b = w->buf;
	int s = w->sht;
	if (r > b->sht[s].used_lines) return NULL;
	return b->sht[s].text[r].p;
}

/* ---
This will be the Redisplay function in the RichtextWidget
*/

void draw_buffer(Display *display, window *w)
{
	int s = w->sht;
	sheet *st = w->buf->sht;
	XtVaSetValues(w->ui->grid,
		XtNrichtextData, w,
		XtNrichtextTopRow, st[s].top.row,
		XtNrichtextTopCol, st[s].top.col,
		XtNrichtextSelectTopRow, st[s].blku.row,
		XtNrichtextSelectTopCol, st[s].blku.col,
		XtNrichtextSelectBottomRow, st[s].blkl.row,
		XtNrichtextSelectBottomCol, st[s].blkl.col,
		XtNrichtextPointRow, st[s].point_pos.row,
		XtNrichtextPointCol, st[s].point_pos.col,
		XtNrichtextPaperWidth, w->buf->paper_width,
		XtNrichtextLeftMargin, w->buf->left_margin,
		XtNrichtextRightMargin, w->buf->right_margin,
		XtNrichtextTabDistance, w->buf->tab_distance,
		XtNrichtextRedisplay, True,
		(char *)0);
}

static GC get_gc(Display * display, Window wind,
       unsigned long fg, unsigned long bg, Font font)
{
	unsigned long valuemask = 0/*GCFunction | GCForeground | GCBackground |
		GCLineWidth | GCFont*/;
	XGCValues values;
	GC gc = XCreateGC(display, wind, valuemask, &values);

	XSetForeground(display, gc, fg);
	XSetBackground(display, gc, bg);
	if (font != -1)
		XSetFont(display, gc, font);
	return gc;
}

/* ---
*/
window *find_window_by_widget(Widget wdg)
{
	window *w = w_list;
	do {
		if (w->ui->viewport == wdg || w->ui->grid == wdg ||
			w->ui->vscroll == wdg || w->ui->hscroll == wdg ||
			w->ui->tab == wdg ||
			w->ui->tabl == wdg || w->ui->tabr)
			return w;
		w = w->next;
	} while (w != w_list);
	return NULL;
}

/* ---
*/
void free_window(window *w)
{
	window *pw;

	for (pw = w_list; pw->next != w && pw->next != pw; pw = pw->next);
	pw->next = w->next;

	if (w_list == w) w_list = w_list->next;
	if (w_list == w) w_list = NULL;
	XtDestroyWidget(w->ui->viewport);

	if (b_list) {
		buffer *b = w->buf;
		int s = w->sht;
		int i;
		for (i = 0; i < b->sht[s].nplugin; i++) {
			if (b->sht[s].plugin[i].displayed) {
				plugin_hide(b->sht[s].plugin[i].ph);
				b->sht[s].plugin[i].displayed = 0;
			}
		}
	}
	cfree(w->ui);
	cfree(w);
}

static void expose_ruler(Widget w, XtPointer p)
{
	window *win = (window *)p;
	buffer *buf = win->buf;
	int i, o;
	int h = height_get(w);
	Display *dpy = XtDisplay(w);
	Screen *s = XtScreen(w);
	Window wi = XtWindow(w);
	unsigned long black = BlackPixelOfScreen(s);
	unsigned long white = WhitePixelOfScreen(s);

	o = -1;
	for (i = buf->left_margin;
	     i < buf->paper_width-buf->right_margin;
	     i += buf->tab_distance) {
		XSetForeground(dpy, cell_gc, white);
		XDrawLine(dpy, wi, cell_gc, i-2+o, 0, i-2+o, h-6);
		XDrawLine(dpy, wi, cell_gc, i-2+o, h-6, i+o, h-4);
		XSetForeground(dpy, cell_gc, black);
		XDrawLine(dpy, wi, cell_gc, i+2+o, 0, i+2+o, h-6);
		XDrawLine(dpy, wi, cell_gc, i+o, h-4, i+2+o, h-6);
	}
}

/* ---
used for the XtNtablePluginCoords resource
*/

static void tplugin_coordinates(Widget w, XtPointer p, int *x, int *y)
{
	window *win = (window *)p;
        buffer *b = win->buf;
        int n, ph;
        *x = *y = 0;
        ph = plugin_find_by_widget(w);
        if (ph == -1) return;
        n = buffer_plugin2index(b, win->sht, ph);
        if (n == -1) return;
        buffer_global_coords(b, win->sht,
			b->sht[win->sht].plugin[n].row,
			b->sht[win->sht].plugin[n].col, x, y);
}

static void select_tab(Widget w, int pos)
{
	buffer *b;
	int s;
	char *name = TabbingPosToText(w, pos);
	if (name == NULL) return;

	w_list = find_window_by_widget(w);
	b = find_sheet_by_name(name, w_list->buf, &s);
	if (b == NULL) return;

	hide_cur(w_list);
	XtVaSetValues(w, XtNtabbingSelected, pos, (char *)0);
	XtVaGetValues(w, XtNtabbingSelected, &pos, (char *)0);

	w_list->buf = b;
	w_list->sht = s;
	pr_scr_flag = TRUE;
	activate_window(w_list);
	show_cur(w_list);
}

/* ---
*/
int remove_window(window *w)
{
	
	if (w == w->next) return FALSE;
	free_window(w);
	return TRUE;
}

/* ---
*/
int split_window(window *w)
{
	window *w2 = new_window(w->buf, w);
	int s = w->sht;

	if (w2 == NULL) return FALSE;
	w2->buf->sht[s].point_pos = w->buf->sht[s].point_pos;
	w2->buf->sht[s].top = w->buf->sht[s].top;
	return TRUE;
}

/* stuff snarfed from main.c */

static void set_point_rc(int row, int col)
{
        position p;

        p.row = row;
        p.col = col;
        set_point(w_list, p);
}

static void set_mark_rc(int row, int col)
{
        position p;

        p.row = row;
        p.col = col;
        set_point(w_list, p);
}

static void set_block(int row1, int col1, int row2, int col2)
{
	int s = w_list->sht;
	sheet *st = w_list->buf->sht;
        int ur = st[s].blku.row, uc = st[s].blku.col;
        int lr = st[s].blkl.row, lc = st[s].blkl.col;

        st[s].blku.row = row1;
        st[s].blkl.row = row2;
        st[s].blku.col = col1;
        st[s].blkl.col = col2;
	w_list->bsht = s;

        /* Redraw if any change */
        if (ur != st[s].blku.row || uc != st[s].blku.col ||
                lr != st[s].blkl.row || lc != st[s].blkl.col) {
                pr_scr_flag = TRUE;
        }

        /* Become selection owner */
        /* this function should be integrated with the one in cmds.c */
        if (XtOwnSelection(w_list->ui->grid, XA_PRIMARY,
		CurrentTime, convert_proc,
                lose_ownership_proc, transfer_done_proc) == False) {
                XtWarning("PW: failed to become selection owner\n");
                st[s].blku.row = st[s].blku.col = -1;
                st[s].blkl.row = st[s].blkl.col = -1;
        }
}

/* ---
*/
void GridButtonAction(Widget w, XEvent *event, String *params, Cardinal *n)
{
        int col, row;
        int x, y;
	int s;
	sheet *st;

        x = event->xbutton.x;
        y = event->xbutton.y;
        hide_cur(w_list);
        activate_window(find_window_by_widget(w));
	s = w_list->sht;
	st = w_list->buf->sht;
        get_coords_cell(w_list, st[s].top.row, st[s].top.col,
                        &row, &col, x, y);
        if (*n < 1 || !strcmp(params[0], "point")) {
                set_point_rc(row, col);
        } else if (!strcmp(params[0], "mark")) {
                set_mark_rc(row, col);
        } else if (!strcmp(params[0], "block")) {
                set_block(row, col,
                        get_mark(w_list).row, get_mark(w_list).col);
                pr_scr_flag = TRUE;
        } else if (!strcmp(params[0], "paste")) {
                set_point_rc(row, col);
                XtGetSelectionValue(w, XA_PRIMARY, target_atom,
                        requestor_callback, event, event->xbutton.time);
                pr_scr_flag = TRUE;
        } else if (!strcmp(params[0], "select")) {
                int r1 = st[s].blku.row, c1 = st[s].blku.col;
                int r2 = st[s].blkl.row, c2 = st[s].blkl.col;
                /* is the selection set to something already? */
                if (r1 == -1 || c1 == -1 || r2 == -1 || c2 == -1) {
			int r = get_point(w_list).row;
			int c = get_point(w_list).col;
			if (row < r || (row == r && col < c)) {
				set_block(row, col, r, c);
			} else {
				set_block(r, c, row, col);
			}
                } else {
			if (row < r1 || (row == r1 && col <= c1)) {
				/* grow at the start */
				r1 = row, c1 = col;
			} else if (row > r2 || (row == r2 && col >= c2)) {
				/* grow at the end */
				r2 = row, c2 = col;
			} else if (ABS(r1-row) < ABS(r2-row)) {
				/* shrink at the start */
				r1 = row, c1 = col;
			} else {
				/* shrink at the end */
				r2 = row, c2 = col;
			}
                        set_block(r1, c1, r2, c2);
                }
        } else if (!strcmp(params[0], "adjust")) {
		int r = get_point(w_list).row, c = get_point(w_list).col;
		if (row < r || (row == r && col < c)) {
			set_block(row, col, r, c);
		} else {
			set_block(r, c, row, col);
		}
        }
        show_cur(w_list);
}

/* ---
*/
void SelfInsertCharAction(Widget w, XEvent *event, String *params, Cardinal *n)
{
        int count, bufsiz = 10;
        unsigned char buf[12];
        KeySym keysym;

        hide_cur(w_list);
	count = RichtextLookupString(w, event, buf, bufsiz, &keysym);
        if (keysym >= ' ' && count == 1) {
		int s = w_list->sht;
		sheet *st = w_list->buf->sht;
                buf[count] = '\0';
                ins_text(w_list->buf, s,
			st[s].point_pos, buf, w_list->current_fmt);
                rebreak_line(w_list->buf, s,
			st[s].point_pos.row);
                st[s].point_pos.col += count;
                /* point position may now be hosed */
                if (st[s].point_pos.col >
                                line_length(w_list->buf, s,
					st[s].point_pos.row)) {
                        st[s].point_pos.col -=
                                line_length(w_list->buf, s,
					st[s].point_pos.row);
                        st[s].point_pos.col -= 1; /* we deleted a space too */
                        st[s].point_pos.row++;
                }
                /* and if that didn't help... */
                if (st[s].point_pos.col >
                                line_length(w_list->buf, s,
					st[s].point_pos.row)) {
                        st[s].point_pos.col =
                                line_length(w_list->buf, s,
					st[s].point_pos.row);
                }
                pr_line_flag = TRUE;
        }
        show_cur(w_list);
}

static LISP ltooltip_mode(LISP newmode)
{
	int mode = get_c_long(newmode);
	XtVaSetValues(tooltip,
		XtNtooltipMode, mode,
		XtNtooltipLabel, label2,
		(char *)0);
	return NIL;
}

static void siaghelp_action(Widget w, XEvent *event,
        String *params, Cardinal *num_params)
{
        char b[256];

        sprintf(b, "file://localhost%s/pw/%s", docdir, params[0]);
        if (!fork()) {
                execlp(siaghelp, "Siaghelp", b, (char *)0);
                exit(0);
        }
}

static void
execute_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
        char *b = (char *) client_data;

        execute(b);
}

static void tabs_left(Widget w, XtPointer client_data, XtPointer call_data)
{
	Widget tw = (Widget)client_data;
	int n;

	XtVaGetValues(tw, XtNtabbingTop, &n, (char *)0);
	XtVaSetValues(tw, XtNtabbingTop, n-1, (char *)0);
}

static void tabs_right(Widget w, XtPointer client_data, XtPointer call_data)
{
	Widget tw = (Widget)client_data;
	int n;

	XtVaGetValues(tw, XtNtabbingTop, &n, (char *)0);
	XtVaSetValues(tw, XtNtabbingTop, n+1, (char *)0);
}

/* ---
*/
window *new_window(buffer *b, window *prev)
{
	Dimension totalwidth, formheight, w1, h1;
	window *w;
	int s;
	sheet *st;

	w = (window *)cmalloc(sizeof(window));

	if (w == NULL) return NULL;	/* can't really happen */

	w->ui = (pw_ui *)cmalloc(sizeof(pw_ui));
	if (w->ui == NULL) {
		cfree(w);
		return NULL;
	}

	w->buf = b;
	w->sht = 0;
	w->current_fmt = 0;

	if (prev == NULL) prev = w;
	else w->next = prev->next;
	prev->next = w;

	/* Figure out how big the new form should be.			*/
	/* The total width must be the width of the gridpane.		*/
	XtVaGetValues(gridpane,
		XtNwidth, &totalwidth, (char *)0);

	/* The form height is whatever we get, but if it is too small
	we cannot create the new window.				*/
	formheight = 100;
	w->ui->viewport = XtVaCreateManagedWidget("viewport",
		rudegridWidgetClass, gridpane,
		XtNwidth, totalwidth,
		(char *)0);
	XtVaGetValues(w->ui->viewport,
		XtNheight, &formheight, (char *)0);

	w1 = totalwidth-20;
	h1 = 20;

	w->ui->rulerframe = XtVaCreateManagedWidget("rulerframe",
		frameWidgetClass, w->ui->viewport,
		(char *)0);
	w->ui->ruler = XtVaCreateManagedWidget("ruler",
		canvasWidgetClass, w->ui->rulerframe,
		XtNcanvasExpose, expose_ruler,
		XtNcanvasData, (XtPointer)w,
		(char *)0);

	h1 = formheight-height_get(w->ui->ruler)-20;
	s = w->sht;
	st = w->buf->sht;
	w->ui->grid = XtVaCreateManagedWidget("grid",
		richtextWidgetClass, w->ui->viewport,
		/* set nonchanging resources here */
		XtNrichtextData, w,
		XtNrichtextRowHeight, trow_height,
		XtNrichtextAdjHoriz, trow_hadj,
		XtNrichtextStyle, trow_style,
		XtNrichtextBop, trow_bop,
		XtNrichtextText, trow_text,
		/* as well as changing ones */
		XtNrichtextTopRow, st[s].top.row,
		XtNrichtextTopCol, st[s].top.col,
		XtNrichtextSelectTopRow, st[s].blku.row,
		XtNrichtextSelectTopCol, st[s].blku.col,
		XtNrichtextSelectBottomRow, st[s].blkl.row,
		XtNrichtextSelectBottomCol, st[s].blkl.col,
		XtNrichtextPointRow, st[s].point_pos.row,
		XtNrichtextPointCol, st[s].point_pos.col,
		XtNrichtextPluginCoords, tplugin_coordinates,
		(char *)0);
	XDefineCursor(display, XtWindow(w->ui->grid), grid_cursor);
	h1 = height_get(w->ui->ruler)+height_get(w->ui->grid);
	w->ui->vscroll = XtVaCreateManagedWidget("vscroll",
		scrollbarWidgetClass, w->ui->viewport,
		(char *)0);
	w1 = width_get(w->ui->grid);
	w->ui->hscroll = XtVaCreateManagedWidget("hscroll",
		scrollbarWidgetClass, w->ui->viewport,
		(char *)0);
	w->ui->tab = XtVaCreateManagedWidget("tab",
		tabbingWidgetClass, w->ui->viewport,
		XtNtabbingSelect, select_tab,
		(char *)0);
	b = b_list;
	XtVaSetValues(w->ui->tab,
		XtNtabbingSelected, TabbingTextToPos(w->ui->tab,
						w->buf->sht[0].name),
		(char *)0);
	w->ui->tabl = XtVaCreateManagedWidget("tabl",
		repeaterWidgetClass, w->ui->viewport,
		(char *)0);
	XtAddCallback(w->ui->tabl, XtNcallback,
			tabs_left, (XtPointer)w->ui->tab);
	w->ui->tabr = XtVaCreateManagedWidget("tabr",
		repeaterWidgetClass, w->ui->viewport,
		(char *)0);
	XtAddCallback(w->ui->tabr, XtNcallback,
			tabs_right, (XtPointer)w->ui->tab);

	XtAddCallback(w->ui->vscroll, XtNjumpProc, vscroll_jump, NULL);
	XtAddCallback(w->ui->vscroll, XtNscrollProc, vscroll_scroll, NULL);

	XtAddCallback(w->ui->hscroll, XtNjumpProc, hscroll_jump, NULL);
	XtAddCallback(w->ui->hscroll, XtNscrollProc, hscroll_scroll, NULL);

	XtVaGetValues(w->ui->viewport,
		XtNheight, &formheight, (char *)0);

	return w;
}

/* ---
*/
void vscroll_jump(Widget w, XtPointer client_data, XtPointer call_data)
{
        float top;
        int gridtop;
	int s;
	sheet *st;

        hide_cur(w_list);
        activate_window(find_window_by_widget(w));
	s = w_list->sht;
	st = w_list->buf->sht;
        XtVaGetValues(w, XtNtopOfThumb, &top, (char *)0);
        gridtop = top*(w_list->buf->sht[w_list->sht].alloc_lines+50)+1;
        if (gridtop < 1) gridtop = 1;
	if (gridtop == get_top(w_list).row) return;
        st[s].point_pos.row = st[s].top.row = gridtop;
        pr_scr_flag = TRUE;
        show_cur(w_list);
}

/* ---
*/
void vscroll_scroll(Widget w, XtPointer client_data, XtPointer call_data)
{
        int i = (long) call_data;
        Dimension length;

        activate_window(find_window_by_widget(w));
        XtVaGetValues(w, XtNlength, &length, (char *)0);
        if (i < 0) {
                if ((length / -i) > 15) {
                        execute("(scroll-char-up)");
                } else {
                        execute("(scroll-up)");
                }
        } else {
                if ((length / i) > 15) {
                        execute("(scroll-char-down)");
                } else {
                        execute("(scroll-down)");
                }
        }
}

/* ---
*/
void hscroll_jump(Widget w, XtPointer client_data, XtPointer call_data)
{
        float top;
        int gridtop;
	int s;
	sheet *st;

        hide_cur(w_list);
        activate_window(find_window_by_widget(w));
	s = w_list->sht;
	st = w_list->buf->sht;
        XtVaGetValues(w, XtNtopOfThumb, &top, (char *)0);
        gridtop = top*BUFFER_COLS;
        if (gridtop < 1) gridtop = 1;
        st[s].point_pos.col = st[s].top.col = gridtop;
        pr_scr_flag = TRUE;
        show_cur(w_list);
}

/* ---
*/
void hscroll_scroll(Widget w, XtPointer client_data, XtPointer call_data)
{
        int i = (long) call_data;
        Dimension length;

        activate_window(find_window_by_widget(w));
        XtVaGetValues(w, XtNlength, &length, (char *)0);
        if (i < 0) {
                if ((length / -i) > 15) {
                        execute("(scroll-char-left)");
                } else {
                        execute("(scroll-left)");
                }
        } else {
                if ((length / i) > 15) {
                        execute("(scroll-char-right)");
                } else {
                        execute("(scroll-right)");
                }
        }
}

static struct {
        char *label;
        Widget button, menu;
} *menubar;

static int menucount = 0;

static LISP add_menu(LISP lisp_label)
{
	char *label;
        char button_name[80];

	label = get_c_string(lisp_label);

        sprintf(button_name, "btn%s", label);
	menubar = crealloc(menubar, (menucount+1)*(sizeof *menubar));
        menubar[menucount].label = cstrdup(label);
        menubar[menucount].button = XtVaCreateManagedWidget(button_name,
                siagMenuButtonWidgetClass, menubox,
		(char *)0);
        menubar[menucount].menu = XtVaCreatePopupShell("menu",
                simpleMenuWidgetClass, menubar[menucount].button, (char *)0);
	label_set(menubar[menucount].button, label);
        menucount++;

	return NIL;
}

static void remake_ylayout(void)
{
	char b[100];
	sprintf(b, "%s %s %s 30 100%% 30",
		(bars&MENUBAR) ? "30" : "0",
		(bars&TOOLBAR) ? "30" : "0",
		(bars&FORMATBAR) ? "30" : "0");
	XtVaSetValues(topbox,
		XtNyLayout, b,
		(char *)0);
}

static void attach_bar(Widget w)
{
	if (w == frame1) bars |= MENUBAR;
	if (w == frame2) bars |= TOOLBAR;
	if (w == frame3) bars |= FORMATBAR;
	remake_ylayout();
}

static void detach_bar(Widget w)
{
	if (w == frame1) bars &= ~MENUBAR;
	if (w == frame2) bars &= ~TOOLBAR;
	if (w == frame3) bars &= ~FORMATBAR;
	remake_ylayout();
}

static Widget make_handle(Widget pw, Widget victim)
{
	Pixmap pm;
	Pixel color;
	XtVaGetValues(pw,
		      XtNbackground, &color,
		      (char *)0);
	pm = load_pixmap(pw, "handlebg.xpm");
	return XtVaCreateManagedWidget("handle",
		handleWidgetClass, pw,
		XtNforeground, color,
		XtNbackgroundPixmap, pm,
		XtNshadowWidth, 1,
		XtNhandleDetach, detach_bar,
		XtNhandleAttach, attach_bar,
		XtNhandleVictim, victim,
		(char *)0);
}

static void init_menu()
{
	frame1 = XtVaCreateManagedWidget("menubar",
		frameWidgetClass, topbox,
		(char *)0);
	bars |= MENUBAR;
	menubox = XtVaCreateManagedWidget("menubox",
		boxWidgetClass, frame1,
		(char *)0);
	make_handle(menubox, frame1);
}

static Widget find_menu_by_name(char *label)
{
        int i;

	if (!cstrcasecmp("Shortcuts", label))
		return shortcuts;

        for (i = 0; i < menucount; i++) {
                if (!cstrcasecmp(menubar[i].label, label))
                        return menubar[i].menu;
        }
        return NULL;
}

static LISP add_menu_entry(LISP menu, LISP label, LISP function)
{
        Widget entry;
        Widget menuw = find_menu_by_name(get_c_string(menu));
        if (!menuw) {
#if 0
                fprintf(stderr, "No menu! Bailing out.\n");
#endif
                return NIL;
        }

        if (!strcmp(get_c_string(label), "-"))  /* line pane */
                entry = XtCreateManagedWidget("-",
                                smeLineObjectClass,
                                menuw,
                                NULL, 0);
        else {
                entry = XtVaCreateManagedWidget(get_c_string(function),
                                smeBSBObjectClass, menuw,
                                (char *)0, 0);
                XtAddCallback(entry,
                        XtNcallback, execute_callback,
			cstrdup(get_c_string(function)));
		label_set(entry, get_c_string(label));
        }
        return NIL;
}

static struct {
	char *label, *sublabel;
	Widget entry, menu;
} *submenus;

static int submenucount = 0;

static Widget find_submenu_by_name(char *label, char *sublabel)
{
        int i;

        for (i = 0; i < submenucount; i++) {
                if (!cstrcasecmp(submenus[i].label, label)
		    && !cstrcasecmp(submenus[i].sublabel, sublabel))
                        return submenus[i].menu;
        }
        return None;
}

static Widget find_submenu_by_entry(Widget entry)
{
        int i;

        for (i = 0; i < submenucount; i++) {
                if (entry == submenus[i].entry)
                        return submenus[i].menu;
        }
        return None;
}

static LISP add_submenu(LISP label, LISP sublabel)
{
        Widget menuw;
        static Pixmap rbm = None;

        menuw = find_menu_by_name(get_c_string(label));

        if (!menuw) {
#if 0
                fprintf(stderr, "add_submenu: No menu! Bailing out.\n");
#endif
                return NIL;
        }
        if (rbm == None) {
                rbm = XCreatePixmapFromBitmapData(XtDisplay(menuw),
                        XDefaultRootWindow(XtDisplay(menuw)),
                        rbm_bits, rbm_width, rbm_height, 1, 0, 1);
        }
        submenus = crealloc(submenus, (submenucount+1)*(sizeof *submenus));
        submenus[submenucount].label = cstrdup(get_c_string(label));
        submenus[submenucount].sublabel = cstrdup(get_c_string(sublabel));
        submenus[submenucount].entry =
                XtVaCreateManagedWidget(get_c_string(sublabel),
                        smeBSBObjectClass, menuw,
                        XtNlabel, translate(get_c_string(sublabel)),
                        XtNrightBitmap, rbm,
                        (char *)0);
        submenus[submenucount].menu = XtVaCreatePopupShell("submenu",
                simpleMenuWidgetClass, menuw, (char *)0);
        submenucount++;
        return NIL;
}

static LISP add_submenu_entry(LISP menu, LISP submenu,
                                LISP label, LISP function)
{
        Widget entry, menuw;

        menuw = find_submenu_by_name(get_c_string(menu),
                                get_c_string(submenu));
        if (!menuw) {
#if 0
                fprintf(stderr, "add_submenu_entry: No menu! Bailing out.\n");
#endif
                return NIL;
        }

        if (!strcmp(get_c_string(label), "-")) {        /* line pane */
                entry = XtCreateManagedWidget("-",
                        smeLineObjectClass,
                        menuw,
                        NULL, 0);
        } else {
                entry = XtVaCreateManagedWidget(get_c_string(function),
                        smeBSBObjectClass, menuw,
                        XtNlabel, translate(get_c_string(label)),
                        (char *)NULL);
                XtAddCallback(entry,
                        XtNcallback, execute_callback,
                        cstrdup(get_c_string(function)));
        }
        return NIL;
}


static Widget make_toggle(char *cmd, Widget pw, char *pm, char *t)
{
        Widget w;
        Pixmap pm_return;
        Pixel color;

        XtVaGetValues(pw, XtNbackground, &color, (char *)0);

        w = XtVaCreateManagedWidget("toolbar_toggle",
                toggleWidgetClass, pw, (char *)NULL);
	pm_return = load_pixmap(pw, pm);

        XtVaSetValues(w,
                XtNbitmap, pm_return,
		XtNforeground, color,
                (char *)0);
        XtAddCallback(w,
                XtNcallback, execute_callback, (XtPointer)cmd);
	TooltipAdd(tooltip, w, translate(t));
        return w;
}

static Widget make_command(char *cmd, Widget pw, char *pm, char *t)
{
        Widget w;
        Pixmap pm_return;
        Pixel color;

        XtVaGetValues(pw, XtNbackground, &color, (char *)0);

        w = XtVaCreateManagedWidget("toolbar_command",
                commandWidgetClass, pw,
		XtNforeground, color,
		(char *)NULL);
	pm_return = load_pixmap(pw, pm);

        XtVaSetValues(w,
                XtNbitmap, pm_return,
                (char *)0);

        XtAddCallback(w,
                XtNcallback, execute_callback, (XtPointer)cmd);
	TooltipAdd(tooltip, w, translate(t));
        return w;
}

static void make_vsep(Widget pw)
{
	XtCreateManagedWidget("vsep", labelWidgetClass, pw, NULL, 0);
}

/* The toolbar */
static void init_toolbar()
{
	frame2 = XtVaCreateManagedWidget("toolbar",
		frameWidgetClass, topbox,
		(char *)0);
	bars |= TOOLBAR;
	toolbox = XtVaCreateManagedWidget("toolbox",
		boxWidgetClass, frame2,
		(char *)0);
	make_handle(toolbox, frame2);
	frame3 = XtVaCreateManagedWidget("formatbar",
		frameWidgetClass, topbox,
		(char *)0);
	bars |= FORMATBAR;
	formatbox = XtVaCreateManagedWidget("formatbox",
		boxWidgetClass, frame3,
		(char *)0);
	make_handle(formatbox, frame3);

	make_command("(new-buffer)", toolbox, "new.xpm",
		     "Start another instance of Pathetic Writer");
        make_command("(load-buffer)", toolbox, "fld_open.xpm",
		     "Open a Pathetic Writer document");
        make_command("(save-buffer-as)", toolbox, "save.xpm",
		     "Save the contents of the current buffer");
        make_command("(preview)", toolbox, "preview.xpm",
		     "Preview the contents of the current buffer");
        make_command("(print)", toolbox, "printer.xpm",
		     "Print the contents of the current buffer");
	make_vsep(toolbox);
	make_command("(cut-block-to-string)", toolbox, "cut.xpm",
		     "Cut");
	make_command("(copy-block-to-string)", toolbox, "copy.xpm",
		     "Copy");
	make_command("(paste-block-from-string)", toolbox, "paste.xpm",
		     "Paste");
	make_command("(spell-buffer)", toolbox, "spell.xpm",
		     "Spelling checker");
	make_vsep(toolbox);
        make_command("(help-contents)", toolbox, "info.xpm",
		     "Display the PW online documentation");
        make_command("(help-copyright)", toolbox, "copyright.xpm",
		     "Display the Gnu general public license");
}

static void init_toggle()
{
        cmdBold = make_toggle("(toggle-format \"bold\")",
                formatbox, "bold.xpm", "Bold text");
        cmdItalic = make_toggle("(toggle-format \"italic\")",
                formatbox, "italic.xpm", "Italic text");
        cmdUline = make_toggle("(toggle-format \"uline\")",
                formatbox, "uchar.xpm", "Underlined text");
	make_vsep(toolbox);
        cmdHLeft = make_toggle("(change-hadjust HADJ_LEFT)",
                toolbox, "hleft.xpm", "Left adjusted text");
        cmdHCenter = make_toggle("(change-hadjust HADJ_CENTER)",
                toolbox, "hcenter.xpm", "Centered text");
        cmdHRight = make_toggle("(change-hadjust HADJ_RIGHT)",
                toolbox, "hright.xpm", "Right adjusted text");
	make_vsep(formatbox);
        cmdVTop = make_toggle("(toggle-vadj VADJ_TOP)",
                formatbox, "vtop.xpm", "Superscript");
        cmdVBottom = make_toggle("(toggle-vadj VADJ_BOTTOM)",
                formatbox, "vbottom.xpm", "Subscript");
}

static char *combo_sizes[] = {
	"8", "9", "10", "11", "12", "14", "16", "18",
	"20", "22", "24", "26", "28", "36", "48", "72"
};

static char *combo_styles[] = {
	"Default", "Header 1", "Header 2", "Header 3",
	"Header 4", "Header 5", "Header 6", "Address",
	"Title", "Abstract", "Preformatted", "User 1",
	"User 2", "User 3", "User 4", "User 5"
};
static char **combo_fonts, **combo_colors;
int ncombo_fonts, ncombo_colors;

static void cb_font(char *p)
{
	va_execute("(new-format \"family\" \"%s\")", p);
	activate_window(w_list);
}

static void cb_size(char *p)
{
	va_execute("(new-format \"size\" (* 10 %s))", p);
	activate_window(w_list);
}

static void cb_style(char *p)
{
	va_execute("(change-style \"%s\")", p);
	activate_window(w_list);
}

static void cb_color(char *p)
{
	va_execute("(new-format \"fg\" \"%s\")", p);
	activate_window(w_list);
}

/* ---
*/
void font_menu(Widget w, char *label, char *cmd)
{
        Widget entry = XtVaCreateManagedWidget(cmd,
                        smeBSBObjectClass, w,
                        (char *)0);
        XtAddCallback(entry,
                XtNcallback, execute_callback, cmd);
	label_set(entry, label);
}

/* ---
*/
void setup_buttons(void)
{
	Widget formatframe;

	if (topbox == None) return;

	formatframe = XtVaCreateManagedWidget("formatframe",
		frameWidgetClass, formatbox,
		(char *)0);
	combo_fonts = font_list(&ncombo_fonts);
        btnFont = XtVaCreateManagedWidget("toolbar_command",
                comboWidgetClass, formatframe,
		XtNlabel, "Font",
		XtNcomboList, cb_font,
		XtNcomboText, cb_font,
		XtNcomboTop, topLevel,
		XtNcomboData, combo_fonts,
		XtNcomboNData, ncombo_fonts,
		XtNwidth, 160,
                (char *)0);
	TooltipAdd(tooltip, btnFont, translate("Change the font family"));

	formatframe = XtVaCreateManagedWidget("formatframe",
		frameWidgetClass, formatbox,
		(char *)0);
        btnSize = XtVaCreateManagedWidget("toolbar_command",
                comboWidgetClass, formatframe,
		XtNlabel, "Size",
		XtNcomboList, cb_size,
		XtNcomboText, cb_size,
		XtNcomboTop, topLevel,
		XtNcomboData, combo_sizes,
		XtNcomboNData, XtNumber(combo_sizes),
		XtNwidth, 40,
                (char *)0);
	TooltipAdd(tooltip, btnSize, translate("Change the font size"));

	formatframe = XtVaCreateManagedWidget("formatframe",
		frameWidgetClass, formatbox,
		(char *)0);
        btnStyle = XtVaCreateManagedWidget("toolbar_command",
                comboWidgetClass, formatframe,
		XtNlabel, "Style",
		XtNcomboList, cb_style,
		XtNcomboText, cb_style,
		XtNcomboTop, topLevel,
		XtNcomboData, combo_styles,
		XtNcomboNData, XtNumber(combo_styles),
		XtNwidth, 100,
                (char *)0);
	TooltipAdd(tooltip, btnStyle, translate("Change the display style"));

	formatframe = XtVaCreateManagedWidget("formatframe",
		frameWidgetClass, formatbox,
		(char *)0);
	combo_colors = color_list(&ncombo_colors);
        btnColor = XtVaCreateManagedWidget("toolbar_command",
                comboWidgetClass, formatframe,
		XtNlabel, "Color",
		XtNcomboList, cb_color,
		XtNcomboText, cb_color,
		XtNcomboTop, topLevel,
		XtNcomboData, combo_colors,
		XtNcomboNData, ncombo_colors,
		XtNwidth, 80,
                (char *)0);
	TooltipAdd(tooltip, btnColor, translate("Change the color"));
}

/* ---
*/
void drop_handler(Widget w, XtPointer data,
                XEvent *event, Boolean *b)
{
        unsigned char *Data, *filename;
        unsigned long DataSize;
        int DataType = DndDataType(event);

        if (DataType == DndNotDnd) return;
        DndGetData(&Data, &DataSize);
        switch (DataType) {
        case DndFile:
                filename = Data;
                if (!fork()) {
                        execlp("pw", "Pw", filename, (char *)0);
                        exit(1);
                }
        default:
                ;
        }
}

static void place_shortcuts(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        XButtonEvent *bev = (XButtonEvent *)event;
        int col, row;
        int x, y;
	int s;
	sheet *st;

        activate_window(find_window_by_widget(w));
	s = w_list->sht;
	st = w_list->buf->sht;
        x = event->xbutton.x;
        y = event->xbutton.y;
        get_coords_cell(w_list, st[s].top.row, st[s].top.col, 
                        &row, &col, x, y);
        XtVaSetValues(shortcuts,
                XtNx, bev->x_root,
                XtNy, bev->y_root,
                (char *)0);
}

static void menu_motion(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        int fromright;
        Dimension width;
        XMotionEvent *mev = (XMotionEvent *)event;
        Widget menu, submenu;

        menu = XawSimpleMenuGetActiveEntry(w);
        if (menu == None) return;
        submenu = find_submenu_by_entry(menu);
        if (submenu == None) return;
        XtVaGetValues(w, XtNwidth, &width, (char *)0);
        fromright = width-mev->x;
        if (fromright > 0 && fromright < 20) {
                Position x, y, x_root, y_root;
                XtVaGetValues(menu,
                        XtNx, &x,
                        XtNy, &y,
                        (char *)0);
                XtTranslateCoords(w,
                        x+width-30, y, &x_root, &y_root);
                XtVaSetValues(submenu,
                        XtNx, x_root,
                        XtNy, y_root,
                        (char *)0);
                XtPopup(submenu, XtGrabNonexclusive);
        }
}

static void popup_shortcuts(Widget w, XEvent *event,
		String *p, Cardinal *n)
{
	XButtonEvent *bev = (XButtonEvent *)event;

	activate_window(find_window_by_widget(w));
	if (!XtIsRealized(shortcuts)) XtRealizeWidget(shortcuts);
	XtVaSetValues(shortcuts,
		XtNx, bev->x_root,
		XtNy, bev->y_root,
		(char *)0);
	XtPopupSpringLoaded(shortcuts);
}

static void popdown_submenu(Widget w, XEvent *event,
                String *p, Cardinal *n)
{
        XtPopdown(w);
        XtPopdown(XtParent(w));
}


static XtActionsRec actions[] =
{
        {"grid-button", GridButtonAction},
        {"self-insert-char", SelfInsertCharAction},
        {"siaghelp", siaghelp_action},
        {"place-shortcuts", place_shortcuts},
	{"menu-motion", menu_motion},
	{"popup-shortcuts", popup_shortcuts},
	{"popdown-submenu", popdown_submenu}
};

static void init_windows1(int *argc, char **argv)
{
        unsigned long highlight_color;
        unsigned long unhighlight_color;

        XtSetLanguageProc(NULL, (XtLanguageProc) NULL, NULL);

        topLevel = XtVaAppInitialize(
                    &app_context,       /* application context */
                    "Pw",               /* application class */
                    options,            /* command line options list */
		    XtNumber(options),
                    argc, argv,        /* command line args */
                    fallback_resources, /* for missing app-defaults file */
                    (char *)0);         /* terminate varargs list */

	t1_init(topLevel);

	XtGetApplicationResources(topLevel, &app_data, resources,
			XtNumber(resources), NULL, 0);
	plugin = app_data.plugin;

        XtAppAddActions(app_context, actions, XtNumber(actions));

	shortcuts = XtVaCreatePopupShell("shortcuts",
                simpleMenuWidgetClass, topLevel, (char *)0);
	XtRegisterGrabAction(popup_shortcuts,
		True, ButtonPressMask | ButtonReleaseMask,
		GrabModeAsync, GrabModeAsync);
        XtVaCreateManagedWidget("-",
                smeLineObjectClass, shortcuts, (char *)0);

        topbox = XtCreateManagedWidget("topbox",
                rudegridWidgetClass, topLevel, NULL, 0);

        highlight_color = WhitePixel(XtDisplay(topLevel),
                                DefaultScreen(XtDisplay(topLevel)));
        XtVaGetValues(topbox,   /* or any old widget with a grey background */
                XtNbackground, &unhighlight_color,
                (char *)0);
        tooltip_init(topLevel, highlight_color, unhighlight_color);
	tooltip = XtVaCreatePopupShell("tooltip",
		tooltipWidgetClass, topLevel,
		(char *)0);

        init_menu();
        init_toolbar();
        setup_buttons();
        init_toggle();

	textbox = XtVaCreateManagedWidget("textbox",
		rudegridWidgetClass, topbox,
		(char *)0);
        label1 = XtVaCreateManagedWidget("label1",
                labelWidgetClass, textbox,
                (char *)0);

        gridpane = XtVaCreateManagedWidget("gridpane",
                panedWidgetClass, topbox,
                XtNallowResize, True,
                (char *)0);

	statusbox = XtVaCreateManagedWidget("statusbox",
		rudegridWidgetClass, topbox,
		(char *)0);
        label2 = XtVaCreateManagedWidget("label2",
                labelWidgetClass, statusbox,
		(char *)0);
	label3 = XtVaCreateManagedWidget("label3",
		labelWidgetClass, statusbox,
		(char *)0);
}

/* ---
this does probably not belong here in window.c,
   because it doesn't depend on X
*/

static void save_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
        else {
                if (savematrix(p, w_list->buf, NULL)) {
                        printf("501 Can't save %s\n", p);
                } else {
			w_list->buf->change = FALSE;
                        printf("250 Saved %s\n", p);
                }
        }
}

static void load_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 File name missing\n");
        else {
                if (loadmatrix(p, w_list->buf, NULL)) {
                        printf("501 Can't load %s\n", p);
                } else {
			w_list->buf->change = FALSE;
                        printf("250 Loaded %s\n", p);
                }
        }
}

static void exec_plugin(char *p)
{
        if (*p++ != ' ' || *p == '\0') printf("501 Command missing\n");
        else {
                execute(p);
                printf("250 OK\n");
        }
}

static void help_plugin(char *p)
{
        printf("214 SAVE LOAD EXEC HELP NOOP QUIT PRNT\n");
}

static void noop_plugin(char *p)
{
        printf("250 OK\n");
}

static void win_plugin(char *p)
{
	Window win = XtWindow(topLevel);
	printf("250 %lx\n", (unsigned long)win);
}

static void quit_plugin(char *p)
{
        printf("221 Over and out\n");
        execute("(quit-program)");
}

static void prnt_plugin(char *p)
{
        printf("502 Can't print yet\n");
}

static struct {
        char *verb;
        void (*cb)(char *);
} plugin_cmds[] = {
        {"SAVE", save_plugin},
        {"LOAD", load_plugin},
        {"EXEC", exec_plugin},
        {"HELP", help_plugin},
        {"NOOP", noop_plugin},
	{"WIN", win_plugin},
        {"QUIT", quit_plugin},
        {"PRNT", prnt_plugin},
        {NULL, NULL}
};

static void read_plugin_cmd(XtPointer client_data, int *fid, XtInputId *id)
{
        char b[1024], *p;
        int i, n;

        if ((n = read(*fid, b, 1020)) == -1) return;

        b[n] = '\0';
        if ((p = strchr(b, '\n')) == NULL) {
                printf("501 Incomplete command\n");
                return;
        }

        *p = '\0';
        for (i = 0; plugin_cmds[i].verb; i++) {
                if (!strncmp(b, plugin_cmds[i].verb,
                                strlen(plugin_cmds[i].verb)))
                        break;
        }
        if (plugin_cmds[i].verb)
                (*plugin_cmds[i].cb)(b+strlen(plugin_cmds[i].verb));
        else
                printf("500 What are you talking about\n");
        fflush(stdout);
}

/* ---
*/
void mainloop(void)
{
	if (app_data.plugin) {
		XtAppAddInput(XtWidgetToApplicationContext(topLevel),
                        fileno(stdin), (XtPointer)XtInputReadMask,
                        read_plugin_cmd, NULL);
		printf("220 %s\n", version);
		fflush(stdout);
        }

        XtAppMainLoop(app_context);

        exit(0);
}

/* ---
handle spontaneous exit of plugins
*/

static void handle_plugin_exit(int ph)
{
	int s;
        buffer *b = b_list;

	for (s = 0; s < b->nsht; s++) {
	        do {
	                int n = buffer_plugin2index(b, s, ph);
	                if (n != -1) {
	                        cfree(b->sht[s].plugin[n].name);
	                        b->sht[s].nplugin--;
	                        for (; n < b->sht[s].nplugin; n++)
	                                b->sht[s].plugin[n] = b->sht[s].plugin[n+1];
	                        b->change = pr_scr_flag = TRUE;
	                }
	                b = b->next;
	        } while (b != b_list);
	}
}

static sfmt fmt0 = {"Helvetica", 100, 0, 0, 0, "black", "white", 0, 0, 0, 0};
static sfmt fmt1 = {"Helvetica", 120, 0, 0, 0, "black", "white", 0, 0, 0, 0};

static void handle_plugin_cmd(char *p)
{
	execute(p);
}

/* ---
   void init_windows(buffer *b)
   Sets up the whole initial window structure and initializes scrupd.
   The window list w_list is set to a list with a single window with the
   buffer b.
*/

void init_windows(buffer *b, int *argc, char **argv)
{
	unsigned long foreground;	/* pixel value */
	unsigned long background;
	Window cell_win;
	XColor screen_color, exact_color;

	start_splash();

	init_windows1(argc, argv);
	interp_startup();

	init_format(XtDisplay(topLevel));
        ok2print = 1;

	do_bp_styles();

	encode_format(~0, &fmt0);
	encode_format(~0, &fmt1);

	XtRealizeWidget(topLevel);

	plugin_init(topLevel, handle_plugin_exit, handle_plugin_cmd);

	display = XtDisplay(topLevel);

	grid_cursor = XCreateFontCursor(display, XC_xterm);

	activate_window(new_window(b, NULL));

	wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);
	XtOverrideTranslations(topLevel,
		XtParseTranslationTable(
			"<Message>WM_PROTOCOLS: execute(quit-program)"));

	XSetWMProtocols(display, XtWindow(topLevel), &wm_delete_window, 1);

	root = DefaultRootWindow(display);

	XAllocNamedColor(display,
			DefaultColormap(display, DefaultScreen(display)),
			"grey", &screen_color, &exact_color);
	blockbg = screen_color.pixel;
	noblockbg = WhitePixelOfScreen(XtScreen(topLevel));

	foreground = BlackPixelOfScreen(XtScreen(topLevel));

	cell_win = XtWindow(w_list->ui->grid);

	XtVaGetValues(w_list->ui->grid,
		XtNbackground, &background, (char *)0);

	clear_gc = get_gc(display, cell_win, background, foreground, -1);

	cell_gc = get_gc(display, cell_win, foreground, blockbg,
			 -1/*get_font(0)*/);

	cursor_gc = get_gc(display, cell_win, foreground ^ background, 0, -1);
	XSetFunction(display, cursor_gc, GXxor);
	XSetLineAttributes(display, cursor_gc, 1, LineSolid, CapButt, JoinMiter);

	draw_scrollbars(display, w_list);
	draw_status(display, "");

	/* Set up selection */
	target_atom = XInternAtom(display, "PW_BLOCK", False);

	/* stuff originally in realmain */

        embed_init(topLevel);

        init_subr_1("add-menu", add_menu);
        init_subr_3("add-menu-entry", add_menu_entry);
        init_subr_2("add-submenu", add_submenu);
        init_subr_4("add-submenu-entry", add_submenu_entry);
        init_subr_1("tooltip-mode", ltooltip_mode);

        init_form(topLevel, app_context);

        icon_set(topLevel, pw_xpm);
        DndInitialize(topLevel);
        DndRegisterOtherDrop(drop_handler);

	stop_splash();
}

/* ---
   void exit_windows()
   Cleans up after Calc before exit.  All buffers and windows are freed.
*/

void exit_windows(void)
{
	/* free all buffers */
	while (b_list != NULL)
		free_buffer(b_list);
	while (w_list != NULL)
		free_window(w_list);
	deletia_mark(0);
	deletia_reap();
}

static void replace_tabs(window *w)
{
	int n, top;
	buffer *b;
	XtVaGetValues(w->ui->tab,
		XtNtabbingCount, &n,
		XtNtabbingTop, &top,
		(char *)0);
	while (n) TabbingRemove(w->ui->tab, --n);
	for (n = 0; n < w->buf->nsht; n++) {
		TabbingInsert(w->ui->tab, w->buf->sht[n].name, n);
	}
	if (top >= w->buf->nsht)
		top = w->buf->nsht-1;

	b = b_list;
	do {
		if (b != w->buf) {
			char p[1000];
			sprintf(p, "%s:", b->name);
			TabbingInsert(w->ui->tab, p, -1);
		}
		b = b->next;
	} while (b != b_list);

	XtVaSetValues(w->ui->tab,
		XtNtabbingSelected, w->sht,
		XtNtabbingTop, top,
		(char *)0);
}


/* ---
   static void pr_scr()
   Prints and refreshes all the windows.
   Sets pr_scr_flag to FALSE.
*/

static void pr_scr(void)
{
	int i;
	window *w;
	buffer *b;
	int s;
	draw_status(display, "");
	w = w_list;
	s = w->sht;
	do {
		/* plugin size fixer-upper */
		for (i = 0; i < w->buf->sht[s].nplugin; i++) {
			int wi, he;
			int r = w->buf->sht[s].plugin[i].row;
			alloc_line(w->buf, s, r);
			plugin_size_get(w->buf->sht[s].plugin[i].ph, &wi, &he);
			w->buf->sht[s].text[r].height = he;
		}
		draw_scrollbars(display, w);
		draw_buffer(display, w);
		XClearWindow(XtDisplay(w->ui->ruler), XtWindow(w->ui->ruler));
		expose_ruler(w->ui->ruler, w);
                /* check for nondisplayed plugins */
		b = w->buf;
		s = w->sht;
                for (i = 0; i < b->sht[s].nplugin; i++) {
                        if (!b->sht[s].plugin[i].displayed) {
                                plugin_show(b->sht[s].plugin[i].ph,
                                                w->ui->grid);
                                b->sht[s].plugin[i].displayed = 1;
                        }
                }
		replace_tabs(w);
		w = w->next;
	} while (w != w_list);
	pr_scr_flag = pr_line_flag = FALSE;
	deletia_reap();
}	/* pr_scr */

/* ---
*/
void get_char_coords(window *w, int top_row, int top_col,
			int cell_row, int cell_col,
			int *cell_x, int *cell_y)
{
	int r;
	int s = w->sht;
	*cell_y = 0;
	for (r = top_row; r < cell_row; r++)
		*cell_y += cell_height(w->buf, s, r);

	*cell_x = line_start(w->buf, s, cell_row)
			+line_width(w->buf, s, cell_row, cell_col);
}

/* ---
*/
void get_coords_cell(window *w, int top_row, int top_col,
			int *cur_row, int *cur_col,
			int cur_x, int cur_y)
{
	int r, c, h;
	int s = w->sht;
	r = top_row;
	h = line_height(w->buf, s, r);
	while (cur_y > h) {
		r++;
		h += line_height(w->buf, s, r);
	}
	*cur_row = r;

	c = 0;
	cur_x -= line_start(w->buf, s, r);
	cur_x += top_col;	/* it's in pixels, y'know */

	while (c < line_length(w->buf, s, r) &&
			cur_x > line_width(w->buf, s, r, c)) {
		c++;
	}
	*cur_col = c;
}

/* ---
*/
void show_format(void)
{
	static int last_fmt = -1;
	int f = w_list->current_fmt;
	sfmt fmt;
	int s = w_list->sht;
	sheet *st = w_list->buf->sht;
	int sty = ret_style(w_list->buf, s, st[s].point_pos.row);
	static int last_hadj = -1;
	static int last_vadj = -1;
	static int last_sty = -1;
	int hadj, vadj;
	char b[100];

	decode_format(f, ~0, &fmt);
	hadj = ret_hadj(w_list->buf, s, st[s].point_pos.row);
	vadj = fmt.vadj;

	/* thou shalt not waste time on things you have already done */
	if (f == last_fmt && hadj == last_hadj && vadj == last_vadj
	    && sty == last_sty && !pr_scr_flag && !pr_line_flag) return;
	last_fmt = f;
	last_hadj = hadj;
	last_vadj = vadj;
	last_sty = sty;

	/* menus */
	combo_text_change(btnFont, translate(fmt.family));
	sprintf(b, "%d", fmt.size/10);
	combo_text_change(btnSize, b);
	combo_text_change(btnStyle, translate(style2name(sty)));
	combo_text_change(btnColor, translate(fmt.fg));

	/* toggle buttons */
	state_set(cmdBold, (fmt.bold?1:0), 1, 0);
	state_set(cmdItalic, (fmt.italic?1:0), 1, 0);
	state_set(cmdUline, (fmt.uline?1:0), 1, 0);
	state_set(cmdHLeft, ((hadj == HADJ_LEFT)?1:0), 1, 0);
	state_set(cmdHCenter, ((hadj == HADJ_CENTER)?1:0), 1, 0);
	state_set(cmdHRight, ((hadj == HADJ_RIGHT)?1:0), 1, 0);
	state_set(cmdVTop, ((vadj == VADJ_TOP)?1:0), 1, 0);
	state_set(cmdVBottom, ((vadj == VADJ_BOTTOM)?1:0), 1, 0);
}

int cursor_visible = FALSE;

/* ---
   void show_cur(window *w)
   Moves the cursor to reflect the position of point in w.
   If point is not visible, the window is moved so that point is in
   the middle of the screen.
*/

void show_cur(window *w)
{
	int s = w->sht;
	sheet *st = w->buf->sht;
	int top_row = st[s].top.row, top_col = st[s].top.col;
	char *p = "";

	XtVaSetValues(w->ui->grid,
		XtNrichtextPointRow, st[s].point_pos.row,
		XtNrichtextPointCol, st[s].point_pos.col,
		XtNrichtextVisibleCursor, True,
		(char *)0);
	if (pr_scr_flag) {
		pr_scr();
	} else {
		/* this may have moved top */
		XtVaGetValues(w->ui->grid,
			XtNrichtextTopRow, &(st[s].top.row),
			XtNrichtextTopCol, &(st[s].top.col),
			(char *)0);
		if (top_row != st[s].top.row) {
			draw_vbar(display, w);
		}
		if (top_col != st[s].top.col) {
			draw_hbar(display, w);
		}
		if (pr_line_flag) {
			rt_pr_line(w->ui->grid, st[s].point_pos.row);
			pr_line_flag = FALSE;
		}
	}
	show_format();
	draw_input(display, p);
	cursor_visible = TRUE;
}	/* show_cur */

/* ---
*/
void hide_cur(window *w)
{
	;
}

