/*
    Copyright (C) 1996-1998  Ulric Eriksson <ulric@edu.stockholm.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston,
    MA 02111-1307, USA.
*/

/* ---
richchar.c

A "rich char" is a normal 8-bit character plus format information
   such as font, colour, size. This module contains functions to handle
   them much like the standard string library does.
--- */

#include <stdio.h>
#include "../common/cmalloc.h"
#include "richchar.h"
#include "traceme.h"

/* ---
calculates the number of chars in p
*/

size_t rc_strlen(rich_char *p)
{
	size_t i = 0;
	if (p == NULL) return i;
	while (p[i].c) i++;
	return i;
}

/* ---
copies from to to
*/

rich_char *rc_strcpy(rich_char *to, rich_char *from)
{
	size_t i = 0;
	if (!to) return NULL;
	if (!from) to[0].c = '\0';
	else for (i = 0; from[i].c; i++)
		to[i] = from[i];
	to[i].c = '\0';
	return to;
}

/* ---
appends from after to
*/

rich_char *rc_strcat(rich_char *to, rich_char *from)
{
	return rc_strcpy(to+rc_strlen(to), from);
}

/* ---
allocate a new copy of p
*/

rich_char *rc_strdup(rich_char *p)
{
	rich_char *q = (rich_char *)cmalloc((rc_strlen(p)+1)
						*sizeof(rich_char));
	return rc_strcpy(q, p);
}

/* ---
allocate a new copy of p with q inserted at where
*/

rich_char *rc_strins(rich_char *p, rich_char *q, size_t where)
{
	size_t i = 0;

	rich_char *s = (rich_char *)cmalloc((rc_strlen(p)+rc_strlen(q)+1)
						*sizeof(rich_char));
	for (i = 0; p && p[i].c && i < where; i++)
		s[i] = p[i];
	rc_strcpy(s+i, q);
	if (p) rc_strcat(s+i, p+i);
	return s;
}

/* ---
allocate a copy of "plain" string into rich char format
*/

rich_char *rc_makerich(unsigned char *p, int fmt)
{
	size_t i;
	rich_char *s = (rich_char *)cmalloc(((p?strlen((char *)p):0)+1)
					*sizeof(rich_char));
	for (i = 0; p && p[i]; i++) {
		s[i].c = p[i];
		s[i].fmt = fmt;
	}
	s[i].c = '\0';
	return s;
}

/* ---
allocate a plain copy of a rich string
*/

unsigned char *rc_makeplain(rich_char *p)
{
	size_t i;

	unsigned char *s = (unsigned char *)cmalloc((rc_strlen(p)+1)*
						sizeof(unsigned char));
	for (i = 0; p && p[i].c; i++) {
		s[i] = p[i].c;
	}
	s[i] = '\0';
	return s;
}

/* ---
For each stretch of text sharing the same format, call the callback.
*/

void rc_forparts(rich_char *p, void (*callback)(rich_char *, size_t))
{
	size_t i = 0;

	while (p[i].c) {
		size_t j = i+1;

		while (p[j].c && p[i].fmt == p[j].fmt) j++;
		callback(p+i, j-i);
		i = j;
	}
}

